/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendraranking.endpoints.internal;

import java.net.URI;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.kendraranking.endpoints.KendraRankingEndpointParams;
import software.amazon.awssdk.services.kendraranking.endpoints.KendraRankingEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultKendraRankingEndpointProvider implements KendraRankingEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(KendraRankingEndpointParams params) {
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        try {
            Region region = params.region();
            String regionId = region == null ? null : region.id();
            RuleResult result = endpointRule0(params, regionId);
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(KendraRankingEndpointParams params, String region) {
        RuleResult result = endpointRule1(params);
        if (result.isResolved()) {
            return result;
        }
        result = endpointRule4(params, region);
        if (result.isResolved()) {
            return result;
        }
        return RuleResult.error("Invalid Configuration: Missing Region");
    }

    private static RuleResult endpointRule1(KendraRankingEndpointParams params) {
        if (params.endpoint() != null) {
            if (params.useFips()) {
                return RuleResult.error("Invalid Configuration: FIPS and custom endpoint are not supported");
            }
            return RuleResult.endpoint(Endpoint.builder().url(URI.create(params.endpoint())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule4(KendraRankingEndpointParams params, String region) {
        if (region != null) {
            return endpointRule5(params, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule5(KendraRankingEndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            RuleResult result = endpointRule6(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule12(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://kendra-ranking." + region + "." + partitionResult.dnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule6(KendraRankingEndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsDualStack()) {
            RuleResult result = endpointRule7(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://kendra-ranking." + region + "." + partitionResult.dualStackDnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule7(KendraRankingEndpointParams params, RulePartition partitionResult, String region) {
        if (params.useFips()) {
            RuleResult result = endpointRule8(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("FIPS is enabled but this partition does not support FIPS");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule8(KendraRankingEndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsFIPS()) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://kendra-ranking-fips." + region + "." + partitionResult.dualStackDnsSuffix()))
                    .build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule12(KendraRankingEndpointParams params, RulePartition partitionResult, String region) {
        if (params.useFips()) {
            RuleResult result = endpointRule13(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("FIPS is enabled but this partition does not support FIPS");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule13(KendraRankingEndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsFIPS()) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://kendra-ranking-fips." + region + "." + partitionResult.dnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }
}
