/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspaces;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.keyspaces.model.AccessDeniedException;
import software.amazon.awssdk.services.keyspaces.model.ConflictException;
import software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.CreateTableRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateTableResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteTableRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteTableResponse;
import software.amazon.awssdk.services.keyspaces.model.GetKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.GetKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.GetTableRequest;
import software.amazon.awssdk.services.keyspaces.model.GetTableResponse;
import software.amazon.awssdk.services.keyspaces.model.InternalServerException;
import software.amazon.awssdk.services.keyspaces.model.KeyspacesException;
import software.amazon.awssdk.services.keyspaces.model.KeyspacesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTablesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTablesResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.ResourceNotFoundException;
import software.amazon.awssdk.services.keyspaces.model.RestoreTableRequest;
import software.amazon.awssdk.services.keyspaces.model.RestoreTableResponse;
import software.amazon.awssdk.services.keyspaces.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.keyspaces.model.TagResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.TagResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.UntagResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.UntagResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.UpdateTableRequest;
import software.amazon.awssdk.services.keyspaces.model.UpdateTableResponse;
import software.amazon.awssdk.services.keyspaces.model.ValidationException;
import software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesIterable;
import software.amazon.awssdk.services.keyspaces.paginators.ListTablesIterable;
import software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.keyspaces.transform.CreateKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.DeleteKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.GetKeyspaceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.GetTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListKeyspacesRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.RestoreTableRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.keyspaces.transform.UpdateTableRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link KeyspacesClient}.
 *
 * @see KeyspacesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKeyspacesClient implements KeyspacesClient {
    private static final Logger log = Logger.loggerFor(DefaultKeyspacesClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final KeyspacesServiceClientConfiguration serviceClientConfiguration;

    protected DefaultKeyspacesClient(KeyspacesServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * The <code>CreateKeyspace</code> operation adds a new keyspace to your account. In an Amazon Web Services account,
     * keyspace names must be unique within each Region.
     * </p>
     * <p>
     * <code>CreateKeyspace</code> is an asynchronous operation. You can monitor the creation status of the new keyspace
     * by using the <code>GetKeyspace</code> operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/working-with-keyspaces.html#keyspaces-create"
     * >Creating keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createKeyspaceRequest
     * @return Result of the CreateKeyspace operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.CreateKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateKeyspaceResponse createKeyspace(CreateKeyspaceRequest createKeyspaceRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKeyspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKeyspace");

            return clientHandler.execute(new ClientExecutionParams<CreateKeyspaceRequest, CreateKeyspaceResponse>()
                    .withOperationName("CreateKeyspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createKeyspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKeyspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateTable</code> operation adds a new table to the specified keyspace. Within a keyspace, table names
     * must be unique.
     * </p>
     * <p>
     * <code>CreateTable</code> is an asynchronous operation. When the request is received, the status of the table is
     * set to <code>CREATING</code>. You can monitor the creation status of the new table by using the
     * <code>GetTable</code> operation, which returns the current <code>status</code> of the table. You can start using
     * a table when the status is <code>ACTIVE</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/working-with-tables.html#tables-create">Creating
     * tables</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createTableRequest
     * @return Result of the CreateTable operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTableResponse createTable(CreateTableRequest createTableRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");

            return clientHandler.execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                    .withOperationName("CreateTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DeleteKeyspace</code> operation deletes a keyspace and all of its tables.
     * </p>
     *
     * @param deleteKeyspaceRequest
     * @return Result of the DeleteKeyspace operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.DeleteKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteKeyspaceResponse deleteKeyspace(DeleteKeyspaceRequest deleteKeyspaceRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteKeyspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKeyspace");

            return clientHandler.execute(new ClientExecutionParams<DeleteKeyspaceRequest, DeleteKeyspaceResponse>()
                    .withOperationName("DeleteKeyspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteKeyspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKeyspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DeleteTable</code> operation deletes a table and all of its data. After a <code>DeleteTable</code>
     * request is received, the specified table is in the <code>DELETING</code> state until Amazon Keyspaces completes
     * the deletion. If the table is in the <code>ACTIVE</code> state, you can delete it. If a table is either in the
     * <code>CREATING</code> or <code>UPDATING</code> states, then Amazon Keyspaces returns a
     * <code>ResourceInUseException</code>. If the specified table does not exist, Amazon Keyspaces returns a
     * <code>ResourceNotFoundException</code>. If the table is already in the <code>DELETING</code> state, no error is
     * returned.
     * </p>
     *
     * @param deleteTableRequest
     * @return Result of the DeleteTable operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTableResponse deleteTable(DeleteTableRequest deleteTableRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");

            return clientHandler.execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                    .withOperationName("DeleteTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the name and the Amazon Resource Name (ARN) of the specified table.
     * </p>
     *
     * @param getKeyspaceRequest
     * @return Result of the GetKeyspace operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.GetKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKeyspaceResponse getKeyspace(GetKeyspaceRequest getKeyspaceRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKeyspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKeyspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKeyspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKeyspace");

            return clientHandler.execute(new ClientExecutionParams<GetKeyspaceRequest, GetKeyspaceResponse>()
                    .withOperationName("GetKeyspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getKeyspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKeyspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the table, including the table's name and current status, the keyspace name,
     * configuration settings, and metadata.
     * </p>
     * <p>
     * To read table metadata using <code>GetTable</code>, <code>Select</code> action permissions for the table and
     * system tables are required to complete the operation.
     * </p>
     *
     * @param getTableRequest
     * @return Result of the GetTable operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.GetTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTableResponse getTable(GetTableRequest getTableRequest) throws ValidationException, ServiceQuotaExceededException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTable");

            return clientHandler.execute(new ClientExecutionParams<GetTableRequest, GetTableResponse>()
                    .withOperationName("GetTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTableRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of keyspaces.
     * </p>
     *
     * @param listKeyspacesRequest
     * @return Result of the ListKeyspaces operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKeyspacesResponse listKeyspaces(ListKeyspacesRequest listKeyspacesRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKeyspacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKeyspacesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKeyspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKeyspaces");

            return clientHandler.execute(new ClientExecutionParams<ListKeyspacesRequest, ListKeyspacesResponse>()
                    .withOperationName("ListKeyspaces").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listKeyspacesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKeyspacesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of keyspaces.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listKeyspaces(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesIterable responses = client.listKeyspacesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesIterable responses = client.listKeyspacesPaginator(request);
     *     for (software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesIterable responses = client.listKeyspacesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listKeyspaces(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest)} operation.</b>
     * </p>
     *
     * @param listKeyspacesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKeyspacesIterable listKeyspacesPaginator(ListKeyspacesRequest listKeyspacesRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KeyspacesException {
        return new ListKeyspacesIterable(this, applyPaginatorUserAgent(listKeyspacesRequest));
    }

    /**
     * <p>
     * Returns a list of tables for a specified keyspace.
     * </p>
     *
     * @param listTablesRequest
     * @return Result of the ListTables operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTablesResponse listTables(ListTablesRequest listTablesRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTablesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tables for a specified keyspace.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.keyspaces.model.ListTablesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspaces.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     *     for (software.amazon.awssdk.services.keyspaces.model.ListTablesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.keyspaces.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTablesIterable listTablesPaginator(ListTablesRequest listTablesRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KeyspacesException {
        return new ListTablesIterable(this, applyPaginatorUserAgent(listTablesRequest));
    }

    /**
     * <p>
     * Returns a list of all tags associated with the specified Amazon Keyspaces resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all tags associated with the specified Amazon Keyspaces resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Restores the specified table to the specified point in time within the <code>earliest_restorable_timestamp</code>
     * and the current time. For more information about restore points, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_window"
     * > Time window for PITR continuous backups</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Any number of users can execute up to 4 concurrent restores (any type of restore) in a given account.
     * </p>
     * <p>
     * When you restore using point in time recovery, Amazon Keyspaces restores your source table's schema and data to
     * the state based on the selected timestamp <code>(day:hour:minute:second)</code> to a new table. The Time to Live
     * (TTL) settings are also restored to the state based on the selected timestamp.
     * </p>
     * <p>
     * In addition to the table's schema, data, and TTL settings, <code>RestoreTable</code> restores the capacity mode,
     * encryption, and point-in-time recovery settings from the source table. Unlike the table's schema data and TTL
     * settings, which are restored based on the selected timestamp, these settings are always restored based on the
     * table's settings as of the current time or when the table was deleted.
     * </p>
     * <p>
     * You can also overwrite these settings during restore:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Read/write capacity mode
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned throughput capacity settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Point-in-time (PITR) settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_settings"
     * >PITR restore settings</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Note that the following settings are not restored, and you must configure them manually for the new table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Automatic scaling policies (for tables that use provisioned capacity mode)
     * </p>
     * </li>
     * <li>
     * <p>
     * Identity and Access Management (IAM) policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableRequest
     * @return Result of the RestoreTable operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.RestoreTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/RestoreTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RestoreTableResponse restoreTable(RestoreTableRequest restoreTableRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RestoreTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreTable");

            return clientHandler.execute(new ClientExecutionParams<RestoreTableRequest, RestoreTableResponse>()
                    .withOperationName("RestoreTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(restoreTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RestoreTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a set of tags with a Amazon Keyspaces resource. You can then activate these user-defined tags so that
     * they appear on the Cost Management Console for cost allocation tracking. For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/tagging-keyspaces.html">Adding tags and labels to
     * Amazon Keyspaces resources</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * For IAM policy examples that show how to control access to Amazon Keyspaces resources based on tags, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/security_iam_id-based-policy-examples-tags">Amazon
     * Keyspaces resource access based on tags</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Amazon Keyspaces resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds new columns to the table or updates one of the table's settings, for example capacity mode, encryption,
     * point-in-time recovery, or ttl settings. Note that you can only update one specific table setting per update
     * operation.
     * </p>
     *
     * @param updateTableRequest
     * @return Result of the UpdateTable operation returned by the service.
     * @throws ValidationException
     *         The operation failed due to an invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The operation exceeded the service quota for this resource. For more information on service quotas, see
     *         <a href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.
     * @throws InternalServerException
     *         Amazon Keyspaces was unable to fully process this request because of an internal server error.
     * @throws ConflictException
     *         Amazon Keyspaces could not complete the requested action. This error may occur if you try to perform an
     *         action and the same or a different action is already in progress, or if you try to create a resource that
     *         already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a keyspace or table that doesn't exist. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UpdateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTableResponse updateTable(UpdateTableRequest updateTableRequest) throws ValidationException,
            ServiceQuotaExceededException, InternalServerException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KeyspacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Keyspaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTable");

            return clientHandler.execute(new ClientExecutionParams<UpdateTableRequest, UpdateTableResponse>()
                    .withOperationName("UpdateTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends KeyspacesRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KeyspacesException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final KeyspacesServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
