/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspaces;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.CreateTableRequest;
import software.amazon.awssdk.services.keyspaces.model.CreateTableResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.DeleteTableRequest;
import software.amazon.awssdk.services.keyspaces.model.DeleteTableResponse;
import software.amazon.awssdk.services.keyspaces.model.GetKeyspaceRequest;
import software.amazon.awssdk.services.keyspaces.model.GetKeyspaceResponse;
import software.amazon.awssdk.services.keyspaces.model.GetTableAutoScalingSettingsRequest;
import software.amazon.awssdk.services.keyspaces.model.GetTableAutoScalingSettingsResponse;
import software.amazon.awssdk.services.keyspaces.model.GetTableRequest;
import software.amazon.awssdk.services.keyspaces.model.GetTableResponse;
import software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTablesRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTablesResponse;
import software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.RestoreTableRequest;
import software.amazon.awssdk.services.keyspaces.model.RestoreTableResponse;
import software.amazon.awssdk.services.keyspaces.model.TagResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.TagResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.UntagResourceRequest;
import software.amazon.awssdk.services.keyspaces.model.UntagResourceResponse;
import software.amazon.awssdk.services.keyspaces.model.UpdateTableRequest;
import software.amazon.awssdk.services.keyspaces.model.UpdateTableResponse;
import software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesPublisher;
import software.amazon.awssdk.services.keyspaces.paginators.ListTablesPublisher;
import software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourcePublisher;

/**
 * Service client for accessing Amazon Keyspaces asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Amazon Keyspaces (for Apache Cassandra) is a scalable, highly available, and managed Apache Cassandra-compatible
 * database service. Amazon Keyspaces makes it easy to migrate, run, and scale Cassandra workloads in the Amazon Web
 * Services Cloud. With just a few clicks on the Amazon Web Services Management Console or a few lines of code, you can
 * create keyspaces and tables in Amazon Keyspaces, without deploying any infrastructure or installing software.
 * </p>
 * <p>
 * In addition to supporting Cassandra Query Language (CQL) requests via open-source Cassandra drivers, Amazon Keyspaces
 * supports data definition language (DDL) operations to manage keyspaces and tables using the Amazon Web Services SDK
 * and CLI, as well as infrastructure as code (IaC) services and tools such as CloudFormation and Terraform. This API
 * reference describes the supported DDL operations in detail.
 * </p>
 * <p>
 * For the list of all supported CQL APIs, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/cassandra-apis.html">Supported Cassandra APIs,
 * operations, and data types in Amazon Keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * To learn how Amazon Keyspaces API actions are recorded with CloudTrail, see <a href=
 * "https://docs.aws.amazon.com/keyspaces/latest/devguide/logging-using-cloudtrail.html#service-name-info-in-cloudtrail"
 * >Amazon Keyspaces information in CloudTrail</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * For more information about Amazon Web Services APIs, for example how to implement retry logic or how to sign Amazon
 * Web Services API requests, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-apis.html">Amazon Web
 * Services APIs</a> in the <i>General Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface KeyspacesAsyncClient extends AwsClient {
    String SERVICE_NAME = "cassandra";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "cassandra";

    /**
     * <p>
     * The <code>CreateKeyspace</code> operation adds a new keyspace to your account. In an Amazon Web Services account,
     * keyspace names must be unique within each Region.
     * </p>
     * <p>
     * <code>CreateKeyspace</code> is an asynchronous operation. You can monitor the creation status of the new keyspace
     * by using the <code>GetKeyspace</code> operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/working-with-keyspaces.html#keyspaces-create"
     * >Creating keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createKeyspaceRequest
     * @return A Java Future containing the result of the CreateKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateKeyspaceResponse> createKeyspace(CreateKeyspaceRequest createKeyspaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The <code>CreateKeyspace</code> operation adds a new keyspace to your account. In an Amazon Web Services account,
     * keyspace names must be unique within each Region.
     * </p>
     * <p>
     * <code>CreateKeyspace</code> is an asynchronous operation. You can monitor the creation status of the new keyspace
     * by using the <code>GetKeyspace</code> operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/working-with-keyspaces.html#keyspaces-create"
     * >Creating keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateKeyspaceRequest.Builder} avoiding the need to
     * create one manually via {@link CreateKeyspaceRequest#builder()}
     * </p>
     *
     * @param createKeyspaceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.CreateKeyspaceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateKeyspaceResponse> createKeyspace(Consumer<CreateKeyspaceRequest.Builder> createKeyspaceRequest) {
        return createKeyspace(CreateKeyspaceRequest.builder().applyMutation(createKeyspaceRequest).build());
    }

    /**
     * <p>
     * The <code>CreateTable</code> operation adds a new table to the specified keyspace. Within a keyspace, table names
     * must be unique.
     * </p>
     * <p>
     * <code>CreateTable</code> is an asynchronous operation. When the request is received, the status of the table is
     * set to <code>CREATING</code>. You can monitor the creation status of the new table by using the
     * <code>GetTable</code> operation, which returns the current <code>status</code> of the table. You can start using
     * a table when the status is <code>ACTIVE</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/working-with-tables.html#tables-create">Creating
     * tables</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param createTableRequest
     * @return A Java Future containing the result of the CreateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateTableResponse> createTable(CreateTableRequest createTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The <code>CreateTable</code> operation adds a new table to the specified keyspace. Within a keyspace, table names
     * must be unique.
     * </p>
     * <p>
     * <code>CreateTable</code> is an asynchronous operation. When the request is received, the status of the table is
     * set to <code>CREATING</code>. You can monitor the creation status of the new table by using the
     * <code>GetTable</code> operation, which returns the current <code>status</code> of the table. You can start using
     * a table when the status is <code>ACTIVE</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/working-with-tables.html#tables-create">Creating
     * tables</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateTableRequest.Builder} avoiding the need to
     * create one manually via {@link CreateTableRequest#builder()}
     * </p>
     *
     * @param createTableRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.CreateTableRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateTableResponse> createTable(Consumer<CreateTableRequest.Builder> createTableRequest) {
        return createTable(CreateTableRequest.builder().applyMutation(createTableRequest).build());
    }

    /**
     * <p>
     * The <code>DeleteKeyspace</code> operation deletes a keyspace and all of its tables.
     * </p>
     *
     * @param deleteKeyspaceRequest
     * @return A Java Future containing the result of the DeleteKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteKeyspaceResponse> deleteKeyspace(DeleteKeyspaceRequest deleteKeyspaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The <code>DeleteKeyspace</code> operation deletes a keyspace and all of its tables.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteKeyspaceRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteKeyspaceRequest#builder()}
     * </p>
     *
     * @param deleteKeyspaceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.DeleteKeyspaceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteKeyspaceResponse> deleteKeyspace(Consumer<DeleteKeyspaceRequest.Builder> deleteKeyspaceRequest) {
        return deleteKeyspace(DeleteKeyspaceRequest.builder().applyMutation(deleteKeyspaceRequest).build());
    }

    /**
     * <p>
     * The <code>DeleteTable</code> operation deletes a table and all of its data. After a <code>DeleteTable</code>
     * request is received, the specified table is in the <code>DELETING</code> state until Amazon Keyspaces completes
     * the deletion. If the table is in the <code>ACTIVE</code> state, you can delete it. If a table is either in the
     * <code>CREATING</code> or <code>UPDATING</code> states, then Amazon Keyspaces returns a
     * <code>ResourceInUseException</code>. If the specified table does not exist, Amazon Keyspaces returns a
     * <code>ResourceNotFoundException</code>. If the table is already in the <code>DELETING</code> state, no error is
     * returned.
     * </p>
     *
     * @param deleteTableRequest
     * @return A Java Future containing the result of the DeleteTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteTableResponse> deleteTable(DeleteTableRequest deleteTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The <code>DeleteTable</code> operation deletes a table and all of its data. After a <code>DeleteTable</code>
     * request is received, the specified table is in the <code>DELETING</code> state until Amazon Keyspaces completes
     * the deletion. If the table is in the <code>ACTIVE</code> state, you can delete it. If a table is either in the
     * <code>CREATING</code> or <code>UPDATING</code> states, then Amazon Keyspaces returns a
     * <code>ResourceInUseException</code>. If the specified table does not exist, Amazon Keyspaces returns a
     * <code>ResourceNotFoundException</code>. If the table is already in the <code>DELETING</code> state, no error is
     * returned.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteTableRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteTableRequest#builder()}
     * </p>
     *
     * @param deleteTableRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.DeleteTableRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteTableResponse> deleteTable(Consumer<DeleteTableRequest.Builder> deleteTableRequest) {
        return deleteTable(DeleteTableRequest.builder().applyMutation(deleteTableRequest).build());
    }

    /**
     * <p>
     * Returns the name and the Amazon Resource Name (ARN) of the specified table.
     * </p>
     *
     * @param getKeyspaceRequest
     * @return A Java Future containing the result of the GetKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetKeyspaceResponse> getKeyspace(GetKeyspaceRequest getKeyspaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the name and the Amazon Resource Name (ARN) of the specified table.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetKeyspaceRequest.Builder} avoiding the need to
     * create one manually via {@link GetKeyspaceRequest#builder()}
     * </p>
     *
     * @param getKeyspaceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.GetKeyspaceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetKeyspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetKeyspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetKeyspace" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetKeyspaceResponse> getKeyspace(Consumer<GetKeyspaceRequest.Builder> getKeyspaceRequest) {
        return getKeyspace(GetKeyspaceRequest.builder().applyMutation(getKeyspaceRequest).build());
    }

    /**
     * <p>
     * Returns information about the table, including the table's name and current status, the keyspace name,
     * configuration settings, and metadata.
     * </p>
     * <p>
     * To read table metadata using <code>GetTable</code>, <code>Select</code> action permissions for the table and
     * system tables are required to complete the operation.
     * </p>
     *
     * @param getTableRequest
     * @return A Java Future containing the result of the GetTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetTableResponse> getTable(GetTableRequest getTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the table, including the table's name and current status, the keyspace name,
     * configuration settings, and metadata.
     * </p>
     * <p>
     * To read table metadata using <code>GetTable</code>, <code>Select</code> action permissions for the table and
     * system tables are required to complete the operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetTableRequest.Builder} avoiding the need to
     * create one manually via {@link GetTableRequest#builder()}
     * </p>
     *
     * @param getTableRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.GetTableRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetTableResponse> getTable(Consumer<GetTableRequest.Builder> getTableRequest) {
        return getTable(GetTableRequest.builder().applyMutation(getTableRequest).build());
    }

    /**
     * <p>
     * Returns auto scaling related settings of the specified table in JSON format. If the table is a multi-Region
     * table, the Amazon Web Services Region specific auto scaling settings of the table are included.
     * </p>
     * <p>
     * Amazon Keyspaces auto scaling helps you provision throughput capacity for variable workloads efficiently by
     * increasing and decreasing your table's read and write capacity automatically in response to application traffic.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/autoscaling.html">Managing throughput capacity
     * automatically with Amazon Keyspaces auto scaling</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <code>GetTableAutoScalingSettings</code> can't be used as an action in an IAM policy.
     * </p>
     * </important>
     * <p>
     * To define permissions for <code>GetTableAutoScalingSettings</code>, you must allow the following two actions in
     * the IAM policy statement's <code>Action</code> element:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalableTargets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalingPolicies</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getTableAutoScalingSettingsRequest
     * @return A Java Future containing the result of the GetTableAutoScalingSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetTableAutoScalingSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTableAutoScalingSettings"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetTableAutoScalingSettingsResponse> getTableAutoScalingSettings(
            GetTableAutoScalingSettingsRequest getTableAutoScalingSettingsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns auto scaling related settings of the specified table in JSON format. If the table is a multi-Region
     * table, the Amazon Web Services Region specific auto scaling settings of the table are included.
     * </p>
     * <p>
     * Amazon Keyspaces auto scaling helps you provision throughput capacity for variable workloads efficiently by
     * increasing and decreasing your table's read and write capacity automatically in response to application traffic.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/autoscaling.html">Managing throughput capacity
     * automatically with Amazon Keyspaces auto scaling</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <code>GetTableAutoScalingSettings</code> can't be used as an action in an IAM policy.
     * </p>
     * </important>
     * <p>
     * To define permissions for <code>GetTableAutoScalingSettings</code>, you must allow the following two actions in
     * the IAM policy statement's <code>Action</code> element:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalableTargets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application-autoscaling:DescribeScalingPolicies</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetTableAutoScalingSettingsRequest.Builder}
     * avoiding the need to create one manually via {@link GetTableAutoScalingSettingsRequest#builder()}
     * </p>
     *
     * @param getTableAutoScalingSettingsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.GetTableAutoScalingSettingsRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the GetTableAutoScalingSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.GetTableAutoScalingSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/GetTableAutoScalingSettings"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetTableAutoScalingSettingsResponse> getTableAutoScalingSettings(
            Consumer<GetTableAutoScalingSettingsRequest.Builder> getTableAutoScalingSettingsRequest) {
        return getTableAutoScalingSettings(GetTableAutoScalingSettingsRequest.builder()
                .applyMutation(getTableAutoScalingSettingsRequest).build());
    }

    /**
     * <p>
     * Returns a list of keyspaces.
     * </p>
     *
     * @param listKeyspacesRequest
     * @return A Java Future containing the result of the ListKeyspaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListKeyspacesResponse> listKeyspaces(ListKeyspacesRequest listKeyspacesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of keyspaces.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListKeyspacesRequest.Builder} avoiding the need to
     * create one manually via {@link ListKeyspacesRequest#builder()}
     * </p>
     *
     * @param listKeyspacesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListKeyspaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListKeyspacesResponse> listKeyspaces(Consumer<ListKeyspacesRequest.Builder> listKeyspacesRequest) {
        return listKeyspaces(ListKeyspacesRequest.builder().applyMutation(listKeyspacesRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listKeyspaces(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesPublisher publisher = client.listKeyspacesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesPublisher publisher = client.listKeyspacesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listKeyspaces(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest)} operation.</b>
     * </p>
     *
     * @param listKeyspacesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    default ListKeyspacesPublisher listKeyspacesPaginator(ListKeyspacesRequest listKeyspacesRequest) {
        return new ListKeyspacesPublisher(this, listKeyspacesRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listKeyspaces(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesPublisher publisher = client.listKeyspacesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListKeyspacesPublisher publisher = client.listKeyspacesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listKeyspaces(software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListKeyspacesRequest.Builder} avoiding the need to
     * create one manually via {@link ListKeyspacesRequest#builder()}
     * </p>
     *
     * @param listKeyspacesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.ListKeyspacesRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListKeyspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListKeyspaces" target="_top">AWS API
     *      Documentation</a>
     */
    default ListKeyspacesPublisher listKeyspacesPaginator(Consumer<ListKeyspacesRequest.Builder> listKeyspacesRequest) {
        return listKeyspacesPaginator(ListKeyspacesRequest.builder().applyMutation(listKeyspacesRequest).build());
    }

    /**
     * <p>
     * Returns a list of tables for a specified keyspace.
     * </p>
     *
     * @param listTablesRequest
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of tables for a specified keyspace.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTablesRequest.Builder} avoiding the need to
     * create one manually via {@link ListTablesRequest#builder()}
     * </p>
     *
     * @param listTablesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.ListTablesRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTablesResponse> listTables(Consumer<ListTablesRequest.Builder> listTablesRequest) {
        return listTables(ListTablesRequest.builder().applyMutation(listTablesRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.keyspaces.model.ListTablesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspaces.model.ListTablesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspaces.model.ListTablesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.keyspaces.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTablesPublisher listTablesPaginator(ListTablesRequest listTablesRequest) {
        return new ListTablesPublisher(this, listTablesRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.keyspaces.model.ListTablesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspaces.model.ListTablesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspaces.model.ListTablesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.keyspaces.model.ListTablesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTablesRequest.Builder} avoiding the need to
     * create one manually via {@link ListTablesRequest#builder()}
     * </p>
     *
     * @param listTablesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.ListTablesRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTablesPublisher listTablesPaginator(Consumer<ListTablesRequest.Builder> listTablesRequest) {
        return listTablesPaginator(ListTablesRequest.builder().applyMutation(listTablesRequest).build());
    }

    /**
     * <p>
     * Returns a list of all tags associated with the specified Amazon Keyspaces resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all tags associated with the specified Amazon Keyspaces resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, listTagsForResourceRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.keyspaces.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourcePublisher listTagsForResourcePaginator(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResourcePaginator(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest)
                .build());
    }

    /**
     * <p>
     * Restores the table to the specified point in time within the <code>earliest_restorable_timestamp</code> and the
     * current time. For more information about restore points, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_window"
     * > Time window for PITR continuous backups</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Any number of users can execute up to 4 concurrent restores (any type of restore) in a given account.
     * </p>
     * <p>
     * When you restore using point in time recovery, Amazon Keyspaces restores your source table's schema and data to
     * the state based on the selected timestamp <code>(day:hour:minute:second)</code> to a new table. The Time to Live
     * (TTL) settings are also restored to the state based on the selected timestamp.
     * </p>
     * <p>
     * In addition to the table's schema, data, and TTL settings, <code>RestoreTable</code> restores the capacity mode,
     * auto scaling settings, encryption settings, and point-in-time recovery settings from the source table. Unlike the
     * table's schema data and TTL settings, which are restored based on the selected timestamp, these settings are
     * always restored based on the table's settings as of the current time or when the table was deleted.
     * </p>
     * <p>
     * You can also overwrite these settings during restore:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Read/write capacity mode
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned throughput capacity units
     * </p>
     * </li>
     * <li>
     * <p>
     * Auto scaling settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Point-in-time (PITR) settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_settings"
     * >PITR restore settings</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Note that the following settings are not restored, and you must configure them manually for the new table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Identity and Access Management (IAM) policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableRequest
     * @return A Java Future containing the result of the RestoreTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.RestoreTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/RestoreTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<RestoreTableResponse> restoreTable(RestoreTableRequest restoreTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Restores the table to the specified point in time within the <code>earliest_restorable_timestamp</code> and the
     * current time. For more information about restore points, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_window"
     * > Time window for PITR continuous backups</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Any number of users can execute up to 4 concurrent restores (any type of restore) in a given account.
     * </p>
     * <p>
     * When you restore using point in time recovery, Amazon Keyspaces restores your source table's schema and data to
     * the state based on the selected timestamp <code>(day:hour:minute:second)</code> to a new table. The Time to Live
     * (TTL) settings are also restored to the state based on the selected timestamp.
     * </p>
     * <p>
     * In addition to the table's schema, data, and TTL settings, <code>RestoreTable</code> restores the capacity mode,
     * auto scaling settings, encryption settings, and point-in-time recovery settings from the source table. Unlike the
     * table's schema data and TTL settings, which are restored based on the selected timestamp, these settings are
     * always restored based on the table's settings as of the current time or when the table was deleted.
     * </p>
     * <p>
     * You can also overwrite these settings during restore:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Read/write capacity mode
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned throughput capacity units
     * </p>
     * </li>
     * <li>
     * <p>
     * Auto scaling settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Point-in-time (PITR) settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/PointInTimeRecovery_HowItWorks.html#howitworks_backup_settings"
     * >PITR restore settings</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * Note that the following settings are not restored, and you must configure them manually for the new table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Identity and Access Management (IAM) policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RestoreTableRequest.Builder} avoiding the need to
     * create one manually via {@link RestoreTableRequest#builder()}
     * </p>
     *
     * @param restoreTableRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.RestoreTableRequest.Builder} to create a request.
     * @return A Java Future containing the result of the RestoreTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.RestoreTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/RestoreTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<RestoreTableResponse> restoreTable(Consumer<RestoreTableRequest.Builder> restoreTableRequest) {
        return restoreTable(RestoreTableRequest.builder().applyMutation(restoreTableRequest).build());
    }

    /**
     * <p>
     * Associates a set of tags with a Amazon Keyspaces resource. You can then activate these user-defined tags so that
     * they appear on the Cost Management Console for cost allocation tracking. For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/tagging-keyspaces.html">Adding tags and labels to
     * Amazon Keyspaces resources</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * For IAM policy examples that show how to control access to Amazon Keyspaces resources based on tags, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-tags"
     * >Amazon Keyspaces resource access based on tags</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates a set of tags with a Amazon Keyspaces resource. You can then activate these user-defined tags so that
     * they appear on the Cost Management Console for cost allocation tracking. For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/tagging-keyspaces.html">Adding tags and labels to
     * Amazon Keyspaces resources</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * For IAM policy examples that show how to control access to Amazon Keyspaces resources based on tags, see <a href=
     * "https://docs.aws.amazon.com/keyspaces/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-tags"
     * >Amazon Keyspaces resource access based on tags</a> in the <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes the association of tags from a Amazon Keyspaces resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the association of tags from a Amazon Keyspaces resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Adds new columns to the table or updates one of the table's settings, for example capacity mode, auto scaling,
     * encryption, point-in-time recovery, or ttl settings. Note that you can only update one specific table setting per
     * update operation.
     * </p>
     *
     * @param updateTableRequest
     * @return A Java Future containing the result of the UpdateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UpdateTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateTableResponse> updateTable(UpdateTableRequest updateTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds new columns to the table or updates one of the table's settings, for example capacity mode, auto scaling,
     * encryption, point-in-time recovery, or ttl settings. Note that you can only update one specific table setting per
     * update operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateTableRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateTableRequest#builder()}
     * </p>
     *
     * @param updateTableRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.keyspaces.model.UpdateTableRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The operation failed due to an invalid or malformed request.</li>
     *         <li>ServiceQuotaExceededException The operation exceeded the service quota for this resource. For more
     *         information on service quotas, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/quotas.html">Quotas</a> in the <i>Amazon
     *         Keyspaces Developer Guide</i>.</li>
     *         <li>InternalServerException Amazon Keyspaces was unable to fully process this request because of an
     *         internal server error.</li>
     *         <li>ConflictException Amazon Keyspaces couldn't complete the requested action. This error may occur if
     *         you try to perform an action and the same or a different action is already in progress, or if you try to
     *         create a resource that already exists.</li>
     *         <li>AccessDeniedException You don't have sufficient access permissions to perform this action.</li>
     *         <li>ResourceNotFoundException The operation tried to access a keyspace or table that doesn't exist. The
     *         resource might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KeyspacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KeyspacesAsyncClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspaces-2022-02-10/UpdateTable" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateTableResponse> updateTable(Consumer<UpdateTableRequest.Builder> updateTableRequest) {
        return updateTable(UpdateTableRequest.builder().applyMutation(updateTableRequest).build());
    }

    @Override
    default KeyspacesServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link KeyspacesAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static KeyspacesAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link KeyspacesAsyncClient}.
     */
    static KeyspacesAsyncClientBuilder builder() {
        return new DefaultKeyspacesAsyncClientBuilder();
    }
}
