/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The settings for the CDC stream of a table. For more information about CDC streams, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/cdc.html">Working with change data capture (CDC) streams
 * in Amazon Keyspaces</a> in the <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CdcSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<CdcSpecification.Builder, CdcSpecification> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(CdcSpecification::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> VIEW_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("viewType").getter(getter(CdcSpecification::viewTypeAsString)).setter(setter(Builder::viewType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("viewType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(CdcSpecification::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROPAGATE_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("propagateTags").getter(getter(CdcSpecification::propagateTagsAsString))
            .setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propagateTags").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD, VIEW_TYPE_FIELD,
            TAGS_FIELD, PROPAGATE_TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final String viewType;

    private final List<Tag> tags;

    private final String propagateTags;

    private CdcSpecification(BuilderImpl builder) {
        this.status = builder.status;
        this.viewType = builder.viewType;
        this.tags = builder.tags;
        this.propagateTags = builder.propagateTags;
    }

    /**
     * <p>
     * The status of the CDC stream. You can enable or disable a stream for a table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CdcStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the CDC stream. You can enable or disable a stream for a table.
     * @see CdcStatus
     */
    public final CdcStatus status() {
        return CdcStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the CDC stream. You can enable or disable a stream for a table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CdcStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the CDC stream. You can enable or disable a stream for a table.
     * @see CdcStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After you create
     * the stream, you can't make changes to this selection.
     * </p>
     * <p>
     * The options are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the default.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_IMAGE</code> - the version of the row after the change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OLD_IMAGE</code> - the version of the row before the change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #viewType} will
     * return {@link ViewType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #viewTypeAsString}.
     * </p>
     * 
     * @return The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After
     *         you create the stream, you can't make changes to this selection. </p>
     *         <p>
     *         The options are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the
     *         default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_IMAGE</code> - the version of the row after the change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OLD_IMAGE</code> - the version of the row before the change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
     *         </p>
     *         </li>
     * @see ViewType
     */
    public final ViewType viewType() {
        return ViewType.fromValue(viewType);
    }

    /**
     * <p>
     * The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After you create
     * the stream, you can't make changes to this selection.
     * </p>
     * <p>
     * The options are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the default.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_IMAGE</code> - the version of the row after the change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OLD_IMAGE</code> - the version of the row before the change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #viewType} will
     * return {@link ViewType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #viewTypeAsString}.
     * </p>
     * 
     * @return The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After
     *         you create the stream, you can't make changes to this selection. </p>
     *         <p>
     *         The options are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the
     *         default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_IMAGE</code> - the version of the row after the change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OLD_IMAGE</code> - the version of the row before the change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
     *         </p>
     *         </li>
     * @see ViewType
     */
    public final String viewTypeAsString() {
        return viewType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags (key-value pairs) that you want to apply to the stream.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags (key-value pairs) that you want to apply to the stream.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies that the stream inherits the tags from the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link CdcPropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies that the stream inherits the tags from the table.
     * @see CdcPropagateTags
     */
    public final CdcPropagateTags propagateTags() {
        return CdcPropagateTags.fromValue(propagateTags);
    }

    /**
     * <p>
     * Specifies that the stream inherits the tags from the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link CdcPropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies that the stream inherits the tags from the table.
     * @see CdcPropagateTags
     */
    public final String propagateTagsAsString() {
        return propagateTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(viewTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(propagateTagsAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CdcSpecification)) {
            return false;
        }
        CdcSpecification other = (CdcSpecification) obj;
        return Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(viewTypeAsString(), other.viewTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(propagateTagsAsString(), other.propagateTagsAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CdcSpecification").add("Status", statusAsString()).add("ViewType", viewTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("PropagateTags", propagateTagsAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "viewType":
            return Optional.ofNullable(clazz.cast(viewTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "propagateTags":
            return Optional.ofNullable(clazz.cast(propagateTagsAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("status", STATUS_FIELD);
        map.put("viewType", VIEW_TYPE_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("propagateTags", PROPAGATE_TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CdcSpecification, T> g) {
        return obj -> g.apply((CdcSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CdcSpecification> {
        /**
         * <p>
         * The status of the CDC stream. You can enable or disable a stream for a table.
         * </p>
         * 
         * @param status
         *        The status of the CDC stream. You can enable or disable a stream for a table.
         * @see CdcStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CdcStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the CDC stream. You can enable or disable a stream for a table.
         * </p>
         * 
         * @param status
         *        The status of the CDC stream. You can enable or disable a stream for a table.
         * @see CdcStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CdcStatus
         */
        Builder status(CdcStatus status);

        /**
         * <p>
         * The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After you
         * create the stream, you can't make changes to this selection.
         * </p>
         * <p>
         * The options are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the default.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_IMAGE</code> - the version of the row after the change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OLD_IMAGE</code> - the version of the row before the change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param viewType
         *        The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After
         *        you create the stream, you can't make changes to this selection. </p>
         *        <p>
         *        The options are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the
         *        default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_IMAGE</code> - the version of the row after the change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OLD_IMAGE</code> - the version of the row before the change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
         *        </p>
         *        </li>
         * @see ViewType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ViewType
         */
        Builder viewType(String viewType);

        /**
         * <p>
         * The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After you
         * create the stream, you can't make changes to this selection.
         * </p>
         * <p>
         * The options are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the default.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_IMAGE</code> - the version of the row after the change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OLD_IMAGE</code> - the version of the row before the change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param viewType
         *        The view type specifies the changes Amazon Keyspaces records for each changed row in the stream. After
         *        you create the stream, you can't make changes to this selection. </p>
         *        <p>
         *        The options are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NEW_AND_OLD_IMAGES</code> - both versions of the row, before and after the change. This is the
         *        default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_IMAGE</code> - the version of the row after the change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OLD_IMAGE</code> - the version of the row before the change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - the partition and clustering keys of the row that was changed.
         *        </p>
         *        </li>
         * @see ViewType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ViewType
         */
        Builder viewType(ViewType viewType);

        /**
         * <p>
         * The tags (key-value pairs) that you want to apply to the stream.
         * </p>
         * 
         * @param tags
         *        The tags (key-value pairs) that you want to apply to the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags (key-value pairs) that you want to apply to the stream.
         * </p>
         * 
         * @param tags
         *        The tags (key-value pairs) that you want to apply to the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags (key-value pairs) that you want to apply to the stream.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.keyspaces.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.keyspaces.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.keyspaces.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.keyspaces.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Specifies that the stream inherits the tags from the table.
         * </p>
         * 
         * @param propagateTags
         *        Specifies that the stream inherits the tags from the table.
         * @see CdcPropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CdcPropagateTags
         */
        Builder propagateTags(String propagateTags);

        /**
         * <p>
         * Specifies that the stream inherits the tags from the table.
         * </p>
         * 
         * @param propagateTags
         *        Specifies that the stream inherits the tags from the table.
         * @see CdcPropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CdcPropagateTags
         */
        Builder propagateTags(CdcPropagateTags propagateTags);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private String viewType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String propagateTags;

        private BuilderImpl() {
        }

        private BuilderImpl(CdcSpecification model) {
            status(model.status);
            viewType(model.viewType);
            tags(model.tags);
            propagateTags(model.propagateTags);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CdcStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getViewType() {
            return viewType;
        }

        public final void setViewType(String viewType) {
            this.viewType = viewType;
        }

        @Override
        public final Builder viewType(String viewType) {
            this.viewType = viewType;
            return this;
        }

        @Override
        public final Builder viewType(ViewType viewType) {
            this.viewType(viewType == null ? null : viewType.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPropagateTags() {
            return propagateTags;
        }

        public final void setPropagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public final Builder propagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        @Override
        public final Builder propagateTags(CdcPropagateTags propagateTags) {
            this.propagateTags(propagateTags == null ? null : propagateTags.toString());
            return this;
        }

        @Override
        public CdcSpecification build() {
            return new CdcSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
