/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Amazon Keyspaces encrypts and decrypts the table data at rest transparently and integrates with Key Management
 * Service for storing and managing the encryption key. You can choose one of the following KMS keys (KMS keys):
 * </p>
 * <ul>
 * <li>
 * <p>
 * Amazon Web Services owned key - This is the default encryption type. The key is owned by Amazon Keyspaces (no
 * additional charge).
 * </p>
 * </li>
 * <li>
 * <p>
 * Customer managed key - This key is stored in your account and is created, owned, and managed by you. You have full
 * control over the customer managed key (KMS charges apply).
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information about encryption at rest in Amazon Keyspaces, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a> in the
 * <i>Amazon Keyspaces Developer Guide</i>.
 * </p>
 * <p>
 * For more information about KMS, see <a
 * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">KMS management service
 * concepts</a> in the <i>Key Management Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EncryptionSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<EncryptionSpecification.Builder, EncryptionSpecification> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(EncryptionSpecification::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> KMS_KEY_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyIdentifier").getter(getter(EncryptionSpecification::kmsKeyIdentifier))
            .setter(setter(Builder::kmsKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyIdentifier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            KMS_KEY_IDENTIFIER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String kmsKeyIdentifier;

    private EncryptionSpecification(BuilderImpl builder) {
        this.type = builder.type;
        this.kmsKeyIdentifier = builder.kmsKeyIdentifier;
    }

    /**
     * <p>
     * The encryption option specified for the table. You can choose one of the following KMS keys (KMS keys):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned, and
     * managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon Resource Name
     * (ARN) format as input.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The default is <code>type:AWS_OWNED_KMS_KEY</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a> in the
     * <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The encryption option specified for the table. You can choose one of the following KMS keys (KMS
     *         keys):</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned,
     *         and managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon
     *         Resource Name (ARN) format as input.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The default is <code>type:AWS_OWNED_KMS_KEY</code>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a>
     *         in the <i>Amazon Keyspaces Developer Guide</i>.
     * @see EncryptionType
     */
    public final EncryptionType type() {
        return EncryptionType.fromValue(type);
    }

    /**
     * <p>
     * The encryption option specified for the table. You can choose one of the following KMS keys (KMS keys):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned, and
     * managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon Resource Name
     * (ARN) format as input.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The default is <code>type:AWS_OWNED_KMS_KEY</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a> in the
     * <i>Amazon Keyspaces Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The encryption option specified for the table. You can choose one of the following KMS keys (KMS
     *         keys):</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned,
     *         and managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon
     *         Resource Name (ARN) format as input.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The default is <code>type:AWS_OWNED_KMS_KEY</code>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a>
     *         in the <i>Amazon Keyspaces Developer Guide</i>.
     * @see EncryptionType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the customer managed KMS key, for example <code>kms_key_identifier:ARN</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the customer managed KMS key, for example
     *         <code>kms_key_identifier:ARN</code>.
     */
    public final String kmsKeyIdentifier() {
        return kmsKeyIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyIdentifier());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EncryptionSpecification)) {
            return false;
        }
        EncryptionSpecification other = (EncryptionSpecification) obj;
        return Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(kmsKeyIdentifier(), other.kmsKeyIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EncryptionSpecification").add("Type", typeAsString())
                .add("KmsKeyIdentifier", kmsKeyIdentifier()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "kmsKeyIdentifier":
            return Optional.ofNullable(clazz.cast(kmsKeyIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("kmsKeyIdentifier", KMS_KEY_IDENTIFIER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EncryptionSpecification, T> g) {
        return obj -> g.apply((EncryptionSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EncryptionSpecification> {
        /**
         * <p>
         * The encryption option specified for the table. You can choose one of the following KMS keys (KMS keys):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned, and
         * managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon Resource
         * Name (ARN) format as input.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The default is <code>type:AWS_OWNED_KMS_KEY</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a> in
         * the <i>Amazon Keyspaces Developer Guide</i>.
         * </p>
         * 
         * @param type
         *        The encryption option specified for the table. You can choose one of the following KMS keys (KMS
         *        keys):</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned,
         *        and managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon
         *        Resource Name (ARN) format as input.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The default is <code>type:AWS_OWNED_KMS_KEY</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at
         *        rest</a> in the <i>Amazon Keyspaces Developer Guide</i>.
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder type(String type);

        /**
         * <p>
         * The encryption option specified for the table. You can choose one of the following KMS keys (KMS keys):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned, and
         * managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon Resource
         * Name (ARN) format as input.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The default is <code>type:AWS_OWNED_KMS_KEY</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at rest</a> in
         * the <i>Amazon Keyspaces Developer Guide</i>.
         * </p>
         * 
         * @param type
         *        The encryption option specified for the table. You can choose one of the following KMS keys (KMS
         *        keys):</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>type:AWS_OWNED_KMS_KEY</code> - This key is owned by Amazon Keyspaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>type:CUSTOMER_MANAGED_KMS_KEY</code> - This key is stored in your account and is created, owned,
         *        and managed by you. This option requires the <code>kms_key_identifier</code> of the KMS key in Amazon
         *        Resource Name (ARN) format as input.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The default is <code>type:AWS_OWNED_KMS_KEY</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/keyspaces/latest/devguide/EncryptionAtRest.html">Encryption at
         *        rest</a> in the <i>Amazon Keyspaces Developer Guide</i>.
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder type(EncryptionType type);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the customer managed KMS key, for example
         * <code>kms_key_identifier:ARN</code>.
         * </p>
         * 
         * @param kmsKeyIdentifier
         *        The Amazon Resource Name (ARN) of the customer managed KMS key, for example
         *        <code>kms_key_identifier:ARN</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyIdentifier(String kmsKeyIdentifier);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String kmsKeyIdentifier;

        private BuilderImpl() {
        }

        private BuilderImpl(EncryptionSpecification model) {
            type(model.type);
            kmsKeyIdentifier(model.kmsKeyIdentifier);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(EncryptionType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getKmsKeyIdentifier() {
            return kmsKeyIdentifier;
        }

        public final void setKmsKeyIdentifier(String kmsKeyIdentifier) {
            this.kmsKeyIdentifier = kmsKeyIdentifier;
        }

        @Override
        public final Builder kmsKeyIdentifier(String kmsKeyIdentifier) {
            this.kmsKeyIdentifier = kmsKeyIdentifier;
            return this;
        }

        @Override
        public EncryptionSpecification build() {
            return new EncryptionSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
