/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspacesstreams;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.keyspacesstreams.internal.KeyspacesStreamsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.keyspacesstreams.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.keyspacesstreams.model.AccessDeniedException;
import software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetRecordsResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.GetStreamRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.GetStreamResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.InternalServerException;
import software.amazon.awssdk.services.keyspacesstreams.model.KeyspacesStreamsException;
import software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsRequest;
import software.amazon.awssdk.services.keyspacesstreams.model.ListStreamsResponse;
import software.amazon.awssdk.services.keyspacesstreams.model.ResourceNotFoundException;
import software.amazon.awssdk.services.keyspacesstreams.model.ThrottlingException;
import software.amazon.awssdk.services.keyspacesstreams.model.ValidationException;
import software.amazon.awssdk.services.keyspacesstreams.transform.GetRecordsRequestMarshaller;
import software.amazon.awssdk.services.keyspacesstreams.transform.GetShardIteratorRequestMarshaller;
import software.amazon.awssdk.services.keyspacesstreams.transform.GetStreamRequestMarshaller;
import software.amazon.awssdk.services.keyspacesstreams.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link KeyspacesStreamsClient}.
 *
 * @see KeyspacesStreamsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKeyspacesStreamsClient implements KeyspacesStreamsClient {
    private static final Logger log = Logger.loggerFor(DefaultKeyspacesStreamsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKeyspacesStreamsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "KeyspacesStreams" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves data records from a specified shard in an Amazon Keyspaces data stream. This operation returns a
     * collection of data records from the shard, including the primary key columns and information about modifications
     * made to the captured table data. Each record represents a single data modification in the Amazon Keyspaces table
     * and includes metadata about when the change occurred.
     * </p>
     *
     * @param getRecordsRequest
     * @return Result of the GetRecords operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRecordsResponse getRecords(GetRecordsRequest getRecordsRequest) throws ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            KeyspacesStreamsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRecordsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecordsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KeyspacesStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecords");

            return clientHandler.execute(new ClientExecutionParams<GetRecordsRequest, GetRecordsResponse>()
                    .withOperationName("GetRecords").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRecordsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a shard iterator that serves as a bookmark for reading data from a specific position in an Amazon
     * Keyspaces data stream's shard. The shard iterator specifies the shard position from which to start reading data
     * records sequentially. You can specify whether to begin reading at the latest record, the oldest record, or at a
     * particular sequence number within the shard.
     * </p>
     *
     * @param getShardIteratorRequest
     * @return Result of the GetShardIterator operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetShardIteratorResponse getShardIterator(GetShardIteratorRequest getShardIteratorRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetShardIteratorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetShardIteratorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getShardIteratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getShardIteratorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KeyspacesStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetShardIterator");

            return clientHandler.execute(new ClientExecutionParams<GetShardIteratorRequest, GetShardIteratorResponse>()
                    .withOperationName("GetShardIterator").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getShardIteratorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetShardIteratorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about a specific data capture stream for an Amazon Keyspaces table. The information
     * includes the stream's Amazon Resource Name (ARN), creation time, current status, retention period, shard
     * composition, and associated table details. This operation helps you monitor and manage the configuration of your
     * Amazon Keyspaces data streams.
     * </p>
     *
     * @param getStreamRequest
     * @return Result of the GetStream operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/GetStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetStreamResponse getStream(GetStreamRequest getStreamRequest) throws ThrottlingException, AccessDeniedException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            KeyspacesStreamsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStreamResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KeyspacesStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStream");

            return clientHandler.execute(new ClientExecutionParams<GetStreamRequest, GetStreamResponse>()
                    .withOperationName("GetStream").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all data capture streams associated with your Amazon Keyspaces account or for a specific
     * keyspace or table. The response includes information such as stream ARNs, table associations, creation
     * timestamps, and current status. This operation helps you discover and manage all active data streams in your
     * Amazon Keyspaces environment.
     * </p>
     *
     * @param listStreamsRequest
     * @return Result of the ListStreams operation returned by the service.
     * @throws ThrottlingException
     *         The request rate is too high and exceeds the service's throughput limits. </p>
     *         <p>
     *         This exception occurs when you send too many requests in a short period of time. Implement exponential
     *         backoff in your retry strategy to handle this exception. Reducing your request frequency or distributing
     *         requests more evenly can help avoid throughput exceptions.
     * @throws AccessDeniedException
     *         You don't have sufficient access permissions to perform this operation.
     *         </p>
     *         <p>
     *         This exception occurs when your IAM user or role lacks the required permissions to access the Amazon
     *         Keyspaces resource or perform the requested action. Check your IAM policies and ensure they grant the
     *         necessary permissions.
     * @throws InternalServerException
     *         The Amazon Keyspaces service encountered an unexpected error while processing the request.
     *         </p>
     *         <p>
     *         This internal server error is not related to your request parameters. Retry your request after a brief
     *         delay. If the issue persists, contact Amazon Web Services Support with details of your request to help
     *         identify and resolve the problem.
     * @throws ValidationException
     *         The request validation failed because one or more input parameters failed validation.
     *         </p>
     *         <p>
     *         This exception occurs when there are syntax errors in the request, field constraints are violated, or
     *         required parameters are missing. To help you fix the issue, the exception message provides details about
     *         which parameter failed and why.
     * @throws ResourceNotFoundException
     *         The requested resource doesn't exist or could not be found.
     *         </p>
     *         <p>
     *         This exception occurs when you attempt to access a keyspace, table, stream, or other Amazon Keyspaces
     *         resource that doesn't exist or that has been deleted. Verify that the resource identifier is correct and
     *         that the resource exists in your account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KeyspacesStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KeyspacesStreamsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/keyspacesstreams-2024-09-09/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, KeyspacesStreamsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStreamsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KeyspacesStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreams");

            return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                    .withOperationName("ListStreams").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStreamsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        KeyspacesStreamsServiceClientConfigurationBuilder serviceConfigBuilder = new KeyspacesStreamsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KeyspacesStreamsException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0");
    }

    @Override
    public final KeyspacesStreamsServiceClientConfiguration serviceClientConfiguration() {
        return new KeyspacesStreamsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
