/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspacesstreams.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a uniquely identified group of change records within a change data capture stream for Amazon Keyspaces.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Shard implements SdkPojo, Serializable, ToCopyableBuilder<Shard.Builder, Shard> {
    private static final SdkField<String> SHARD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("shardId").getter(getter(Shard::shardId)).setter(setter(Builder::shardId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("shardId").build()).build();

    private static final SdkField<SequenceNumberRange> SEQUENCE_NUMBER_RANGE_FIELD = SdkField
            .<SequenceNumberRange> builder(MarshallingType.SDK_POJO).memberName("sequenceNumberRange")
            .getter(getter(Shard::sequenceNumberRange)).setter(setter(Builder::sequenceNumberRange))
            .constructor(SequenceNumberRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sequenceNumberRange").build())
            .build();

    private static final SdkField<List<String>> PARENT_SHARD_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("parentShardIds")
            .getter(getter(Shard::parentShardIds))
            .setter(setter(Builder::parentShardIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parentShardIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SHARD_ID_FIELD,
            SEQUENCE_NUMBER_RANGE_FIELD, PARENT_SHARD_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String shardId;

    private final SequenceNumberRange sequenceNumberRange;

    private final List<String> parentShardIds;

    private Shard(BuilderImpl builder) {
        this.shardId = builder.shardId;
        this.sequenceNumberRange = builder.sequenceNumberRange;
        this.parentShardIds = builder.parentShardIds;
    }

    /**
     * <p>
     * A unique identifier for this shard within the stream.
     * </p>
     * 
     * @return A unique identifier for this shard within the stream.
     */
    public final String shardId() {
        return shardId;
    }

    /**
     * <p>
     * The range of sequence numbers contained within this shard.
     * </p>
     * 
     * @return The range of sequence numbers contained within this shard.
     */
    public final SequenceNumberRange sequenceNumberRange() {
        return sequenceNumberRange;
    }

    /**
     * For responses, this returns true if the service returned a value for the ParentShardIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasParentShardIds() {
        return parentShardIds != null && !(parentShardIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of parent shards that this shard evolved from, if this shard was created through resharding.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParentShardIds} method.
     * </p>
     * 
     * @return The identifiers of parent shards that this shard evolved from, if this shard was created through
     *         resharding.
     */
    public final List<String> parentShardIds() {
        return parentShardIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(shardId());
        hashCode = 31 * hashCode + Objects.hashCode(sequenceNumberRange());
        hashCode = 31 * hashCode + Objects.hashCode(hasParentShardIds() ? parentShardIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Shard)) {
            return false;
        }
        Shard other = (Shard) obj;
        return Objects.equals(shardId(), other.shardId()) && Objects.equals(sequenceNumberRange(), other.sequenceNumberRange())
                && hasParentShardIds() == other.hasParentShardIds() && Objects.equals(parentShardIds(), other.parentShardIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Shard").add("ShardId", shardId()).add("SequenceNumberRange", sequenceNumberRange())
                .add("ParentShardIds", hasParentShardIds() ? parentShardIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "shardId":
            return Optional.ofNullable(clazz.cast(shardId()));
        case "sequenceNumberRange":
            return Optional.ofNullable(clazz.cast(sequenceNumberRange()));
        case "parentShardIds":
            return Optional.ofNullable(clazz.cast(parentShardIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("shardId", SHARD_ID_FIELD);
        map.put("sequenceNumberRange", SEQUENCE_NUMBER_RANGE_FIELD);
        map.put("parentShardIds", PARENT_SHARD_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Shard, T> g) {
        return obj -> g.apply((Shard) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Shard> {
        /**
         * <p>
         * A unique identifier for this shard within the stream.
         * </p>
         * 
         * @param shardId
         *        A unique identifier for this shard within the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shardId(String shardId);

        /**
         * <p>
         * The range of sequence numbers contained within this shard.
         * </p>
         * 
         * @param sequenceNumberRange
         *        The range of sequence numbers contained within this shard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceNumberRange(SequenceNumberRange sequenceNumberRange);

        /**
         * <p>
         * The range of sequence numbers contained within this shard.
         * </p>
         * This is a convenience method that creates an instance of the {@link SequenceNumberRange.Builder} avoiding the
         * need to create one manually via {@link SequenceNumberRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SequenceNumberRange.Builder#build()} is called immediately and
         * its result is passed to {@link #sequenceNumberRange(SequenceNumberRange)}.
         * 
         * @param sequenceNumberRange
         *        a consumer that will call methods on {@link SequenceNumberRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sequenceNumberRange(SequenceNumberRange)
         */
        default Builder sequenceNumberRange(Consumer<SequenceNumberRange.Builder> sequenceNumberRange) {
            return sequenceNumberRange(SequenceNumberRange.builder().applyMutation(sequenceNumberRange).build());
        }

        /**
         * <p>
         * The identifiers of parent shards that this shard evolved from, if this shard was created through resharding.
         * </p>
         * 
         * @param parentShardIds
         *        The identifiers of parent shards that this shard evolved from, if this shard was created through
         *        resharding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentShardIds(Collection<String> parentShardIds);

        /**
         * <p>
         * The identifiers of parent shards that this shard evolved from, if this shard was created through resharding.
         * </p>
         * 
         * @param parentShardIds
         *        The identifiers of parent shards that this shard evolved from, if this shard was created through
         *        resharding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentShardIds(String... parentShardIds);
    }

    static final class BuilderImpl implements Builder {
        private String shardId;

        private SequenceNumberRange sequenceNumberRange;

        private List<String> parentShardIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Shard model) {
            shardId(model.shardId);
            sequenceNumberRange(model.sequenceNumberRange);
            parentShardIds(model.parentShardIds);
        }

        public final String getShardId() {
            return shardId;
        }

        public final void setShardId(String shardId) {
            this.shardId = shardId;
        }

        @Override
        public final Builder shardId(String shardId) {
            this.shardId = shardId;
            return this;
        }

        public final SequenceNumberRange.Builder getSequenceNumberRange() {
            return sequenceNumberRange != null ? sequenceNumberRange.toBuilder() : null;
        }

        public final void setSequenceNumberRange(SequenceNumberRange.BuilderImpl sequenceNumberRange) {
            this.sequenceNumberRange = sequenceNumberRange != null ? sequenceNumberRange.build() : null;
        }

        @Override
        public final Builder sequenceNumberRange(SequenceNumberRange sequenceNumberRange) {
            this.sequenceNumberRange = sequenceNumberRange;
            return this;
        }

        public final Collection<String> getParentShardIds() {
            if (parentShardIds instanceof SdkAutoConstructList) {
                return null;
            }
            return parentShardIds;
        }

        public final void setParentShardIds(Collection<String> parentShardIds) {
            this.parentShardIds = ShardIdListCopier.copy(parentShardIds);
        }

        @Override
        public final Builder parentShardIds(Collection<String> parentShardIds) {
            this.parentShardIds = ShardIdListCopier.copy(parentShardIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parentShardIds(String... parentShardIds) {
            parentShardIds(Arrays.asList(parentShardIds));
            return this;
        }

        @Override
        public Shard build() {
            return new Shard(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
