/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.nio.ByteBuffer;
import java.util.Date;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.runtime.StandardMemberCopier;
import software.amazon.awssdk.services.kinesis.transform.RecordMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The unit of data of the Amazon Kinesis stream, which is composed of a sequence number, a partition key, and a data
 * blob.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Record implements StructuredPojo, ToCopyableBuilder<Record.Builder, Record> {
    private final String sequenceNumber;

    private final Date approximateArrivalTimestamp;

    private final ByteBuffer data;

    private final String partitionKey;

    private Record(BuilderImpl builder) {
        this.sequenceNumber = builder.sequenceNumber;
        this.approximateArrivalTimestamp = builder.approximateArrivalTimestamp;
        this.data = builder.data;
        this.partitionKey = builder.partitionKey;
    }

    /**
     * <p>
     * The unique identifier of the record in the stream.
     * </p>
     * 
     * @return The unique identifier of the record in the stream.
     */
    public String sequenceNumber() {
        return sequenceNumber;
    }

    /**
     * <p>
     * The approximate time that the record was inserted into the stream.
     * </p>
     * 
     * @return The approximate time that the record was inserted into the stream.
     */
    public Date approximateArrivalTimestamp() {
        return approximateArrivalTimestamp;
    }

    /**
     * <p>
     * The data blob. The data in the blob is both opaque and immutable to the Amazon Kinesis service, which does not
     * inspect, interpret, or change the data in the blob in any way. When the data blob (the payload before
     * base64-encoding) is added to the partition key size, the total size must not exceed the maximum record size (1
     * MB).
     * </p>
     * <p>
     * {@code ByteBuffer}s are stateful. Calling their {@code get} methods changes their {@code position}. We recommend
     * using {@link java.nio.ByteBuffer#asReadOnlyBuffer()} to create a read-only view of the buffer with an independent
     * {@code position}, and calling {@code get} methods on this rather than directly on the returned {@code ByteBuffer}
     * . Doing so will ensure that anyone else using the {@code ByteBuffer} will not be affected by changes to the
     * {@code position}.
     * </p>
     * 
     * @return The data blob. The data in the blob is both opaque and immutable to the Amazon Kinesis service, which
     *         does not inspect, interpret, or change the data in the blob in any way. When the data blob (the payload
     *         before base64-encoding) is added to the partition key size, the total size must not exceed the maximum
     *         record size (1 MB).
     */
    public ByteBuffer data() {
        return data;
    }

    /**
     * <p>
     * Identifies which shard in the stream the data record is assigned to.
     * </p>
     * 
     * @return Identifies which shard in the stream the data record is assigned to.
     */
    public String partitionKey() {
        return partitionKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((sequenceNumber() == null) ? 0 : sequenceNumber().hashCode());
        hashCode = 31 * hashCode + ((approximateArrivalTimestamp() == null) ? 0 : approximateArrivalTimestamp().hashCode());
        hashCode = 31 * hashCode + ((data() == null) ? 0 : data().hashCode());
        hashCode = 31 * hashCode + ((partitionKey() == null) ? 0 : partitionKey().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Record)) {
            return false;
        }
        Record other = (Record) obj;
        if (other.sequenceNumber() == null ^ this.sequenceNumber() == null) {
            return false;
        }
        if (other.sequenceNumber() != null && !other.sequenceNumber().equals(this.sequenceNumber())) {
            return false;
        }
        if (other.approximateArrivalTimestamp() == null ^ this.approximateArrivalTimestamp() == null) {
            return false;
        }
        if (other.approximateArrivalTimestamp() != null
                && !other.approximateArrivalTimestamp().equals(this.approximateArrivalTimestamp())) {
            return false;
        }
        if (other.data() == null ^ this.data() == null) {
            return false;
        }
        if (other.data() != null && !other.data().equals(this.data())) {
            return false;
        }
        if (other.partitionKey() == null ^ this.partitionKey() == null) {
            return false;
        }
        if (other.partitionKey() != null && !other.partitionKey().equals(this.partitionKey())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (sequenceNumber() != null) {
            sb.append("SequenceNumber: ").append(sequenceNumber()).append(",");
        }
        if (approximateArrivalTimestamp() != null) {
            sb.append("ApproximateArrivalTimestamp: ").append(approximateArrivalTimestamp()).append(",");
        }
        if (data() != null) {
            sb.append("Data: ").append(data()).append(",");
        }
        if (partitionKey() != null) {
            sb.append("PartitionKey: ").append(partitionKey()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        RecordMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Record> {
        /**
         * <p>
         * The unique identifier of the record in the stream.
         * </p>
         * 
         * @param sequenceNumber
         *        The unique identifier of the record in the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceNumber(String sequenceNumber);

        /**
         * <p>
         * The approximate time that the record was inserted into the stream.
         * </p>
         * 
         * @param approximateArrivalTimestamp
         *        The approximate time that the record was inserted into the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approximateArrivalTimestamp(Date approximateArrivalTimestamp);

        /**
         * <p>
         * The data blob. The data in the blob is both opaque and immutable to the Amazon Kinesis service, which does
         * not inspect, interpret, or change the data in the blob in any way. When the data blob (the payload before
         * base64-encoding) is added to the partition key size, the total size must not exceed the maximum record size
         * (1 MB).
         * </p>
         * 
         * @param data
         *        The data blob. The data in the blob is both opaque and immutable to the Amazon Kinesis service, which
         *        does not inspect, interpret, or change the data in the blob in any way. When the data blob (the
         *        payload before base64-encoding) is added to the partition key size, the total size must not exceed the
         *        maximum record size (1 MB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(ByteBuffer data);

        /**
         * <p>
         * Identifies which shard in the stream the data record is assigned to.
         * </p>
         * 
         * @param partitionKey
         *        Identifies which shard in the stream the data record is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKey(String partitionKey);
    }

    private static final class BuilderImpl implements Builder {
        private String sequenceNumber;

        private Date approximateArrivalTimestamp;

        private ByteBuffer data;

        private String partitionKey;

        private BuilderImpl() {
        }

        private BuilderImpl(Record model) {
            setSequenceNumber(model.sequenceNumber);
            setApproximateArrivalTimestamp(model.approximateArrivalTimestamp);
            setData(model.data);
            setPartitionKey(model.partitionKey);
        }

        public final String getSequenceNumber() {
            return sequenceNumber;
        }

        @Override
        public final Builder sequenceNumber(String sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
            return this;
        }

        public final void setSequenceNumber(String sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
        }

        public final Date getApproximateArrivalTimestamp() {
            return approximateArrivalTimestamp;
        }

        @Override
        public final Builder approximateArrivalTimestamp(Date approximateArrivalTimestamp) {
            this.approximateArrivalTimestamp = StandardMemberCopier.copy(approximateArrivalTimestamp);
            return this;
        }

        public final void setApproximateArrivalTimestamp(Date approximateArrivalTimestamp) {
            this.approximateArrivalTimestamp = StandardMemberCopier.copy(approximateArrivalTimestamp);
        }

        public final ByteBuffer getData() {
            return data;
        }

        @Override
        public final Builder data(ByteBuffer data) {
            this.data = StandardMemberCopier.copy(data);
            return this;
        }

        public final void setData(ByteBuffer data) {
            this.data = StandardMemberCopier.copy(data);
        }

        public final String getPartitionKey() {
            return partitionKey;
        }

        @Override
        public final Builder partitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
            return this;
        }

        public final void setPartitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
        }

        @Override
        public Record build() {
            return new Record(this);
        }
    }
}
