/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.firehose.transform.DeliveryStreamDescriptionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a delivery stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DeliveryStreamDescription implements StructuredPojo,
        ToCopyableBuilder<DeliveryStreamDescription.Builder, DeliveryStreamDescription> {
    private final String deliveryStreamName;

    private final String deliveryStreamARN;

    private final String deliveryStreamStatus;

    private final String deliveryStreamType;

    private final String versionId;

    private final Instant createTimestamp;

    private final Instant lastUpdateTimestamp;

    private final SourceDescription source;

    private final List<DestinationDescription> destinations;

    private final Boolean hasMoreDestinations;

    private DeliveryStreamDescription(BuilderImpl builder) {
        this.deliveryStreamName = builder.deliveryStreamName;
        this.deliveryStreamARN = builder.deliveryStreamARN;
        this.deliveryStreamStatus = builder.deliveryStreamStatus;
        this.deliveryStreamType = builder.deliveryStreamType;
        this.versionId = builder.versionId;
        this.createTimestamp = builder.createTimestamp;
        this.lastUpdateTimestamp = builder.lastUpdateTimestamp;
        this.source = builder.source;
        this.destinations = builder.destinations;
        this.hasMoreDestinations = builder.hasMoreDestinations;
    }

    /**
     * <p>
     * The name of the delivery stream.
     * </p>
     * 
     * @return The name of the delivery stream.
     */
    public String deliveryStreamName() {
        return deliveryStreamName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the delivery stream.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the delivery stream.
     */
    public String deliveryStreamARN() {
        return deliveryStreamARN;
    }

    /**
     * <p>
     * The status of the delivery stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamStatus} will return {@link DeliveryStreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #deliveryStreamStatusAsString}.
     * </p>
     * 
     * @return The status of the delivery stream.
     * @see DeliveryStreamStatus
     */
    public DeliveryStreamStatus deliveryStreamStatus() {
        return DeliveryStreamStatus.fromValue(deliveryStreamStatus);
    }

    /**
     * <p>
     * The status of the delivery stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamStatus} will return {@link DeliveryStreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #deliveryStreamStatusAsString}.
     * </p>
     * 
     * @return The status of the delivery stream.
     * @see DeliveryStreamStatus
     */
    public String deliveryStreamStatusAsString() {
        return deliveryStreamStatus;
    }

    /**
     * <p>
     * The delivery stream type. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DirectPut</code>: Provider applications access the delivery stream directly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamType} will return {@link DeliveryStreamType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #deliveryStreamTypeAsString}.
     * </p>
     * 
     * @return The delivery stream type. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DirectPut</code>: Provider applications access the delivery stream directly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
     *         </p>
     *         </li>
     * @see DeliveryStreamType
     */
    public DeliveryStreamType deliveryStreamType() {
        return DeliveryStreamType.fromValue(deliveryStreamType);
    }

    /**
     * <p>
     * The delivery stream type. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DirectPut</code>: Provider applications access the delivery stream directly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamType} will return {@link DeliveryStreamType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #deliveryStreamTypeAsString}.
     * </p>
     * 
     * @return The delivery stream type. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DirectPut</code>: Provider applications access the delivery stream directly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
     *         </p>
     *         </li>
     * @see DeliveryStreamType
     */
    public String deliveryStreamTypeAsString() {
        return deliveryStreamType;
    }

    /**
     * <p>
     * Each time the destination is updated for a delivery stream, the version ID is changed, and the current version ID
     * is required when updating the destination. This is so that the service knows it is applying the changes to the
     * correct version of the delivery stream.
     * </p>
     * 
     * @return Each time the destination is updated for a delivery stream, the version ID is changed, and the current
     *         version ID is required when updating the destination. This is so that the service knows it is applying
     *         the changes to the correct version of the delivery stream.
     */
    public String versionId() {
        return versionId;
    }

    /**
     * <p>
     * The date and time that the delivery stream was created.
     * </p>
     * 
     * @return The date and time that the delivery stream was created.
     */
    public Instant createTimestamp() {
        return createTimestamp;
    }

    /**
     * <p>
     * The date and time that the delivery stream was last updated.
     * </p>
     * 
     * @return The date and time that the delivery stream was last updated.
     */
    public Instant lastUpdateTimestamp() {
        return lastUpdateTimestamp;
    }

    /**
     * <p>
     * If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
     * <a>SourceDescription</a> object describing the source Kinesis stream.
     * </p>
     * 
     * @return If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
     *         <a>SourceDescription</a> object describing the source Kinesis stream.
     */
    public SourceDescription source() {
        return source;
    }

    /**
     * <p>
     * The destinations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The destinations.
     */
    public List<DestinationDescription> destinations() {
        return destinations;
    }

    /**
     * <p>
     * Indicates whether there are more destinations available to list.
     * </p>
     * 
     * @return Indicates whether there are more destinations available to list.
     */
    public Boolean hasMoreDestinations() {
        return hasMoreDestinations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamName());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamARN());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(createTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(destinations());
        hashCode = 31 * hashCode + Objects.hashCode(hasMoreDestinations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeliveryStreamDescription)) {
            return false;
        }
        DeliveryStreamDescription other = (DeliveryStreamDescription) obj;
        return Objects.equals(deliveryStreamName(), other.deliveryStreamName())
                && Objects.equals(deliveryStreamARN(), other.deliveryStreamARN())
                && Objects.equals(deliveryStreamStatusAsString(), other.deliveryStreamStatusAsString())
                && Objects.equals(deliveryStreamTypeAsString(), other.deliveryStreamTypeAsString())
                && Objects.equals(versionId(), other.versionId()) && Objects.equals(createTimestamp(), other.createTimestamp())
                && Objects.equals(lastUpdateTimestamp(), other.lastUpdateTimestamp()) && Objects.equals(source(), other.source())
                && Objects.equals(destinations(), other.destinations())
                && Objects.equals(hasMoreDestinations(), other.hasMoreDestinations());
    }

    @Override
    public String toString() {
        return ToString.builder("DeliveryStreamDescription").add("DeliveryStreamName", deliveryStreamName())
                .add("DeliveryStreamARN", deliveryStreamARN()).add("DeliveryStreamStatus", deliveryStreamStatusAsString())
                .add("DeliveryStreamType", deliveryStreamTypeAsString()).add("VersionId", versionId())
                .add("CreateTimestamp", createTimestamp()).add("LastUpdateTimestamp", lastUpdateTimestamp())
                .add("Source", source()).add("Destinations", destinations()).add("HasMoreDestinations", hasMoreDestinations())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeliveryStreamName":
            return Optional.of(clazz.cast(deliveryStreamName()));
        case "DeliveryStreamARN":
            return Optional.of(clazz.cast(deliveryStreamARN()));
        case "DeliveryStreamStatus":
            return Optional.of(clazz.cast(deliveryStreamStatusAsString()));
        case "DeliveryStreamType":
            return Optional.of(clazz.cast(deliveryStreamTypeAsString()));
        case "VersionId":
            return Optional.of(clazz.cast(versionId()));
        case "CreateTimestamp":
            return Optional.of(clazz.cast(createTimestamp()));
        case "LastUpdateTimestamp":
            return Optional.of(clazz.cast(lastUpdateTimestamp()));
        case "Source":
            return Optional.of(clazz.cast(source()));
        case "Destinations":
            return Optional.of(clazz.cast(destinations()));
        case "HasMoreDestinations":
            return Optional.of(clazz.cast(hasMoreDestinations()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DeliveryStreamDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DeliveryStreamDescription> {
        /**
         * <p>
         * The name of the delivery stream.
         * </p>
         * 
         * @param deliveryStreamName
         *        The name of the delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryStreamName(String deliveryStreamName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the delivery stream.
         * </p>
         * 
         * @param deliveryStreamARN
         *        The Amazon Resource Name (ARN) of the delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryStreamARN(String deliveryStreamARN);

        /**
         * <p>
         * The status of the delivery stream.
         * </p>
         * 
         * @param deliveryStreamStatus
         *        The status of the delivery stream.
         * @see DeliveryStreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamStatus
         */
        Builder deliveryStreamStatus(String deliveryStreamStatus);

        /**
         * <p>
         * The status of the delivery stream.
         * </p>
         * 
         * @param deliveryStreamStatus
         *        The status of the delivery stream.
         * @see DeliveryStreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamStatus
         */
        Builder deliveryStreamStatus(DeliveryStreamStatus deliveryStreamStatus);

        /**
         * <p>
         * The delivery stream type. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DirectPut</code>: Provider applications access the delivery stream directly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
         * </p>
         * </li>
         * </ul>
         * 
         * @param deliveryStreamType
         *        The delivery stream type. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DirectPut</code>: Provider applications access the delivery stream directly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
         *        </p>
         *        </li>
         * @see DeliveryStreamType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamType
         */
        Builder deliveryStreamType(String deliveryStreamType);

        /**
         * <p>
         * The delivery stream type. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DirectPut</code>: Provider applications access the delivery stream directly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
         * </p>
         * </li>
         * </ul>
         * 
         * @param deliveryStreamType
         *        The delivery stream type. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DirectPut</code>: Provider applications access the delivery stream directly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis stream as a source.
         *        </p>
         *        </li>
         * @see DeliveryStreamType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamType
         */
        Builder deliveryStreamType(DeliveryStreamType deliveryStreamType);

        /**
         * <p>
         * Each time the destination is updated for a delivery stream, the version ID is changed, and the current
         * version ID is required when updating the destination. This is so that the service knows it is applying the
         * changes to the correct version of the delivery stream.
         * </p>
         * 
         * @param versionId
         *        Each time the destination is updated for a delivery stream, the version ID is changed, and the current
         *        version ID is required when updating the destination. This is so that the service knows it is applying
         *        the changes to the correct version of the delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * The date and time that the delivery stream was created.
         * </p>
         * 
         * @param createTimestamp
         *        The date and time that the delivery stream was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTimestamp(Instant createTimestamp);

        /**
         * <p>
         * The date and time that the delivery stream was last updated.
         * </p>
         * 
         * @param lastUpdateTimestamp
         *        The date and time that the delivery stream was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimestamp(Instant lastUpdateTimestamp);

        /**
         * <p>
         * If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
         * <a>SourceDescription</a> object describing the source Kinesis stream.
         * </p>
         * 
         * @param source
         *        If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
         *        <a>SourceDescription</a> object describing the source Kinesis stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(SourceDescription source);

        /**
         * <p>
         * If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
         * <a>SourceDescription</a> object describing the source Kinesis stream.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceDescription.Builder} avoiding the need to
         * create one manually via {@link SourceDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #source(SourceDescription)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link SourceDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(SourceDescription)
         */
        default Builder source(Consumer<SourceDescription.Builder> source) {
            return source(SourceDescription.builder().apply(source).build());
        }

        /**
         * <p>
         * The destinations.
         * </p>
         * 
         * @param destinations
         *        The destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<DestinationDescription> destinations);

        /**
         * <p>
         * The destinations.
         * </p>
         * 
         * @param destinations
         *        The destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(DestinationDescription... destinations);

        /**
         * <p>
         * The destinations.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DestinationDescription>.Builder} avoiding
         * the need to create one manually via {@link List<DestinationDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DestinationDescription>.Builder#build()} is called
         * immediately and its result is passed to {@link #destinations(List<DestinationDescription>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<DestinationDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<DestinationDescription>)
         */
        Builder destinations(Consumer<DestinationDescription.Builder>... destinations);

        /**
         * <p>
         * Indicates whether there are more destinations available to list.
         * </p>
         * 
         * @param hasMoreDestinations
         *        Indicates whether there are more destinations available to list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hasMoreDestinations(Boolean hasMoreDestinations);
    }

    static final class BuilderImpl implements Builder {
        private String deliveryStreamName;

        private String deliveryStreamARN;

        private String deliveryStreamStatus;

        private String deliveryStreamType;

        private String versionId;

        private Instant createTimestamp;

        private Instant lastUpdateTimestamp;

        private SourceDescription source;

        private List<DestinationDescription> destinations;

        private Boolean hasMoreDestinations;

        private BuilderImpl() {
        }

        private BuilderImpl(DeliveryStreamDescription model) {
            deliveryStreamName(model.deliveryStreamName);
            deliveryStreamARN(model.deliveryStreamARN);
            deliveryStreamStatus(model.deliveryStreamStatus);
            deliveryStreamType(model.deliveryStreamType);
            versionId(model.versionId);
            createTimestamp(model.createTimestamp);
            lastUpdateTimestamp(model.lastUpdateTimestamp);
            source(model.source);
            destinations(model.destinations);
            hasMoreDestinations(model.hasMoreDestinations);
        }

        public final String getDeliveryStreamName() {
            return deliveryStreamName;
        }

        @Override
        public final Builder deliveryStreamName(String deliveryStreamName) {
            this.deliveryStreamName = deliveryStreamName;
            return this;
        }

        public final void setDeliveryStreamName(String deliveryStreamName) {
            this.deliveryStreamName = deliveryStreamName;
        }

        public final String getDeliveryStreamARN() {
            return deliveryStreamARN;
        }

        @Override
        public final Builder deliveryStreamARN(String deliveryStreamARN) {
            this.deliveryStreamARN = deliveryStreamARN;
            return this;
        }

        public final void setDeliveryStreamARN(String deliveryStreamARN) {
            this.deliveryStreamARN = deliveryStreamARN;
        }

        public final String getDeliveryStreamStatus() {
            return deliveryStreamStatus;
        }

        @Override
        public final Builder deliveryStreamStatus(String deliveryStreamStatus) {
            this.deliveryStreamStatus = deliveryStreamStatus;
            return this;
        }

        @Override
        public final Builder deliveryStreamStatus(DeliveryStreamStatus deliveryStreamStatus) {
            this.deliveryStreamStatus(deliveryStreamStatus.toString());
            return this;
        }

        public final void setDeliveryStreamStatus(String deliveryStreamStatus) {
            this.deliveryStreamStatus = deliveryStreamStatus;
        }

        public final String getDeliveryStreamType() {
            return deliveryStreamType;
        }

        @Override
        public final Builder deliveryStreamType(String deliveryStreamType) {
            this.deliveryStreamType = deliveryStreamType;
            return this;
        }

        @Override
        public final Builder deliveryStreamType(DeliveryStreamType deliveryStreamType) {
            this.deliveryStreamType(deliveryStreamType.toString());
            return this;
        }

        public final void setDeliveryStreamType(String deliveryStreamType) {
            this.deliveryStreamType = deliveryStreamType;
        }

        public final String getVersionId() {
            return versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        public final Instant getCreateTimestamp() {
            return createTimestamp;
        }

        @Override
        public final Builder createTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
            return this;
        }

        public final void setCreateTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
        }

        public final Instant getLastUpdateTimestamp() {
            return lastUpdateTimestamp;
        }

        @Override
        public final Builder lastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
            return this;
        }

        public final void setLastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
        }

        public final SourceDescription.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        @Override
        public final Builder source(SourceDescription source) {
            this.source = source;
            return this;
        }

        public final void setSource(SourceDescription.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        public final Collection<DestinationDescription.Builder> getDestinations() {
            return destinations != null ? destinations.stream().map(DestinationDescription::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder destinations(Collection<DestinationDescription> destinations) {
            this.destinations = DestinationDescriptionListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(DestinationDescription... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<DestinationDescription.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> DestinationDescription.builder().apply(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinations(Collection<DestinationDescription.BuilderImpl> destinations) {
            this.destinations = DestinationDescriptionListCopier.copyFromBuilder(destinations);
        }

        public final Boolean getHasMoreDestinations() {
            return hasMoreDestinations;
        }

        @Override
        public final Builder hasMoreDestinations(Boolean hasMoreDestinations) {
            this.hasMoreDestinations = hasMoreDestinations;
            return this;
        }

        public final void setHasMoreDestinations(Boolean hasMoreDestinations) {
            this.hasMoreDestinations = hasMoreDestinations;
        }

        @Override
        public DeliveryStreamDescription build() {
            return new DeliveryStreamDescription(this);
        }
    }
}
