/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.kinesis.transform.RecordMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The unit of data of the Kinesis stream, which is composed of a sequence number, a partition key, and a data blob.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Record implements StructuredPojo, ToCopyableBuilder<Record.Builder, Record> {
    private final String sequenceNumber;

    private final Instant approximateArrivalTimestamp;

    private final SdkBytes data;

    private final String partitionKey;

    private final String encryptionType;

    private Record(BuilderImpl builder) {
        this.sequenceNumber = builder.sequenceNumber;
        this.approximateArrivalTimestamp = builder.approximateArrivalTimestamp;
        this.data = builder.data;
        this.partitionKey = builder.partitionKey;
        this.encryptionType = builder.encryptionType;
    }

    /**
     * <p>
     * The unique identifier of the record within its shard.
     * </p>
     * 
     * @return The unique identifier of the record within its shard.
     */
    public String sequenceNumber() {
        return sequenceNumber;
    }

    /**
     * <p>
     * The approximate time that the record was inserted into the stream.
     * </p>
     * 
     * @return The approximate time that the record was inserted into the stream.
     */
    public Instant approximateArrivalTimestamp() {
        return approximateArrivalTimestamp;
    }

    /**
     * <p>
     * The data blob. The data in the blob is both opaque and immutable to Kinesis Streams, which does not inspect,
     * interpret, or change the data in the blob in any way. When the data blob (the payload before base64-encoding) is
     * added to the partition key size, the total size must not exceed the maximum record size (1 MB).
     * </p>
     * 
     * @return The data blob. The data in the blob is both opaque and immutable to Kinesis Streams, which does not
     *         inspect, interpret, or change the data in the blob in any way. When the data blob (the payload before
     *         base64-encoding) is added to the partition key size, the total size must not exceed the maximum record
     *         size (1 MB).
     */
    public SdkBytes data() {
        return data;
    }

    /**
     * <p>
     * Identifies which shard in the stream the data record is assigned to.
     * </p>
     * 
     * @return Identifies which shard in the stream the data record is assigned to.
     */
    public String partitionKey() {
        return partitionKey;
    }

    /**
     * <p>
     * The encryption type used on the record. This parameter can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code>: Do not encrypt the records in the stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS key.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The encryption type used on the record. This parameter can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Do not encrypt the records in the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS
     *         key.
     *         </p>
     *         </li>
     * @see EncryptionType
     */
    public EncryptionType encryptionType() {
        return EncryptionType.fromValue(encryptionType);
    }

    /**
     * <p>
     * The encryption type used on the record. This parameter can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code>: Do not encrypt the records in the stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS key.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The encryption type used on the record. This parameter can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Do not encrypt the records in the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS
     *         key.
     *         </p>
     *         </li>
     * @see EncryptionType
     */
    public String encryptionTypeAsString() {
        return encryptionType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sequenceNumber());
        hashCode = 31 * hashCode + Objects.hashCode(approximateArrivalTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(partitionKey());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Record)) {
            return false;
        }
        Record other = (Record) obj;
        return Objects.equals(sequenceNumber(), other.sequenceNumber())
                && Objects.equals(approximateArrivalTimestamp(), other.approximateArrivalTimestamp())
                && Objects.equals(data(), other.data()) && Objects.equals(partitionKey(), other.partitionKey())
                && Objects.equals(encryptionTypeAsString(), other.encryptionTypeAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("Record").add("SequenceNumber", sequenceNumber())
                .add("ApproximateArrivalTimestamp", approximateArrivalTimestamp()).add("Data", data())
                .add("PartitionKey", partitionKey()).add("EncryptionType", encryptionTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SequenceNumber":
            return Optional.ofNullable(clazz.cast(sequenceNumber()));
        case "ApproximateArrivalTimestamp":
            return Optional.ofNullable(clazz.cast(approximateArrivalTimestamp()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "PartitionKey":
            return Optional.ofNullable(clazz.cast(partitionKey()));
        case "EncryptionType":
            return Optional.ofNullable(clazz.cast(encryptionTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        RecordMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Record> {
        /**
         * <p>
         * The unique identifier of the record within its shard.
         * </p>
         * 
         * @param sequenceNumber
         *        The unique identifier of the record within its shard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceNumber(String sequenceNumber);

        /**
         * <p>
         * The approximate time that the record was inserted into the stream.
         * </p>
         * 
         * @param approximateArrivalTimestamp
         *        The approximate time that the record was inserted into the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approximateArrivalTimestamp(Instant approximateArrivalTimestamp);

        /**
         * <p>
         * The data blob. The data in the blob is both opaque and immutable to Kinesis Streams, which does not inspect,
         * interpret, or change the data in the blob in any way. When the data blob (the payload before base64-encoding)
         * is added to the partition key size, the total size must not exceed the maximum record size (1 MB).
         * </p>
         * 
         * @param data
         *        The data blob. The data in the blob is both opaque and immutable to Kinesis Streams, which does not
         *        inspect, interpret, or change the data in the blob in any way. When the data blob (the payload before
         *        base64-encoding) is added to the partition key size, the total size must not exceed the maximum record
         *        size (1 MB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(SdkBytes data);

        /**
         * <p>
         * Identifies which shard in the stream the data record is assigned to.
         * </p>
         * 
         * @param partitionKey
         *        Identifies which shard in the stream the data record is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKey(String partitionKey);

        /**
         * <p>
         * The encryption type used on the record. This parameter can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code>: Do not encrypt the records in the stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS key.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionType
         *        The encryption type used on the record. This parameter can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Do not encrypt the records in the stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS
         *        key.
         *        </p>
         *        </li>
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(String encryptionType);

        /**
         * <p>
         * The encryption type used on the record. This parameter can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code>: Do not encrypt the records in the stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS key.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionType
         *        The encryption type used on the record. This parameter can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Do not encrypt the records in the stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed KMS
         *        key.
         *        </p>
         *        </li>
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(EncryptionType encryptionType);
    }

    static final class BuilderImpl implements Builder {
        private String sequenceNumber;

        private Instant approximateArrivalTimestamp;

        private SdkBytes data;

        private String partitionKey;

        private String encryptionType;

        private BuilderImpl() {
        }

        private BuilderImpl(Record model) {
            sequenceNumber(model.sequenceNumber);
            approximateArrivalTimestamp(model.approximateArrivalTimestamp);
            data(model.data);
            partitionKey(model.partitionKey);
            encryptionType(model.encryptionType);
        }

        public final String getSequenceNumber() {
            return sequenceNumber;
        }

        @Override
        public final Builder sequenceNumber(String sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
            return this;
        }

        public final void setSequenceNumber(String sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
        }

        public final Instant getApproximateArrivalTimestamp() {
            return approximateArrivalTimestamp;
        }

        @Override
        public final Builder approximateArrivalTimestamp(Instant approximateArrivalTimestamp) {
            this.approximateArrivalTimestamp = approximateArrivalTimestamp;
            return this;
        }

        public final void setApproximateArrivalTimestamp(Instant approximateArrivalTimestamp) {
            this.approximateArrivalTimestamp = approximateArrivalTimestamp;
        }

        public final ByteBuffer getData() {
            return data == null ? null : data.asByteBuffer();
        }

        @Override
        public final Builder data(SdkBytes data) {
            this.data = StandardMemberCopier.copy(data);
            return this;
        }

        public final void setData(ByteBuffer data) {
            data(data == null ? null : SdkBytes.fromByteBuffer(data));
        }

        public final String getPartitionKey() {
            return partitionKey;
        }

        @Override
        public final Builder partitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
            return this;
        }

        public final void setPartitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
        }

        public final String getEncryptionType() {
            return encryptionType;
        }

        @Override
        public final Builder encryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        @Override
        public final Builder encryptionType(EncryptionType encryptionType) {
            this.encryptionType(encryptionType.toString());
            return this;
        }

        public final void setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
        }

        @Override
        public Record build() {
            return new Record(this);
        }
    }
}
