/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.kinesis.transform.SubscribeToShardEventMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * After you call <a>SubscribeToShard</a>, Kinesis Data Streams sends events of this type to your consumer.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SubscribeToShardEvent implements StructuredPojo,
        ToCopyableBuilder<SubscribeToShardEvent.Builder, SubscribeToShardEvent>, SubscribeToShardEventStream {
    private final List<Record> records;

    private final String continuationSequenceNumber;

    private final Long millisBehindLatest;

    private SubscribeToShardEvent(BuilderImpl builder) {
        this.records = builder.records;
        this.continuationSequenceNumber = builder.continuationSequenceNumber;
        this.millisBehindLatest = builder.millisBehindLatest;
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return
     */
    public List<Record> records() {
        return records;
    }

    /**
     * <p>
     * Use this as <code>StartingSequenceNumber</code> in the next call to <a>SubscribeToShard</a>.
     * </p>
     * 
     * @return Use this as <code>StartingSequenceNumber</code> in the next call to <a>SubscribeToShard</a>.
     */
    public String continuationSequenceNumber() {
        return continuationSequenceNumber;
    }

    /**
     * <p>
     * The number of milliseconds the read records are from the tip of the stream, indicating how far behind current
     * time the consumer is. A value of zero indicates that record processing is caught up, and there are no new records
     * to process at this moment.
     * </p>
     * 
     * @return The number of milliseconds the read records are from the tip of the stream, indicating how far behind
     *         current time the consumer is. A value of zero indicates that record processing is caught up, and there
     *         are no new records to process at this moment.
     */
    public Long millisBehindLatest() {
        return millisBehindLatest;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(records());
        hashCode = 31 * hashCode + Objects.hashCode(continuationSequenceNumber());
        hashCode = 31 * hashCode + Objects.hashCode(millisBehindLatest());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SubscribeToShardEvent)) {
            return false;
        }
        SubscribeToShardEvent other = (SubscribeToShardEvent) obj;
        return Objects.equals(records(), other.records())
                && Objects.equals(continuationSequenceNumber(), other.continuationSequenceNumber())
                && Objects.equals(millisBehindLatest(), other.millisBehindLatest());
    }

    @Override
    public String toString() {
        return ToString.builder("SubscribeToShardEvent").add("Records", records())
                .add("ContinuationSequenceNumber", continuationSequenceNumber()).add("MillisBehindLatest", millisBehindLatest())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Records":
            return Optional.ofNullable(clazz.cast(records()));
        case "ContinuationSequenceNumber":
            return Optional.ofNullable(clazz.cast(continuationSequenceNumber()));
        case "MillisBehindLatest":
            return Optional.ofNullable(clazz.cast(millisBehindLatest()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SubscribeToShardEventMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    /**
     * Calls the appropriate visit method depending on the subtype of {@link SubscribeToShardEvent}.
     *
     * @param visitor
     *        Visitor to invoke.
     */
    @Override
    public void accept(SubscribeToShardResponseHandler.Visitor visitor) {
        visitor.visit(this);
    }

    public interface Builder extends CopyableBuilder<Builder, SubscribeToShardEvent> {
        /**
         * <p/>
         * 
         * @param records
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder records(Collection<Record> records);

        /**
         * <p/>
         * 
         * @param records
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder records(Record... records);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link List<Record>.Builder} avoiding the need to
         * create one manually via {@link List<Record>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Record>.Builder#build()} is called immediately and its
         * result is passed to {@link #records(List<Record>)}.
         * 
         * @param records
         *        a consumer that will call methods on {@link List<Record>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #records(List<Record>)
         */
        Builder records(Consumer<Record.Builder>... records);

        /**
         * <p>
         * Use this as <code>StartingSequenceNumber</code> in the next call to <a>SubscribeToShard</a>.
         * </p>
         * 
         * @param continuationSequenceNumber
         *        Use this as <code>StartingSequenceNumber</code> in the next call to <a>SubscribeToShard</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder continuationSequenceNumber(String continuationSequenceNumber);

        /**
         * <p>
         * The number of milliseconds the read records are from the tip of the stream, indicating how far behind current
         * time the consumer is. A value of zero indicates that record processing is caught up, and there are no new
         * records to process at this moment.
         * </p>
         * 
         * @param millisBehindLatest
         *        The number of milliseconds the read records are from the tip of the stream, indicating how far behind
         *        current time the consumer is. A value of zero indicates that record processing is caught up, and there
         *        are no new records to process at this moment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder millisBehindLatest(Long millisBehindLatest);
    }

    static final class BuilderImpl implements Builder {
        private List<Record> records = DefaultSdkAutoConstructList.getInstance();

        private String continuationSequenceNumber;

        private Long millisBehindLatest;

        private BuilderImpl() {
        }

        private BuilderImpl(SubscribeToShardEvent model) {
            records(model.records);
            continuationSequenceNumber(model.continuationSequenceNumber);
            millisBehindLatest(model.millisBehindLatest);
        }

        public final Collection<Record.Builder> getRecords() {
            return records != null ? records.stream().map(Record::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder records(Collection<Record> records) {
            this.records = RecordListCopier.copy(records);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder records(Record... records) {
            records(Arrays.asList(records));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder records(Consumer<Record.Builder>... records) {
            records(Stream.of(records).map(c -> Record.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRecords(Collection<Record.BuilderImpl> records) {
            this.records = RecordListCopier.copyFromBuilder(records);
        }

        public final String getContinuationSequenceNumber() {
            return continuationSequenceNumber;
        }

        @Override
        public final Builder continuationSequenceNumber(String continuationSequenceNumber) {
            this.continuationSequenceNumber = continuationSequenceNumber;
            return this;
        }

        public final void setContinuationSequenceNumber(String continuationSequenceNumber) {
            this.continuationSequenceNumber = continuationSequenceNumber;
        }

        public final Long getMillisBehindLatest() {
            return millisBehindLatest;
        }

        @Override
        public final Builder millisBehindLatest(Long millisBehindLatest) {
            this.millisBehindLatest = millisBehindLatest;
            return this;
        }

        public final void setMillisBehindLatest(Long millisBehindLatest) {
            this.millisBehindLatest = millisBehindLatest;
        }

        @Override
        public SubscribeToShardEvent build() {
            return new SubscribeToShardEvent(this);
        }
    }
}
