/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis;

import java.util.concurrent.CompletableFuture;
import javax.annotation.Generated;
import software.amazon.awssdk.AmazonServiceException;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.client.AsyncClientHandler;
import software.amazon.awssdk.client.ClientExecutionParams;
import software.amazon.awssdk.client.SdkAsyncClientHandler;
import software.amazon.awssdk.config.AsyncClientConfiguration;
import software.amazon.awssdk.http.HttpResponseHandler;
import software.amazon.awssdk.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.protocol.json.SdkJsonProtocolFactory;
import software.amazon.awssdk.services.kinesis.model.AddTagsToStreamRequest;
import software.amazon.awssdk.services.kinesis.model.AddTagsToStreamResponse;
import software.amazon.awssdk.services.kinesis.model.CreateStreamRequest;
import software.amazon.awssdk.services.kinesis.model.CreateStreamResponse;
import software.amazon.awssdk.services.kinesis.model.DecreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.DecreaseStreamRetentionPeriodResponse;
import software.amazon.awssdk.services.kinesis.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DeleteStreamResponse;
import software.amazon.awssdk.services.kinesis.model.DescribeLimitsRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeLimitsResponse;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamResponse;
import software.amazon.awssdk.services.kinesis.model.DisableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.DisableEnhancedMonitoringResponse;
import software.amazon.awssdk.services.kinesis.model.EnableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.EnableEnhancedMonitoringResponse;
import software.amazon.awssdk.services.kinesis.model.ExpiredIteratorException;
import software.amazon.awssdk.services.kinesis.model.GetRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.GetRecordsResponse;
import software.amazon.awssdk.services.kinesis.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.kinesis.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.kinesis.model.IncreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.IncreaseStreamRetentionPeriodResponse;
import software.amazon.awssdk.services.kinesis.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesis.model.LimitExceededException;
import software.amazon.awssdk.services.kinesis.model.ListStreamsRequest;
import software.amazon.awssdk.services.kinesis.model.ListStreamsResponse;
import software.amazon.awssdk.services.kinesis.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesis.model.ListTagsForStreamResponse;
import software.amazon.awssdk.services.kinesis.model.MergeShardsRequest;
import software.amazon.awssdk.services.kinesis.model.MergeShardsResponse;
import software.amazon.awssdk.services.kinesis.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.kinesis.model.PutRecordRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordResponse;
import software.amazon.awssdk.services.kinesis.model.PutRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordsResponse;
import software.amazon.awssdk.services.kinesis.model.RemoveTagsFromStreamRequest;
import software.amazon.awssdk.services.kinesis.model.RemoveTagsFromStreamResponse;
import software.amazon.awssdk.services.kinesis.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesis.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesis.model.SplitShardRequest;
import software.amazon.awssdk.services.kinesis.model.SplitShardResponse;
import software.amazon.awssdk.services.kinesis.model.UpdateShardCountRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateShardCountResponse;
import software.amazon.awssdk.services.kinesis.transform.AddTagsToStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.AddTagsToStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.CreateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.CreateStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.DecreaseStreamRetentionPeriodRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DecreaseStreamRetentionPeriodResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.DeleteStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DeleteStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeLimitsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeLimitsResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.DisableEnhancedMonitoringRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DisableEnhancedMonitoringResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.EnableEnhancedMonitoringRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.EnableEnhancedMonitoringResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.GetRecordsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.GetRecordsResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.GetShardIteratorRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.GetShardIteratorResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.IncreaseStreamRetentionPeriodRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.IncreaseStreamRetentionPeriodResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListStreamsResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListTagsForStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListTagsForStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.MergeShardsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.MergeShardsResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.PutRecordRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.PutRecordResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.PutRecordsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.PutRecordsResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.RemoveTagsFromStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.RemoveTagsFromStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.SplitShardRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.SplitShardResponseUnmarshaller;
import software.amazon.awssdk.services.kinesis.transform.UpdateShardCountRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.UpdateShardCountResponseUnmarshaller;

/**
 * Internal implementation of {@link KinesisAsyncClient}.
 *
 * @see KinesisAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisAsyncClient implements KinesisAsyncClient {
    private final AsyncClientHandler clientHandler;

    private final SdkJsonProtocolFactory protocolFactory;

    protected DefaultKinesisAsyncClient(AsyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkAsyncClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
    }

    /**
     * <p>
     * Adds or updates tags for the specified Amazon Kinesis stream. Each stream can have up to 10 tags.
     * </p>
     * <p>
     * If tags have already been assigned to the stream, <code>AddTagsToStream</code> overwrites any existing tags that
     * correspond to the specified tag keys.
     * </p>
     *
     * @param addTagsToStreamRequest
     *        Represents the input for <code>AddTagsToStream</code>.
     * @return A Java Future containing the result of the AddTagsToStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.AddTagsToStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/AddTagsToStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddTagsToStreamResponse> addTagsToStream(AddTagsToStreamRequest addTagsToStreamRequest) {

        HttpResponseHandler<AddTagsToStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new AddTagsToStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<AddTagsToStreamRequest, AddTagsToStreamResponse>()
                .withMarshaller(new AddTagsToStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(addTagsToStreamRequest));
    }

    /**
     * <p>
     * Creates an Amazon Kinesis stream. A stream captures and transports data records that are continuously emitted
     * from different data sources or <i>producers</i>. Scale-out within a stream is explicitly supported by means of
     * shards, which are uniquely identified groups of data records in a stream.
     * </p>
     * <p>
     * You specify and control the number of shards that a stream is composed of. Each shard can support reads up to 5
     * transactions per second, up to a maximum data read total of 2 MB per second. Each shard can support writes up to
     * 1,000 records per second, up to a maximum data write total of 1 MB per second. You can add shards to a stream if
     * the amount of data input increases and you can remove shards if the amount of data input decreases.
     * </p>
     * <p>
     * The stream name identifies the stream. The name is scoped to the AWS account used by the application. It is also
     * scoped by region. That is, two streams in two different accounts can have the same name, and two streams in the
     * same account, but in two different regions, can have the same name.
     * </p>
     * <p>
     * <code>CreateStream</code> is an asynchronous operation. Upon receiving a <code>CreateStream</code> request,
     * Amazon Kinesis immediately returns and sets the stream status to <code>CREATING</code>. After the stream is
     * created, Amazon Kinesis sets the stream status to <code>ACTIVE</code>. You should perform read and write
     * operations only on an <code>ACTIVE</code> stream.
     * </p>
     * <p>
     * You receive a <code>LimitExceededException</code> when making a <code>CreateStream</code> request if you try to
     * do one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Have more than five streams in the <code>CREATING</code> state at any point in time.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create more shards than are authorized for your account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the default shard limit for an AWS account, see <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in the
     * <i>Amazon Kinesis Streams Developer Guide</i>. If you need to increase this limit, <a
     * href="http://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">contact AWS Support</a>.
     * </p>
     * <p>
     * You can use <code>DescribeStream</code> to check the stream status, which is returned in
     * <code>StreamStatus</code>.
     * </p>
     * <p>
     * <a>CreateStream</a> has a limit of 5 transactions per second per account.
     * </p>
     *
     * @param createStreamRequest
     *        Represents the input for <code>CreateStream</code>.
     * @return A Java Future containing the result of the CreateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.CreateStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/CreateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamResponse> createStream(CreateStreamRequest createStreamRequest) {

        HttpResponseHandler<CreateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateStreamRequest, CreateStreamResponse>()
                .withMarshaller(new CreateStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createStreamRequest));
    }

    /**
     * <p>
     * Decreases the Amazon Kinesis stream's retention period, which is the length of time data records are accessible
     * after they are added to the stream. The minimum value of a stream's retention period is 24 hours.
     * </p>
     * <p>
     * This operation may result in lost data. For example, if the stream's retention period is 48 hours and is
     * decreased to 24 hours, any data already in the stream that is older than 24 hours is inaccessible.
     * </p>
     *
     * @param decreaseStreamRetentionPeriodRequest
     *        Represents the input for <a>DecreaseStreamRetentionPeriod</a>.
     * @return A Java Future containing the result of the DecreaseStreamRetentionPeriod operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.DecreaseStreamRetentionPeriod
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DecreaseStreamRetentionPeriod"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DecreaseStreamRetentionPeriodResponse> decreaseStreamRetentionPeriod(
            DecreaseStreamRetentionPeriodRequest decreaseStreamRetentionPeriodRequest) {

        HttpResponseHandler<DecreaseStreamRetentionPeriodResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DecreaseStreamRetentionPeriodResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<DecreaseStreamRetentionPeriodRequest, DecreaseStreamRetentionPeriodResponse>()
                        .withMarshaller(new DecreaseStreamRetentionPeriodRequestMarshaller(protocolFactory))
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(decreaseStreamRetentionPeriodRequest));
    }

    /**
     * <p>
     * Deletes an Amazon Kinesis stream and all its shards and data. You must shut down any applications that are
     * operating on the stream before you delete the stream. If an application attempts to operate on a deleted stream,
     * it will receive the exception <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * If the stream is in the <code>ACTIVE</code> state, you can delete it. After a <code>DeleteStream</code> request,
     * the specified stream is in the <code>DELETING</code> state until Amazon Kinesis completes the deletion.
     * </p>
     * <p>
     * <b>Note:</b> Amazon Kinesis might continue to accept data read and write operations, such as <a>PutRecord</a>,
     * <a>PutRecords</a>, and <a>GetRecords</a>, on a stream in the <code>DELETING</code> state until the stream
     * deletion is complete.
     * </p>
     * <p>
     * When you delete a stream, any shards in that stream are also deleted, and any tags are dissociated from the
     * stream.
     * </p>
     * <p>
     * You can use the <a>DescribeStream</a> operation to check the state of the stream, which is returned in
     * <code>StreamStatus</code>.
     * </p>
     * <p>
     * <a>DeleteStream</a> has a limit of 5 transactions per second per account.
     * </p>
     *
     * @param deleteStreamRequest
     *        Represents the input for <a>DeleteStream</a>.
     * @return A Java Future containing the result of the DeleteStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.DeleteStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DeleteStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStreamResponse> deleteStream(DeleteStreamRequest deleteStreamRequest) {

        HttpResponseHandler<DeleteStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteStreamRequest, DeleteStreamResponse>()
                .withMarshaller(new DeleteStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteStreamRequest));
    }

    /**
     * <p>
     * Describes the shard limits and usage for the account.
     * </p>
     * <p>
     * If you update your account limits, the old limits might be returned for a few minutes.
     * </p>
     * <p>
     * This operation has a limit of 1 transaction per second per account.
     * </p>
     *
     * @param describeLimitsRequest
     * @return A Java Future containing the result of the DescribeLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.DescribeLimits
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DescribeLimits" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLimitsResponse> describeLimits(DescribeLimitsRequest describeLimitsRequest) {

        HttpResponseHandler<DescribeLimitsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeLimitsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeLimitsRequest, DescribeLimitsResponse>()
                .withMarshaller(new DescribeLimitsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeLimitsRequest));
    }

    /**
     * <p>
     * Describes the specified Amazon Kinesis stream.
     * </p>
     * <p>
     * The information returned includes the stream name, Amazon Resource Name (ARN), creation time, enhanced metric
     * configuration, and shard map. The shard map is an array of shard objects. For each shard object, there is the
     * hash key and sequence number ranges that the shard spans, and the IDs of any earlier shards that played in a role
     * in creating the shard. Every record ingested in the stream is identified by a sequence number, which is assigned
     * when the record is put into the stream.
     * </p>
     * <p>
     * You can limit the number of shards returned by each call. For more information, see <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-retrieve-shards.html">Retrieving
     * Shards from a Stream</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * There are no guarantees about the chronological order shards returned. To process shards in chronological order,
     * use the ID of the parent shard to track the lineage to the oldest shard.
     * </p>
     * <p>
     * This operation has a limit of 10 transactions per second per account.
     * </p>
     *
     * @param describeStreamRequest
     *        Represents the input for <code>DescribeStream</code>.
     * @return A Java Future containing the result of the DescribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.DescribeStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DescribeStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStreamResponse> describeStream(DescribeStreamRequest describeStreamRequest) {

        HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeStreamRequest));
    }

    /**
     * <p>
     * Disables enhanced monitoring.
     * </p>
     *
     * @param disableEnhancedMonitoringRequest
     *        Represents the input for <a>DisableEnhancedMonitoring</a>.
     * @return A Java Future containing the result of the DisableEnhancedMonitoring operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.DisableEnhancedMonitoring
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DisableEnhancedMonitoring"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableEnhancedMonitoringResponse> disableEnhancedMonitoring(
            DisableEnhancedMonitoringRequest disableEnhancedMonitoringRequest) {

        HttpResponseHandler<DisableEnhancedMonitoringResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DisableEnhancedMonitoringResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<DisableEnhancedMonitoringRequest, DisableEnhancedMonitoringResponse>()
                        .withMarshaller(new DisableEnhancedMonitoringRequestMarshaller(protocolFactory))
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(disableEnhancedMonitoringRequest));
    }

    /**
     * <p>
     * Enables enhanced Amazon Kinesis stream monitoring for shard-level metrics.
     * </p>
     *
     * @param enableEnhancedMonitoringRequest
     *        Represents the input for <a>EnableEnhancedMonitoring</a>.
     * @return A Java Future containing the result of the EnableEnhancedMonitoring operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.EnableEnhancedMonitoring
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/EnableEnhancedMonitoring"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableEnhancedMonitoringResponse> enableEnhancedMonitoring(
            EnableEnhancedMonitoringRequest enableEnhancedMonitoringRequest) {

        HttpResponseHandler<EnableEnhancedMonitoringResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new EnableEnhancedMonitoringResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<EnableEnhancedMonitoringRequest, EnableEnhancedMonitoringResponse>()
                        .withMarshaller(new EnableEnhancedMonitoringRequestMarshaller(protocolFactory))
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(enableEnhancedMonitoringRequest));
    }

    /**
     * <p>
     * Gets data records from an Amazon Kinesis stream's shard.
     * </p>
     * <p>
     * Specify a shard iterator using the <code>ShardIterator</code> parameter. The shard iterator specifies the
     * position in the shard from which you want to start reading data records sequentially. If there are no records
     * available in the portion of the shard that the iterator points to, <a>GetRecords</a> returns an empty list. Note
     * that it might take multiple calls to get to a portion of the shard that contains records.
     * </p>
     * <p>
     * You can scale by provisioning multiple shards per stream while considering service limits (for more information,
     * see <a href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     * the <i>Amazon Kinesis Streams Developer Guide</i>). Your application should have one thread per shard, each
     * reading continuously from its stream. To read from a stream continually, call <a>GetRecords</a> in a loop. Use
     * <a>GetShardIterator</a> to get the shard iterator to specify in the first <a>GetRecords</a> call.
     * <a>GetRecords</a> returns a new shard iterator in <code>NextShardIterator</code>. Specify the shard iterator
     * returned in <code>NextShardIterator</code> in subsequent calls to <a>GetRecords</a>. Note that if the shard has
     * been closed, the shard iterator can't return more data and <a>GetRecords</a> returns <code>null</code> in
     * <code>NextShardIterator</code>. You can terminate the loop when the shard is closed, or when the shard iterator
     * reaches the record with the sequence number or other attribute that marks it as the last record to process.
     * </p>
     * <p>
     * Each data record can be up to 1 MB in size, and each shard can read up to 2 MB per second. You can ensure that
     * your calls don't exceed the maximum supported size or throughput by using the <code>Limit</code> parameter to
     * specify the maximum number of records that <a>GetRecords</a> can return. Consider your average record size when
     * determining this limit.
     * </p>
     * <p>
     * The size of the data returned by <a>GetRecords</a> varies depending on the utilization of the shard. The maximum
     * size of data that <a>GetRecords</a> can return is 10 MB. If a call returns this amount of data, subsequent calls
     * made within the next 5 seconds throw <code>ProvisionedThroughputExceededException</code>. If there is
     * insufficient provisioned throughput on the shard, subsequent calls made within the next 1 second throw
     * <code>ProvisionedThroughputExceededException</code>. Note that <a>GetRecords</a> won't return any data when it
     * throws an exception. For this reason, we recommend that you wait one second between calls to <a>GetRecords</a>;
     * however, it's possible that the application will get exceptions for longer than 1 second.
     * </p>
     * <p>
     * To detect whether the application is falling behind in processing, you can use the
     * <code>MillisBehindLatest</code> response attribute. You can also monitor the stream using CloudWatch metrics and
     * other mechanisms (see <a href="http://docs.aws.amazon.com/kinesis/latest/dev/monitoring.html">Monitoring</a> in
     * the <i>Amazon Kinesis Streams Developer Guide</i>).
     * </p>
     * <p>
     * Each Amazon Kinesis record includes a value, <code>ApproximateArrivalTimestamp</code>, that is set when a stream
     * successfully receives and stores a record. This is commonly referred to as a server-side timestamp, whereas a
     * client-side timestamp is set when a data producer creates or sends the record to a stream (a data producer is any
     * data source putting data records into a stream, for example with <a>PutRecords</a>). The timestamp has
     * millisecond precision. There are no guarantees about the timestamp accuracy, or that the timestamp is always
     * increasing. For example, records in a shard or across a stream might have timestamps that are out of order.
     * </p>
     *
     * @param getRecordsRequest
     *        Represents the input for <a>GetRecords</a>.
     * @return A Java Future containing the result of the GetRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>ProvisionedThroughputExceededException The request rate for the stream is too high, or the requested
     *         data is too large for the available throughput. Reduce the frequency or size of your requests. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Streams Developer Guide</i>, and <a
     *         href="http://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in AWS</a> in the <i>AWS General Reference</i>.</li>
     *         <li>ExpiredIteratorException The provided iterator exceeds the maximum age allowed.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.GetRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/GetRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecordsResponse> getRecords(GetRecordsRequest getRecordsRequest) {

        HttpResponseHandler<GetRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetRecordsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetRecordsRequest, GetRecordsResponse>()
                .withMarshaller(new GetRecordsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getRecordsRequest));
    }

    /**
     * <p>
     * Gets an Amazon Kinesis shard iterator. A shard iterator expires five minutes after it is returned to the
     * requester.
     * </p>
     * <p>
     * A shard iterator specifies the shard position from which to start reading data records sequentially. The position
     * is specified using the sequence number of a data record in a shard. A sequence number is the identifier
     * associated with every record ingested in the stream, and is assigned when a record is put into the stream. Each
     * stream has one or more shards.
     * </p>
     * <p>
     * You must specify the shard iterator type. For example, you can set the <code>ShardIteratorType</code> parameter
     * to read exactly from the position denoted by a specific sequence number by using the
     * <code>AT_SEQUENCE_NUMBER</code> shard iterator type, or right after the sequence number by using the
     * <code>AFTER_SEQUENCE_NUMBER</code> shard iterator type, using sequence numbers returned by earlier calls to
     * <a>PutRecord</a>, <a>PutRecords</a>, <a>GetRecords</a>, or <a>DescribeStream</a>. In the request, you can specify
     * the shard iterator type <code>AT_TIMESTAMP</code> to read records from an arbitrary point in time,
     * <code>TRIM_HORIZON</code> to cause <code>ShardIterator</code> to point to the last untrimmed record in the shard
     * in the system (the oldest data record in the shard), or <code>LATEST</code> so that you always read the most
     * recent data in the shard.
     * </p>
     * <p>
     * When you read repeatedly from a stream, use a <a>GetShardIterator</a> request to get the first shard iterator for
     * use in your first <a>GetRecords</a> request and for subsequent reads use the shard iterator returned by the
     * <a>GetRecords</a> request in <code>NextShardIterator</code>. A new shard iterator is returned by every
     * <a>GetRecords</a> request in <code>NextShardIterator</code>, which you use in the <code>ShardIterator</code>
     * parameter of the next <a>GetRecords</a> request.
     * </p>
     * <p>
     * If a <a>GetShardIterator</a> request is made too often, you receive a
     * <code>ProvisionedThroughputExceededException</code>. For more information about throughput limits, see
     * <a>GetRecords</a>, and <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in the
     * <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * If the shard is closed, <a>GetShardIterator</a> returns a valid iterator for the last sequence number of the
     * shard. Note that a shard can be closed as a result of using <a>SplitShard</a> or <a>MergeShards</a>.
     * </p>
     * <p>
     * <a>GetShardIterator</a> has a limit of 5 transactions per second per account per open shard.
     * </p>
     *
     * @param getShardIteratorRequest
     *        Represents the input for <code>GetShardIterator</code>.
     * @return A Java Future containing the result of the GetShardIterator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>ProvisionedThroughputExceededException The request rate for the stream is too high, or the requested
     *         data is too large for the available throughput. Reduce the frequency or size of your requests. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Streams Developer Guide</i>, and <a
     *         href="http://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in AWS</a> in the <i>AWS General Reference</i>.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.GetShardIterator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/GetShardIterator" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetShardIteratorResponse> getShardIterator(GetShardIteratorRequest getShardIteratorRequest) {

        HttpResponseHandler<GetShardIteratorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetShardIteratorResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetShardIteratorRequest, GetShardIteratorResponse>()
                .withMarshaller(new GetShardIteratorRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getShardIteratorRequest));
    }

    /**
     * <p>
     * Increases the Amazon Kinesis stream's retention period, which is the length of time data records are accessible
     * after they are added to the stream. The maximum value of a stream's retention period is 168 hours (7 days).
     * </p>
     * <p>
     * Upon choosing a longer stream retention period, this operation will increase the time period records are
     * accessible that have not yet expired. However, it will not make previous data that has expired (older than the
     * stream's previous retention period) accessible after the operation has been called. For example, if a stream's
     * retention period is set to 24 hours and is increased to 168 hours, any data that is older than 24 hours will
     * remain inaccessible to consumer applications.
     * </p>
     *
     * @param increaseStreamRetentionPeriodRequest
     *        Represents the input for <a>IncreaseStreamRetentionPeriod</a>.
     * @return A Java Future containing the result of the IncreaseStreamRetentionPeriod operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.IncreaseStreamRetentionPeriod
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/IncreaseStreamRetentionPeriod"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<IncreaseStreamRetentionPeriodResponse> increaseStreamRetentionPeriod(
            IncreaseStreamRetentionPeriodRequest increaseStreamRetentionPeriodRequest) {

        HttpResponseHandler<IncreaseStreamRetentionPeriodResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new IncreaseStreamRetentionPeriodResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<IncreaseStreamRetentionPeriodRequest, IncreaseStreamRetentionPeriodResponse>()
                        .withMarshaller(new IncreaseStreamRetentionPeriodRequestMarshaller(protocolFactory))
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(increaseStreamRetentionPeriodRequest));
    }

    /**
     * <p>
     * Lists your Amazon Kinesis streams.
     * </p>
     * <p>
     * The number of streams may be too large to return from a single call to <code>ListStreams</code>. You can limit
     * the number of returned streams using the <code>Limit</code> parameter. If you do not specify a value for the
     * <code>Limit</code> parameter, Amazon Kinesis uses the default limit, which is currently 10.
     * </p>
     * <p>
     * You can detect if there are more streams available to list by using the <code>HasMoreStreams</code> flag from the
     * returned output. If there are more streams available, you can request more streams by using the name of the last
     * stream returned by the <code>ListStreams</code> request in the <code>ExclusiveStartStreamName</code> parameter in
     * a subsequent request to <code>ListStreams</code>. The group of stream names returned by the subsequent request is
     * then added to the list. You can continue this process until all the stream names have been collected in the list.
     * </p>
     * <p>
     * <a>ListStreams</a> has a limit of 5 transactions per second per account.
     * </p>
     *
     * @param listStreamsRequest
     *        Represents the input for <code>ListStreams</code>.
     * @return A Java Future containing the result of the ListStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.ListStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamsResponse> listStreams(ListStreamsRequest listStreamsRequest) {

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListStreamsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listStreamsRequest));
    }

    /**
     * <p>
     * Lists the tags for the specified Amazon Kinesis stream.
     * </p>
     *
     * @param listTagsForStreamRequest
     *        Represents the input for <code>ListTagsForStream</code>.
     * @return A Java Future containing the result of the ListTagsForStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.ListTagsForStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/ListTagsForStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForStreamResponse> listTagsForStream(ListTagsForStreamRequest listTagsForStreamRequest) {

        HttpResponseHandler<ListTagsForStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListTagsForStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListTagsForStreamRequest, ListTagsForStreamResponse>()
                .withMarshaller(new ListTagsForStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForStreamRequest));
    }

    /**
     * <p>
     * Merges two adjacent shards in an Amazon Kinesis stream and combines them into a single shard to reduce the
     * stream's capacity to ingest and transport data. Two shards are considered adjacent if the union of the hash key
     * ranges for the two shards form a contiguous set with no gaps. For example, if you have two shards, one with a
     * hash key range of 276...381 and the other with a hash key range of 382...454, then you could merge these two
     * shards into a single shard that would have a hash key range of 276...454. After the merge, the single child shard
     * receives data for all hash key values covered by the two parent shards.
     * </p>
     * <p>
     * <code>MergeShards</code> is called when there is a need to reduce the overall capacity of a stream because of
     * excess capacity that is not being used. You must specify the shard to be merged and the adjacent shard for a
     * stream. For more information about merging shards, see <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-resharding-merge.html">Merge Two
     * Shards</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * If the stream is in the <code>ACTIVE</code> state, you can call <code>MergeShards</code>. If a stream is in the
     * <code>CREATING</code>, <code>UPDATING</code>, or <code>DELETING</code> state, <code>MergeShards</code> returns a
     * <code>ResourceInUseException</code>. If the specified stream does not exist, <code>MergeShards</code> returns a
     * <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * You can use <a>DescribeStream</a> to check the state of the stream, which is returned in
     * <code>StreamStatus</code>.
     * </p>
     * <p>
     * <code>MergeShards</code> is an asynchronous operation. Upon receiving a <code>MergeShards</code> request, Amazon
     * Kinesis immediately returns a response and sets the <code>StreamStatus</code> to <code>UPDATING</code>. After the
     * operation is completed, Amazon Kinesis sets the <code>StreamStatus</code> to <code>ACTIVE</code>. Read and write
     * operations continue to work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * <p>
     * You use <a>DescribeStream</a> to determine the shard IDs that are specified in the <code>MergeShards</code>
     * request.
     * </p>
     * <p>
     * If you try to operate on too many streams in parallel using <a>CreateStream</a>, <a>DeleteStream</a>,
     * <code>MergeShards</code> or <a>SplitShard</a>, you will receive a <code>LimitExceededException</code>.
     * </p>
     * <p>
     * <code>MergeShards</code> has limit of 5 transactions per second per account.
     * </p>
     *
     * @param mergeShardsRequest
     *        Represents the input for <code>MergeShards</code>.
     * @return A Java Future containing the result of the MergeShards operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.MergeShards
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/MergeShards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<MergeShardsResponse> mergeShards(MergeShardsRequest mergeShardsRequest) {

        HttpResponseHandler<MergeShardsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new MergeShardsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<MergeShardsRequest, MergeShardsResponse>()
                .withMarshaller(new MergeShardsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(mergeShardsRequest));
    }

    /**
     * <p>
     * Writes a single data record into an Amazon Kinesis stream. Call <code>PutRecord</code> to send data into the
     * stream for real-time ingestion and subsequent processing, one record at a time. Each shard can support writes up
     * to 1,000 records per second, up to a maximum data write total of 1 MB per second.
     * </p>
     * <p>
     * You must specify the name of the stream that captures, stores, and transports the data; a partition key; and the
     * data blob itself.
     * </p>
     * <p>
     * The data blob can be any type of data; for example, a segment from a log file, geographic/location data, website
     * clickstream data, and so on.
     * </p>
     * <p>
     * The partition key is used by Amazon Kinesis to distribute data across shards. Amazon Kinesis segregates the data
     * records that belong to a stream into multiple shards, using the partition key associated with each data record to
     * determine which shard a given data record belongs to.
     * </p>
     * <p>
     * Partition keys are Unicode strings, with a maximum length limit of 256 characters for each key. An MD5 hash
     * function is used to map partition keys to 128-bit integer values and to map associated data records to shards
     * using the hash key ranges of the shards. You can override hashing the partition key to determine the shard by
     * explicitly specifying a hash value using the <code>ExplicitHashKey</code> parameter. For more information, see <a
     * href=
     * "http://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-add-data-to-stream"
     * >Adding Data to a Stream</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * <code>PutRecord</code> returns the shard ID of where the data record was placed and the sequence number that was
     * assigned to the data record.
     * </p>
     * <p>
     * Sequence numbers increase over time and are specific to a shard within a stream, not across all shards within a
     * stream. To guarantee strictly increasing ordering, write serially to a shard and use the
     * <code>SequenceNumberForOrdering</code> parameter. For more information, see <a href=
     * "http://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-add-data-to-stream"
     * >Adding Data to a Stream</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * If a <code>PutRecord</code> request cannot be processed because of insufficient provisioned throughput on the
     * shard involved in the request, <code>PutRecord</code> throws <code>ProvisionedThroughputExceededException</code>.
     * </p>
     * <p>
     * Data records are accessible for only 24 hours from the time that they are added to a stream.
     * </p>
     *
     * @param putRecordRequest
     *        Represents the input for <code>PutRecord</code>.
     * @return A Java Future containing the result of the PutRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>ProvisionedThroughputExceededException The request rate for the stream is too high, or the requested
     *         data is too large for the available throughput. Reduce the frequency or size of your requests. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Streams Developer Guide</i>, and <a
     *         href="http://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in AWS</a> in the <i>AWS General Reference</i>.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.PutRecord
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/PutRecord" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecordResponse> putRecord(PutRecordRequest putRecordRequest) {

        HttpResponseHandler<PutRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new PutRecordResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutRecordRequest, PutRecordResponse>()
                .withMarshaller(new PutRecordRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putRecordRequest));
    }

    /**
     * <p>
     * Writes multiple data records into an Amazon Kinesis stream in a single call (also referred to as a
     * <code>PutRecords</code> request). Use this operation to send data into the stream for data ingestion and
     * processing.
     * </p>
     * <p>
     * Each <code>PutRecords</code> request can support up to 500 records. Each record in the request can be as large as
     * 1 MB, up to a limit of 5 MB for the entire request, including partition keys. Each shard can support writes up to
     * 1,000 records per second, up to a maximum data write total of 1 MB per second.
     * </p>
     * <p>
     * You must specify the name of the stream that captures, stores, and transports the data; and an array of request
     * <code>Records</code>, with each record in the array requiring a partition key and data blob. The record size
     * limit applies to the total size of the partition key and data blob.
     * </p>
     * <p>
     * The data blob can be any type of data; for example, a segment from a log file, geographic/location data, website
     * clickstream data, and so on.
     * </p>
     * <p>
     * The partition key is used by Amazon Kinesis as input to a hash function that maps the partition key and
     * associated data to a specific shard. An MD5 hash function is used to map partition keys to 128-bit integer values
     * and to map associated data records to shards. As a result of this hashing mechanism, all data records with the
     * same partition key map to the same shard within the stream. For more information, see <a href=
     * "http://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-add-data-to-stream"
     * >Adding Data to a Stream</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * Each record in the <code>Records</code> array may include an optional parameter, <code>ExplicitHashKey</code>,
     * which overrides the partition key to shard mapping. This parameter allows a data producer to determine explicitly
     * the shard where the record is stored. For more information, see <a href=
     * "http://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-putrecords"
     * >Adding Multiple Records with PutRecords</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * The <code>PutRecords</code> response includes an array of response <code>Records</code>. Each record in the
     * response array directly correlates with a record in the request array using natural ordering, from the top to the
     * bottom of the request and response. The response <code>Records</code> array always includes the same number of
     * records as the request array.
     * </p>
     * <p>
     * The response <code>Records</code> array includes both successfully and unsuccessfully processed records. Amazon
     * Kinesis attempts to process all records in each <code>PutRecords</code> request. A single record failure does not
     * stop the processing of subsequent records.
     * </p>
     * <p>
     * A successfully-processed record includes <code>ShardId</code> and <code>SequenceNumber</code> values. The
     * <code>ShardId</code> parameter identifies the shard in the stream where the record is stored. The
     * <code>SequenceNumber</code> parameter is an identifier assigned to the put record, unique to all records in the
     * stream.
     * </p>
     * <p>
     * An unsuccessfully-processed record includes <code>ErrorCode</code> and <code>ErrorMessage</code> values.
     * <code>ErrorCode</code> reflects the type of error and can be one of the following values:
     * <code>ProvisionedThroughputExceededException</code> or <code>InternalFailure</code>. <code>ErrorMessage</code>
     * provides more detailed information about the <code>ProvisionedThroughputExceededException</code> exception
     * including the account ID, stream name, and shard ID of the record that was throttled. For more information about
     * partially successful responses, see <a href=
     * "http://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-add-data-to-stream.html#kinesis-using-sdk-java-putrecords"
     * >Adding Multiple Records with PutRecords</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * By default, data records are accessible for only 24 hours from the time that they are added to an Amazon Kinesis
     * stream. This retention period can be modified using the <a>DecreaseStreamRetentionPeriod</a> and
     * <a>IncreaseStreamRetentionPeriod</a> operations.
     * </p>
     *
     * @param putRecordsRequest
     *        A <code>PutRecords</code> request.
     * @return A Java Future containing the result of the PutRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>ProvisionedThroughputExceededException The request rate for the stream is too high, or the requested
     *         data is too large for the available throughput. Reduce the frequency or size of your requests. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Streams Developer Guide</i>, and <a
     *         href="http://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in AWS</a> in the <i>AWS General Reference</i>.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.PutRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/PutRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecordsResponse> putRecords(PutRecordsRequest putRecordsRequest) {

        HttpResponseHandler<PutRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new PutRecordsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutRecordsRequest, PutRecordsResponse>()
                .withMarshaller(new PutRecordsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putRecordsRequest));
    }

    /**
     * <p>
     * Removes tags from the specified Amazon Kinesis stream. Removed tags are deleted and cannot be recovered after
     * this operation successfully completes.
     * </p>
     * <p>
     * If you specify a tag that does not exist, it is ignored.
     * </p>
     *
     * @param removeTagsFromStreamRequest
     *        Represents the input for <code>RemoveTagsFromStream</code>.
     * @return A Java Future containing the result of the RemoveTagsFromStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.RemoveTagsFromStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/RemoveTagsFromStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveTagsFromStreamResponse> removeTagsFromStream(
            RemoveTagsFromStreamRequest removeTagsFromStreamRequest) {

        HttpResponseHandler<RemoveTagsFromStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new RemoveTagsFromStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<RemoveTagsFromStreamRequest, RemoveTagsFromStreamResponse>()
                .withMarshaller(new RemoveTagsFromStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removeTagsFromStreamRequest));
    }

    /**
     * <p>
     * Splits a shard into two new shards in the Amazon Kinesis stream to increase the stream's capacity to ingest and
     * transport data. <code>SplitShard</code> is called when there is a need to increase the overall capacity of a
     * stream because of an expected increase in the volume of data records being ingested.
     * </p>
     * <p>
     * You can also use <code>SplitShard</code> when a shard appears to be approaching its maximum utilization; for
     * example, the producers sending data into the specific shard are suddenly sending more than previously
     * anticipated. You can also call <code>SplitShard</code> to increase stream capacity, so that more Amazon Kinesis
     * applications can simultaneously read data from the stream for real-time processing.
     * </p>
     * <p>
     * You must specify the shard to be split and the new hash key, which is the position in the shard where the shard
     * gets split in two. In many cases, the new hash key might simply be the average of the beginning and ending hash
     * key, but it can be any hash key value in the range being mapped into the shard. For more information about
     * splitting shards, see <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-resharding-split.html">Split a
     * Shard</a> in the <i>Amazon Kinesis Streams Developer Guide</i>.
     * </p>
     * <p>
     * You can use <a>DescribeStream</a> to determine the shard ID and hash key values for the <code>ShardToSplit</code>
     * and <code>NewStartingHashKey</code> parameters that are specified in the <code>SplitShard</code> request.
     * </p>
     * <p>
     * <code>SplitShard</code> is an asynchronous operation. Upon receiving a <code>SplitShard</code> request, Amazon
     * Kinesis immediately returns a response and sets the stream status to <code>UPDATING</code>. After the operation
     * is completed, Amazon Kinesis sets the stream status to <code>ACTIVE</code>. Read and write operations continue to
     * work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * <p>
     * You can use <code>DescribeStream</code> to check the status of the stream, which is returned in
     * <code>StreamStatus</code>. If the stream is in the <code>ACTIVE</code> state, you can call
     * <code>SplitShard</code>. If a stream is in <code>CREATING</code> or <code>UPDATING</code> or
     * <code>DELETING</code> states, <code>DescribeStream</code> returns a <code>ResourceInUseException</code>.
     * </p>
     * <p>
     * If the specified stream does not exist, <code>DescribeStream</code> returns a
     * <code>ResourceNotFoundException</code>. If you try to create more shards than are authorized for your account,
     * you receive a <code>LimitExceededException</code>.
     * </p>
     * <p>
     * For the default shard limit for an AWS account, see <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in the
     * <i>Amazon Kinesis Streams Developer Guide</i>. If you need to increase this limit, <a
     * href="http://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">contact AWS Support</a>.
     * </p>
     * <p>
     * If you try to operate on too many streams simultaneously using <a>CreateStream</a>, <a>DeleteStream</a>,
     * <a>MergeShards</a>, and/or <a>SplitShard</a>, you receive a <code>LimitExceededException</code>.
     * </p>
     * <p>
     * <code>SplitShard</code> has limit of 5 transactions per second per account.
     * </p>
     *
     * @param splitShardRequest
     *        Represents the input for <code>SplitShard</code>.
     * @return A Java Future containing the result of the SplitShard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.SplitShard
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/SplitShard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SplitShardResponse> splitShard(SplitShardRequest splitShardRequest) {

        HttpResponseHandler<SplitShardResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new SplitShardResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SplitShardRequest, SplitShardResponse>()
                .withMarshaller(new SplitShardRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(splitShardRequest));
    }

    /**
     * <p>
     * Updates the shard count of the specified stream to the specified number of shards.
     * </p>
     * <p>
     * Updating the shard count is an asynchronous operation. Upon receiving the request, Amazon Kinesis returns
     * immediately and sets the status of the stream to <code>UPDATING</code>. After the update is complete, Amazon
     * Kinesis sets the status of the stream back to <code>ACTIVE</code>. Depending on the size of the stream, the
     * scaling action could take a few minutes to complete. You can continue to read and write data to your stream while
     * its status is <code>UPDATING</code>.
     * </p>
     * <p>
     * To update the shard count, Amazon Kinesis performs splits and merges and individual shards. This can cause
     * short-lived shards to be created, in addition to the final shards. We recommend that you double or halve the
     * shard count, as this results in the fewest number of splits or merges.
     * </p>
     * <p>
     * This operation has a rate limit of twice per rolling 24 hour period. You cannot scale above double your current
     * shard count, scale below half your current shard count, or exceed the shard limits for your account.
     * </p>
     * <p>
     * For the default limits for an AWS account, see <a
     * href="http://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in the
     * <i>Amazon Kinesis Streams Developer Guide</i>. If you need to increase a limit, <a
     * href="http://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">contact AWS Support</a>.
     * </p>
     *
     * @param updateShardCountRequest
     * @return A Java Future containing the result of the UpdateShardCount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used. For more information, see the returned message.</li>
     *         <li>LimitExceededException The requested resource exceeds the maximum number allowed, or the number of
     *         concurrent stream requests exceeds the maximum number allowed (5).</li>
     *         <li>ResourceInUseException The resource is not available for this operation. For successful operation,
     *         the resource needs to be in the <code>ACTIVE</code> state.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found. The stream might not be
     *         specified correctly.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisAsyncClient.UpdateShardCount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/UpdateShardCount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateShardCountResponse> updateShardCount(UpdateShardCountRequest updateShardCountRequest) {

        HttpResponseHandler<UpdateShardCountResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateShardCountResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<UpdateShardCountRequest, UpdateShardCountResponse>()
                .withMarshaller(new UpdateShardCountRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateShardCountRequest));
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private software.amazon.awssdk.protocol.json.SdkJsonProtocolFactory init() {
        return new SdkJsonProtocolFactory(new JsonClientMetadata()
                .withProtocolVersion("1.1")
                .withSupportsCbor(true)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.kinesis.model.KinesisException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidArgumentException").withModeledClass(
                                InvalidArgumentException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceInUseException").withModeledClass(
                                ResourceInUseException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                ResourceNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ExpiredIteratorException").withModeledClass(
                                ExpiredIteratorException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ProvisionedThroughputExceededException").withModeledClass(
                                ProvisionedThroughputExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("LimitExceededException").withModeledClass(
                                LimitExceededException.class)));
    }

    private HttpResponseHandler<AmazonServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }
}
