/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AwsRequestOverrideConfig;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input for <code>PutRecord</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class PutRecordRequest extends KinesisRequest implements ToCopyableBuilder<PutRecordRequest.Builder, PutRecordRequest> {
    private final String streamName;

    private final ByteBuffer data;

    private final String partitionKey;

    private final String explicitHashKey;

    private final String sequenceNumberForOrdering;

    private PutRecordRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.data = builder.data;
        this.partitionKey = builder.partitionKey;
        this.explicitHashKey = builder.explicitHashKey;
        this.sequenceNumberForOrdering = builder.sequenceNumberForOrdering;
    }

    /**
     * <p>
     * The name of the stream to put the data record into.
     * </p>
     * 
     * @return The name of the stream to put the data record into.
     */
    public String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data blob
     * (the payload before base64-encoding) is added to the partition key size, the total size must not exceed the
     * maximum record size (1 MB).
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data
     *         blob (the payload before base64-encoding) is added to the partition key size, the total size must not
     *         exceed the maximum record size (1 MB).
     */
    public ByteBuffer data() {
        return data == null ? null : data.asReadOnlyBuffer();
    }

    /**
     * <p>
     * Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings with a
     * maximum length limit of 256 characters for each key. Amazon Kinesis uses the partition key as input to a hash
     * function that maps the partition key and associated data to a specific shard. Specifically, an MD5 hash function
     * is used to map partition keys to 128-bit integer values and to map associated data records to shards. As a result
     * of this hashing mechanism, all data records with the same partition key map to the same shard within the stream.
     * </p>
     * 
     * @return Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings
     *         with a maximum length limit of 256 characters for each key. Amazon Kinesis uses the partition key as
     *         input to a hash function that maps the partition key and associated data to a specific shard.
     *         Specifically, an MD5 hash function is used to map partition keys to 128-bit integer values and to map
     *         associated data records to shards. As a result of this hashing mechanism, all data records with the same
     *         partition key map to the same shard within the stream.
     */
    public String partitionKey() {
        return partitionKey;
    }

    /**
     * <p>
     * The hash value used to explicitly determine the shard the data record is assigned to by overriding the partition
     * key hash.
     * </p>
     * 
     * @return The hash value used to explicitly determine the shard the data record is assigned to by overriding the
     *         partition key hash.
     */
    public String explicitHashKey() {
        return explicitHashKey;
    }

    /**
     * <p>
     * Guarantees strictly increasing sequence numbers, for puts from the same client and to the same partition key.
     * Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the sequence number of record
     * <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If this parameter is not set, records are
     * coarsely ordered based on arrival time.
     * </p>
     * 
     * @return Guarantees strictly increasing sequence numbers, for puts from the same client and to the same partition
     *         key. Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the sequence number of
     *         record <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If this parameter is not
     *         set, records are coarsely ordered based on arrival time.
     */
    public String sequenceNumberForOrdering() {
        return sequenceNumberForOrdering;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(partitionKey());
        hashCode = 31 * hashCode + Objects.hashCode(explicitHashKey());
        hashCode = 31 * hashCode + Objects.hashCode(sequenceNumberForOrdering());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutRecordRequest)) {
            return false;
        }
        PutRecordRequest other = (PutRecordRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(data(), other.data())
                && Objects.equals(partitionKey(), other.partitionKey())
                && Objects.equals(explicitHashKey(), other.explicitHashKey())
                && Objects.equals(sequenceNumberForOrdering(), other.sequenceNumberForOrdering());
    }

    @Override
    public String toString() {
        return ToString.builder("PutRecordRequest").add("StreamName", streamName()).add("Data", data())
                .add("PartitionKey", partitionKey()).add("ExplicitHashKey", explicitHashKey())
                .add("SequenceNumberForOrdering", sequenceNumberForOrdering()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.of(clazz.cast(streamName()));
        case "Data":
            return Optional.of(clazz.cast(data()));
        case "PartitionKey":
            return Optional.of(clazz.cast(partitionKey()));
        case "ExplicitHashKey":
            return Optional.of(clazz.cast(explicitHashKey()));
        case "SequenceNumberForOrdering":
            return Optional.of(clazz.cast(sequenceNumberForOrdering()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends KinesisRequest.Builder, CopyableBuilder<Builder, PutRecordRequest> {
        /**
         * <p>
         * The name of the stream to put the data record into.
         * </p>
         * 
         * @param streamName
         *        The name of the stream to put the data record into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data blob
         * (the payload before base64-encoding) is added to the partition key size, the total size must not exceed the
         * maximum record size (1 MB).
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param data
         *        The data blob to put into the record, which is base64-encoded when the blob is serialized. When the
         *        data blob (the payload before base64-encoding) is added to the partition key size, the total size must
         *        not exceed the maximum record size (1 MB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(ByteBuffer data);

        /**
         * <p>
         * Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings with
         * a maximum length limit of 256 characters for each key. Amazon Kinesis uses the partition key as input to a
         * hash function that maps the partition key and associated data to a specific shard. Specifically, an MD5 hash
         * function is used to map partition keys to 128-bit integer values and to map associated data records to
         * shards. As a result of this hashing mechanism, all data records with the same partition key map to the same
         * shard within the stream.
         * </p>
         * 
         * @param partitionKey
         *        Determines which shard in the stream the data record is assigned to. Partition keys are Unicode
         *        strings with a maximum length limit of 256 characters for each key. Amazon Kinesis uses the partition
         *        key as input to a hash function that maps the partition key and associated data to a specific shard.
         *        Specifically, an MD5 hash function is used to map partition keys to 128-bit integer values and to map
         *        associated data records to shards. As a result of this hashing mechanism, all data records with the
         *        same partition key map to the same shard within the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKey(String partitionKey);

        /**
         * <p>
         * The hash value used to explicitly determine the shard the data record is assigned to by overriding the
         * partition key hash.
         * </p>
         * 
         * @param explicitHashKey
         *        The hash value used to explicitly determine the shard the data record is assigned to by overriding the
         *        partition key hash.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitHashKey(String explicitHashKey);

        /**
         * <p>
         * Guarantees strictly increasing sequence numbers, for puts from the same client and to the same partition key.
         * Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the sequence number of record
         * <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If this parameter is not set, records
         * are coarsely ordered based on arrival time.
         * </p>
         * 
         * @param sequenceNumberForOrdering
         *        Guarantees strictly increasing sequence numbers, for puts from the same client and to the same
         *        partition key. Usage: set the <code>SequenceNumberForOrdering</code> of record <i>n</i> to the
         *        sequence number of record <i>n-1</i> (as returned in the result when putting record <i>n-1</i>). If
         *        this parameter is not set, records are coarsely ordered based on arrival time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceNumberForOrdering(String sequenceNumberForOrdering);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);
    }

    static final class BuilderImpl extends KinesisRequest.BuilderImpl implements Builder {
        private String streamName;

        private ByteBuffer data;

        private String partitionKey;

        private String explicitHashKey;

        private String sequenceNumberForOrdering;

        private BuilderImpl() {
        }

        private BuilderImpl(PutRecordRequest model) {
            streamName(model.streamName);
            data(model.data);
            partitionKey(model.partitionKey);
            explicitHashKey(model.explicitHashKey);
            sequenceNumberForOrdering(model.sequenceNumberForOrdering);
        }

        public final String getStreamName() {
            return streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        public final ByteBuffer getData() {
            return data;
        }

        @Override
        public final Builder data(ByteBuffer data) {
            this.data = StandardMemberCopier.copy(data);
            return this;
        }

        public final void setData(ByteBuffer data) {
            this.data = StandardMemberCopier.copy(data);
        }

        public final String getPartitionKey() {
            return partitionKey;
        }

        @Override
        public final Builder partitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
            return this;
        }

        public final void setPartitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
        }

        public final String getExplicitHashKey() {
            return explicitHashKey;
        }

        @Override
        public final Builder explicitHashKey(String explicitHashKey) {
            this.explicitHashKey = explicitHashKey;
            return this;
        }

        public final void setExplicitHashKey(String explicitHashKey) {
            this.explicitHashKey = explicitHashKey;
        }

        public final String getSequenceNumberForOrdering() {
            return sequenceNumberForOrdering;
        }

        @Override
        public final Builder sequenceNumberForOrdering(String sequenceNumberForOrdering) {
            this.sequenceNumberForOrdering = sequenceNumberForOrdering;
            return this;
        }

        public final void setSequenceNumberForOrdering(String sequenceNumberForOrdering) {
            this.sequenceNumberForOrdering = sequenceNumberForOrdering;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public PutRecordRequest build() {
            return new PutRecordRequest(this);
        }
    }
}
