/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.kinesis.transform.PutRecordsRequestEntryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output for <code>PutRecords</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutRecordsRequestEntry implements StructuredPojo,
        ToCopyableBuilder<PutRecordsRequestEntry.Builder, PutRecordsRequestEntry> {
    private final SdkBytes data;

    private final String explicitHashKey;

    private final String partitionKey;

    private PutRecordsRequestEntry(BuilderImpl builder) {
        this.data = builder.data;
        this.explicitHashKey = builder.explicitHashKey;
        this.partitionKey = builder.partitionKey;
    }

    /**
     * <p>
     * The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data blob
     * (the payload before base64-encoding) is added to the partition key size, the total size must not exceed the
     * maximum record size (1 MB).
     * </p>
     * 
     * @return The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data
     *         blob (the payload before base64-encoding) is added to the partition key size, the total size must not
     *         exceed the maximum record size (1 MB).
     */
    public SdkBytes data() {
        return data;
    }

    /**
     * <p>
     * The hash value used to determine explicitly the shard that the data record is assigned to by overriding the
     * partition key hash.
     * </p>
     * 
     * @return The hash value used to determine explicitly the shard that the data record is assigned to by overriding
     *         the partition key hash.
     */
    public String explicitHashKey() {
        return explicitHashKey;
    }

    /**
     * <p>
     * Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings with a
     * maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses the partition key as input
     * to a hash function that maps the partition key and associated data to a specific shard. Specifically, an MD5 hash
     * function is used to map partition keys to 128-bit integer values and to map associated data records to shards. As
     * a result of this hashing mechanism, all data records with the same partition key map to the same shard within the
     * stream.
     * </p>
     * 
     * @return Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings
     *         with a maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses the
     *         partition key as input to a hash function that maps the partition key and associated data to a specific
     *         shard. Specifically, an MD5 hash function is used to map partition keys to 128-bit integer values and to
     *         map associated data records to shards. As a result of this hashing mechanism, all data records with the
     *         same partition key map to the same shard within the stream.
     */
    public String partitionKey() {
        return partitionKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(explicitHashKey());
        hashCode = 31 * hashCode + Objects.hashCode(partitionKey());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutRecordsRequestEntry)) {
            return false;
        }
        PutRecordsRequestEntry other = (PutRecordsRequestEntry) obj;
        return Objects.equals(data(), other.data()) && Objects.equals(explicitHashKey(), other.explicitHashKey())
                && Objects.equals(partitionKey(), other.partitionKey());
    }

    @Override
    public String toString() {
        return ToString.builder("PutRecordsRequestEntry").add("Data", data()).add("ExplicitHashKey", explicitHashKey())
                .add("PartitionKey", partitionKey()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "ExplicitHashKey":
            return Optional.ofNullable(clazz.cast(explicitHashKey()));
        case "PartitionKey":
            return Optional.ofNullable(clazz.cast(partitionKey()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        PutRecordsRequestEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, PutRecordsRequestEntry> {
        /**
         * <p>
         * The data blob to put into the record, which is base64-encoded when the blob is serialized. When the data blob
         * (the payload before base64-encoding) is added to the partition key size, the total size must not exceed the
         * maximum record size (1 MB).
         * </p>
         * 
         * @param data
         *        The data blob to put into the record, which is base64-encoded when the blob is serialized. When the
         *        data blob (the payload before base64-encoding) is added to the partition key size, the total size must
         *        not exceed the maximum record size (1 MB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(SdkBytes data);

        /**
         * <p>
         * The hash value used to determine explicitly the shard that the data record is assigned to by overriding the
         * partition key hash.
         * </p>
         * 
         * @param explicitHashKey
         *        The hash value used to determine explicitly the shard that the data record is assigned to by
         *        overriding the partition key hash.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder explicitHashKey(String explicitHashKey);

        /**
         * <p>
         * Determines which shard in the stream the data record is assigned to. Partition keys are Unicode strings with
         * a maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses the partition key as
         * input to a hash function that maps the partition key and associated data to a specific shard. Specifically,
         * an MD5 hash function is used to map partition keys to 128-bit integer values and to map associated data
         * records to shards. As a result of this hashing mechanism, all data records with the same partition key map to
         * the same shard within the stream.
         * </p>
         * 
         * @param partitionKey
         *        Determines which shard in the stream the data record is assigned to. Partition keys are Unicode
         *        strings with a maximum length limit of 256 characters for each key. Amazon Kinesis Data Streams uses
         *        the partition key as input to a hash function that maps the partition key and associated data to a
         *        specific shard. Specifically, an MD5 hash function is used to map partition keys to 128-bit integer
         *        values and to map associated data records to shards. As a result of this hashing mechanism, all data
         *        records with the same partition key map to the same shard within the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKey(String partitionKey);
    }

    static final class BuilderImpl implements Builder {
        private SdkBytes data;

        private String explicitHashKey;

        private String partitionKey;

        private BuilderImpl() {
        }

        private BuilderImpl(PutRecordsRequestEntry model) {
            data(model.data);
            explicitHashKey(model.explicitHashKey);
            partitionKey(model.partitionKey);
        }

        public final ByteBuffer getData() {
            return data == null ? null : data.asByteBuffer();
        }

        @Override
        public final Builder data(SdkBytes data) {
            this.data = StandardMemberCopier.copy(data);
            return this;
        }

        public final void setData(ByteBuffer data) {
            data(data == null ? null : SdkBytes.fromByteBuffer(data));
        }

        public final String getExplicitHashKey() {
            return explicitHashKey;
        }

        @Override
        public final Builder explicitHashKey(String explicitHashKey) {
            this.explicitHashKey = explicitHashKey;
            return this;
        }

        public final void setExplicitHashKey(String explicitHashKey) {
            this.explicitHashKey = explicitHashKey;
        }

        public final String getPartitionKey() {
            return partitionKey;
        }

        @Override
        public final Builder partitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
            return this;
        }

        public final void setPartitionKey(String partitionKey) {
            this.partitionKey = partitionKey;
        }

        @Override
        public PutRecordsRequestEntry build() {
            return new PutRecordsRequestEntry(this);
        }
    }
}
