/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.kinesis.transform.ConsumerMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents the details of the consumer you registered.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Consumer implements StructuredPojo, ToCopyableBuilder<Consumer.Builder, Consumer> {
    private final String consumerName;

    private final String consumerARN;

    private final String consumerStatus;

    private final Instant consumerCreationTimestamp;

    private Consumer(BuilderImpl builder) {
        this.consumerName = builder.consumerName;
        this.consumerARN = builder.consumerARN;
        this.consumerStatus = builder.consumerStatus;
        this.consumerCreationTimestamp = builder.consumerCreationTimestamp;
    }

    /**
     * <p>
     * The name of the consumer is something you choose when you register the consumer.
     * </p>
     * 
     * @return The name of the consumer is something you choose when you register the consumer.
     */
    public String consumerName() {
        return consumerName;
    }

    /**
     * <p>
     * When you register a consumer, Kinesis Data Streams generates an ARN for it. You need this ARN to be able to call
     * <a>SubscribeToShard</a>.
     * </p>
     * <p>
     * If you delete a consumer and then create a new one with the same name, it won't have the same ARN. That's because
     * consumer ARNs contain the creation timestamp. This is important to keep in mind if you have IAM policies that
     * reference consumer ARNs.
     * </p>
     * 
     * @return When you register a consumer, Kinesis Data Streams generates an ARN for it. You need this ARN to be able
     *         to call <a>SubscribeToShard</a>.</p>
     *         <p>
     *         If you delete a consumer and then create a new one with the same name, it won't have the same ARN. That's
     *         because consumer ARNs contain the creation timestamp. This is important to keep in mind if you have IAM
     *         policies that reference consumer ARNs.
     */
    public String consumerARN() {
        return consumerARN;
    }

    /**
     * <p>
     * A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #consumerStatus}
     * will return {@link ConsumerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #consumerStatusAsString}.
     * </p>
     * 
     * @return A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
     * @see ConsumerStatus
     */
    public ConsumerStatus consumerStatus() {
        return ConsumerStatus.fromValue(consumerStatus);
    }

    /**
     * <p>
     * A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #consumerStatus}
     * will return {@link ConsumerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #consumerStatusAsString}.
     * </p>
     * 
     * @return A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
     * @see ConsumerStatus
     */
    public String consumerStatusAsString() {
        return consumerStatus;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public Instant consumerCreationTimestamp() {
        return consumerCreationTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(consumerName());
        hashCode = 31 * hashCode + Objects.hashCode(consumerARN());
        hashCode = 31 * hashCode + Objects.hashCode(consumerStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(consumerCreationTimestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Consumer)) {
            return false;
        }
        Consumer other = (Consumer) obj;
        return Objects.equals(consumerName(), other.consumerName()) && Objects.equals(consumerARN(), other.consumerARN())
                && Objects.equals(consumerStatusAsString(), other.consumerStatusAsString())
                && Objects.equals(consumerCreationTimestamp(), other.consumerCreationTimestamp());
    }

    @Override
    public String toString() {
        return ToString.builder("Consumer").add("ConsumerName", consumerName()).add("ConsumerARN", consumerARN())
                .add("ConsumerStatus", consumerStatusAsString()).add("ConsumerCreationTimestamp", consumerCreationTimestamp())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConsumerName":
            return Optional.ofNullable(clazz.cast(consumerName()));
        case "ConsumerARN":
            return Optional.ofNullable(clazz.cast(consumerARN()));
        case "ConsumerStatus":
            return Optional.ofNullable(clazz.cast(consumerStatusAsString()));
        case "ConsumerCreationTimestamp":
            return Optional.ofNullable(clazz.cast(consumerCreationTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ConsumerMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Consumer> {
        /**
         * <p>
         * The name of the consumer is something you choose when you register the consumer.
         * </p>
         * 
         * @param consumerName
         *        The name of the consumer is something you choose when you register the consumer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumerName(String consumerName);

        /**
         * <p>
         * When you register a consumer, Kinesis Data Streams generates an ARN for it. You need this ARN to be able to
         * call <a>SubscribeToShard</a>.
         * </p>
         * <p>
         * If you delete a consumer and then create a new one with the same name, it won't have the same ARN. That's
         * because consumer ARNs contain the creation timestamp. This is important to keep in mind if you have IAM
         * policies that reference consumer ARNs.
         * </p>
         * 
         * @param consumerARN
         *        When you register a consumer, Kinesis Data Streams generates an ARN for it. You need this ARN to be
         *        able to call <a>SubscribeToShard</a>.</p>
         *        <p>
         *        If you delete a consumer and then create a new one with the same name, it won't have the same ARN.
         *        That's because consumer ARNs contain the creation timestamp. This is important to keep in mind if you
         *        have IAM policies that reference consumer ARNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumerARN(String consumerARN);

        /**
         * <p>
         * A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
         * </p>
         * 
         * @param consumerStatus
         *        A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
         * @see ConsumerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConsumerStatus
         */
        Builder consumerStatus(String consumerStatus);

        /**
         * <p>
         * A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
         * </p>
         * 
         * @param consumerStatus
         *        A consumer can't read data while in the <code>CREATING</code> or <code>DELETING</code> states.
         * @see ConsumerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConsumerStatus
         */
        Builder consumerStatus(ConsumerStatus consumerStatus);

        /**
         * <p/>
         * 
         * @param consumerCreationTimestamp
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumerCreationTimestamp(Instant consumerCreationTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String consumerName;

        private String consumerARN;

        private String consumerStatus;

        private Instant consumerCreationTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(Consumer model) {
            consumerName(model.consumerName);
            consumerARN(model.consumerARN);
            consumerStatus(model.consumerStatus);
            consumerCreationTimestamp(model.consumerCreationTimestamp);
        }

        public final String getConsumerName() {
            return consumerName;
        }

        @Override
        public final Builder consumerName(String consumerName) {
            this.consumerName = consumerName;
            return this;
        }

        public final void setConsumerName(String consumerName) {
            this.consumerName = consumerName;
        }

        public final String getConsumerARN() {
            return consumerARN;
        }

        @Override
        public final Builder consumerARN(String consumerARN) {
            this.consumerARN = consumerARN;
            return this;
        }

        public final void setConsumerARN(String consumerARN) {
            this.consumerARN = consumerARN;
        }

        public final String getConsumerStatus() {
            return consumerStatus;
        }

        @Override
        public final Builder consumerStatus(String consumerStatus) {
            this.consumerStatus = consumerStatus;
            return this;
        }

        @Override
        public final Builder consumerStatus(ConsumerStatus consumerStatus) {
            this.consumerStatus(consumerStatus.toString());
            return this;
        }

        public final void setConsumerStatus(String consumerStatus) {
            this.consumerStatus = consumerStatus;
        }

        public final Instant getConsumerCreationTimestamp() {
            return consumerCreationTimestamp;
        }

        @Override
        public final Builder consumerCreationTimestamp(Instant consumerCreationTimestamp) {
            this.consumerCreationTimestamp = consumerCreationTimestamp;
            return this;
        }

        public final void setConsumerCreationTimestamp(Instant consumerCreationTimestamp) {
            this.consumerCreationTimestamp = consumerCreationTimestamp;
        }

        @Override
        public Consumer build() {
            return new Consumer(this);
        }
    }
}
