/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListShardsResponse extends KinesisResponse implements
        ToCopyableBuilder<ListShardsResponse.Builder, ListShardsResponse> {
    private static final SdkField<List<Shard>> SHARDS_FIELD = SdkField
            .<List<Shard>> builder(MarshallingType.LIST)
            .memberName("Shards")
            .getter(getter(ListShardsResponse::shards))
            .setter(setter(Builder::shards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Shards").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Shard> builder(MarshallingType.SDK_POJO)
                                            .constructor(Shard::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextToken").getter(getter(ListShardsResponse::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(SHARDS_FIELD, NEXT_TOKEN_FIELD));

    private final List<Shard> shards;

    private final String nextToken;

    private ListShardsResponse(BuilderImpl builder) {
        super(builder);
        this.shards = builder.shards;
        this.nextToken = builder.nextToken;
    }

    /**
     * Returns true if the Shards property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasShards() {
        return shards != null && !(shards instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the shard's
     * parent, and the shard that's adjacent to the shard's parent. Each object also contains the starting and ending
     * hash keys and the starting and ending sequence numbers for the shard.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasShards()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the
     *         shard's parent, and the shard that's adjacent to the shard's parent. Each object also contains the
     *         starting and ending hash keys and the starting and ending sequence numbers for the shard.
     */
    public List<Shard> shards() {
        return shards;
    }

    /**
     * <p>
     * When the number of shards in the data stream is greater than the default value for the <code>MaxResults</code>
     * parameter, or if you explicitly specify a value for <code>MaxResults</code> that is less than the number of
     * shards in the data stream, the response includes a pagination token named <code>NextToken</code>. You can specify
     * this <code>NextToken</code> value in a subsequent call to <code>ListShards</code> to list the next set of shards.
     * For more information about the use of this pagination token when calling the <code>ListShards</code> operation,
     * see <a>ListShardsInput$NextToken</a>.
     * </p>
     * <important>
     * <p>
     * Tokens expire after 300 seconds. When you obtain a value for <code>NextToken</code> in the response to a call to
     * <code>ListShards</code>, you have 300 seconds to use that value. If you specify an expired token in a call to
     * <code>ListShards</code>, you get <code>ExpiredNextTokenException</code>.
     * </p>
     * </important>
     * 
     * @return When the number of shards in the data stream is greater than the default value for the
     *         <code>MaxResults</code> parameter, or if you explicitly specify a value for <code>MaxResults</code> that
     *         is less than the number of shards in the data stream, the response includes a pagination token named
     *         <code>NextToken</code>. You can specify this <code>NextToken</code> value in a subsequent call to
     *         <code>ListShards</code> to list the next set of shards. For more information about the use of this
     *         pagination token when calling the <code>ListShards</code> operation, see
     *         <a>ListShardsInput$NextToken</a>.</p> <important>
     *         <p>
     *         Tokens expire after 300 seconds. When you obtain a value for <code>NextToken</code> in the response to a
     *         call to <code>ListShards</code>, you have 300 seconds to use that value. If you specify an expired token
     *         in a call to <code>ListShards</code>, you get <code>ExpiredNextTokenException</code>.
     *         </p>
     */
    public String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(hasShards() ? shards() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListShardsResponse)) {
            return false;
        }
        ListShardsResponse other = (ListShardsResponse) obj;
        return hasShards() == other.hasShards() && Objects.equals(shards(), other.shards())
                && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ListShardsResponse").add("Shards", hasShards() ? shards() : null).add("NextToken", nextToken())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Shards":
            return Optional.ofNullable(clazz.cast(shards()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ListShardsResponse, T> g) {
        return obj -> g.apply((ListShardsResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisResponse.Builder, SdkPojo, CopyableBuilder<Builder, ListShardsResponse> {
        /**
         * <p>
         * An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the shard's
         * parent, and the shard that's adjacent to the shard's parent. Each object also contains the starting and
         * ending hash keys and the starting and ending sequence numbers for the shard.
         * </p>
         * 
         * @param shards
         *        An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the
         *        shard's parent, and the shard that's adjacent to the shard's parent. Each object also contains the
         *        starting and ending hash keys and the starting and ending sequence numbers for the shard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shards(Collection<Shard> shards);

        /**
         * <p>
         * An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the shard's
         * parent, and the shard that's adjacent to the shard's parent. Each object also contains the starting and
         * ending hash keys and the starting and ending sequence numbers for the shard.
         * </p>
         * 
         * @param shards
         *        An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the
         *        shard's parent, and the shard that's adjacent to the shard's parent. Each object also contains the
         *        starting and ending hash keys and the starting and ending sequence numbers for the shard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shards(Shard... shards);

        /**
         * <p>
         * An array of JSON objects. Each object represents one shard and specifies the IDs of the shard, the shard's
         * parent, and the shard that's adjacent to the shard's parent. Each object also contains the starting and
         * ending hash keys and the starting and ending sequence numbers for the shard.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Shard>.Builder} avoiding the need to create
         * one manually via {@link List<Shard>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Shard>.Builder#build()} is called immediately and its result
         * is passed to {@link #shards(List<Shard>)}.
         * 
         * @param shards
         *        a consumer that will call methods on {@link List<Shard>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shards(List<Shard>)
         */
        Builder shards(Consumer<Shard.Builder>... shards);

        /**
         * <p>
         * When the number of shards in the data stream is greater than the default value for the
         * <code>MaxResults</code> parameter, or if you explicitly specify a value for <code>MaxResults</code> that is
         * less than the number of shards in the data stream, the response includes a pagination token named
         * <code>NextToken</code>. You can specify this <code>NextToken</code> value in a subsequent call to
         * <code>ListShards</code> to list the next set of shards. For more information about the use of this pagination
         * token when calling the <code>ListShards</code> operation, see <a>ListShardsInput$NextToken</a>.
         * </p>
         * <important>
         * <p>
         * Tokens expire after 300 seconds. When you obtain a value for <code>NextToken</code> in the response to a call
         * to <code>ListShards</code>, you have 300 seconds to use that value. If you specify an expired token in a call
         * to <code>ListShards</code>, you get <code>ExpiredNextTokenException</code>.
         * </p>
         * </important>
         * 
         * @param nextToken
         *        When the number of shards in the data stream is greater than the default value for the
         *        <code>MaxResults</code> parameter, or if you explicitly specify a value for <code>MaxResults</code>
         *        that is less than the number of shards in the data stream, the response includes a pagination token
         *        named <code>NextToken</code>. You can specify this <code>NextToken</code> value in a subsequent call
         *        to <code>ListShards</code> to list the next set of shards. For more information about the use of this
         *        pagination token when calling the <code>ListShards</code> operation, see
         *        <a>ListShardsInput$NextToken</a>.</p> <important>
         *        <p>
         *        Tokens expire after 300 seconds. When you obtain a value for <code>NextToken</code> in the response to
         *        a call to <code>ListShards</code>, you have 300 seconds to use that value. If you specify an expired
         *        token in a call to <code>ListShards</code>, you get <code>ExpiredNextTokenException</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);
    }

    static final class BuilderImpl extends KinesisResponse.BuilderImpl implements Builder {
        private List<Shard> shards = DefaultSdkAutoConstructList.getInstance();

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(ListShardsResponse model) {
            super(model);
            shards(model.shards);
            nextToken(model.nextToken);
        }

        public final Collection<Shard.Builder> getShards() {
            if (shards instanceof SdkAutoConstructList) {
                return null;
            }
            return shards != null ? shards.stream().map(Shard::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder shards(Collection<Shard> shards) {
            this.shards = ShardListCopier.copy(shards);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder shards(Shard... shards) {
            shards(Arrays.asList(shards));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder shards(Consumer<Shard.Builder>... shards) {
            shards(Stream.of(shards).map(c -> Shard.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setShards(Collection<Shard.BuilderImpl> shards) {
            this.shards = ShardListCopier.copyFromBuilder(shards);
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public ListShardsResponse build() {
            return new ListShardsResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
