/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.endpoints.internal;

import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.kinesis.endpoints.KinesisEndpointParams;
import software.amazon.awssdk.services.kinesis.endpoints.KinesisEndpointProvider;
import software.amazon.awssdk.services.kinesis.model.AddTagsToStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DecreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DeregisterStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamSummaryRequest;
import software.amazon.awssdk.services.kinesis.model.DisableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.EnableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.GetRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.kinesis.model.IncreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.ListShardsRequest;
import software.amazon.awssdk.services.kinesis.model.ListStreamConsumersRequest;
import software.amazon.awssdk.services.kinesis.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesis.model.MergeShardsRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.RegisterStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.RemoveTagsFromStreamRequest;
import software.amazon.awssdk.services.kinesis.model.SplitShardRequest;
import software.amazon.awssdk.services.kinesis.model.StartStreamEncryptionRequest;
import software.amazon.awssdk.services.kinesis.model.StopStreamEncryptionRequest;
import software.amazon.awssdk.services.kinesis.model.SubscribeToShardRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateShardCountRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateStreamModeRequest;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class KinesisResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return context.request();
        }
        KinesisEndpointProvider provider = (KinesisEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            Endpoint result = provider.resolveEndpoint(ruleParams(context, executionAttributes)).join();
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        context.request());
                if (hostPrefix.isPresent()) {
                    result = AwsEndpointProviderUtils.addHostPrefix(result, hostPrefix.get());
                }
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, result);
            return context.request();
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    private static KinesisEndpointParams ruleParams(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        KinesisEndpointParams.Builder builder = KinesisEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), context.request());
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        return builder.build();
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, String operationName, SdkRequest request) {
        switch (operationName) {
        case "AddTagsToStream":
            setContextParams(params, (AddTagsToStreamRequest) request);
            break;
        case "DecreaseStreamRetentionPeriod":
            setContextParams(params, (DecreaseStreamRetentionPeriodRequest) request);
            break;
        case "DeleteStream":
            setContextParams(params, (DeleteStreamRequest) request);
            break;
        case "DeregisterStreamConsumer":
            setContextParams(params, (DeregisterStreamConsumerRequest) request);
            break;
        case "DescribeStream":
            setContextParams(params, (DescribeStreamRequest) request);
            break;
        case "DescribeStreamConsumer":
            setContextParams(params, (DescribeStreamConsumerRequest) request);
            break;
        case "DescribeStreamSummary":
            setContextParams(params, (DescribeStreamSummaryRequest) request);
            break;
        case "DisableEnhancedMonitoring":
            setContextParams(params, (DisableEnhancedMonitoringRequest) request);
            break;
        case "EnableEnhancedMonitoring":
            setContextParams(params, (EnableEnhancedMonitoringRequest) request);
            break;
        case "GetRecords":
            setContextParams(params, (GetRecordsRequest) request);
            break;
        case "GetShardIterator":
            setContextParams(params, (GetShardIteratorRequest) request);
            break;
        case "IncreaseStreamRetentionPeriod":
            setContextParams(params, (IncreaseStreamRetentionPeriodRequest) request);
            break;
        case "ListShards":
            setContextParams(params, (ListShardsRequest) request);
            break;
        case "ListStreamConsumers":
            setContextParams(params, (ListStreamConsumersRequest) request);
            break;
        case "ListTagsForStream":
            setContextParams(params, (ListTagsForStreamRequest) request);
            break;
        case "MergeShards":
            setContextParams(params, (MergeShardsRequest) request);
            break;
        case "PutRecord":
            setContextParams(params, (PutRecordRequest) request);
            break;
        case "PutRecords":
            setContextParams(params, (PutRecordsRequest) request);
            break;
        case "RegisterStreamConsumer":
            setContextParams(params, (RegisterStreamConsumerRequest) request);
            break;
        case "RemoveTagsFromStream":
            setContextParams(params, (RemoveTagsFromStreamRequest) request);
            break;
        case "SplitShard":
            setContextParams(params, (SplitShardRequest) request);
            break;
        case "StartStreamEncryption":
            setContextParams(params, (StartStreamEncryptionRequest) request);
            break;
        case "StopStreamEncryption":
            setContextParams(params, (StopStreamEncryptionRequest) request);
            break;
        case "SubscribeToShard":
            setContextParams(params, (SubscribeToShardRequest) request);
            break;
        case "UpdateShardCount":
            setContextParams(params, (UpdateShardCountRequest) request);
            break;
        case "UpdateStreamMode":
            setContextParams(params, (UpdateStreamModeRequest) request);
            break;
        default:
            break;
        }
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, AddTagsToStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DecreaseStreamRetentionPeriodRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DeleteStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DeregisterStreamConsumerRequest request) {
        params.streamArn(request.streamARN());
        params.consumerArn(request.consumerARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DescribeStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DescribeStreamConsumerRequest request) {
        params.streamArn(request.streamARN());
        params.consumerArn(request.consumerARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DescribeStreamSummaryRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DisableEnhancedMonitoringRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, EnableEnhancedMonitoringRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, GetRecordsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, GetShardIteratorRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, IncreaseStreamRetentionPeriodRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, ListShardsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, ListStreamConsumersRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, ListTagsForStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, MergeShardsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, PutRecordRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, PutRecordsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, RegisterStreamConsumerRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, RemoveTagsFromStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, SplitShardRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, StartStreamEncryptionRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, StopStreamEncryptionRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, SubscribeToShardRequest request) {
        params.consumerArn(request.consumerARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, UpdateShardCountRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, UpdateStreamModeRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setStaticContextParams(KinesisEndpointParams.Builder params, String operationName) {
        switch (operationName) {
        case "AddTagsToStream":
            addTagsToStreamStaticContextParams(params);
            break;
        case "DecreaseStreamRetentionPeriod":
            decreaseStreamRetentionPeriodStaticContextParams(params);
            break;
        case "DeleteStream":
            deleteStreamStaticContextParams(params);
            break;
        case "DeregisterStreamConsumer":
            deregisterStreamConsumerStaticContextParams(params);
            break;
        case "DescribeStream":
            describeStreamStaticContextParams(params);
            break;
        case "DescribeStreamConsumer":
            describeStreamConsumerStaticContextParams(params);
            break;
        case "DescribeStreamSummary":
            describeStreamSummaryStaticContextParams(params);
            break;
        case "DisableEnhancedMonitoring":
            disableEnhancedMonitoringStaticContextParams(params);
            break;
        case "EnableEnhancedMonitoring":
            enableEnhancedMonitoringStaticContextParams(params);
            break;
        case "GetRecords":
            getRecordsStaticContextParams(params);
            break;
        case "GetShardIterator":
            getShardIteratorStaticContextParams(params);
            break;
        case "IncreaseStreamRetentionPeriod":
            increaseStreamRetentionPeriodStaticContextParams(params);
            break;
        case "ListShards":
            listShardsStaticContextParams(params);
            break;
        case "ListStreamConsumers":
            listStreamConsumersStaticContextParams(params);
            break;
        case "ListTagsForStream":
            listTagsForStreamStaticContextParams(params);
            break;
        case "MergeShards":
            mergeShardsStaticContextParams(params);
            break;
        case "PutRecord":
            putRecordStaticContextParams(params);
            break;
        case "PutRecords":
            putRecordsStaticContextParams(params);
            break;
        case "RegisterStreamConsumer":
            registerStreamConsumerStaticContextParams(params);
            break;
        case "RemoveTagsFromStream":
            removeTagsFromStreamStaticContextParams(params);
            break;
        case "SplitShard":
            splitShardStaticContextParams(params);
            break;
        case "StartStreamEncryption":
            startStreamEncryptionStaticContextParams(params);
            break;
        case "StopStreamEncryption":
            stopStreamEncryptionStaticContextParams(params);
            break;
        case "SubscribeToShard":
            subscribeToShardStaticContextParams(params);
            break;
        case "UpdateShardCount":
            updateShardCountStaticContextParams(params);
            break;
        case "UpdateStreamMode":
            updateStreamModeStaticContextParams(params);
            break;
        default:
            break;
        }
    }

    private static void addTagsToStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void decreaseStreamRetentionPeriodStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void deleteStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void deregisterStreamConsumerStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void describeStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void describeStreamConsumerStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void describeStreamSummaryStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void disableEnhancedMonitoringStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void enableEnhancedMonitoringStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void getRecordsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void getShardIteratorStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void increaseStreamRetentionPeriodStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void listShardsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void listStreamConsumersStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void listTagsForStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void mergeShardsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void putRecordStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void putRecordsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void registerStreamConsumerStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void removeTagsFromStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void splitShardStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void startStreamEncryptionStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void stopStreamEncryptionStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void subscribeToShardStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void updateShardCountStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void updateStreamModeStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        return Optional.empty();
    }
}
