/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.cbor.AwsCborProtocolFactory;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesis.internal.KinesisServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.kinesis.model.AccessDeniedException;
import software.amazon.awssdk.services.kinesis.model.AddTagsToStreamRequest;
import software.amazon.awssdk.services.kinesis.model.AddTagsToStreamResponse;
import software.amazon.awssdk.services.kinesis.model.CreateStreamRequest;
import software.amazon.awssdk.services.kinesis.model.CreateStreamResponse;
import software.amazon.awssdk.services.kinesis.model.DecreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.DecreaseStreamRetentionPeriodResponse;
import software.amazon.awssdk.services.kinesis.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DeleteStreamResponse;
import software.amazon.awssdk.services.kinesis.model.DeregisterStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.DeregisterStreamConsumerResponse;
import software.amazon.awssdk.services.kinesis.model.DescribeLimitsRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeLimitsResponse;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamConsumerResponse;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamResponse;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamSummaryRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamSummaryResponse;
import software.amazon.awssdk.services.kinesis.model.DisableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.DisableEnhancedMonitoringResponse;
import software.amazon.awssdk.services.kinesis.model.EnableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.EnableEnhancedMonitoringResponse;
import software.amazon.awssdk.services.kinesis.model.ExpiredIteratorException;
import software.amazon.awssdk.services.kinesis.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.kinesis.model.GetRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.GetRecordsResponse;
import software.amazon.awssdk.services.kinesis.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.kinesis.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.kinesis.model.IncreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.IncreaseStreamRetentionPeriodResponse;
import software.amazon.awssdk.services.kinesis.model.InternalFailureException;
import software.amazon.awssdk.services.kinesis.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesis.model.KinesisException;
import software.amazon.awssdk.services.kinesis.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.kinesis.model.KmsDisabledException;
import software.amazon.awssdk.services.kinesis.model.KmsInvalidStateException;
import software.amazon.awssdk.services.kinesis.model.KmsNotFoundException;
import software.amazon.awssdk.services.kinesis.model.KmsOptInRequiredException;
import software.amazon.awssdk.services.kinesis.model.KmsThrottlingException;
import software.amazon.awssdk.services.kinesis.model.LimitExceededException;
import software.amazon.awssdk.services.kinesis.model.ListShardsRequest;
import software.amazon.awssdk.services.kinesis.model.ListShardsResponse;
import software.amazon.awssdk.services.kinesis.model.ListStreamConsumersRequest;
import software.amazon.awssdk.services.kinesis.model.ListStreamConsumersResponse;
import software.amazon.awssdk.services.kinesis.model.ListStreamsRequest;
import software.amazon.awssdk.services.kinesis.model.ListStreamsResponse;
import software.amazon.awssdk.services.kinesis.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesis.model.ListTagsForStreamResponse;
import software.amazon.awssdk.services.kinesis.model.MergeShardsRequest;
import software.amazon.awssdk.services.kinesis.model.MergeShardsResponse;
import software.amazon.awssdk.services.kinesis.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.kinesis.model.PutRecordRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordResponse;
import software.amazon.awssdk.services.kinesis.model.PutRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordsResponse;
import software.amazon.awssdk.services.kinesis.model.RegisterStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.RegisterStreamConsumerResponse;
import software.amazon.awssdk.services.kinesis.model.RemoveTagsFromStreamRequest;
import software.amazon.awssdk.services.kinesis.model.RemoveTagsFromStreamResponse;
import software.amazon.awssdk.services.kinesis.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesis.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesis.model.SplitShardRequest;
import software.amazon.awssdk.services.kinesis.model.SplitShardResponse;
import software.amazon.awssdk.services.kinesis.model.StartStreamEncryptionRequest;
import software.amazon.awssdk.services.kinesis.model.StartStreamEncryptionResponse;
import software.amazon.awssdk.services.kinesis.model.StopStreamEncryptionRequest;
import software.amazon.awssdk.services.kinesis.model.StopStreamEncryptionResponse;
import software.amazon.awssdk.services.kinesis.model.UpdateShardCountRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateShardCountResponse;
import software.amazon.awssdk.services.kinesis.model.UpdateStreamModeRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateStreamModeResponse;
import software.amazon.awssdk.services.kinesis.model.ValidationException;
import software.amazon.awssdk.services.kinesis.transform.AddTagsToStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.CreateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DecreaseStreamRetentionPeriodRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DeleteStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DeregisterStreamConsumerRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeLimitsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeStreamConsumerRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DescribeStreamSummaryRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.DisableEnhancedMonitoringRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.EnableEnhancedMonitoringRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.GetRecordsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.GetShardIteratorRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.IncreaseStreamRetentionPeriodRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListShardsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListStreamConsumersRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.ListTagsForStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.MergeShardsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.PutRecordRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.PutRecordsRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.RegisterStreamConsumerRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.RemoveTagsFromStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.SplitShardRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.StartStreamEncryptionRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.StopStreamEncryptionRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.UpdateShardCountRequestMarshaller;
import software.amazon.awssdk.services.kinesis.transform.UpdateStreamModeRequestMarshaller;
import software.amazon.awssdk.services.kinesis.waiters.KinesisWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link KinesisClient}.
 *
 * @see KinesisClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisClient implements KinesisClient {
    private static final Logger log = Logger.loggerFor(DefaultKinesisClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsCborProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final KinesisServiceClientConfiguration serviceClientConfiguration;

    protected DefaultKinesisClient(KinesisServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsCborProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds or updates tags for the specified Kinesis data stream. You can assign up to 50 tags to a data stream.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * If tags have already been assigned to the stream, <code>AddTagsToStream</code> overwrites any existing tags that
     * correspond to the specified tag keys.
     * </p>
     * <p>
     * <a>AddTagsToStream</a> has a limit of five transactions per second per account.
     * </p>
     *
     * @param addTagsToStreamRequest
     *        Represents the input for <code>AddTagsToStream</code>.
     * @return Result of the AddTagsToStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.AddTagsToStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/AddTagsToStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddTagsToStreamResponse addTagsToStream(AddTagsToStreamRequest addTagsToStreamRequest)
            throws ResourceNotFoundException, ResourceInUseException, InvalidArgumentException, LimitExceededException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsToStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsToStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addTagsToStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsToStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTagsToStream");

            return clientHandler.execute(new ClientExecutionParams<AddTagsToStreamRequest, AddTagsToStreamResponse>()
                    .withOperationName("AddTagsToStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(addTagsToStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddTagsToStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Kinesis data stream. A stream captures and transports data records that are continuously emitted from
     * different data sources or <i>producers</i>. Scale-out within a stream is explicitly supported by means of shards,
     * which are uniquely identified groups of data records in a stream.
     * </p>
     * <p>
     * You can create your data stream using either on-demand or provisioned capacity mode. Data streams with an
     * on-demand mode require no capacity planning and automatically scale to handle gigabytes of write and read
     * throughput per minute. With the on-demand mode, Kinesis Data Streams automatically manages the shards in order to
     * provide the necessary throughput. For the data streams with a provisioned mode, you must specify the number of
     * shards for the data stream. Each shard can support reads up to five transactions per second, up to a maximum data
     * read total of 2 MiB per second. Each shard can support writes up to 1,000 records per second, up to a maximum
     * data write total of 1 MiB per second. If the amount of data input increases or decreases, you can add or remove
     * shards.
     * </p>
     * <p>
     * The stream name identifies the stream. The name is scoped to the Amazon Web Services account used by the
     * application. It is also scoped by Amazon Web Services Region. That is, two streams in two different accounts can
     * have the same name, and two streams in the same account, but in two different Regions, can have the same name.
     * </p>
     * <p>
     * <code>CreateStream</code> is an asynchronous operation. Upon receiving a <code>CreateStream</code> request,
     * Kinesis Data Streams immediately returns and sets the stream status to <code>CREATING</code>. After the stream is
     * created, Kinesis Data Streams sets the stream status to <code>ACTIVE</code>. You should perform read and write
     * operations only on an <code>ACTIVE</code> stream.
     * </p>
     * <p>
     * You receive a <code>LimitExceededException</code> when making a <code>CreateStream</code> request when you try to
     * do one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Have more than five streams in the <code>CREATING</code> state at any point in time.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create more shards than are authorized for your account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the default shard limit for an Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Amazon Kinesis Data Streams
     * Limits</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>. To increase this limit, <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">contact Amazon Web Services
     * Support</a>.
     * </p>
     * <p>
     * You can use <a>DescribeStreamSummary</a> to check the stream status, which is returned in
     * <code>StreamStatus</code>.
     * </p>
     * <p>
     * <a>CreateStream</a> has a limit of five transactions per second per account.
     * </p>
     *
     * @param createStreamRequest
     *        Represents the input for <code>CreateStream</code>.
     * @return Result of the CreateStream operation returned by the service.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.CreateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/CreateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStreamResponse createStream(CreateStreamRequest createStreamRequest) throws ResourceInUseException,
            LimitExceededException, InvalidArgumentException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStream");

            return clientHandler.execute(new ClientExecutionParams<CreateStreamRequest, CreateStreamResponse>()
                    .withOperationName("CreateStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Decreases the Kinesis data stream's retention period, which is the length of time data records are accessible
     * after they are added to the stream. The minimum value of a stream's retention period is 24 hours.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * This operation may result in lost data. For example, if the stream's retention period is 48 hours and is
     * decreased to 24 hours, any data already in the stream that is older than 24 hours is inaccessible.
     * </p>
     *
     * @param decreaseStreamRetentionPeriodRequest
     *        Represents the input for <a>DecreaseStreamRetentionPeriod</a>.
     * @return Result of the DecreaseStreamRetentionPeriod operation returned by the service.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DecreaseStreamRetentionPeriod
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DecreaseStreamRetentionPeriod"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DecreaseStreamRetentionPeriodResponse decreaseStreamRetentionPeriod(
            DecreaseStreamRetentionPeriodRequest decreaseStreamRetentionPeriodRequest) throws ResourceInUseException,
            ResourceNotFoundException, LimitExceededException, InvalidArgumentException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DecreaseStreamRetentionPeriodResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DecreaseStreamRetentionPeriodResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(decreaseStreamRetentionPeriodRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                decreaseStreamRetentionPeriodRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DecreaseStreamRetentionPeriod");

            return clientHandler
                    .execute(new ClientExecutionParams<DecreaseStreamRetentionPeriodRequest, DecreaseStreamRetentionPeriodResponse>()
                            .withOperationName("DecreaseStreamRetentionPeriod").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(decreaseStreamRetentionPeriodRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DecreaseStreamRetentionPeriodRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Kinesis data stream and all its shards and data. You must shut down any applications that are operating
     * on the stream before you delete the stream. If an application attempts to operate on a deleted stream, it
     * receives the exception <code>ResourceNotFoundException</code>.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * If the stream is in the <code>ACTIVE</code> state, you can delete it. After a <code>DeleteStream</code> request,
     * the specified stream is in the <code>DELETING</code> state until Kinesis Data Streams completes the deletion.
     * </p>
     * <p>
     * <b>Note:</b> Kinesis Data Streams might continue to accept data read and write operations, such as
     * <a>PutRecord</a>, <a>PutRecords</a>, and <a>GetRecords</a>, on a stream in the <code>DELETING</code> state until
     * the stream deletion is complete.
     * </p>
     * <p>
     * When you delete a stream, any shards in that stream are also deleted, and any tags are dissociated from the
     * stream.
     * </p>
     * <p>
     * You can use the <a>DescribeStreamSummary</a> operation to check the state of the stream, which is returned in
     * <code>StreamStatus</code>.
     * </p>
     * <p>
     * <a>DeleteStream</a> has a limit of five transactions per second per account.
     * </p>
     *
     * @param deleteStreamRequest
     *        Represents the input for <a>DeleteStream</a>.
     * @return Result of the DeleteStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DeleteStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DeleteStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStreamResponse deleteStream(DeleteStreamRequest deleteStreamRequest) throws ResourceNotFoundException,
            LimitExceededException, ResourceInUseException, InvalidArgumentException, AccessDeniedException, AwsServiceException,
            SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStream");

            return clientHandler.execute(new ClientExecutionParams<DeleteStreamRequest, DeleteStreamResponse>()
                    .withOperationName("DeleteStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To deregister a consumer, provide its ARN. Alternatively, you can provide the ARN of the data stream and the name
     * you gave the consumer when you registered it. You may also provide all three parameters, as long as they don't
     * conflict with each other. If you don't know the name or ARN of the consumer that you want to deregister, you can
     * use the <a>ListStreamConsumers</a> operation to get a list of the descriptions of all the consumers that are
     * currently registered with a given data stream. The description of a consumer contains its name and ARN.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second per stream.
     * </p>
     *
     * @param deregisterStreamConsumerRequest
     * @return Result of the DeregisterStreamConsumer operation returned by the service.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DeregisterStreamConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DeregisterStreamConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterStreamConsumerResponse deregisterStreamConsumer(
            DeregisterStreamConsumerRequest deregisterStreamConsumerRequest) throws LimitExceededException,
            ResourceNotFoundException, InvalidArgumentException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterStreamConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterStreamConsumerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterStreamConsumerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterStreamConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterStreamConsumer");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterStreamConsumerRequest, DeregisterStreamConsumerResponse>()
                            .withOperationName("DeregisterStreamConsumer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(deregisterStreamConsumerRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterStreamConsumerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the shard limits and usage for the account.
     * </p>
     * <p>
     * If you update your account limits, the old limits might be returned for a few minutes.
     * </p>
     * <p>
     * This operation has a limit of one transaction per second per account.
     * </p>
     *
     * @param describeLimitsRequest
     * @return Result of the DescribeLimits operation returned by the service.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DescribeLimits
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DescribeLimits" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeLimitsResponse describeLimits(DescribeLimitsRequest describeLimitsRequest) throws LimitExceededException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLimitsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeLimitsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLimitsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLimits");

            return clientHandler.execute(new ClientExecutionParams<DescribeLimitsRequest, DescribeLimitsResponse>()
                    .withOperationName("DescribeLimits").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeLimitsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLimitsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified Kinesis data stream.
     * </p>
     * <note>
     * <p>
     * This API has been revised. It's highly recommended that you use the <a>DescribeStreamSummary</a> API to get a
     * summarized description of the specified Kinesis data stream and the <a>ListShards</a> API to list the shards in a
     * specified data stream and obtain information about each shard.
     * </p>
     * </note> <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * The information returned includes the stream name, Amazon Resource Name (ARN), creation time, enhanced metric
     * configuration, and shard map. The shard map is an array of shard objects. For each shard object, there is the
     * hash key and sequence number ranges that the shard spans, and the IDs of any earlier shards that played in a role
     * in creating the shard. Every record ingested in the stream is identified by a sequence number, which is assigned
     * when the record is put into the stream.
     * </p>
     * <p>
     * You can limit the number of shards returned by each call. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-retrieve-shards.html">Retrieving
     * Shards from a Stream</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * There are no guarantees about the chronological order shards returned. To process shards in chronological order,
     * use the ID of the parent shard to track the lineage to the oldest shard.
     * </p>
     * <p>
     * This operation has a limit of 10 transactions per second per account.
     * </p>
     *
     * @param describeStreamRequest
     *        Represents the input for <code>DescribeStream</code>.
     * @return Result of the DescribeStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DescribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DescribeStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeStreamResponse describeStream(DescribeStreamRequest describeStreamRequest) throws ResourceNotFoundException,
            LimitExceededException, InvalidArgumentException, AccessDeniedException, AwsServiceException, SdkClientException,
            KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStream");

            return clientHandler.execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                    .withOperationName("DescribeStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To get the description of a registered consumer, provide the ARN of the consumer. Alternatively, you can provide
     * the ARN of the data stream and the name you gave the consumer when you registered it. You may also provide all
     * three parameters, as long as they don't conflict with each other. If you don't know the name or ARN of the
     * consumer that you want to describe, you can use the <a>ListStreamConsumers</a> operation to get a list of the
     * descriptions of all the consumers that are currently registered with a given data stream.
     * </p>
     * <p>
     * This operation has a limit of 20 transactions per second per stream.
     * </p>
     *
     * @param describeStreamConsumerRequest
     * @return Result of the DescribeStreamConsumer operation returned by the service.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DescribeStreamConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DescribeStreamConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStreamConsumerResponse describeStreamConsumer(DescribeStreamConsumerRequest describeStreamConsumerRequest)
            throws LimitExceededException, ResourceNotFoundException, InvalidArgumentException, AwsServiceException,
            SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStreamConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStreamConsumerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStreamConsumerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStreamConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStreamConsumer");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeStreamConsumerRequest, DescribeStreamConsumerResponse>()
                            .withOperationName("DescribeStreamConsumer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(describeStreamConsumerRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeStreamConsumerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a summarized description of the specified Kinesis data stream without the shard list.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * The information returned includes the stream name, Amazon Resource Name (ARN), status, record retention period,
     * approximate creation time, monitoring, encryption details, and open shard count.
     * </p>
     * <p>
     * <a>DescribeStreamSummary</a> has a limit of 20 transactions per second per account.
     * </p>
     *
     * @param describeStreamSummaryRequest
     * @return Result of the DescribeStreamSummary operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DescribeStreamSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DescribeStreamSummary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeStreamSummaryResponse describeStreamSummary(DescribeStreamSummaryRequest describeStreamSummaryRequest)
            throws ResourceNotFoundException, LimitExceededException, InvalidArgumentException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStreamSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStreamSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStreamSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStreamSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStreamSummary");

            return clientHandler.execute(new ClientExecutionParams<DescribeStreamSummaryRequest, DescribeStreamSummaryResponse>()
                    .withOperationName("DescribeStreamSummary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeStreamSummaryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStreamSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables enhanced monitoring.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     *
     * @param disableEnhancedMonitoringRequest
     *        Represents the input for <a>DisableEnhancedMonitoring</a>.
     * @return Result of the DisableEnhancedMonitoring operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.DisableEnhancedMonitoring
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/DisableEnhancedMonitoring"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableEnhancedMonitoringResponse disableEnhancedMonitoring(
            DisableEnhancedMonitoringRequest disableEnhancedMonitoringRequest) throws InvalidArgumentException,
            LimitExceededException, ResourceInUseException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableEnhancedMonitoringResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableEnhancedMonitoringResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableEnhancedMonitoringRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableEnhancedMonitoringRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableEnhancedMonitoring");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableEnhancedMonitoringRequest, DisableEnhancedMonitoringResponse>()
                            .withOperationName("DisableEnhancedMonitoring").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(disableEnhancedMonitoringRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableEnhancedMonitoringRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables enhanced Kinesis data stream monitoring for shard-level metrics.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     *
     * @param enableEnhancedMonitoringRequest
     *        Represents the input for <a>EnableEnhancedMonitoring</a>.
     * @return Result of the EnableEnhancedMonitoring operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.EnableEnhancedMonitoring
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/EnableEnhancedMonitoring"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableEnhancedMonitoringResponse enableEnhancedMonitoring(
            EnableEnhancedMonitoringRequest enableEnhancedMonitoringRequest) throws InvalidArgumentException,
            LimitExceededException, ResourceInUseException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableEnhancedMonitoringResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableEnhancedMonitoringResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableEnhancedMonitoringRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableEnhancedMonitoringRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableEnhancedMonitoring");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableEnhancedMonitoringRequest, EnableEnhancedMonitoringResponse>()
                            .withOperationName("EnableEnhancedMonitoring").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(enableEnhancedMonitoringRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableEnhancedMonitoringRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets data records from a Kinesis data stream's shard.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter in addition to the
     * <code>ShardIterator</code> parameter.
     * </p>
     * </note>
     * <p>
     * Specify a shard iterator using the <code>ShardIterator</code> parameter. The shard iterator specifies the
     * position in the shard from which you want to start reading data records sequentially. If there are no records
     * available in the portion of the shard that the iterator points to, <a>GetRecords</a> returns an empty list. It
     * might take multiple calls to get to a portion of the shard that contains records.
     * </p>
     * <p>
     * You can scale by provisioning multiple shards per stream while considering service limits (for more information,
     * see <a href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Amazon Kinesis Data
     * Streams Limits</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>). Your application should have one
     * thread per shard, each reading continuously from its stream. To read from a stream continually, call
     * <a>GetRecords</a> in a loop. Use <a>GetShardIterator</a> to get the shard iterator to specify in the first
     * <a>GetRecords</a> call. <a>GetRecords</a> returns a new shard iterator in <code>NextShardIterator</code>. Specify
     * the shard iterator returned in <code>NextShardIterator</code> in subsequent calls to <a>GetRecords</a>. If the
     * shard has been closed, the shard iterator can't return more data and <a>GetRecords</a> returns <code>null</code>
     * in <code>NextShardIterator</code>. You can terminate the loop when the shard is closed, or when the shard
     * iterator reaches the record with the sequence number or other attribute that marks it as the last record to
     * process.
     * </p>
     * <p>
     * Each data record can be up to 1 MiB in size, and each shard can read up to 2 MiB per second. You can ensure that
     * your calls don't exceed the maximum supported size or throughput by using the <code>Limit</code> parameter to
     * specify the maximum number of records that <a>GetRecords</a> can return. Consider your average record size when
     * determining this limit. The maximum number of records that can be returned per call is 10,000.
     * </p>
     * <p>
     * The size of the data returned by <a>GetRecords</a> varies depending on the utilization of the shard. It is
     * recommended that consumer applications retrieve records via the <code>GetRecords</code> command using the 5 TPS
     * limit to remain caught up. Retrieving records less frequently can lead to consumer applications falling behind.
     * The maximum size of data that <a>GetRecords</a> can return is 10 MiB. If a call returns this amount of data,
     * subsequent calls made within the next 5 seconds throw <code>ProvisionedThroughputExceededException</code>. If
     * there is insufficient provisioned throughput on the stream, subsequent calls made within the next 1 second throw
     * <code>ProvisionedThroughputExceededException</code>. <a>GetRecords</a> doesn't return any data when it throws an
     * exception. For this reason, we recommend that you wait 1 second between calls to <a>GetRecords</a>. However, it's
     * possible that the application will get exceptions for longer than 1 second.
     * </p>
     * <p>
     * To detect whether the application is falling behind in processing, you can use the
     * <code>MillisBehindLatest</code> response attribute. You can also monitor the stream using CloudWatch metrics and
     * other mechanisms (see <a href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring.html">Monitoring</a> in
     * the <i>Amazon Kinesis Data Streams Developer Guide</i>).
     * </p>
     * <p>
     * Each Amazon Kinesis record includes a value, <code>ApproximateArrivalTimestamp</code>, that is set when a stream
     * successfully receives and stores a record. This is commonly referred to as a server-side time stamp, whereas a
     * client-side time stamp is set when a data producer creates or sends the record to a stream (a data producer is
     * any data source putting data records into a stream, for example with <a>PutRecords</a>). The time stamp has
     * millisecond precision. There are no guarantees about the time stamp accuracy, or that the time stamp is always
     * increasing. For example, records in a shard or across a stream might have time stamps that are out of order.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second per shard.
     * </p>
     *
     * @param getRecordsRequest
     *        Represents the input for <a>GetRecords</a>.
     * @return Result of the GetRecords operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws ProvisionedThroughputExceededException
     *         The request rate for the stream is too high, or the requested data is too large for the available
     *         throughput. Reduce the frequency or size of your requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Data Streams Developer Guide</i>, and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in Amazon Web Services</a> in the <i>Amazon Web Services General Reference</i>.
     * @throws ExpiredIteratorException
     *         The provided iterator exceeds the maximum age allowed.
     * @throws KmsDisabledException
     *         The request was rejected because the specified customer master key (CMK) isn't enabled.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource isn't valid for this request. For
     *         more information, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How
     *         Key State Affects Use of a Customer Master Key</a> in the <i>Amazon Web Services Key Management Service
     *         Developer Guide</i>.
     * @throws KmsAccessDeniedException
     *         The ciphertext references a key that doesn't exist or that you don't have access to.
     * @throws KmsNotFoundException
     *         The request was rejected because the specified entity or resource can't be found.
     * @throws KmsOptInRequiredException
     *         The Amazon Web Services access key ID needs a subscription for the service.
     * @throws KmsThrottlingException
     *         The request was denied due to request throttling. For more information about throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/GetRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRecordsResponse getRecords(GetRecordsRequest getRecordsRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ProvisionedThroughputExceededException, ExpiredIteratorException, KmsDisabledException,
            KmsInvalidStateException, KmsAccessDeniedException, KmsNotFoundException, KmsOptInRequiredException,
            KmsThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecordsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecords");

            return clientHandler.execute(new ClientExecutionParams<GetRecordsRequest, GetRecordsResponse>()
                    .withOperationName("GetRecords").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRecordsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets an Amazon Kinesis shard iterator. A shard iterator expires 5 minutes after it is returned to the requester.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * A shard iterator specifies the shard position from which to start reading data records sequentially. The position
     * is specified using the sequence number of a data record in a shard. A sequence number is the identifier
     * associated with every record ingested in the stream, and is assigned when a record is put into the stream. Each
     * stream has one or more shards.
     * </p>
     * <p>
     * You must specify the shard iterator type. For example, you can set the <code>ShardIteratorType</code> parameter
     * to read exactly from the position denoted by a specific sequence number by using the
     * <code>AT_SEQUENCE_NUMBER</code> shard iterator type. Alternatively, the parameter can read right after the
     * sequence number by using the <code>AFTER_SEQUENCE_NUMBER</code> shard iterator type, using sequence numbers
     * returned by earlier calls to <a>PutRecord</a>, <a>PutRecords</a>, <a>GetRecords</a>, or <a>DescribeStream</a>. In
     * the request, you can specify the shard iterator type <code>AT_TIMESTAMP</code> to read records from an arbitrary
     * point in time, <code>TRIM_HORIZON</code> to cause <code>ShardIterator</code> to point to the last untrimmed
     * record in the shard in the system (the oldest data record in the shard), or <code>LATEST</code> so that you
     * always read the most recent data in the shard.
     * </p>
     * <p>
     * When you read repeatedly from a stream, use a <a>GetShardIterator</a> request to get the first shard iterator for
     * use in your first <a>GetRecords</a> request and for subsequent reads use the shard iterator returned by the
     * <a>GetRecords</a> request in <code>NextShardIterator</code>. A new shard iterator is returned by every
     * <a>GetRecords</a> request in <code>NextShardIterator</code>, which you use in the <code>ShardIterator</code>
     * parameter of the next <a>GetRecords</a> request.
     * </p>
     * <p>
     * If a <a>GetShardIterator</a> request is made too often, you receive a
     * <code>ProvisionedThroughputExceededException</code>. For more information about throughput limits, see
     * <a>GetRecords</a>, and <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in the
     * <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * If the shard is closed, <a>GetShardIterator</a> returns a valid iterator for the last sequence number of the
     * shard. A shard can be closed as a result of using <a>SplitShard</a> or <a>MergeShards</a>.
     * </p>
     * <p>
     * <a>GetShardIterator</a> has a limit of five transactions per second per account per open shard.
     * </p>
     *
     * @param getShardIteratorRequest
     *        Represents the input for <code>GetShardIterator</code>.
     * @return Result of the GetShardIterator operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws ProvisionedThroughputExceededException
     *         The request rate for the stream is too high, or the requested data is too large for the available
     *         throughput. Reduce the frequency or size of your requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Data Streams Developer Guide</i>, and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in Amazon Web Services</a> in the <i>Amazon Web Services General Reference</i>.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/GetShardIterator" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetShardIteratorResponse getShardIterator(GetShardIteratorRequest getShardIteratorRequest)
            throws ResourceNotFoundException, InvalidArgumentException, ProvisionedThroughputExceededException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetShardIteratorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetShardIteratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getShardIteratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getShardIteratorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetShardIterator");

            return clientHandler.execute(new ClientExecutionParams<GetShardIteratorRequest, GetShardIteratorResponse>()
                    .withOperationName("GetShardIterator").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getShardIteratorRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetShardIteratorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Increases the Kinesis data stream's retention period, which is the length of time data records are accessible
     * after they are added to the stream. The maximum value of a stream's retention period is 8760 hours (365 days).
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * If you choose a longer stream retention period, this operation increases the time period during which records
     * that have not yet expired are accessible. However, it does not make previous, expired data (older than the
     * stream's previous retention period) accessible after the operation has been called. For example, if a stream's
     * retention period is set to 24 hours and is increased to 168 hours, any data that is older than 24 hours remains
     * inaccessible to consumer applications.
     * </p>
     *
     * @param increaseStreamRetentionPeriodRequest
     *        Represents the input for <a>IncreaseStreamRetentionPeriod</a>.
     * @return Result of the IncreaseStreamRetentionPeriod operation returned by the service.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.IncreaseStreamRetentionPeriod
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/IncreaseStreamRetentionPeriod"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public IncreaseStreamRetentionPeriodResponse increaseStreamRetentionPeriod(
            IncreaseStreamRetentionPeriodRequest increaseStreamRetentionPeriodRequest) throws ResourceInUseException,
            ResourceNotFoundException, LimitExceededException, InvalidArgumentException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<IncreaseStreamRetentionPeriodResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, IncreaseStreamRetentionPeriodResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(increaseStreamRetentionPeriodRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                increaseStreamRetentionPeriodRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IncreaseStreamRetentionPeriod");

            return clientHandler
                    .execute(new ClientExecutionParams<IncreaseStreamRetentionPeriodRequest, IncreaseStreamRetentionPeriodResponse>()
                            .withOperationName("IncreaseStreamRetentionPeriod").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(increaseStreamRetentionPeriodRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new IncreaseStreamRetentionPeriodRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the shards in a stream and provides information about each shard. This operation has a limit of 1000
     * transactions per second per data stream.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * This action does not list expired shards. For information about expired shards, see <a href=
     * "https://docs.aws.amazon.com/streams/latest/dev/kinesis-using-sdk-java-after-resharding.html#kinesis-using-sdk-java-resharding-data-routing"
     * >Data Routing, Data Persistence, and Shard State after a Reshard</a>.
     * </p>
     * <important>
     * <p>
     * This API is a new operation that is used by the Amazon Kinesis Client Library (KCL). If you have a fine-grained
     * IAM policy that only allows specific operations, you must update your policy to allow calls to this API. For more
     * information, see <a href="https://docs.aws.amazon.com/streams/latest/dev/controlling-access.html">Controlling
     * Access to Amazon Kinesis Data Streams Resources Using IAM</a>.
     * </p>
     * </important>
     *
     * @param listShardsRequest
     * @return Result of the ListShards operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ExpiredNextTokenException
     *         The pagination token passed to the operation is expired.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.ListShards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/ListShards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListShardsResponse listShards(ListShardsRequest listShardsRequest) throws ResourceNotFoundException,
            InvalidArgumentException, LimitExceededException, ExpiredNextTokenException, ResourceInUseException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListShardsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListShardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listShardsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listShardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListShards");

            return clientHandler.execute(new ClientExecutionParams<ListShardsRequest, ListShardsResponse>()
                    .withOperationName("ListShards").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listShardsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListShardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the consumers registered to receive data from a stream using enhanced fan-out, and provides information
     * about each consumer.
     * </p>
     * <p>
     * This operation has a limit of 5 transactions per second per stream.
     * </p>
     *
     * @param listStreamConsumersRequest
     * @return Result of the ListStreamConsumers operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ExpiredNextTokenException
     *         The pagination token passed to the operation is expired.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.ListStreamConsumers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/ListStreamConsumers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStreamConsumersResponse listStreamConsumers(ListStreamConsumersRequest listStreamConsumersRequest)
            throws ResourceNotFoundException, InvalidArgumentException, LimitExceededException, ExpiredNextTokenException,
            ResourceInUseException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamConsumersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListStreamConsumersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamConsumersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamConsumersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamConsumers");

            return clientHandler.execute(new ClientExecutionParams<ListStreamConsumersRequest, ListStreamConsumersResponse>()
                    .withOperationName("ListStreamConsumers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStreamConsumersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamConsumersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your Kinesis data streams.
     * </p>
     * <p>
     * The number of streams may be too large to return from a single call to <code>ListStreams</code>. You can limit
     * the number of returned streams using the <code>Limit</code> parameter. If you do not specify a value for the
     * <code>Limit</code> parameter, Kinesis Data Streams uses the default limit, which is currently 100.
     * </p>
     * <p>
     * You can detect if there are more streams available to list by using the <code>HasMoreStreams</code> flag from the
     * returned output. If there are more streams available, you can request more streams by using the name of the last
     * stream returned by the <code>ListStreams</code> request in the <code>ExclusiveStartStreamName</code> parameter in
     * a subsequent request to <code>ListStreams</code>. The group of stream names returned by the subsequent request is
     * then added to the list. You can continue this process until all the stream names have been collected in the list.
     * </p>
     * <p>
     * <a>ListStreams</a> has a limit of five transactions per second per account.
     * </p>
     *
     * @param listStreamsRequest
     *        Represents the input for <code>ListStreams</code>.
     * @return Result of the ListStreams operation returned by the service.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ExpiredNextTokenException
     *         The pagination token passed to the operation is expired.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws LimitExceededException,
            ExpiredNextTokenException, InvalidArgumentException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreams");

            return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                    .withOperationName("ListStreams").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStreamsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified Kinesis data stream. This operation has a limit of five transactions per second
     * per account.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     *
     * @param listTagsForStreamRequest
     *        Represents the input for <code>ListTagsForStream</code>.
     * @return Result of the ListTagsForStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.ListTagsForStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/ListTagsForStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForStreamResponse listTagsForStream(ListTagsForStreamRequest listTagsForStreamRequest)
            throws ResourceNotFoundException, InvalidArgumentException, LimitExceededException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsForStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForStream");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForStreamRequest, ListTagsForStreamResponse>()
                    .withOperationName("ListTagsForStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsForStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Merges two adjacent shards in a Kinesis data stream and combines them into a single shard to reduce the stream's
     * capacity to ingest and transport data. This API is only supported for the data streams with the provisioned
     * capacity mode. Two shards are considered adjacent if the union of the hash key ranges for the two shards form a
     * contiguous set with no gaps. For example, if you have two shards, one with a hash key range of 276...381 and the
     * other with a hash key range of 382...454, then you could merge these two shards into a single shard that would
     * have a hash key range of 276...454. After the merge, the single child shard receives data for all hash key values
     * covered by the two parent shards.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * <code>MergeShards</code> is called when there is a need to reduce the overall capacity of a stream because of
     * excess capacity that is not being used. You must specify the shard to be merged and the adjacent shard for a
     * stream. For more information about merging shards, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-resharding-merge.html">Merge Two
     * Shards</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * If the stream is in the <code>ACTIVE</code> state, you can call <code>MergeShards</code>. If a stream is in the
     * <code>CREATING</code>, <code>UPDATING</code>, or <code>DELETING</code> state, <code>MergeShards</code> returns a
     * <code>ResourceInUseException</code>. If the specified stream does not exist, <code>MergeShards</code> returns a
     * <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * You can use <a>DescribeStreamSummary</a> to check the state of the stream, which is returned in
     * <code>StreamStatus</code>.
     * </p>
     * <p>
     * <code>MergeShards</code> is an asynchronous operation. Upon receiving a <code>MergeShards</code> request, Amazon
     * Kinesis Data Streams immediately returns a response and sets the <code>StreamStatus</code> to
     * <code>UPDATING</code>. After the operation is completed, Kinesis Data Streams sets the <code>StreamStatus</code>
     * to <code>ACTIVE</code>. Read and write operations continue to work while the stream is in the
     * <code>UPDATING</code> state.
     * </p>
     * <p>
     * You use <a>DescribeStreamSummary</a> and the <a>ListShards</a> APIs to determine the shard IDs that are specified
     * in the <code>MergeShards</code> request.
     * </p>
     * <p>
     * If you try to operate on too many streams in parallel using <a>CreateStream</a>, <a>DeleteStream</a>,
     * <code>MergeShards</code>, or <a>SplitShard</a>, you receive a <code>LimitExceededException</code>.
     * </p>
     * <p>
     * <code>MergeShards</code> has a limit of five transactions per second per account.
     * </p>
     *
     * @param mergeShardsRequest
     *        Represents the input for <code>MergeShards</code>.
     * @return Result of the MergeShards operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ValidationException
     *         Specifies that you tried to invoke this API for a data stream with the on-demand capacity mode. This API
     *         is only supported for data streams with the provisioned capacity mode.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.MergeShards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/MergeShards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public MergeShardsResponse mergeShards(MergeShardsRequest mergeShardsRequest) throws ResourceNotFoundException,
            ResourceInUseException, InvalidArgumentException, LimitExceededException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<MergeShardsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                MergeShardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergeShardsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeShardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeShards");

            return clientHandler.execute(new ClientExecutionParams<MergeShardsRequest, MergeShardsResponse>()
                    .withOperationName("MergeShards").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(mergeShardsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new MergeShardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Writes a single data record into an Amazon Kinesis data stream. Call <code>PutRecord</code> to send data into the
     * stream for real-time ingestion and subsequent processing, one record at a time. Each shard can support writes up
     * to 1,000 records per second, up to a maximum data write total of 1 MiB per second.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * You must specify the name of the stream that captures, stores, and transports the data; a partition key; and the
     * data blob itself.
     * </p>
     * <p>
     * The data blob can be any type of data; for example, a segment from a log file, geographic/location data, website
     * clickstream data, and so on.
     * </p>
     * <p>
     * The partition key is used by Kinesis Data Streams to distribute data across shards. Kinesis Data Streams
     * segregates the data records that belong to a stream into multiple shards, using the partition key associated with
     * each data record to determine the shard to which a given data record belongs.
     * </p>
     * <p>
     * Partition keys are Unicode strings, with a maximum length limit of 256 characters for each key. An MD5 hash
     * function is used to map partition keys to 128-bit integer values and to map associated data records to shards
     * using the hash key ranges of the shards. You can override hashing the partition key to determine the shard by
     * explicitly specifying a hash value using the <code>ExplicitHashKey</code> parameter. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-add-data-to-stream"
     * >Adding Data to a Stream</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * <code>PutRecord</code> returns the shard ID of where the data record was placed and the sequence number that was
     * assigned to the data record.
     * </p>
     * <p>
     * Sequence numbers increase over time and are specific to a shard within a stream, not across all shards within a
     * stream. To guarantee strictly increasing ordering, write serially to a shard and use the
     * <code>SequenceNumberForOrdering</code> parameter. For more information, see <a href=
     * "https://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-add-data-to-stream"
     * >Adding Data to a Stream</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * After you write a record to a stream, you cannot modify that record or its order within the stream.
     * </p>
     * </important>
     * <p>
     * If a <code>PutRecord</code> request cannot be processed because of insufficient provisioned throughput on the
     * shard involved in the request, <code>PutRecord</code> throws <code>ProvisionedThroughputExceededException</code>.
     * </p>
     * <p>
     * By default, data records are accessible for 24 hours from the time that they are added to a stream. You can use
     * <a>IncreaseStreamRetentionPeriod</a> or <a>DecreaseStreamRetentionPeriod</a> to modify this retention period.
     * </p>
     *
     * @param putRecordRequest
     *        Represents the input for <code>PutRecord</code>.
     * @return Result of the PutRecord operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws ProvisionedThroughputExceededException
     *         The request rate for the stream is too high, or the requested data is too large for the available
     *         throughput. Reduce the frequency or size of your requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Data Streams Developer Guide</i>, and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in Amazon Web Services</a> in the <i>Amazon Web Services General Reference</i>.
     * @throws KmsDisabledException
     *         The request was rejected because the specified customer master key (CMK) isn't enabled.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource isn't valid for this request. For
     *         more information, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How
     *         Key State Affects Use of a Customer Master Key</a> in the <i>Amazon Web Services Key Management Service
     *         Developer Guide</i>.
     * @throws KmsAccessDeniedException
     *         The ciphertext references a key that doesn't exist or that you don't have access to.
     * @throws KmsNotFoundException
     *         The request was rejected because the specified entity or resource can't be found.
     * @throws KmsOptInRequiredException
     *         The Amazon Web Services access key ID needs a subscription for the service.
     * @throws KmsThrottlingException
     *         The request was denied due to request throttling. For more information about throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.PutRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/PutRecord" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRecordResponse putRecord(PutRecordRequest putRecordRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ProvisionedThroughputExceededException, KmsDisabledException, KmsInvalidStateException,
            KmsAccessDeniedException, KmsNotFoundException, KmsOptInRequiredException, KmsThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRecordResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRecordRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecord");

            return clientHandler.execute(new ClientExecutionParams<PutRecordRequest, PutRecordResponse>()
                    .withOperationName("PutRecord").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putRecordRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRecordRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Writes multiple data records into a Kinesis data stream in a single call (also referred to as a
     * <code>PutRecords</code> request). Use this operation to send data into the stream for data ingestion and
     * processing.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * Each <code>PutRecords</code> request can support up to 500 records. Each record in the request can be as large as
     * 1 MiB, up to a limit of 5 MiB for the entire request, including partition keys. Each shard can support writes up
     * to 1,000 records per second, up to a maximum data write total of 1 MiB per second.
     * </p>
     * <p>
     * You must specify the name of the stream that captures, stores, and transports the data; and an array of request
     * <code>Records</code>, with each record in the array requiring a partition key and data blob. The record size
     * limit applies to the total size of the partition key and data blob.
     * </p>
     * <p>
     * The data blob can be any type of data; for example, a segment from a log file, geographic/location data, website
     * clickstream data, and so on.
     * </p>
     * <p>
     * The partition key is used by Kinesis Data Streams as input to a hash function that maps the partition key and
     * associated data to a specific shard. An MD5 hash function is used to map partition keys to 128-bit integer values
     * and to map associated data records to shards. As a result of this hashing mechanism, all data records with the
     * same partition key map to the same shard within the stream. For more information, see <a href=
     * "https://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-add-data-to-stream"
     * >Adding Data to a Stream</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * Each record in the <code>Records</code> array may include an optional parameter, <code>ExplicitHashKey</code>,
     * which overrides the partition key to shard mapping. This parameter allows a data producer to determine explicitly
     * the shard where the record is stored. For more information, see <a href=
     * "https://docs.aws.amazon.com/kinesis/latest/dev/developing-producers-with-sdk.html#kinesis-using-sdk-java-putrecords"
     * >Adding Multiple Records with PutRecords</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * The <code>PutRecords</code> response includes an array of response <code>Records</code>. Each record in the
     * response array directly correlates with a record in the request array using natural ordering, from the top to the
     * bottom of the request and response. The response <code>Records</code> array always includes the same number of
     * records as the request array.
     * </p>
     * <p>
     * The response <code>Records</code> array includes both successfully and unsuccessfully processed records. Kinesis
     * Data Streams attempts to process all records in each <code>PutRecords</code> request. A single record failure
     * does not stop the processing of subsequent records. As a result, PutRecords doesn't guarantee the ordering of
     * records. If you need to read records in the same order they are written to the stream, use <a>PutRecord</a>
     * instead of <code>PutRecords</code>, and write to the same shard.
     * </p>
     * <p>
     * A successfully processed record includes <code>ShardId</code> and <code>SequenceNumber</code> values. The
     * <code>ShardId</code> parameter identifies the shard in the stream where the record is stored. The
     * <code>SequenceNumber</code> parameter is an identifier assigned to the put record, unique to all records in the
     * stream.
     * </p>
     * <p>
     * An unsuccessfully processed record includes <code>ErrorCode</code> and <code>ErrorMessage</code> values.
     * <code>ErrorCode</code> reflects the type of error and can be one of the following values:
     * <code>ProvisionedThroughputExceededException</code> or <code>InternalFailure</code>. <code>ErrorMessage</code>
     * provides more detailed information about the <code>ProvisionedThroughputExceededException</code> exception
     * including the account ID, stream name, and shard ID of the record that was throttled. For more information about
     * partially successful responses, see <a href=
     * "https://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-add-data-to-stream.html#kinesis-using-sdk-java-putrecords"
     * >Adding Multiple Records with PutRecords</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * After you write a record to a stream, you cannot modify that record or its order within the stream.
     * </p>
     * </important>
     * <p>
     * By default, data records are accessible for 24 hours from the time that they are added to a stream. You can use
     * <a>IncreaseStreamRetentionPeriod</a> or <a>DecreaseStreamRetentionPeriod</a> to modify this retention period.
     * </p>
     *
     * @param putRecordsRequest
     *        A <code>PutRecords</code> request.
     * @return Result of the PutRecords operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws ProvisionedThroughputExceededException
     *         The request rate for the stream is too high, or the requested data is too large for the available
     *         throughput. Reduce the frequency or size of your requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in
     *         the <i>Amazon Kinesis Data Streams Developer Guide</i>, and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/api-retries.html">Error Retries and Exponential
     *         Backoff in Amazon Web Services</a> in the <i>Amazon Web Services General Reference</i>.
     * @throws KmsDisabledException
     *         The request was rejected because the specified customer master key (CMK) isn't enabled.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource isn't valid for this request. For
     *         more information, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How
     *         Key State Affects Use of a Customer Master Key</a> in the <i>Amazon Web Services Key Management Service
     *         Developer Guide</i>.
     * @throws KmsAccessDeniedException
     *         The ciphertext references a key that doesn't exist or that you don't have access to.
     * @throws KmsNotFoundException
     *         The request was rejected because the specified entity or resource can't be found.
     * @throws KmsOptInRequiredException
     *         The Amazon Web Services access key ID needs a subscription for the service.
     * @throws KmsThrottlingException
     *         The request was denied due to request throttling. For more information about throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.PutRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/PutRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRecordsResponse putRecords(PutRecordsRequest putRecordsRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ProvisionedThroughputExceededException, KmsDisabledException, KmsInvalidStateException,
            KmsAccessDeniedException, KmsNotFoundException, KmsOptInRequiredException, KmsThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRecordsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecords");

            return clientHandler.execute(new ClientExecutionParams<PutRecordsRequest, PutRecordsResponse>()
                    .withOperationName("PutRecords").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putRecordsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a consumer with a Kinesis data stream. When you use this operation, the consumer you register can then
     * call <a>SubscribeToShard</a> to receive data from the stream using enhanced fan-out, at a rate of up to 2 MiB per
     * second for every shard you subscribe to. This rate is unaffected by the total number of consumers that read from
     * the same stream.
     * </p>
     * <p>
     * You can register up to 20 consumers per stream. A given consumer can only be registered with one stream at a
     * time.
     * </p>
     * <p>
     * For an example of how to use this operations, see <a
     * href="/streams/latest/dev/building-enhanced-consumers-api.html">Enhanced Fan-Out Using the Kinesis Data Streams
     * API</a>.
     * </p>
     * <p>
     * The use of this operation has a limit of five transactions per second per account. Also, only 5 consumers can be
     * created simultaneously. In other words, you cannot have more than 5 consumers in a <code>CREATING</code> status
     * at the same time. Registering a 6th consumer while there are 5 in a <code>CREATING</code> status results in a
     * <code>LimitExceededException</code>.
     * </p>
     *
     * @param registerStreamConsumerRequest
     * @return Result of the RegisterStreamConsumer operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.RegisterStreamConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/RegisterStreamConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterStreamConsumerResponse registerStreamConsumer(RegisterStreamConsumerRequest registerStreamConsumerRequest)
            throws InvalidArgumentException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterStreamConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterStreamConsumerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerStreamConsumerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerStreamConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterStreamConsumer");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterStreamConsumerRequest, RegisterStreamConsumerResponse>()
                            .withOperationName("RegisterStreamConsumer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(registerStreamConsumerRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterStreamConsumerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified Kinesis data stream. Removed tags are deleted and cannot be recovered after this
     * operation successfully completes.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * If you specify a tag that does not exist, it is ignored.
     * </p>
     * <p>
     * <a>RemoveTagsFromStream</a> has a limit of five transactions per second per account.
     * </p>
     *
     * @param removeTagsFromStreamRequest
     *        Represents the input for <code>RemoveTagsFromStream</code>.
     * @return Result of the RemoveTagsFromStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.RemoveTagsFromStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/RemoveTagsFromStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RemoveTagsFromStreamResponse removeTagsFromStream(RemoveTagsFromStreamRequest removeTagsFromStreamRequest)
            throws ResourceNotFoundException, ResourceInUseException, InvalidArgumentException, LimitExceededException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsFromStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveTagsFromStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeTagsFromStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsFromStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTagsFromStream");

            return clientHandler.execute(new ClientExecutionParams<RemoveTagsFromStreamRequest, RemoveTagsFromStreamResponse>()
                    .withOperationName("RemoveTagsFromStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(removeTagsFromStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RemoveTagsFromStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Splits a shard into two new shards in the Kinesis data stream, to increase the stream's capacity to ingest and
     * transport data. <code>SplitShard</code> is called when there is a need to increase the overall capacity of a
     * stream because of an expected increase in the volume of data records being ingested. This API is only supported
     * for the data streams with the provisioned capacity mode.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * You can also use <code>SplitShard</code> when a shard appears to be approaching its maximum utilization; for
     * example, the producers sending data into the specific shard are suddenly sending more than previously
     * anticipated. You can also call <code>SplitShard</code> to increase stream capacity, so that more Kinesis Data
     * Streams applications can simultaneously read data from the stream for real-time processing.
     * </p>
     * <p>
     * You must specify the shard to be split and the new hash key, which is the position in the shard where the shard
     * gets split in two. In many cases, the new hash key might be the average of the beginning and ending hash key, but
     * it can be any hash key value in the range being mapped into the shard. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/kinesis-using-sdk-java-resharding-split.html">Split a
     * Shard</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>.
     * </p>
     * <p>
     * You can use <a>DescribeStreamSummary</a> and the <a>ListShards</a> APIs to determine the shard ID and hash key
     * values for the <code>ShardToSplit</code> and <code>NewStartingHashKey</code> parameters that are specified in the
     * <code>SplitShard</code> request.
     * </p>
     * <p>
     * <code>SplitShard</code> is an asynchronous operation. Upon receiving a <code>SplitShard</code> request, Kinesis
     * Data Streams immediately returns a response and sets the stream status to <code>UPDATING</code>. After the
     * operation is completed, Kinesis Data Streams sets the stream status to <code>ACTIVE</code>. Read and write
     * operations continue to work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * <p>
     * You can use <a>DescribeStreamSummary</a> to check the status of the stream, which is returned in
     * <code>StreamStatus</code>. If the stream is in the <code>ACTIVE</code> state, you can call
     * <code>SplitShard</code>.
     * </p>
     * <p>
     * If the specified stream does not exist, <a>DescribeStreamSummary</a> returns a
     * <code>ResourceNotFoundException</code>. If you try to create more shards than are authorized for your account,
     * you receive a <code>LimitExceededException</code>.
     * </p>
     * <p>
     * For the default shard limit for an Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Kinesis Data Streams
     * Limits</a> in the <i>Amazon Kinesis Data Streams Developer Guide</i>. To increase this limit, <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">contact Amazon Web Services
     * Support</a>.
     * </p>
     * <p>
     * If you try to operate on too many streams simultaneously using <a>CreateStream</a>, <a>DeleteStream</a>,
     * <a>MergeShards</a>, and/or <a>SplitShard</a>, you receive a <code>LimitExceededException</code>.
     * </p>
     * <p>
     * <code>SplitShard</code> has a limit of five transactions per second per account.
     * </p>
     *
     * @param splitShardRequest
     *        Represents the input for <code>SplitShard</code>.
     * @return Result of the SplitShard operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ValidationException
     *         Specifies that you tried to invoke this API for a data stream with the on-demand capacity mode. This API
     *         is only supported for data streams with the provisioned capacity mode.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.SplitShard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/SplitShard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SplitShardResponse splitShard(SplitShardRequest splitShardRequest) throws ResourceNotFoundException,
            ResourceInUseException, InvalidArgumentException, LimitExceededException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SplitShardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SplitShardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(splitShardRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, splitShardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SplitShard");

            return clientHandler.execute(new ClientExecutionParams<SplitShardRequest, SplitShardResponse>()
                    .withOperationName("SplitShard").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(splitShardRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SplitShardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables or updates server-side encryption using an Amazon Web Services KMS key for a specified stream.
     * </p>
     * <p>
     * Starting encryption is an asynchronous operation. Upon receiving the request, Kinesis Data Streams returns
     * immediately and sets the status of the stream to <code>UPDATING</code>. After the update is complete, Kinesis
     * Data Streams sets the status of the stream back to <code>ACTIVE</code>. Updating or applying encryption normally
     * takes a few seconds to complete, but it can take minutes. You can continue to read and write data to your stream
     * while its status is <code>UPDATING</code>. Once the status of the stream is <code>ACTIVE</code>, encryption
     * begins for records written to the stream.
     * </p>
     * <p>
     * API Limits: You can successfully apply a new Amazon Web Services KMS key for server-side encryption 25 times in a
     * rolling 24-hour period.
     * </p>
     * <p>
     * Note: It can take up to 5 seconds after the stream is in an <code>ACTIVE</code> status before all records written
     * to the stream are encrypted. After you enable encryption, you can verify that encryption is applied by inspecting
     * the API response from <code>PutRecord</code> or <code>PutRecords</code>.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     *
     * @param startStreamEncryptionRequest
     * @return Result of the StartStreamEncryption operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws KmsDisabledException
     *         The request was rejected because the specified customer master key (CMK) isn't enabled.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource isn't valid for this request. For
     *         more information, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How
     *         Key State Affects Use of a Customer Master Key</a> in the <i>Amazon Web Services Key Management Service
     *         Developer Guide</i>.
     * @throws KmsAccessDeniedException
     *         The ciphertext references a key that doesn't exist or that you don't have access to.
     * @throws KmsNotFoundException
     *         The request was rejected because the specified entity or resource can't be found.
     * @throws KmsOptInRequiredException
     *         The Amazon Web Services access key ID needs a subscription for the service.
     * @throws KmsThrottlingException
     *         The request was denied due to request throttling. For more information about throttling, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#requests-per-second">Limits</a>
     *         in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.StartStreamEncryption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/StartStreamEncryption" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartStreamEncryptionResponse startStreamEncryption(StartStreamEncryptionRequest startStreamEncryptionRequest)
            throws InvalidArgumentException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            KmsDisabledException, KmsInvalidStateException, KmsAccessDeniedException, KmsNotFoundException,
            KmsOptInRequiredException, KmsThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartStreamEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartStreamEncryptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startStreamEncryptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startStreamEncryptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartStreamEncryption");

            return clientHandler.execute(new ClientExecutionParams<StartStreamEncryptionRequest, StartStreamEncryptionResponse>()
                    .withOperationName("StartStreamEncryption").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startStreamEncryptionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartStreamEncryptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables server-side encryption for a specified stream.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * Stopping encryption is an asynchronous operation. Upon receiving the request, Kinesis Data Streams returns
     * immediately and sets the status of the stream to <code>UPDATING</code>. After the update is complete, Kinesis
     * Data Streams sets the status of the stream back to <code>ACTIVE</code>. Stopping encryption normally takes a few
     * seconds to complete, but it can take minutes. You can continue to read and write data to your stream while its
     * status is <code>UPDATING</code>. Once the status of the stream is <code>ACTIVE</code>, records written to the
     * stream are no longer encrypted by Kinesis Data Streams.
     * </p>
     * <p>
     * API Limits: You can successfully disable server-side encryption 25 times in a rolling 24-hour period.
     * </p>
     * <p>
     * Note: It can take up to 5 seconds after the stream is in an <code>ACTIVE</code> status before all records written
     * to the stream are no longer subject to encryption. After you disabled encryption, you can verify that encryption
     * is not applied by inspecting the API response from <code>PutRecord</code> or <code>PutRecords</code>.
     * </p>
     *
     * @param stopStreamEncryptionRequest
     * @return Result of the StopStreamEncryption operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.StopStreamEncryption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/StopStreamEncryption" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopStreamEncryptionResponse stopStreamEncryption(StopStreamEncryptionRequest stopStreamEncryptionRequest)
            throws InvalidArgumentException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopStreamEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopStreamEncryptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopStreamEncryptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopStreamEncryptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopStreamEncryption");

            return clientHandler.execute(new ClientExecutionParams<StopStreamEncryptionRequest, StopStreamEncryptionResponse>()
                    .withOperationName("StopStreamEncryption").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopStreamEncryptionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopStreamEncryptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the shard count of the specified stream to the specified number of shards. This API is only supported for
     * the data streams with the provisioned capacity mode.
     * </p>
     * <note>
     * <p>
     * When invoking this API, it is recommended you use the <code>StreamARN</code> input parameter rather than the
     * <code>StreamName</code> input parameter.
     * </p>
     * </note>
     * <p>
     * Updating the shard count is an asynchronous operation. Upon receiving the request, Kinesis Data Streams returns
     * immediately and sets the status of the stream to <code>UPDATING</code>. After the update is complete, Kinesis
     * Data Streams sets the status of the stream back to <code>ACTIVE</code>. Depending on the size of the stream, the
     * scaling action could take a few minutes to complete. You can continue to read and write data to your stream while
     * its status is <code>UPDATING</code>.
     * </p>
     * <p>
     * To update the shard count, Kinesis Data Streams performs splits or merges on individual shards. This can cause
     * short-lived shards to be created, in addition to the final shards. These short-lived shards count towards your
     * total shard limit for your account in the Region.
     * </p>
     * <p>
     * When using this operation, we recommend that you specify a target shard count that is a multiple of 25% (25%,
     * 50%, 75%, 100%). You can specify any target value within your shard limit. However, if you specify a target that
     * isn't a multiple of 25%, the scaling action might take longer to complete.
     * </p>
     * <p>
     * This operation has the following default limits. By default, you cannot do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Scale more than ten times per rolling 24-hour period per stream
     * </p>
     * </li>
     * <li>
     * <p>
     * Scale up to more than double your current shard count for a stream
     * </p>
     * </li>
     * <li>
     * <p>
     * Scale down below half your current shard count for a stream
     * </p>
     * </li>
     * <li>
     * <p>
     * Scale up to more than 10000 shards in a stream
     * </p>
     * </li>
     * <li>
     * <p>
     * Scale a stream with more than 10000 shards down unless the result is less than 10000 shards
     * </p>
     * </li>
     * <li>
     * <p>
     * Scale up to more than the shard limit for your account
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the default limits for an Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/service-sizes-and-limits.html">Streams Limits</a> in the
     * <i>Amazon Kinesis Data Streams Developer Guide</i>. To request an increase in the call rate limit, the shard
     * limit for this API, or your overall shard limit, use the <a href=
     * "https://console.aws.amazon.com/support/v1#/case/create?issueType=service-limit-increase&amp;limitType=service-code-kinesis"
     * >limits form</a>.
     * </p>
     *
     * @param updateShardCountRequest
     * @return Result of the UpdateShardCount operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws ValidationException
     *         Specifies that you tried to invoke this API for a data stream with the on-demand capacity mode. This API
     *         is only supported for data streams with the provisioned capacity mode.
     * @throws AccessDeniedException
     *         Specifies that you do not have the permissions required to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.UpdateShardCount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/UpdateShardCount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateShardCountResponse updateShardCount(UpdateShardCountRequest updateShardCountRequest)
            throws InvalidArgumentException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateShardCountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateShardCountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateShardCountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateShardCountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateShardCount");

            return clientHandler.execute(new ClientExecutionParams<UpdateShardCountRequest, UpdateShardCountResponse>()
                    .withOperationName("UpdateShardCount").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateShardCountRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateShardCountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the capacity mode of the data stream. Currently, in Kinesis Data Streams, you can choose between an
     * <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data stream.
     * </p>
     *
     * @param updateStreamModeRequest
     * @return Result of the UpdateStreamMode operation returned by the service.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used. For more information,
     *         see the returned message.
     * @throws LimitExceededException
     *         The requested resource exceeds the maximum number allowed, or the number of concurrent stream requests
     *         exceeds the maximum number allowed.
     * @throws ResourceInUseException
     *         The resource is not available for this operation. For successful operation, the resource must be in the
     *         <code>ACTIVE</code> state.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found. The stream might not be specified correctly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisClient.UpdateStreamMode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/UpdateStreamMode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateStreamModeResponse updateStreamMode(UpdateStreamModeRequest updateStreamModeRequest)
            throws InvalidArgumentException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, KinesisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStreamModeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateStreamModeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStreamModeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStreamModeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStreamMode");

            return clientHandler.execute(new ClientExecutionParams<UpdateStreamModeRequest, UpdateStreamModeResponse>()
                    .withOperationName("UpdateStreamMode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateStreamModeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStreamModeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link KinesisWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link KinesisWaiter}
     */
    @Override
    public KinesisWaiter waiter() {
        return KinesisWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        KinesisServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = KinesisServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSThrottlingException")
                                .exceptionBuilderSupplier(KmsThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidStateException")
                                .exceptionBuilderSupplier(KmsInvalidStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSNotFoundException")
                                .exceptionBuilderSupplier(KmsNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredIteratorException")
                                .exceptionBuilderSupplier(ExpiredIteratorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSOptInRequired")
                                .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredNextTokenException")
                                .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSAccessDeniedException")
                                .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSDisabledException")
                                .exceptionBuilderSupplier(KmsDisabledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                                .exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build());
    }

    @Override
    public final KinesisServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
