/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.endpoints.internal;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.kinesis.endpoints.KinesisEndpointParams;
import software.amazon.awssdk.services.kinesis.endpoints.KinesisEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultKinesisEndpointProvider implements KinesisEndpointProvider {
    private static final EndpointRuleset ENDPOINT_RULE_SET = ruleSet();

    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(KinesisEndpointParams endpointParams) {
        Validate.notNull(endpointParams.useDualStack(), "Parameter 'UseDualStack' must not be null");
        Validate.notNull(endpointParams.useFips(), "Parameter 'UseFIPS' must not be null");
        Value res = new DefaultRuleEngine().evaluate(ENDPOINT_RULE_SET, toIdentifierValueMap(endpointParams));
        try {
            return CompletableFuture.completedFuture(AwsEndpointProviderUtils.valueAsEndpointOrThrow(res));
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static Map<Identifier, Value> toIdentifierValueMap(KinesisEndpointParams params) {
        Map<Identifier, Value> paramsMap = new HashMap<>();
        if (params.region() != null) {
            paramsMap.put(Identifier.of("Region"), Value.fromStr(params.region().id()));
        }
        if (params.useDualStack() != null) {
            paramsMap.put(Identifier.of("UseDualStack"), Value.fromBool(params.useDualStack()));
        }
        if (params.useFips() != null) {
            paramsMap.put(Identifier.of("UseFIPS"), Value.fromBool(params.useFips()));
        }
        if (params.endpoint() != null) {
            paramsMap.put(Identifier.of("Endpoint"), Value.fromStr(params.endpoint()));
        }
        if (params.streamArn() != null) {
            paramsMap.put(Identifier.of("StreamARN"), Value.fromStr(params.streamArn()));
        }
        if (params.operationType() != null) {
            paramsMap.put(Identifier.of("OperationType"), Value.fromStr(params.operationType()));
        }
        if (params.consumerArn() != null) {
            paramsMap.put(Identifier.of("ConsumerARN"), Value.fromStr(params.consumerArn()));
        }
        if (params.resourceArn() != null) {
            paramsMap.put(Identifier.of("ResourceARN"), Value.fromStr(params.resourceArn()));
        }
        return paramsMap;
    }

    private static Rule endpointRule_12() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis-fips.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_11() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_12()));
    }

    private static Rule endpointRule_13() {
        return Rule.builder().error("DualStack is enabled, but this partition does not support DualStack.");
    }

    private static Rule endpointRule_10() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_11(), endpointRule_13()));
    }

    private static Rule endpointRule_14() {
        return Rule.builder().error("FIPS is enabled, but this partition does not support FIPS.");
    }

    private static Rule endpointRule_9() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_10(), endpointRule_14()));
    }

    private static Rule endpointRule_17() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis-fips.{Region}.{PartitionResult#dnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_16() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_17()));
    }

    private static Rule endpointRule_18() {
        return Rule.builder().error("FIPS is enabled but this partition does not support FIPS");
    }

    private static Rule endpointRule_15() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_16(), endpointRule_18()));
    }

    private static Rule endpointRule_21() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_20() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_21()));
    }

    private static Rule endpointRule_22() {
        return Rule.builder().error("DualStack is enabled but this partition does not support DualStack");
    }

    private static Rule endpointRule_19() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_20(), endpointRule_22()));
    }

    private static Rule endpointRule_23() {
        return Rule.builder().endpoint(
                EndpointResult.builder()
                        .url(Expr.of("https://{arn#accountId}.{OperationType}-kinesis.{Region}.{PartitionResult#dnsSuffix}"))
                        .build());
    }

    private static Rule endpointRule_8() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("OperationType"))))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_9(), endpointRule_15(), endpointRule_19(), endpointRule_23()));
    }

    private static Rule endpointRule_24() {
        return Rule.builder().error("Operation Type is not set. Please contact service team for resolution.");
    }

    private static Rule endpointRule_7() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("{arn#partition}")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_8(), endpointRule_24()));
    }

    private static Rule endpointRule_25() {
        return Rule.builder().error(
                "Partition: {arn#partition} from ARN doesn't match with partition name: {PartitionResult#name}.");
    }

    private static Rule endpointRule_6() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("stream"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_7(), endpointRule_25()));
    }

    private static Rule endpointRule_26() {
        return Rule.builder().error("Invalid ARN: Kinesis ARNs don't support `{arnType}` arn types.");
    }

    private static Rule endpointRule_5() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("resourceId[0]"))).build()
                                        .validate()).result("arnType").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of(""))).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_6(), endpointRule_26()));
    }

    private static Rule endpointRule_27() {
        return Rule.builder().error("Invalid ARN: No ARN type specified");
    }

    private static Rule endpointRule_4() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("service")))
                                                        .build().validate(), Expr.of("kinesis"))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_5(), endpointRule_27()));
    }

    private static Rule endpointRule_28() {
        return Rule.builder().error("Invalid ARN: The ARN was not for the Kinesis service, found: {arn#service}.");
    }

    private static Rule endpointRule_3() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("region")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_4(), endpointRule_28()));
    }

    private static Rule endpointRule_29() {
        return Rule.builder().error("Invalid ARN: Invalid region.");
    }

    private static Rule endpointRule_2() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("accountId")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_3(), endpointRule_29()));
    }

    private static Rule endpointRule_30() {
        return Rule.builder().error("Invalid ARN: Invalid account id.");
    }

    private static Rule endpointRule_1() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.parseArn").argv(Arrays.asList(Expr.ref(Identifier.of("StreamARN"))))
                                        .build().validate()).result("arn").build())
                .treeRule(Arrays.asList(endpointRule_2(), endpointRule_30()));
    }

    private static Rule endpointRule_31() {
        return Rule.builder().error("Invalid ARN: Failed to parse ARN.");
    }

    private static Rule endpointRule_0() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("StreamARN"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("PartitionResult").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                        Expr.of("name"))).build().validate(), Expr.of("aws-iso")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                        Expr.of("name"))).build().validate(), Expr
                                                                .of("aws-iso-b"))).build().validate())).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_1(), endpointRule_31()));
    }

    private static Rule endpointRule_44() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis-fips.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_43() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_44()));
    }

    private static Rule endpointRule_45() {
        return Rule.builder().error("DualStack is enabled, but this partition does not support DualStack.");
    }

    private static Rule endpointRule_42() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_43(), endpointRule_45()));
    }

    private static Rule endpointRule_46() {
        return Rule.builder().error("FIPS is enabled, but this partition does not support FIPS.");
    }

    private static Rule endpointRule_41() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_42(), endpointRule_46()));
    }

    private static Rule endpointRule_49() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis-fips.{Region}.{PartitionResult#dnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_48() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_49()));
    }

    private static Rule endpointRule_50() {
        return Rule.builder().error("FIPS is enabled but this partition does not support FIPS");
    }

    private static Rule endpointRule_47() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_48(), endpointRule_50()));
    }

    private static Rule endpointRule_53() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_52() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_53()));
    }

    private static Rule endpointRule_54() {
        return Rule.builder().error("DualStack is enabled but this partition does not support DualStack");
    }

    private static Rule endpointRule_51() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_52(), endpointRule_54()));
    }

    private static Rule endpointRule_55() {
        return Rule.builder().endpoint(
                EndpointResult.builder()
                        .url(Expr.of("https://{arn#accountId}.{OperationType}-kinesis.{Region}.{PartitionResult#dnsSuffix}"))
                        .build());
    }

    private static Rule endpointRule_40() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("OperationType"))))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_41(), endpointRule_47(), endpointRule_51(), endpointRule_55()));
    }

    private static Rule endpointRule_56() {
        return Rule.builder().error("Operation Type is not set. Please contact service team for resolution.");
    }

    private static Rule endpointRule_39() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("{arn#partition}")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_40(), endpointRule_56()));
    }

    private static Rule endpointRule_57() {
        return Rule.builder().error(
                "Partition: {arn#partition} from ARN doesn't match with partition name: {PartitionResult#name}.");
    }

    private static Rule endpointRule_38() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("stream"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_39(), endpointRule_57()));
    }

    private static Rule endpointRule_58() {
        return Rule.builder().error("Invalid ARN: Kinesis ARNs don't support `{arnType}` arn types.");
    }

    private static Rule endpointRule_37() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("resourceId[0]"))).build()
                                        .validate()).result("arnType").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of(""))).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_38(), endpointRule_58()));
    }

    private static Rule endpointRule_59() {
        return Rule.builder().error("Invalid ARN: No ARN type specified");
    }

    private static Rule endpointRule_36() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("service")))
                                                        .build().validate(), Expr.of("kinesis"))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_37(), endpointRule_59()));
    }

    private static Rule endpointRule_60() {
        return Rule.builder().error("Invalid ARN: The ARN was not for the Kinesis service, found: {arn#service}.");
    }

    private static Rule endpointRule_35() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("region")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_36(), endpointRule_60()));
    }

    private static Rule endpointRule_61() {
        return Rule.builder().error("Invalid ARN: Invalid region.");
    }

    private static Rule endpointRule_34() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("accountId")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_35(), endpointRule_61()));
    }

    private static Rule endpointRule_62() {
        return Rule.builder().error("Invalid ARN: Invalid account id.");
    }

    private static Rule endpointRule_33() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.parseArn")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("ConsumerARN")))).build().validate())
                                .result("arn").build()).treeRule(Arrays.asList(endpointRule_34(), endpointRule_62()));
    }

    private static Rule endpointRule_63() {
        return Rule.builder().error("Invalid ARN: Failed to parse ARN.");
    }

    private static Rule endpointRule_32() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("ConsumerARN"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("PartitionResult").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                        Expr.of("name"))).build().validate(), Expr.of("aws-iso")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                        Expr.of("name"))).build().validate(), Expr
                                                                .of("aws-iso-b"))).build().validate())).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_33(), endpointRule_63()));
    }

    private static Rule endpointRule_76() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis-fips.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_75() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_76()));
    }

    private static Rule endpointRule_77() {
        return Rule.builder().error("DualStack is enabled, but this partition does not support DualStack.");
    }

    private static Rule endpointRule_74() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_75(), endpointRule_77()));
    }

    private static Rule endpointRule_78() {
        return Rule.builder().error("FIPS is enabled, but this partition does not support FIPS.");
    }

    private static Rule endpointRule_73() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_74(), endpointRule_78()));
    }

    private static Rule endpointRule_81() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis-fips.{Region}.{PartitionResult#dnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_80() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_81()));
    }

    private static Rule endpointRule_82() {
        return Rule.builder().error("FIPS is enabled but this partition does not support FIPS");
    }

    private static Rule endpointRule_79() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_80(), endpointRule_82()));
    }

    private static Rule endpointRule_85() {
        return Rule
                .builder()
                .endpoint(
                        EndpointResult
                                .builder()
                                .url(Expr
                                        .of("https://{arn#accountId}.{OperationType}-kinesis.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_84() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build()).treeRule(Arrays.asList(endpointRule_85()));
    }

    private static Rule endpointRule_86() {
        return Rule.builder().error("DualStack is enabled but this partition does not support DualStack");
    }

    private static Rule endpointRule_83() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_84(), endpointRule_86()));
    }

    private static Rule endpointRule_87() {
        return Rule.builder().endpoint(
                EndpointResult.builder()
                        .url(Expr.of("https://{arn#accountId}.{OperationType}-kinesis.{Region}.{PartitionResult#dnsSuffix}"))
                        .build());
    }

    private static Rule endpointRule_72() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("OperationType"))))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_73(), endpointRule_79(), endpointRule_83(), endpointRule_87()));
    }

    private static Rule endpointRule_88() {
        return Rule.builder().error("Operation Type is not set. Please contact service team for resolution.");
    }

    private static Rule endpointRule_71() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("{arn#partition}")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_72(), endpointRule_88()));
    }

    private static Rule endpointRule_89() {
        return Rule.builder().error(
                "Partition: {arn#partition} from ARN doesn't match with partition name: {PartitionResult#name}.");
    }

    private static Rule endpointRule_70() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("stream"))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_71(), endpointRule_89()));
    }

    private static Rule endpointRule_90() {
        return Rule.builder().error("Invalid ARN: Kinesis ARNs don't support `{arnType}` arn types.");
    }

    private static Rule endpointRule_69() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("resourceId[0]"))).build()
                                        .validate()).result("arnType").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of(""))).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_70(), endpointRule_90()));
    }

    private static Rule endpointRule_91() {
        return Rule.builder().error("Invalid ARN: No ARN type specified");
    }

    private static Rule endpointRule_68() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("service")))
                                                        .build().validate(), Expr.of("kinesis"))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_69(), endpointRule_91()));
    }

    private static Rule endpointRule_92() {
        return Rule.builder().error("Invalid ARN: The ARN was not for the Kinesis service, found: {arn#service}.");
    }

    private static Rule endpointRule_67() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder().fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("region")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_68(), endpointRule_92()));
    }

    private static Rule endpointRule_93() {
        return Rule.builder().error("Invalid ARN: Invalid region.");
    }

    private static Rule endpointRule_66() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("isValidHostLabel")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arn")), Expr.of("accountId")))
                                                        .build().validate(), Expr.of(false))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_67(), endpointRule_93()));
    }

    private static Rule endpointRule_94() {
        return Rule.builder().error("Invalid ARN: Invalid account id.");
    }

    private static Rule endpointRule_65() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.parseArn")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("ResourceARN")))).build().validate())
                                .result("arn").build()).treeRule(Arrays.asList(endpointRule_66(), endpointRule_94()));
    }

    private static Rule endpointRule_95() {
        return Rule.builder().error("Invalid ARN: Failed to parse ARN.");
    }

    private static Rule endpointRule_64() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("ResourceARN"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build().validate()))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("PartitionResult").build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                        Expr.of("name"))).build().validate(), Expr.of("aws-iso")))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode
                                                .builder()
                                                .fn("stringEquals")
                                                .argv(Arrays.asList(
                                                        FnNode.builder()
                                                                .fn("getAttr")
                                                                .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                        Expr.of("name"))).build().validate(), Expr
                                                                .of("aws-iso-b"))).build().validate())).build().validate())
                                .build()).treeRule(Arrays.asList(endpointRule_65(), endpointRule_95()));
    }

    private static Rule endpointRule_97() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Invalid Configuration: FIPS and custom endpoint are not supported");
    }

    private static Rule endpointRule_98() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Invalid Configuration: Dualstack and custom endpoint are not supported");
    }

    private static Rule endpointRule_99() {
        return Rule.builder().endpoint(EndpointResult.builder().url(Expr.ref(Identifier.of("Endpoint"))).build());
    }

    private static Rule endpointRule_96() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .treeRule(Arrays.asList(endpointRule_97(), endpointRule_98(), endpointRule_99()));
    }

    private static Rule endpointRule_104() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://kinesis-fips.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                        .build());
    }

    private static Rule endpointRule_103() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                Expr.of(true),
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate())).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                Expr.of(true),
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate())).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_104()));
    }

    private static Rule endpointRule_105() {
        return Rule.builder().error("FIPS and DualStack are enabled, but this partition does not support one or both");
    }

    private static Rule endpointRule_102() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_103(), endpointRule_105()));
    }

    private static Rule endpointRule_108() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-us-gov")))
                                        .build().validate()).build())
                .endpoint(EndpointResult.builder().url(Expr.of("https://kinesis.{Region}.amazonaws.com")).build());
    }

    private static Rule endpointRule_109() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://kinesis-fips.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_107() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_108(), endpointRule_109()));
    }

    private static Rule endpointRule_110() {
        return Rule.builder().error("FIPS is enabled but this partition does not support FIPS");
    }

    private static Rule endpointRule_106() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_107(), endpointRule_110()));
    }

    private static Rule endpointRule_113() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://kinesis.{Region}.{PartitionResult#dualStackDnsSuffix}")).build());
    }

    private static Rule endpointRule_112() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                Expr.of(true),
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate())).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_113()));
    }

    private static Rule endpointRule_114() {
        return Rule.builder().error("DualStack is enabled but this partition does not support DualStack");
    }

    private static Rule endpointRule_111() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_112(), endpointRule_114()));
    }

    private static Rule endpointRule_115() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://kinesis.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_101() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("PartitionResult").build())
                .treeRule(Arrays.asList(endpointRule_102(), endpointRule_106(), endpointRule_111(), endpointRule_115()));
    }

    private static Rule endpointRule_100() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_101()));
    }

    private static Rule endpointRule_116() {
        return Rule.builder().error("Invalid Configuration: Missing Region");
    }

    private static EndpointRuleset ruleSet() {
        return EndpointRuleset
                .builder()
                .version("1.0")
                .serviceId(null)
                .parameters(
                        Parameters
                                .builder()
                                .addParameter(
                                        Parameter.builder().name("Region").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("AWS::Region")
                                                .documentation("The AWS region used to dispatch the request.").build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseDualStack")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseDualStack")
                                                .documentation(
                                                        "When true, use the dual-stack endpoint. If the configured endpoint does not support dual-stack, dispatching the request MAY return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseFIPS")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseFIPS")
                                                .documentation(
                                                        "When true, send this request to the FIPS-compliant regional endpoint. If the configured endpoint does not have a FIPS compliant endpoint, dispatching the request will return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter.builder().name("Endpoint").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("SDK::Endpoint")
                                                .documentation("Override the endpoint used to send this request").build())
                                .addParameter(
                                        Parameter.builder().name("StreamARN").type(ParameterType.fromValue("String"))
                                                .required(false).documentation("The ARN of the Kinesis stream").build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("OperationType")
                                                .type(ParameterType.fromValue("String"))
                                                .required(false)
                                                .documentation(
                                                        "Internal parameter to distinguish between Control/Data plane API and accordingly generate control/data plane endpoint")
                                                .build())
                                .addParameter(
                                        Parameter.builder().name("ConsumerARN").type(ParameterType.fromValue("String"))
                                                .required(false).documentation("The ARN of the Kinesis consumer").build())
                                .addParameter(
                                        Parameter.builder().name("ResourceARN").type(ParameterType.fromValue("String"))
                                                .required(false).documentation("The ARN of the Kinesis resource").build())
                                .build()).addRule(endpointRule_0()).addRule(endpointRule_32()).addRule(endpointRule_64())
                .addRule(endpointRule_96()).addRule(endpointRule_100()).addRule(endpointRule_116()).build();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }
}
