/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input for <a>EnableEnhancedMonitoring</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EnableEnhancedMonitoringRequest extends KinesisRequest implements
        ToCopyableBuilder<EnableEnhancedMonitoringRequest.Builder, EnableEnhancedMonitoringRequest> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(EnableEnhancedMonitoringRequest::streamName))
            .setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<List<String>> SHARD_LEVEL_METRICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ShardLevelMetrics")
            .getter(getter(EnableEnhancedMonitoringRequest::shardLevelMetricsAsStrings))
            .setter(setter(Builder::shardLevelMetricsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShardLevelMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamARN").getter(getter(EnableEnhancedMonitoringRequest::streamARN))
            .setter(setter(Builder::streamARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD,
            SHARD_LEVEL_METRICS_FIELD, STREAM_ARN_FIELD));

    private final String streamName;

    private final List<String> shardLevelMetrics;

    private final String streamARN;

    private EnableEnhancedMonitoringRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.shardLevelMetrics = builder.shardLevelMetrics;
        this.streamARN = builder.streamARN;
    }

    /**
     * <p>
     * The name of the stream for which to enable enhanced monitoring.
     * </p>
     * 
     * @return The name of the stream for which to enable enhanced monitoring.
     */
    public final String streamName() {
        return streamName;
    }

    /**
     * <p>
     * List of shard-level metrics to enable.
     * </p>
     * <p>
     * The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IncomingBytes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IncomingRecords</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutgoingBytes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutgoingRecords</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WriteProvisionedThroughputExceeded</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadProvisionedThroughputExceeded</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IteratorAgeMilliseconds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the Amazon
     * Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams Developer
     * Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasShardLevelMetrics} method.
     * </p>
     * 
     * @return List of shard-level metrics to enable.</p>
     *         <p>
     *         The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IncomingBytes</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IncomingRecords</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OutgoingBytes</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OutgoingRecords</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WriteProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IteratorAgeMilliseconds</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the
     *         Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams
     *         Developer Guide</i>.
     */
    public final List<MetricsName> shardLevelMetrics() {
        return MetricsNameListCopier.copyStringToEnum(shardLevelMetrics);
    }

    /**
     * For responses, this returns true if the service returned a value for the ShardLevelMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasShardLevelMetrics() {
        return shardLevelMetrics != null && !(shardLevelMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of shard-level metrics to enable.
     * </p>
     * <p>
     * The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IncomingBytes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IncomingRecords</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutgoingBytes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutgoingRecords</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WriteProvisionedThroughputExceeded</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadProvisionedThroughputExceeded</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IteratorAgeMilliseconds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the Amazon
     * Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams Developer
     * Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasShardLevelMetrics} method.
     * </p>
     * 
     * @return List of shard-level metrics to enable.</p>
     *         <p>
     *         The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IncomingBytes</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IncomingRecords</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OutgoingBytes</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OutgoingRecords</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WriteProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IteratorAgeMilliseconds</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the
     *         Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams
     *         Developer Guide</i>.
     */
    public final List<String> shardLevelMetricsAsStrings() {
        return shardLevelMetrics;
    }

    /**
     * <p>
     * The ARN of the stream.
     * </p>
     * 
     * @return The ARN of the stream.
     */
    public final String streamARN() {
        return streamARN;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(hasShardLevelMetrics() ? shardLevelMetricsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(streamARN());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EnableEnhancedMonitoringRequest)) {
            return false;
        }
        EnableEnhancedMonitoringRequest other = (EnableEnhancedMonitoringRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && hasShardLevelMetrics() == other.hasShardLevelMetrics()
                && Objects.equals(shardLevelMetricsAsStrings(), other.shardLevelMetricsAsStrings())
                && Objects.equals(streamARN(), other.streamARN());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EnableEnhancedMonitoringRequest").add("StreamName", streamName())
                .add("ShardLevelMetrics", hasShardLevelMetrics() ? shardLevelMetricsAsStrings() : null)
                .add("StreamARN", streamARN()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "ShardLevelMetrics":
            return Optional.ofNullable(clazz.cast(shardLevelMetricsAsStrings()));
        case "StreamARN":
            return Optional.ofNullable(clazz.cast(streamARN()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EnableEnhancedMonitoringRequest, T> g) {
        return obj -> g.apply((EnableEnhancedMonitoringRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisRequest.Builder, SdkPojo, CopyableBuilder<Builder, EnableEnhancedMonitoringRequest> {
        /**
         * <p>
         * The name of the stream for which to enable enhanced monitoring.
         * </p>
         * 
         * @param streamName
         *        The name of the stream for which to enable enhanced monitoring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * List of shard-level metrics to enable.
         * </p>
         * <p>
         * The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IncomingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IncomingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WriteProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IteratorAgeMilliseconds</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the Amazon
         * Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams Developer
         * Guide</i>.
         * </p>
         * 
         * @param shardLevelMetrics
         *        List of shard-level metrics to enable.</p>
         *        <p>
         *        The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IncomingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IncomingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WriteProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IteratorAgeMilliseconds</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the
         *        Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shardLevelMetricsWithStrings(Collection<String> shardLevelMetrics);

        /**
         * <p>
         * List of shard-level metrics to enable.
         * </p>
         * <p>
         * The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IncomingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IncomingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WriteProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IteratorAgeMilliseconds</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the Amazon
         * Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams Developer
         * Guide</i>.
         * </p>
         * 
         * @param shardLevelMetrics
         *        List of shard-level metrics to enable.</p>
         *        <p>
         *        The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IncomingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IncomingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WriteProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IteratorAgeMilliseconds</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the
         *        Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shardLevelMetricsWithStrings(String... shardLevelMetrics);

        /**
         * <p>
         * List of shard-level metrics to enable.
         * </p>
         * <p>
         * The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IncomingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IncomingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WriteProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IteratorAgeMilliseconds</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the Amazon
         * Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams Developer
         * Guide</i>.
         * </p>
         * 
         * @param shardLevelMetrics
         *        List of shard-level metrics to enable.</p>
         *        <p>
         *        The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IncomingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IncomingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WriteProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IteratorAgeMilliseconds</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the
         *        Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shardLevelMetrics(Collection<MetricsName> shardLevelMetrics);

        /**
         * <p>
         * List of shard-level metrics to enable.
         * </p>
         * <p>
         * The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IncomingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IncomingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingBytes</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OutgoingRecords</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WriteProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadProvisionedThroughputExceeded</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IteratorAgeMilliseconds</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the Amazon
         * Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams Developer
         * Guide</i>.
         * </p>
         * 
         * @param shardLevelMetrics
         *        List of shard-level metrics to enable.</p>
         *        <p>
         *        The following are the valid shard-level metrics. The value "<code>ALL</code>" enables every metric.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IncomingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IncomingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingBytes</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OutgoingRecords</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WriteProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadProvisionedThroughputExceeded</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IteratorAgeMilliseconds</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesis/latest/dev/monitoring-with-cloudwatch.html">Monitoring the
         *        Amazon Kinesis Data Streams Service with Amazon CloudWatch</a> in the <i>Amazon Kinesis Data Streams
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shardLevelMetrics(MetricsName... shardLevelMetrics);

        /**
         * <p>
         * The ARN of the stream.
         * </p>
         * 
         * @param streamARN
         *        The ARN of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamARN(String streamARN);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KinesisRequest.BuilderImpl implements Builder {
        private String streamName;

        private List<String> shardLevelMetrics = DefaultSdkAutoConstructList.getInstance();

        private String streamARN;

        private BuilderImpl() {
        }

        private BuilderImpl(EnableEnhancedMonitoringRequest model) {
            super(model);
            streamName(model.streamName);
            shardLevelMetricsWithStrings(model.shardLevelMetrics);
            streamARN(model.streamARN);
        }

        public final String getStreamName() {
            return streamName;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final Collection<String> getShardLevelMetrics() {
            if (shardLevelMetrics instanceof SdkAutoConstructList) {
                return null;
            }
            return shardLevelMetrics;
        }

        public final void setShardLevelMetrics(Collection<String> shardLevelMetrics) {
            this.shardLevelMetrics = MetricsNameListCopier.copy(shardLevelMetrics);
        }

        @Override
        public final Builder shardLevelMetricsWithStrings(Collection<String> shardLevelMetrics) {
            this.shardLevelMetrics = MetricsNameListCopier.copy(shardLevelMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder shardLevelMetricsWithStrings(String... shardLevelMetrics) {
            shardLevelMetricsWithStrings(Arrays.asList(shardLevelMetrics));
            return this;
        }

        @Override
        public final Builder shardLevelMetrics(Collection<MetricsName> shardLevelMetrics) {
            this.shardLevelMetrics = MetricsNameListCopier.copyEnumToString(shardLevelMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder shardLevelMetrics(MetricsName... shardLevelMetrics) {
            shardLevelMetrics(Arrays.asList(shardLevelMetrics));
            return this;
        }

        public final String getStreamARN() {
            return streamARN;
        }

        public final void setStreamARN(String streamARN) {
            this.streamARN = streamARN;
        }

        @Override
        public final Builder streamARN(String streamARN) {
            this.streamARN = streamARN;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public EnableEnhancedMonitoringRequest build() {
            return new EnableEnhancedMonitoringRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
