/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalytics;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationCloudWatchLoggingOptionRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationCloudWatchLoggingOptionResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationInputProcessingConfigurationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationInputProcessingConfigurationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationInputRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationInputResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationOutputRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationOutputResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationReferenceDataSourceRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.AddApplicationReferenceDataSourceResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.CodeValidationException;
import software.amazon.awssdk.services.kinesisanalytics.model.ConcurrentModificationException;
import software.amazon.awssdk.services.kinesisanalytics.model.CreateApplicationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.CreateApplicationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationCloudWatchLoggingOptionRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationCloudWatchLoggingOptionResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationInputProcessingConfigurationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationInputProcessingConfigurationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationOutputRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationOutputResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationReferenceDataSourceRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationReferenceDataSourceResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DescribeApplicationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DescribeApplicationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.DiscoverInputSchemaRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.DiscoverInputSchemaResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.InvalidApplicationConfigurationException;
import software.amazon.awssdk.services.kinesisanalytics.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisanalytics.model.KinesisAnalyticsException;
import software.amazon.awssdk.services.kinesisanalytics.model.LimitExceededException;
import software.amazon.awssdk.services.kinesisanalytics.model.ListApplicationsRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.ListApplicationsResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesisanalytics.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisanalytics.model.ResourceProvisionedThroughputExceededException;
import software.amazon.awssdk.services.kinesisanalytics.model.ServiceUnavailableException;
import software.amazon.awssdk.services.kinesisanalytics.model.StartApplicationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.StartApplicationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.StopApplicationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.StopApplicationResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.TagResourceRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.TagResourceResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.TooManyTagsException;
import software.amazon.awssdk.services.kinesisanalytics.model.UnableToDetectSchemaException;
import software.amazon.awssdk.services.kinesisanalytics.model.UnsupportedOperationException;
import software.amazon.awssdk.services.kinesisanalytics.model.UntagResourceRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.UntagResourceResponse;
import software.amazon.awssdk.services.kinesisanalytics.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.kinesisanalytics.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.kinesisanalytics.transform.AddApplicationCloudWatchLoggingOptionRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.AddApplicationInputProcessingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.AddApplicationInputRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.AddApplicationOutputRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.AddApplicationReferenceDataSourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DeleteApplicationCloudWatchLoggingOptionRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DeleteApplicationInputProcessingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DeleteApplicationOutputRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DeleteApplicationReferenceDataSourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DescribeApplicationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.DiscoverInputSchemaRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.StartApplicationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.StopApplicationRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisanalytics.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisAnalyticsAsyncClient}.
 *
 * @see KinesisAnalyticsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisAnalyticsAsyncClient implements KinesisAnalyticsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisAnalyticsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKinesisAnalyticsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Adds a CloudWatch log stream to monitor application configuration errors. For more information about using
     * CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
     * CloudWatch Logs</a>.
     * </p>
     *
     * @param addApplicationCloudWatchLoggingOptionRequest
     * @return A Java Future containing the result of the AddApplicationCloudWatchLoggingOption operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.AddApplicationCloudWatchLoggingOption
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/AddApplicationCloudWatchLoggingOption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddApplicationCloudWatchLoggingOptionResponse> addApplicationCloudWatchLoggingOption(
            AddApplicationCloudWatchLoggingOptionRequest addApplicationCloudWatchLoggingOptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                addApplicationCloudWatchLoggingOptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddApplicationCloudWatchLoggingOption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddApplicationCloudWatchLoggingOptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AddApplicationCloudWatchLoggingOptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddApplicationCloudWatchLoggingOptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddApplicationCloudWatchLoggingOptionRequest, AddApplicationCloudWatchLoggingOptionResponse>()
                            .withOperationName("AddApplicationCloudWatchLoggingOption")
                            .withMarshaller(new AddApplicationCloudWatchLoggingOptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addApplicationCloudWatchLoggingOptionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addApplicationCloudWatchLoggingOptionRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<AddApplicationCloudWatchLoggingOptionResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Adds a streaming source to your Amazon Kinesis application. For conceptual information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring Application
     * Input</a>.
     * </p>
     * <p>
     * You can add a streaming source either when you create an application or you can use this operation to add a
     * streaming source after you create an application. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_CreateApplication.html">CreateApplication</a>.
     * </p>
     * <p>
     * Any configuration update, including adding a streaming source using this operation, results in a new version of
     * the application. You can use the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html"
     * >DescribeApplication</a> operation to find the current application version.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:AddApplicationInput</code> action.
     * </p>
     *
     * @param addApplicationInputRequest
     * @return A Java Future containing the result of the AddApplicationInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>CodeValidationException User-provided application code (query) is invalid. This can be a simple
     *         syntax error.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.AddApplicationInput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/AddApplicationInput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddApplicationInputResponse> addApplicationInput(
            AddApplicationInputRequest addApplicationInputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addApplicationInputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddApplicationInput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddApplicationInputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddApplicationInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddApplicationInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddApplicationInputRequest, AddApplicationInputResponse>()
                            .withOperationName("AddApplicationInput")
                            .withMarshaller(new AddApplicationInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addApplicationInputRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addApplicationInputRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AddApplicationInputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Adds an <a href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_InputProcessingConfiguration.html">
     * InputProcessingConfiguration</a> to an application. An input processor preprocesses records on the input stream
     * before the application's SQL code executes. Currently, the only input processor available is <a
     * href="https://docs.aws.amazon.com/lambda/">AWS Lambda</a>.
     * </p>
     *
     * @param addApplicationInputProcessingConfigurationRequest
     * @return A Java Future containing the result of the AddApplicationInputProcessingConfiguration operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.AddApplicationInputProcessingConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/AddApplicationInputProcessingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddApplicationInputProcessingConfigurationResponse> addApplicationInputProcessingConfiguration(
            AddApplicationInputProcessingConfigurationRequest addApplicationInputProcessingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                addApplicationInputProcessingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddApplicationInputProcessingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddApplicationInputProcessingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AddApplicationInputProcessingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddApplicationInputProcessingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddApplicationInputProcessingConfigurationRequest, AddApplicationInputProcessingConfigurationResponse>()
                            .withOperationName("AddApplicationInputProcessingConfiguration")
                            .withMarshaller(new AddApplicationInputProcessingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(addApplicationInputProcessingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addApplicationInputProcessingConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<AddApplicationInputProcessingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Adds an external destination to your Amazon Kinesis Analytics application.
     * </p>
     * <p>
     * If you want Amazon Kinesis Analytics to deliver data from an in-application stream within your application to an
     * external destination (such as an Amazon Kinesis stream, an Amazon Kinesis Firehose delivery stream, or an AWS
     * Lambda function), you add the relevant configuration to your application using this operation. You can configure
     * one or more outputs for your application. Each output configuration maps an in-application stream and an external
     * destination.
     * </p>
     * <p>
     * You can use one of the output configurations to deliver data from your in-application error stream to an external
     * destination so that you can analyze the errors. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Understanding Application
     * Output (Destination)</a>.
     * </p>
     * <p>
     * Any configuration update, including adding a streaming source using this operation, results in a new version of
     * the application. You can use the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html"
     * >DescribeApplication</a> operation to find the current application version.
     * </p>
     * <p>
     * For the limits on the number of application inputs and outputs you can configure, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/limits.html">Limits</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:AddApplicationOutput</code> action.
     * </p>
     *
     * @param addApplicationOutputRequest
     * @return A Java Future containing the result of the AddApplicationOutput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.AddApplicationOutput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/AddApplicationOutput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddApplicationOutputResponse> addApplicationOutput(
            AddApplicationOutputRequest addApplicationOutputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addApplicationOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddApplicationOutput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddApplicationOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddApplicationOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddApplicationOutputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddApplicationOutputRequest, AddApplicationOutputResponse>()
                            .withOperationName("AddApplicationOutput")
                            .withMarshaller(new AddApplicationOutputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addApplicationOutputRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addApplicationOutputRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AddApplicationOutputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Adds a reference data source to an existing application.
     * </p>
     * <p>
     * Amazon Kinesis Analytics reads reference data (that is, an Amazon S3 object) and creates an in-application table
     * within your application. In the request, you provide the source (S3 bucket name and object key name), name of the
     * in-application table to create, and the necessary mapping information that describes how data in Amazon S3 object
     * maps to columns in the resulting in-application table.
     * </p>
     * <p>
     * For conceptual information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring Application
     * Input</a>. For the limits on data sources you can add to your application, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/limits.html">Limits</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:AddApplicationOutput</code> action.
     * </p>
     *
     * @param addApplicationReferenceDataSourceRequest
     * @return A Java Future containing the result of the AddApplicationReferenceDataSource operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.AddApplicationReferenceDataSource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/AddApplicationReferenceDataSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddApplicationReferenceDataSourceResponse> addApplicationReferenceDataSource(
            AddApplicationReferenceDataSourceRequest addApplicationReferenceDataSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                addApplicationReferenceDataSourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddApplicationReferenceDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddApplicationReferenceDataSourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AddApplicationReferenceDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddApplicationReferenceDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddApplicationReferenceDataSourceRequest, AddApplicationReferenceDataSourceResponse>()
                            .withOperationName("AddApplicationReferenceDataSource")
                            .withMarshaller(new AddApplicationReferenceDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addApplicationReferenceDataSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addApplicationReferenceDataSourceRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<AddApplicationReferenceDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Creates an Amazon Kinesis Analytics application. You can configure each application with one streaming source as
     * input, application code to process the input, and up to three destinations where you want Amazon Kinesis
     * Analytics to write the output data from your application. For an overview, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works.html">How it Works</a>.
     * </p>
     * <p>
     * In the input configuration, you map the streaming source to an in-application stream, which you can think of as a
     * constantly updating table. In the mapping, you must provide a schema for the in-application stream and map each
     * data column in the in-application stream to a data element in the streaming source.
     * </p>
     * <p>
     * Your application code is one or more SQL statements that read input data, transform it, and generate output. Your
     * application code can create one or more SQL artifacts like SQL streams or pumps.
     * </p>
     * <p>
     * In the output configuration, you can configure the application to write data from in-application streams created
     * in your applications to up to three destinations.
     * </p>
     * <p>
     * To read data from your source stream or write data to destination streams, Amazon Kinesis Analytics needs your
     * permissions. You grant these permissions by creating IAM roles. This operation requires permissions to perform
     * the <code>kinesisanalytics:CreateApplication</code> action.
     * </p>
     * <p>
     * For introductory exercises to create an Amazon Kinesis Analytics application, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/getting-started.html">Getting Started</a>.
     * </p>
     *
     * @param createApplicationRequest
     *        TBD
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CodeValidationException User-provided application code (query) is invalid. This can be a simple
     *         syntax error.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>LimitExceededException Exceeded the number of applications allowed.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>TooManyTagsException Application created with too many tags, or too many tags added to an
     *         application. Note that the maximum number of application tags includes system tags. The maximum number of
     *         user-defined application tags is 50.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication")
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Deletes the specified application. Amazon Kinesis Analytics halts application execution and deletes the
     * application, including any application artifacts (such as in-application streams, reference table, and
     * application code).
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:DeleteApplication</code> action.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication")
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Deletes a CloudWatch log stream from an application. For more information about using CloudWatch log streams with
     * Amazon Kinesis Analytics applications, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
     * CloudWatch Logs</a>.
     * </p>
     *
     * @param deleteApplicationCloudWatchLoggingOptionRequest
     * @return A Java Future containing the result of the DeleteApplicationCloudWatchLoggingOption operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DeleteApplicationCloudWatchLoggingOption
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DeleteApplicationCloudWatchLoggingOption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationCloudWatchLoggingOptionResponse> deleteApplicationCloudWatchLoggingOption(
            DeleteApplicationCloudWatchLoggingOptionRequest deleteApplicationCloudWatchLoggingOptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteApplicationCloudWatchLoggingOptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplicationCloudWatchLoggingOption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationCloudWatchLoggingOptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteApplicationCloudWatchLoggingOptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationCloudWatchLoggingOptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationCloudWatchLoggingOptionRequest, DeleteApplicationCloudWatchLoggingOptionResponse>()
                            .withOperationName("DeleteApplicationCloudWatchLoggingOption")
                            .withMarshaller(new DeleteApplicationCloudWatchLoggingOptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApplicationCloudWatchLoggingOptionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApplicationCloudWatchLoggingOptionRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApplicationCloudWatchLoggingOptionResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Deletes an <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_InputProcessingConfiguration.html"
     * >InputProcessingConfiguration</a> from an input.
     * </p>
     *
     * @param deleteApplicationInputProcessingConfigurationRequest
     * @return A Java Future containing the result of the DeleteApplicationInputProcessingConfiguration operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DeleteApplicationInputProcessingConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DeleteApplicationInputProcessingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationInputProcessingConfigurationResponse> deleteApplicationInputProcessingConfiguration(
            DeleteApplicationInputProcessingConfigurationRequest deleteApplicationInputProcessingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteApplicationInputProcessingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplicationInputProcessingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationInputProcessingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteApplicationInputProcessingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationInputProcessingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationInputProcessingConfigurationRequest, DeleteApplicationInputProcessingConfigurationResponse>()
                            .withOperationName("DeleteApplicationInputProcessingConfiguration")
                            .withMarshaller(new DeleteApplicationInputProcessingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApplicationInputProcessingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApplicationInputProcessingConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApplicationInputProcessingConfigurationResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Deletes output destination configuration from your application configuration. Amazon Kinesis Analytics will no
     * longer write data from the corresponding in-application stream to the external output destination.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:DeleteApplicationOutput</code> action.
     * </p>
     *
     * @param deleteApplicationOutputRequest
     * @return A Java Future containing the result of the DeleteApplicationOutput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DeleteApplicationOutput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DeleteApplicationOutput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationOutputResponse> deleteApplicationOutput(
            DeleteApplicationOutputRequest deleteApplicationOutputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplicationOutput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationOutputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationOutputRequest, DeleteApplicationOutputResponse>()
                            .withOperationName("DeleteApplicationOutput")
                            .withMarshaller(new DeleteApplicationOutputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApplicationOutputRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApplicationOutputRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteApplicationOutputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Deletes a reference data source configuration from the specified application configuration.
     * </p>
     * <p>
     * If the application is running, Amazon Kinesis Analytics immediately removes the in-application table that you
     * created using the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_AddApplicationReferenceDataSource.html"
     * >AddApplicationReferenceDataSource</a> operation.
     * </p>
     * <p>
     * This operation requires permissions to perform the
     * <code>kinesisanalytics.DeleteApplicationReferenceDataSource</code> action.
     * </p>
     *
     * @param deleteApplicationReferenceDataSourceRequest
     * @return A Java Future containing the result of the DeleteApplicationReferenceDataSource operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DeleteApplicationReferenceDataSource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DeleteApplicationReferenceDataSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationReferenceDataSourceResponse> deleteApplicationReferenceDataSource(
            DeleteApplicationReferenceDataSourceRequest deleteApplicationReferenceDataSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteApplicationReferenceDataSourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplicationReferenceDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationReferenceDataSourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteApplicationReferenceDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationReferenceDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationReferenceDataSourceRequest, DeleteApplicationReferenceDataSourceResponse>()
                            .withOperationName("DeleteApplicationReferenceDataSource")
                            .withMarshaller(new DeleteApplicationReferenceDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApplicationReferenceDataSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApplicationReferenceDataSourceRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApplicationReferenceDataSourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Returns information about a specific Amazon Kinesis Analytics application.
     * </p>
     * <p>
     * If you want to retrieve a list of all applications in your account, use the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_ListApplications.html">ListApplications</a>
     * operation.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:DescribeApplication</code> action. You
     * can use <code>DescribeApplication</code> to get the current application versionId, which you need to call other
     * operations such as <code>Update</code>.
     * </p>
     *
     * @param describeApplicationRequest
     * @return A Java Future containing the result of the DescribeApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DescribeApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DescribeApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeApplicationResponse> describeApplication(
            DescribeApplicationRequest describeApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationRequest, DescribeApplicationResponse>()
                            .withOperationName("DescribeApplication")
                            .withMarshaller(new DescribeApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeApplicationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Infers a schema by evaluating sample records on the specified streaming source (Amazon Kinesis stream or Amazon
     * Kinesis Firehose delivery stream) or S3 object. In the response, the operation returns the inferred schema and
     * also the sample records that the operation used to infer the schema.
     * </p>
     * <p>
     * You can use the inferred schema when configuring a streaming source for your application. For conceptual
     * information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring Application
     * Input</a>. Note that when you create an application using the Amazon Kinesis Analytics console, the console uses
     * this operation to infer a schema and show it in the console user interface.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:DiscoverInputSchema</code> action.
     * </p>
     *
     * @param discoverInputSchemaRequest
     * @return A Java Future containing the result of the DiscoverInputSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>UnableToDetectSchemaException Data format is not valid. Amazon Kinesis Analytics is not able to
     *         detect schema for the given streaming source.</li>
     *         <li>ResourceProvisionedThroughputExceededException Discovery failed to get a record from the streaming
     *         source because of the Amazon Kinesis Streams ProvisionedThroughputExceededException. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html">GetRecords</a> in the
     *         Amazon Kinesis Streams API Reference.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.DiscoverInputSchema
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/DiscoverInputSchema"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DiscoverInputSchemaResponse> discoverInputSchema(
            DiscoverInputSchemaRequest discoverInputSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, discoverInputSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DiscoverInputSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DiscoverInputSchemaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DiscoverInputSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DiscoverInputSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DiscoverInputSchemaRequest, DiscoverInputSchemaResponse>()
                            .withOperationName("DiscoverInputSchema")
                            .withMarshaller(new DiscoverInputSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(discoverInputSchemaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = discoverInputSchemaRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DiscoverInputSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Returns a list of Amazon Kinesis Analytics applications in your account. For each application, the response
     * includes the application name, Amazon Resource Name (ARN), and status. If the response returns the
     * <code>HasMoreApplications</code> value as true, you can send another request by adding the
     * <code>ExclusiveStartApplicationName</code> in the request body, and set the value of this to the last application
     * name from the previous response.
     * </p>
     * <p>
     * If you want detailed information about a specific application, use <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html"
     * >DescribeApplication</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:ListApplications</code> action.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications")
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listApplicationsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of key-value tags assigned to the application. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-tagging.html">Using Tagging</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Starts the specified Amazon Kinesis Analytics application. After creating an application, you must exclusively
     * call this operation to start your application.
     * </p>
     * <p>
     * After the application starts, it begins consuming the input data, processes it, and writes the output to the
     * configured destination.
     * </p>
     * <p>
     * The application status must be <code>READY</code> for you to start an application. You can get the application
     * status in the console or using the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html"
     * >DescribeApplication</a> operation.
     * </p>
     * <p>
     * After you start the application, you can stop the application from processing the input by calling the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_StopApplication.html">StopApplication</a>
     * operation.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:StartApplication</code> action.
     * </p>
     *
     * @param startApplicationRequest
     * @return A Java Future containing the result of the StartApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>InvalidApplicationConfigurationException User-provided application configuration is not valid.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.StartApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/StartApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartApplicationResponse> startApplication(StartApplicationRequest startApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartApplicationRequest, StartApplicationResponse>()
                            .withOperationName("StartApplication")
                            .withMarshaller(new StartApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Stops the application from processing input data. You can stop an application only if it is in the running state.
     * You can use the <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html">DescribeApplication
     * </a> operation to find the application state. After the application is stopped, Amazon Kinesis Analytics stops
     * reading data from the input, the application stops processing data, and there is no output written to the
     * destination.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>kinesisanalytics:StopApplication</code> action.
     * </p>
     *
     * @param stopApplicationRequest
     * @return A Java Future containing the result of the StopApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.StopApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/StopApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopApplicationResponse> stopApplication(StopApplicationRequest stopApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopApplicationRequest, StopApplicationResponse>()
                            .withOperationName("StopApplication")
                            .withMarshaller(new StopApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more key-value tags to a Kinesis Analytics application. Note that the maximum number of application
     * tags includes system tags. The maximum number of user-defined application tags is 50. For more information, see
     * <a href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-tagging.html">Using Tagging</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>TooManyTagsException Application created with too many tags, or too many tags added to an
     *         application. Note that the maximum number of application tags includes system tags. The maximum number of
     *         user-defined application tags is 50.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a Kinesis Analytics application. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-tagging.html">Using Tagging</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>TooManyTagsException Application created with too many tags, or too many tags added to an
     *         application. Note that the maximum number of application tags includes system tags. The maximum number of
     *         user-defined application tags is 50.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL
     * applications. Version 2 of the API supports SQL and Java applications. For more information about version 2, see
     * <a href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
     * </p>
     * </note>
     * <p>
     * Updates an existing Amazon Kinesis Analytics application. Using this API, you can update application code, input
     * configuration, and output configuration.
     * </p>
     * <p>
     * Note that Amazon Kinesis Analytics updates the <code>CurrentApplicationVersionId</code> each time you update your
     * application.
     * </p>
     * <p>
     * This operation requires permission for the <code>kinesisanalytics:UpdateApplication</code> action.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CodeValidationException User-provided application code (query) is invalid. This can be a simple
     *         syntax error.</li>
     *         <li>ResourceNotFoundException Specified application can't be found.</li>
     *         <li>ResourceInUseException Application is not available for this operation.</li>
     *         <li>InvalidArgumentException Specified input parameter value is invalid.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>UnsupportedOperationException The request was rejected because a specified parameter is not supported
     *         or a specified resource is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample KinesisAnalyticsAsyncClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisanalytics-2015-08-14/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication")
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisAnalyticsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnableToDetectSchemaException")
                                .exceptionBuilderSupplier(UnableToDetectSchemaException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceProvisionedThroughputExceededException")
                                .exceptionBuilderSupplier(ResourceProvisionedThroughputExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApplicationConfigurationException")
                                .exceptionBuilderSupplier(InvalidApplicationConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CodeValidationException")
                                .exceptionBuilderSupplier(CodeValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
