/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalytics.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the format of the data in the streaming source, and how each data element maps to corresponding columns
 * created in the in-application stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceSchema implements SdkPojo, Serializable, ToCopyableBuilder<SourceSchema.Builder, SourceSchema> {
    private static final SdkField<RecordFormat> RECORD_FORMAT_FIELD = SdkField.<RecordFormat> builder(MarshallingType.SDK_POJO)
            .memberName("RecordFormat").getter(getter(SourceSchema::recordFormat)).setter(setter(Builder::recordFormat))
            .constructor(RecordFormat::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordFormat").build()).build();

    private static final SdkField<String> RECORD_ENCODING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecordEncoding").getter(getter(SourceSchema::recordEncoding)).setter(setter(Builder::recordEncoding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordEncoding").build()).build();

    private static final SdkField<List<RecordColumn>> RECORD_COLUMNS_FIELD = SdkField
            .<List<RecordColumn>> builder(MarshallingType.LIST)
            .memberName("RecordColumns")
            .getter(getter(SourceSchema::recordColumns))
            .setter(setter(Builder::recordColumns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordColumns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RecordColumn> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecordColumn::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RECORD_FORMAT_FIELD,
            RECORD_ENCODING_FIELD, RECORD_COLUMNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final RecordFormat recordFormat;

    private final String recordEncoding;

    private final List<RecordColumn> recordColumns;

    private SourceSchema(BuilderImpl builder) {
        this.recordFormat = builder.recordFormat;
        this.recordEncoding = builder.recordEncoding;
        this.recordColumns = builder.recordColumns;
    }

    /**
     * <p>
     * Specifies the format of the records on the streaming source.
     * </p>
     * 
     * @return Specifies the format of the records on the streaming source.
     */
    public final RecordFormat recordFormat() {
        return recordFormat;
    }

    /**
     * <p>
     * Specifies the encoding of the records in the streaming source. For example, UTF-8.
     * </p>
     * 
     * @return Specifies the encoding of the records in the streaming source. For example, UTF-8.
     */
    public final String recordEncoding() {
        return recordEncoding;
    }

    /**
     * Returns true if the RecordColumns property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasRecordColumns() {
        return recordColumns != null && !(recordColumns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>RecordColumn</code> objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecordColumns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>RecordColumn</code> objects.
     */
    public final List<RecordColumn> recordColumns() {
        return recordColumns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(recordFormat());
        hashCode = 31 * hashCode + Objects.hashCode(recordEncoding());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecordColumns() ? recordColumns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceSchema)) {
            return false;
        }
        SourceSchema other = (SourceSchema) obj;
        return Objects.equals(recordFormat(), other.recordFormat()) && Objects.equals(recordEncoding(), other.recordEncoding())
                && hasRecordColumns() == other.hasRecordColumns() && Objects.equals(recordColumns(), other.recordColumns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceSchema").add("RecordFormat", recordFormat()).add("RecordEncoding", recordEncoding())
                .add("RecordColumns", hasRecordColumns() ? recordColumns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RecordFormat":
            return Optional.ofNullable(clazz.cast(recordFormat()));
        case "RecordEncoding":
            return Optional.ofNullable(clazz.cast(recordEncoding()));
        case "RecordColumns":
            return Optional.ofNullable(clazz.cast(recordColumns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SourceSchema, T> g) {
        return obj -> g.apply((SourceSchema) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceSchema> {
        /**
         * <p>
         * Specifies the format of the records on the streaming source.
         * </p>
         * 
         * @param recordFormat
         *        Specifies the format of the records on the streaming source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordFormat(RecordFormat recordFormat);

        /**
         * <p>
         * Specifies the format of the records on the streaming source.
         * </p>
         * This is a convenience that creates an instance of the {@link RecordFormat.Builder} avoiding the need to
         * create one manually via {@link RecordFormat#builder()}.
         *
         * When the {@link Consumer} completes, {@link RecordFormat.Builder#build()} is called immediately and its
         * result is passed to {@link #recordFormat(RecordFormat)}.
         * 
         * @param recordFormat
         *        a consumer that will call methods on {@link RecordFormat.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordFormat(RecordFormat)
         */
        default Builder recordFormat(Consumer<RecordFormat.Builder> recordFormat) {
            return recordFormat(RecordFormat.builder().applyMutation(recordFormat).build());
        }

        /**
         * <p>
         * Specifies the encoding of the records in the streaming source. For example, UTF-8.
         * </p>
         * 
         * @param recordEncoding
         *        Specifies the encoding of the records in the streaming source. For example, UTF-8.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordEncoding(String recordEncoding);

        /**
         * <p>
         * A list of <code>RecordColumn</code> objects.
         * </p>
         * 
         * @param recordColumns
         *        A list of <code>RecordColumn</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordColumns(Collection<RecordColumn> recordColumns);

        /**
         * <p>
         * A list of <code>RecordColumn</code> objects.
         * </p>
         * 
         * @param recordColumns
         *        A list of <code>RecordColumn</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordColumns(RecordColumn... recordColumns);

        /**
         * <p>
         * A list of <code>RecordColumn</code> objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RecordColumn>.Builder} avoiding the need to
         * create one manually via {@link List<RecordColumn>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RecordColumn>.Builder#build()} is called immediately and its
         * result is passed to {@link #recordColumns(List<RecordColumn>)}.
         * 
         * @param recordColumns
         *        a consumer that will call methods on {@link List<RecordColumn>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordColumns(List<RecordColumn>)
         */
        Builder recordColumns(Consumer<RecordColumn.Builder>... recordColumns);
    }

    static final class BuilderImpl implements Builder {
        private RecordFormat recordFormat;

        private String recordEncoding;

        private List<RecordColumn> recordColumns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SourceSchema model) {
            recordFormat(model.recordFormat);
            recordEncoding(model.recordEncoding);
            recordColumns(model.recordColumns);
        }

        public final RecordFormat.Builder getRecordFormat() {
            return recordFormat != null ? recordFormat.toBuilder() : null;
        }

        @Override
        public final Builder recordFormat(RecordFormat recordFormat) {
            this.recordFormat = recordFormat;
            return this;
        }

        public final void setRecordFormat(RecordFormat.BuilderImpl recordFormat) {
            this.recordFormat = recordFormat != null ? recordFormat.build() : null;
        }

        public final String getRecordEncoding() {
            return recordEncoding;
        }

        @Override
        public final Builder recordEncoding(String recordEncoding) {
            this.recordEncoding = recordEncoding;
            return this;
        }

        public final void setRecordEncoding(String recordEncoding) {
            this.recordEncoding = recordEncoding;
        }

        public final List<RecordColumn.Builder> getRecordColumns() {
            List<RecordColumn.Builder> result = RecordColumnsCopier.copyToBuilder(this.recordColumns);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder recordColumns(Collection<RecordColumn> recordColumns) {
            this.recordColumns = RecordColumnsCopier.copy(recordColumns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordColumns(RecordColumn... recordColumns) {
            recordColumns(Arrays.asList(recordColumns));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordColumns(Consumer<RecordColumn.Builder>... recordColumns) {
            recordColumns(Stream.of(recordColumns).map(c -> RecordColumn.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRecordColumns(Collection<RecordColumn.BuilderImpl> recordColumns) {
            this.recordColumns = RecordColumnsCopier.copyFromBuilder(recordColumns);
        }

        @Override
        public SourceSchema build() {
            return new SourceSchema(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
