/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalyticsv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the application input configuration for a SQL-based Kinesis Data Analytics application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InputDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<InputDescription.Builder, InputDescription> {
    private static final SdkField<String> INPUT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputId").getter(getter(InputDescription::inputId)).setter(setter(Builder::inputId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputId").build()).build();

    private static final SdkField<String> NAME_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NamePrefix").getter(getter(InputDescription::namePrefix)).setter(setter(Builder::namePrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NamePrefix").build()).build();

    private static final SdkField<List<String>> IN_APP_STREAM_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InAppStreamNames")
            .getter(getter(InputDescription::inAppStreamNames))
            .setter(setter(Builder::inAppStreamNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InAppStreamNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<InputProcessingConfigurationDescription> INPUT_PROCESSING_CONFIGURATION_DESCRIPTION_FIELD = SdkField
            .<InputProcessingConfigurationDescription> builder(MarshallingType.SDK_POJO)
            .memberName("InputProcessingConfigurationDescription")
            .getter(getter(InputDescription::inputProcessingConfigurationDescription))
            .setter(setter(Builder::inputProcessingConfigurationDescription))
            .constructor(InputProcessingConfigurationDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("InputProcessingConfigurationDescription").build()).build();

    private static final SdkField<KinesisStreamsInputDescription> KINESIS_STREAMS_INPUT_DESCRIPTION_FIELD = SdkField
            .<KinesisStreamsInputDescription> builder(MarshallingType.SDK_POJO)
            .memberName("KinesisStreamsInputDescription")
            .getter(getter(InputDescription::kinesisStreamsInputDescription))
            .setter(setter(Builder::kinesisStreamsInputDescription))
            .constructor(KinesisStreamsInputDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KinesisStreamsInputDescription")
                    .build()).build();

    private static final SdkField<KinesisFirehoseInputDescription> KINESIS_FIREHOSE_INPUT_DESCRIPTION_FIELD = SdkField
            .<KinesisFirehoseInputDescription> builder(MarshallingType.SDK_POJO)
            .memberName("KinesisFirehoseInputDescription")
            .getter(getter(InputDescription::kinesisFirehoseInputDescription))
            .setter(setter(Builder::kinesisFirehoseInputDescription))
            .constructor(KinesisFirehoseInputDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KinesisFirehoseInputDescription")
                    .build()).build();

    private static final SdkField<SourceSchema> INPUT_SCHEMA_FIELD = SdkField.<SourceSchema> builder(MarshallingType.SDK_POJO)
            .memberName("InputSchema").getter(getter(InputDescription::inputSchema)).setter(setter(Builder::inputSchema))
            .constructor(SourceSchema::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputSchema").build()).build();

    private static final SdkField<InputParallelism> INPUT_PARALLELISM_FIELD = SdkField
            .<InputParallelism> builder(MarshallingType.SDK_POJO).memberName("InputParallelism")
            .getter(getter(InputDescription::inputParallelism)).setter(setter(Builder::inputParallelism))
            .constructor(InputParallelism::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputParallelism").build()).build();

    private static final SdkField<InputStartingPositionConfiguration> INPUT_STARTING_POSITION_CONFIGURATION_FIELD = SdkField
            .<InputStartingPositionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("InputStartingPositionConfiguration")
            .getter(getter(InputDescription::inputStartingPositionConfiguration))
            .setter(setter(Builder::inputStartingPositionConfiguration))
            .constructor(InputStartingPositionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputStartingPositionConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INPUT_ID_FIELD,
            NAME_PREFIX_FIELD, IN_APP_STREAM_NAMES_FIELD, INPUT_PROCESSING_CONFIGURATION_DESCRIPTION_FIELD,
            KINESIS_STREAMS_INPUT_DESCRIPTION_FIELD, KINESIS_FIREHOSE_INPUT_DESCRIPTION_FIELD, INPUT_SCHEMA_FIELD,
            INPUT_PARALLELISM_FIELD, INPUT_STARTING_POSITION_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String inputId;

    private final String namePrefix;

    private final List<String> inAppStreamNames;

    private final InputProcessingConfigurationDescription inputProcessingConfigurationDescription;

    private final KinesisStreamsInputDescription kinesisStreamsInputDescription;

    private final KinesisFirehoseInputDescription kinesisFirehoseInputDescription;

    private final SourceSchema inputSchema;

    private final InputParallelism inputParallelism;

    private final InputStartingPositionConfiguration inputStartingPositionConfiguration;

    private InputDescription(BuilderImpl builder) {
        this.inputId = builder.inputId;
        this.namePrefix = builder.namePrefix;
        this.inAppStreamNames = builder.inAppStreamNames;
        this.inputProcessingConfigurationDescription = builder.inputProcessingConfigurationDescription;
        this.kinesisStreamsInputDescription = builder.kinesisStreamsInputDescription;
        this.kinesisFirehoseInputDescription = builder.kinesisFirehoseInputDescription;
        this.inputSchema = builder.inputSchema;
        this.inputParallelism = builder.inputParallelism;
        this.inputStartingPositionConfiguration = builder.inputStartingPositionConfiguration;
    }

    /**
     * <p>
     * The input ID that is associated with the application input. This is the ID that Kinesis Data Analytics assigns to
     * each input configuration that you add to your application.
     * </p>
     * 
     * @return The input ID that is associated with the application input. This is the ID that Kinesis Data Analytics
     *         assigns to each input configuration that you add to your application.
     */
    public final String inputId() {
        return inputId;
    }

    /**
     * <p>
     * The in-application name prefix.
     * </p>
     * 
     * @return The in-application name prefix.
     */
    public final String namePrefix() {
        return namePrefix;
    }

    /**
     * For responses, this returns true if the service returned a value for the InAppStreamNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInAppStreamNames() {
        return inAppStreamNames != null && !(inAppStreamNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Returns the in-application stream names that are mapped to the stream source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInAppStreamNames} method.
     * </p>
     * 
     * @return Returns the in-application stream names that are mapped to the stream source.
     */
    public final List<String> inAppStreamNames() {
        return inAppStreamNames;
    }

    /**
     * <p>
     * The description of the preprocessor that executes on records in this input before the application's code is run.
     * </p>
     * 
     * @return The description of the preprocessor that executes on records in this input before the application's code
     *         is run.
     */
    public final InputProcessingConfigurationDescription inputProcessingConfigurationDescription() {
        return inputProcessingConfigurationDescription;
    }

    /**
     * <p>
     * If a Kinesis data stream is configured as a streaming source, provides the Kinesis data stream's Amazon Resource
     * Name (ARN).
     * </p>
     * 
     * @return If a Kinesis data stream is configured as a streaming source, provides the Kinesis data stream's Amazon
     *         Resource Name (ARN).
     */
    public final KinesisStreamsInputDescription kinesisStreamsInputDescription() {
        return kinesisStreamsInputDescription;
    }

    /**
     * <p>
     * If a Kinesis Data Firehose delivery stream is configured as a streaming source, provides the delivery stream's
     * ARN.
     * </p>
     * 
     * @return If a Kinesis Data Firehose delivery stream is configured as a streaming source, provides the delivery
     *         stream's ARN.
     */
    public final KinesisFirehoseInputDescription kinesisFirehoseInputDescription() {
        return kinesisFirehoseInputDescription;
    }

    /**
     * <p>
     * Describes the format of the data in the streaming source, and how each data element maps to corresponding columns
     * in the in-application stream that is being created.
     * </p>
     * 
     * @return Describes the format of the data in the streaming source, and how each data element maps to corresponding
     *         columns in the in-application stream that is being created.
     */
    public final SourceSchema inputSchema() {
        return inputSchema;
    }

    /**
     * <p>
     * Describes the configured parallelism (number of in-application streams mapped to the streaming source).
     * </p>
     * 
     * @return Describes the configured parallelism (number of in-application streams mapped to the streaming source).
     */
    public final InputParallelism inputParallelism() {
        return inputParallelism;
    }

    /**
     * <p>
     * The point at which the application is configured to read from the input stream.
     * </p>
     * 
     * @return The point at which the application is configured to read from the input stream.
     */
    public final InputStartingPositionConfiguration inputStartingPositionConfiguration() {
        return inputStartingPositionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(inputId());
        hashCode = 31 * hashCode + Objects.hashCode(namePrefix());
        hashCode = 31 * hashCode + Objects.hashCode(hasInAppStreamNames() ? inAppStreamNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(inputProcessingConfigurationDescription());
        hashCode = 31 * hashCode + Objects.hashCode(kinesisStreamsInputDescription());
        hashCode = 31 * hashCode + Objects.hashCode(kinesisFirehoseInputDescription());
        hashCode = 31 * hashCode + Objects.hashCode(inputSchema());
        hashCode = 31 * hashCode + Objects.hashCode(inputParallelism());
        hashCode = 31 * hashCode + Objects.hashCode(inputStartingPositionConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InputDescription)) {
            return false;
        }
        InputDescription other = (InputDescription) obj;
        return Objects.equals(inputId(), other.inputId()) && Objects.equals(namePrefix(), other.namePrefix())
                && hasInAppStreamNames() == other.hasInAppStreamNames()
                && Objects.equals(inAppStreamNames(), other.inAppStreamNames())
                && Objects.equals(inputProcessingConfigurationDescription(), other.inputProcessingConfigurationDescription())
                && Objects.equals(kinesisStreamsInputDescription(), other.kinesisStreamsInputDescription())
                && Objects.equals(kinesisFirehoseInputDescription(), other.kinesisFirehoseInputDescription())
                && Objects.equals(inputSchema(), other.inputSchema())
                && Objects.equals(inputParallelism(), other.inputParallelism())
                && Objects.equals(inputStartingPositionConfiguration(), other.inputStartingPositionConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InputDescription").add("InputId", inputId()).add("NamePrefix", namePrefix())
                .add("InAppStreamNames", hasInAppStreamNames() ? inAppStreamNames() : null)
                .add("InputProcessingConfigurationDescription", inputProcessingConfigurationDescription())
                .add("KinesisStreamsInputDescription", kinesisStreamsInputDescription())
                .add("KinesisFirehoseInputDescription", kinesisFirehoseInputDescription()).add("InputSchema", inputSchema())
                .add("InputParallelism", inputParallelism())
                .add("InputStartingPositionConfiguration", inputStartingPositionConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InputId":
            return Optional.ofNullable(clazz.cast(inputId()));
        case "NamePrefix":
            return Optional.ofNullable(clazz.cast(namePrefix()));
        case "InAppStreamNames":
            return Optional.ofNullable(clazz.cast(inAppStreamNames()));
        case "InputProcessingConfigurationDescription":
            return Optional.ofNullable(clazz.cast(inputProcessingConfigurationDescription()));
        case "KinesisStreamsInputDescription":
            return Optional.ofNullable(clazz.cast(kinesisStreamsInputDescription()));
        case "KinesisFirehoseInputDescription":
            return Optional.ofNullable(clazz.cast(kinesisFirehoseInputDescription()));
        case "InputSchema":
            return Optional.ofNullable(clazz.cast(inputSchema()));
        case "InputParallelism":
            return Optional.ofNullable(clazz.cast(inputParallelism()));
        case "InputStartingPositionConfiguration":
            return Optional.ofNullable(clazz.cast(inputStartingPositionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InputDescription, T> g) {
        return obj -> g.apply((InputDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InputDescription> {
        /**
         * <p>
         * The input ID that is associated with the application input. This is the ID that Kinesis Data Analytics
         * assigns to each input configuration that you add to your application.
         * </p>
         * 
         * @param inputId
         *        The input ID that is associated with the application input. This is the ID that Kinesis Data Analytics
         *        assigns to each input configuration that you add to your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputId(String inputId);

        /**
         * <p>
         * The in-application name prefix.
         * </p>
         * 
         * @param namePrefix
         *        The in-application name prefix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namePrefix(String namePrefix);

        /**
         * <p>
         * Returns the in-application stream names that are mapped to the stream source.
         * </p>
         * 
         * @param inAppStreamNames
         *        Returns the in-application stream names that are mapped to the stream source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inAppStreamNames(Collection<String> inAppStreamNames);

        /**
         * <p>
         * Returns the in-application stream names that are mapped to the stream source.
         * </p>
         * 
         * @param inAppStreamNames
         *        Returns the in-application stream names that are mapped to the stream source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inAppStreamNames(String... inAppStreamNames);

        /**
         * <p>
         * The description of the preprocessor that executes on records in this input before the application's code is
         * run.
         * </p>
         * 
         * @param inputProcessingConfigurationDescription
         *        The description of the preprocessor that executes on records in this input before the application's
         *        code is run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputProcessingConfigurationDescription(
                InputProcessingConfigurationDescription inputProcessingConfigurationDescription);

        /**
         * <p>
         * The description of the preprocessor that executes on records in this input before the application's code is
         * run.
         * </p>
         * This is a convenience that creates an instance of the {@link InputProcessingConfigurationDescription.Builder}
         * avoiding the need to create one manually via {@link InputProcessingConfigurationDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link InputProcessingConfigurationDescription.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #inputProcessingConfigurationDescription(InputProcessingConfigurationDescription)}.
         * 
         * @param inputProcessingConfigurationDescription
         *        a consumer that will call methods on {@link InputProcessingConfigurationDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputProcessingConfigurationDescription(InputProcessingConfigurationDescription)
         */
        default Builder inputProcessingConfigurationDescription(
                Consumer<InputProcessingConfigurationDescription.Builder> inputProcessingConfigurationDescription) {
            return inputProcessingConfigurationDescription(InputProcessingConfigurationDescription.builder()
                    .applyMutation(inputProcessingConfigurationDescription).build());
        }

        /**
         * <p>
         * If a Kinesis data stream is configured as a streaming source, provides the Kinesis data stream's Amazon
         * Resource Name (ARN).
         * </p>
         * 
         * @param kinesisStreamsInputDescription
         *        If a Kinesis data stream is configured as a streaming source, provides the Kinesis data stream's
         *        Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisStreamsInputDescription(KinesisStreamsInputDescription kinesisStreamsInputDescription);

        /**
         * <p>
         * If a Kinesis data stream is configured as a streaming source, provides the Kinesis data stream's Amazon
         * Resource Name (ARN).
         * </p>
         * This is a convenience that creates an instance of the {@link KinesisStreamsInputDescription.Builder} avoiding
         * the need to create one manually via {@link KinesisStreamsInputDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link KinesisStreamsInputDescription.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #kinesisStreamsInputDescription(KinesisStreamsInputDescription)}.
         * 
         * @param kinesisStreamsInputDescription
         *        a consumer that will call methods on {@link KinesisStreamsInputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisStreamsInputDescription(KinesisStreamsInputDescription)
         */
        default Builder kinesisStreamsInputDescription(
                Consumer<KinesisStreamsInputDescription.Builder> kinesisStreamsInputDescription) {
            return kinesisStreamsInputDescription(KinesisStreamsInputDescription.builder()
                    .applyMutation(kinesisStreamsInputDescription).build());
        }

        /**
         * <p>
         * If a Kinesis Data Firehose delivery stream is configured as a streaming source, provides the delivery
         * stream's ARN.
         * </p>
         * 
         * @param kinesisFirehoseInputDescription
         *        If a Kinesis Data Firehose delivery stream is configured as a streaming source, provides the delivery
         *        stream's ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisFirehoseInputDescription(KinesisFirehoseInputDescription kinesisFirehoseInputDescription);

        /**
         * <p>
         * If a Kinesis Data Firehose delivery stream is configured as a streaming source, provides the delivery
         * stream's ARN.
         * </p>
         * This is a convenience that creates an instance of the {@link KinesisFirehoseInputDescription.Builder}
         * avoiding the need to create one manually via {@link KinesisFirehoseInputDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link KinesisFirehoseInputDescription.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #kinesisFirehoseInputDescription(KinesisFirehoseInputDescription)}.
         * 
         * @param kinesisFirehoseInputDescription
         *        a consumer that will call methods on {@link KinesisFirehoseInputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisFirehoseInputDescription(KinesisFirehoseInputDescription)
         */
        default Builder kinesisFirehoseInputDescription(
                Consumer<KinesisFirehoseInputDescription.Builder> kinesisFirehoseInputDescription) {
            return kinesisFirehoseInputDescription(KinesisFirehoseInputDescription.builder()
                    .applyMutation(kinesisFirehoseInputDescription).build());
        }

        /**
         * <p>
         * Describes the format of the data in the streaming source, and how each data element maps to corresponding
         * columns in the in-application stream that is being created.
         * </p>
         * 
         * @param inputSchema
         *        Describes the format of the data in the streaming source, and how each data element maps to
         *        corresponding columns in the in-application stream that is being created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputSchema(SourceSchema inputSchema);

        /**
         * <p>
         * Describes the format of the data in the streaming source, and how each data element maps to corresponding
         * columns in the in-application stream that is being created.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceSchema.Builder} avoiding the need to
         * create one manually via {@link SourceSchema#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceSchema.Builder#build()} is called immediately and its
         * result is passed to {@link #inputSchema(SourceSchema)}.
         * 
         * @param inputSchema
         *        a consumer that will call methods on {@link SourceSchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputSchema(SourceSchema)
         */
        default Builder inputSchema(Consumer<SourceSchema.Builder> inputSchema) {
            return inputSchema(SourceSchema.builder().applyMutation(inputSchema).build());
        }

        /**
         * <p>
         * Describes the configured parallelism (number of in-application streams mapped to the streaming source).
         * </p>
         * 
         * @param inputParallelism
         *        Describes the configured parallelism (number of in-application streams mapped to the streaming
         *        source).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputParallelism(InputParallelism inputParallelism);

        /**
         * <p>
         * Describes the configured parallelism (number of in-application streams mapped to the streaming source).
         * </p>
         * This is a convenience that creates an instance of the {@link InputParallelism.Builder} avoiding the need to
         * create one manually via {@link InputParallelism#builder()}.
         *
         * When the {@link Consumer} completes, {@link InputParallelism.Builder#build()} is called immediately and its
         * result is passed to {@link #inputParallelism(InputParallelism)}.
         * 
         * @param inputParallelism
         *        a consumer that will call methods on {@link InputParallelism.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputParallelism(InputParallelism)
         */
        default Builder inputParallelism(Consumer<InputParallelism.Builder> inputParallelism) {
            return inputParallelism(InputParallelism.builder().applyMutation(inputParallelism).build());
        }

        /**
         * <p>
         * The point at which the application is configured to read from the input stream.
         * </p>
         * 
         * @param inputStartingPositionConfiguration
         *        The point at which the application is configured to read from the input stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputStartingPositionConfiguration(InputStartingPositionConfiguration inputStartingPositionConfiguration);

        /**
         * <p>
         * The point at which the application is configured to read from the input stream.
         * </p>
         * This is a convenience that creates an instance of the {@link InputStartingPositionConfiguration.Builder}
         * avoiding the need to create one manually via {@link InputStartingPositionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link InputStartingPositionConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #inputStartingPositionConfiguration(InputStartingPositionConfiguration)}.
         * 
         * @param inputStartingPositionConfiguration
         *        a consumer that will call methods on {@link InputStartingPositionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputStartingPositionConfiguration(InputStartingPositionConfiguration)
         */
        default Builder inputStartingPositionConfiguration(
                Consumer<InputStartingPositionConfiguration.Builder> inputStartingPositionConfiguration) {
            return inputStartingPositionConfiguration(InputStartingPositionConfiguration.builder()
                    .applyMutation(inputStartingPositionConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String inputId;

        private String namePrefix;

        private List<String> inAppStreamNames = DefaultSdkAutoConstructList.getInstance();

        private InputProcessingConfigurationDescription inputProcessingConfigurationDescription;

        private KinesisStreamsInputDescription kinesisStreamsInputDescription;

        private KinesisFirehoseInputDescription kinesisFirehoseInputDescription;

        private SourceSchema inputSchema;

        private InputParallelism inputParallelism;

        private InputStartingPositionConfiguration inputStartingPositionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(InputDescription model) {
            inputId(model.inputId);
            namePrefix(model.namePrefix);
            inAppStreamNames(model.inAppStreamNames);
            inputProcessingConfigurationDescription(model.inputProcessingConfigurationDescription);
            kinesisStreamsInputDescription(model.kinesisStreamsInputDescription);
            kinesisFirehoseInputDescription(model.kinesisFirehoseInputDescription);
            inputSchema(model.inputSchema);
            inputParallelism(model.inputParallelism);
            inputStartingPositionConfiguration(model.inputStartingPositionConfiguration);
        }

        public final String getInputId() {
            return inputId;
        }

        public final void setInputId(String inputId) {
            this.inputId = inputId;
        }

        @Override
        @Transient
        public final Builder inputId(String inputId) {
            this.inputId = inputId;
            return this;
        }

        public final String getNamePrefix() {
            return namePrefix;
        }

        public final void setNamePrefix(String namePrefix) {
            this.namePrefix = namePrefix;
        }

        @Override
        @Transient
        public final Builder namePrefix(String namePrefix) {
            this.namePrefix = namePrefix;
            return this;
        }

        public final Collection<String> getInAppStreamNames() {
            if (inAppStreamNames instanceof SdkAutoConstructList) {
                return null;
            }
            return inAppStreamNames;
        }

        public final void setInAppStreamNames(Collection<String> inAppStreamNames) {
            this.inAppStreamNames = InAppStreamNamesCopier.copy(inAppStreamNames);
        }

        @Override
        @Transient
        public final Builder inAppStreamNames(Collection<String> inAppStreamNames) {
            this.inAppStreamNames = InAppStreamNamesCopier.copy(inAppStreamNames);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder inAppStreamNames(String... inAppStreamNames) {
            inAppStreamNames(Arrays.asList(inAppStreamNames));
            return this;
        }

        public final InputProcessingConfigurationDescription.Builder getInputProcessingConfigurationDescription() {
            return inputProcessingConfigurationDescription != null ? inputProcessingConfigurationDescription.toBuilder() : null;
        }

        public final void setInputProcessingConfigurationDescription(
                InputProcessingConfigurationDescription.BuilderImpl inputProcessingConfigurationDescription) {
            this.inputProcessingConfigurationDescription = inputProcessingConfigurationDescription != null ? inputProcessingConfigurationDescription
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder inputProcessingConfigurationDescription(
                InputProcessingConfigurationDescription inputProcessingConfigurationDescription) {
            this.inputProcessingConfigurationDescription = inputProcessingConfigurationDescription;
            return this;
        }

        public final KinesisStreamsInputDescription.Builder getKinesisStreamsInputDescription() {
            return kinesisStreamsInputDescription != null ? kinesisStreamsInputDescription.toBuilder() : null;
        }

        public final void setKinesisStreamsInputDescription(
                KinesisStreamsInputDescription.BuilderImpl kinesisStreamsInputDescription) {
            this.kinesisStreamsInputDescription = kinesisStreamsInputDescription != null ? kinesisStreamsInputDescription.build()
                    : null;
        }

        @Override
        @Transient
        public final Builder kinesisStreamsInputDescription(KinesisStreamsInputDescription kinesisStreamsInputDescription) {
            this.kinesisStreamsInputDescription = kinesisStreamsInputDescription;
            return this;
        }

        public final KinesisFirehoseInputDescription.Builder getKinesisFirehoseInputDescription() {
            return kinesisFirehoseInputDescription != null ? kinesisFirehoseInputDescription.toBuilder() : null;
        }

        public final void setKinesisFirehoseInputDescription(
                KinesisFirehoseInputDescription.BuilderImpl kinesisFirehoseInputDescription) {
            this.kinesisFirehoseInputDescription = kinesisFirehoseInputDescription != null ? kinesisFirehoseInputDescription
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder kinesisFirehoseInputDescription(KinesisFirehoseInputDescription kinesisFirehoseInputDescription) {
            this.kinesisFirehoseInputDescription = kinesisFirehoseInputDescription;
            return this;
        }

        public final SourceSchema.Builder getInputSchema() {
            return inputSchema != null ? inputSchema.toBuilder() : null;
        }

        public final void setInputSchema(SourceSchema.BuilderImpl inputSchema) {
            this.inputSchema = inputSchema != null ? inputSchema.build() : null;
        }

        @Override
        @Transient
        public final Builder inputSchema(SourceSchema inputSchema) {
            this.inputSchema = inputSchema;
            return this;
        }

        public final InputParallelism.Builder getInputParallelism() {
            return inputParallelism != null ? inputParallelism.toBuilder() : null;
        }

        public final void setInputParallelism(InputParallelism.BuilderImpl inputParallelism) {
            this.inputParallelism = inputParallelism != null ? inputParallelism.build() : null;
        }

        @Override
        @Transient
        public final Builder inputParallelism(InputParallelism inputParallelism) {
            this.inputParallelism = inputParallelism;
            return this;
        }

        public final InputStartingPositionConfiguration.Builder getInputStartingPositionConfiguration() {
            return inputStartingPositionConfiguration != null ? inputStartingPositionConfiguration.toBuilder() : null;
        }

        public final void setInputStartingPositionConfiguration(
                InputStartingPositionConfiguration.BuilderImpl inputStartingPositionConfiguration) {
            this.inputStartingPositionConfiguration = inputStartingPositionConfiguration != null ? inputStartingPositionConfiguration
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder inputStartingPositionConfiguration(
                InputStartingPositionConfiguration inputStartingPositionConfiguration) {
            this.inputStartingPositionConfiguration = inputStartingPositionConfiguration;
            return this;
        }

        @Override
        public InputDescription build() {
            return new InputDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
