/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalyticsv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * For a SQL-based Kinesis Data Analytics application, describes the reference data source by providing the source
 * information (Amazon S3 bucket name and object key name), the resulting in-application table name that is created, and
 * the necessary schema to map the data elements in the Amazon S3 object to the in-application table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReferenceDataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<ReferenceDataSource.Builder, ReferenceDataSource> {
    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TableName").getter(getter(ReferenceDataSource::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableName").build()).build();

    private static final SdkField<S3ReferenceDataSource> S3_REFERENCE_DATA_SOURCE_FIELD = SdkField
            .<S3ReferenceDataSource> builder(MarshallingType.SDK_POJO).memberName("S3ReferenceDataSource")
            .getter(getter(ReferenceDataSource::s3ReferenceDataSource)).setter(setter(Builder::s3ReferenceDataSource))
            .constructor(S3ReferenceDataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3ReferenceDataSource").build())
            .build();

    private static final SdkField<SourceSchema> REFERENCE_SCHEMA_FIELD = SdkField
            .<SourceSchema> builder(MarshallingType.SDK_POJO).memberName("ReferenceSchema")
            .getter(getter(ReferenceDataSource::referenceSchema)).setter(setter(Builder::referenceSchema))
            .constructor(SourceSchema::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferenceSchema").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TABLE_NAME_FIELD,
            S3_REFERENCE_DATA_SOURCE_FIELD, REFERENCE_SCHEMA_FIELD));

    private static final long serialVersionUID = 1L;

    private final String tableName;

    private final S3ReferenceDataSource s3ReferenceDataSource;

    private final SourceSchema referenceSchema;

    private ReferenceDataSource(BuilderImpl builder) {
        this.tableName = builder.tableName;
        this.s3ReferenceDataSource = builder.s3ReferenceDataSource;
        this.referenceSchema = builder.referenceSchema;
    }

    /**
     * <p>
     * The name of the in-application table to create.
     * </p>
     * 
     * @return The name of the in-application table to create.
     */
    public final String tableName() {
        return tableName;
    }

    /**
     * <p>
     * Identifies the S3 bucket and object that contains the reference data. A Kinesis Data Analytics application loads
     * reference data only once. If the data changes, you call the <a>UpdateApplication</a> operation to trigger
     * reloading of data into your application.
     * </p>
     * 
     * @return Identifies the S3 bucket and object that contains the reference data. A Kinesis Data Analytics
     *         application loads reference data only once. If the data changes, you call the <a>UpdateApplication</a>
     *         operation to trigger reloading of data into your application.
     */
    public final S3ReferenceDataSource s3ReferenceDataSource() {
        return s3ReferenceDataSource;
    }

    /**
     * <p>
     * Describes the format of the data in the streaming source, and how each data element maps to corresponding columns
     * created in the in-application stream.
     * </p>
     * 
     * @return Describes the format of the data in the streaming source, and how each data element maps to corresponding
     *         columns created in the in-application stream.
     */
    public final SourceSchema referenceSchema() {
        return referenceSchema;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(s3ReferenceDataSource());
        hashCode = 31 * hashCode + Objects.hashCode(referenceSchema());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReferenceDataSource)) {
            return false;
        }
        ReferenceDataSource other = (ReferenceDataSource) obj;
        return Objects.equals(tableName(), other.tableName())
                && Objects.equals(s3ReferenceDataSource(), other.s3ReferenceDataSource())
                && Objects.equals(referenceSchema(), other.referenceSchema());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReferenceDataSource").add("TableName", tableName())
                .add("S3ReferenceDataSource", s3ReferenceDataSource()).add("ReferenceSchema", referenceSchema()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "S3ReferenceDataSource":
            return Optional.ofNullable(clazz.cast(s3ReferenceDataSource()));
        case "ReferenceSchema":
            return Optional.ofNullable(clazz.cast(referenceSchema()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReferenceDataSource, T> g) {
        return obj -> g.apply((ReferenceDataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReferenceDataSource> {
        /**
         * <p>
         * The name of the in-application table to create.
         * </p>
         * 
         * @param tableName
         *        The name of the in-application table to create.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * Identifies the S3 bucket and object that contains the reference data. A Kinesis Data Analytics application
         * loads reference data only once. If the data changes, you call the <a>UpdateApplication</a> operation to
         * trigger reloading of data into your application.
         * </p>
         * 
         * @param s3ReferenceDataSource
         *        Identifies the S3 bucket and object that contains the reference data. A Kinesis Data Analytics
         *        application loads reference data only once. If the data changes, you call the <a>UpdateApplication</a>
         *        operation to trigger reloading of data into your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ReferenceDataSource(S3ReferenceDataSource s3ReferenceDataSource);

        /**
         * <p>
         * Identifies the S3 bucket and object that contains the reference data. A Kinesis Data Analytics application
         * loads reference data only once. If the data changes, you call the <a>UpdateApplication</a> operation to
         * trigger reloading of data into your application.
         * </p>
         * This is a convenience that creates an instance of the {@link S3ReferenceDataSource.Builder} avoiding the need
         * to create one manually via {@link S3ReferenceDataSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3ReferenceDataSource.Builder#build()} is called immediately and
         * its result is passed to {@link #s3ReferenceDataSource(S3ReferenceDataSource)}.
         * 
         * @param s3ReferenceDataSource
         *        a consumer that will call methods on {@link S3ReferenceDataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3ReferenceDataSource(S3ReferenceDataSource)
         */
        default Builder s3ReferenceDataSource(Consumer<S3ReferenceDataSource.Builder> s3ReferenceDataSource) {
            return s3ReferenceDataSource(S3ReferenceDataSource.builder().applyMutation(s3ReferenceDataSource).build());
        }

        /**
         * <p>
         * Describes the format of the data in the streaming source, and how each data element maps to corresponding
         * columns created in the in-application stream.
         * </p>
         * 
         * @param referenceSchema
         *        Describes the format of the data in the streaming source, and how each data element maps to
         *        corresponding columns created in the in-application stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceSchema(SourceSchema referenceSchema);

        /**
         * <p>
         * Describes the format of the data in the streaming source, and how each data element maps to corresponding
         * columns created in the in-application stream.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceSchema.Builder} avoiding the need to
         * create one manually via {@link SourceSchema#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceSchema.Builder#build()} is called immediately and its
         * result is passed to {@link #referenceSchema(SourceSchema)}.
         * 
         * @param referenceSchema
         *        a consumer that will call methods on {@link SourceSchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #referenceSchema(SourceSchema)
         */
        default Builder referenceSchema(Consumer<SourceSchema.Builder> referenceSchema) {
            return referenceSchema(SourceSchema.builder().applyMutation(referenceSchema).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String tableName;

        private S3ReferenceDataSource s3ReferenceDataSource;

        private SourceSchema referenceSchema;

        private BuilderImpl() {
        }

        private BuilderImpl(ReferenceDataSource model) {
            tableName(model.tableName);
            s3ReferenceDataSource(model.s3ReferenceDataSource);
            referenceSchema(model.referenceSchema);
        }

        public final String getTableName() {
            return tableName;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        @Override
        @Transient
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final S3ReferenceDataSource.Builder getS3ReferenceDataSource() {
            return s3ReferenceDataSource != null ? s3ReferenceDataSource.toBuilder() : null;
        }

        public final void setS3ReferenceDataSource(S3ReferenceDataSource.BuilderImpl s3ReferenceDataSource) {
            this.s3ReferenceDataSource = s3ReferenceDataSource != null ? s3ReferenceDataSource.build() : null;
        }

        @Override
        @Transient
        public final Builder s3ReferenceDataSource(S3ReferenceDataSource s3ReferenceDataSource) {
            this.s3ReferenceDataSource = s3ReferenceDataSource;
            return this;
        }

        public final SourceSchema.Builder getReferenceSchema() {
            return referenceSchema != null ? referenceSchema.toBuilder() : null;
        }

        public final void setReferenceSchema(SourceSchema.BuilderImpl referenceSchema) {
            this.referenceSchema = referenceSchema != null ? referenceSchema.build() : null;
        }

        @Override
        @Transient
        public final Builder referenceSchema(SourceSchema referenceSchema) {
            this.referenceSchema = referenceSchema;
            return this;
        }

        @Override
        public ReferenceDataSource build() {
            return new ReferenceDataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
