/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalyticsv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes parameters for how a Flink-based Kinesis Data Analytics application executes multiple tasks simultaneously.
 * For more information about parallelism, see <a
 * href="https://ci.apache.org/projects/flink/flink-docs-release-1.8/dev/parallel.html">Parallel Execution</a> in the <a
 * href="https://ci.apache.org/projects/flink/flink-docs-release-1.8/">Apache Flink Documentation</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParallelismConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ParallelismConfiguration.Builder, ParallelismConfiguration> {
    private static final SdkField<String> CONFIGURATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationType").getter(getter(ParallelismConfiguration::configurationTypeAsString))
            .setter(setter(Builder::configurationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationType").build()).build();

    private static final SdkField<Integer> PARALLELISM_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Parallelism").getter(getter(ParallelismConfiguration::parallelism)).setter(setter(Builder::parallelism))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parallelism").build()).build();

    private static final SdkField<Integer> PARALLELISM_PER_KPU_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ParallelismPerKPU").getter(getter(ParallelismConfiguration::parallelismPerKPU))
            .setter(setter(Builder::parallelismPerKPU))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParallelismPerKPU").build()).build();

    private static final SdkField<Boolean> AUTO_SCALING_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoScalingEnabled").getter(getter(ParallelismConfiguration::autoScalingEnabled))
            .setter(setter(Builder::autoScalingEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingEnabled").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_TYPE_FIELD,
            PARALLELISM_FIELD, PARALLELISM_PER_KPU_FIELD, AUTO_SCALING_ENABLED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String configurationType;

    private final Integer parallelism;

    private final Integer parallelismPerKPU;

    private final Boolean autoScalingEnabled;

    private ParallelismConfiguration(BuilderImpl builder) {
        this.configurationType = builder.configurationType;
        this.parallelism = builder.parallelism;
        this.parallelismPerKPU = builder.parallelismPerKPU;
        this.autoScalingEnabled = builder.autoScalingEnabled;
    }

    /**
     * <p>
     * Describes whether the application uses the default parallelism for the Kinesis Data Analytics service. You must
     * set this property to <code>CUSTOM</code> in order to change your application's <code>AutoScalingEnabled</code>,
     * <code>Parallelism</code>, or <code>ParallelismPerKPU</code> properties.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #configurationType}
     * will return {@link ConfigurationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #configurationTypeAsString}.
     * </p>
     * 
     * @return Describes whether the application uses the default parallelism for the Kinesis Data Analytics service.
     *         You must set this property to <code>CUSTOM</code> in order to change your application's
     *         <code>AutoScalingEnabled</code>, <code>Parallelism</code>, or <code>ParallelismPerKPU</code> properties.
     * @see ConfigurationType
     */
    public final ConfigurationType configurationType() {
        return ConfigurationType.fromValue(configurationType);
    }

    /**
     * <p>
     * Describes whether the application uses the default parallelism for the Kinesis Data Analytics service. You must
     * set this property to <code>CUSTOM</code> in order to change your application's <code>AutoScalingEnabled</code>,
     * <code>Parallelism</code>, or <code>ParallelismPerKPU</code> properties.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #configurationType}
     * will return {@link ConfigurationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #configurationTypeAsString}.
     * </p>
     * 
     * @return Describes whether the application uses the default parallelism for the Kinesis Data Analytics service.
     *         You must set this property to <code>CUSTOM</code> in order to change your application's
     *         <code>AutoScalingEnabled</code>, <code>Parallelism</code>, or <code>ParallelismPerKPU</code> properties.
     * @see ConfigurationType
     */
    public final String configurationTypeAsString() {
        return configurationType;
    }

    /**
     * <p>
     * Describes the initial number of parallel tasks that a Flink-based Kinesis Data Analytics application can perform.
     * If <code>AutoScalingEnabled</code> is set to True, Kinesis Data Analytics increases the
     * <code>CurrentParallelism</code> value in response to application load. The service can increase the
     * <code>CurrentParallelism</code> value up to the maximum parallelism, which is <code>ParalellismPerKPU</code>
     * times the maximum KPUs for the application. The maximum KPUs for an application is 32 by default, and can be
     * increased by requesting a limit increase. If application load is reduced, the service can reduce the
     * <code>CurrentParallelism</code> value down to the <code>Parallelism</code> setting.
     * </p>
     * 
     * @return Describes the initial number of parallel tasks that a Flink-based Kinesis Data Analytics application can
     *         perform. If <code>AutoScalingEnabled</code> is set to True, Kinesis Data Analytics increases the
     *         <code>CurrentParallelism</code> value in response to application load. The service can increase the
     *         <code>CurrentParallelism</code> value up to the maximum parallelism, which is
     *         <code>ParalellismPerKPU</code> times the maximum KPUs for the application. The maximum KPUs for an
     *         application is 32 by default, and can be increased by requesting a limit increase. If application load is
     *         reduced, the service can reduce the <code>CurrentParallelism</code> value down to the
     *         <code>Parallelism</code> setting.
     */
    public final Integer parallelism() {
        return parallelism;
    }

    /**
     * <p>
     * Describes the number of parallel tasks that a Flink-based Kinesis Data Analytics application can perform per
     * Kinesis Processing Unit (KPU) used by the application. For more information about KPUs, see <a
     * href="http://aws.amazon.com/kinesis/data-analytics/pricing/">Amazon Kinesis Data Analytics Pricing</a>.
     * </p>
     * 
     * @return Describes the number of parallel tasks that a Flink-based Kinesis Data Analytics application can perform
     *         per Kinesis Processing Unit (KPU) used by the application. For more information about KPUs, see <a
     *         href="http://aws.amazon.com/kinesis/data-analytics/pricing/">Amazon Kinesis Data Analytics Pricing</a>.
     */
    public final Integer parallelismPerKPU() {
        return parallelismPerKPU;
    }

    /**
     * <p>
     * Describes whether the Kinesis Data Analytics service can increase the parallelism of the application in response
     * to increased throughput.
     * </p>
     * 
     * @return Describes whether the Kinesis Data Analytics service can increase the parallelism of the application in
     *         response to increased throughput.
     */
    public final Boolean autoScalingEnabled() {
        return autoScalingEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(configurationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(parallelism());
        hashCode = 31 * hashCode + Objects.hashCode(parallelismPerKPU());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingEnabled());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParallelismConfiguration)) {
            return false;
        }
        ParallelismConfiguration other = (ParallelismConfiguration) obj;
        return Objects.equals(configurationTypeAsString(), other.configurationTypeAsString())
                && Objects.equals(parallelism(), other.parallelism())
                && Objects.equals(parallelismPerKPU(), other.parallelismPerKPU())
                && Objects.equals(autoScalingEnabled(), other.autoScalingEnabled());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ParallelismConfiguration").add("ConfigurationType", configurationTypeAsString())
                .add("Parallelism", parallelism()).add("ParallelismPerKPU", parallelismPerKPU())
                .add("AutoScalingEnabled", autoScalingEnabled()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConfigurationType":
            return Optional.ofNullable(clazz.cast(configurationTypeAsString()));
        case "Parallelism":
            return Optional.ofNullable(clazz.cast(parallelism()));
        case "ParallelismPerKPU":
            return Optional.ofNullable(clazz.cast(parallelismPerKPU()));
        case "AutoScalingEnabled":
            return Optional.ofNullable(clazz.cast(autoScalingEnabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ParallelismConfiguration, T> g) {
        return obj -> g.apply((ParallelismConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParallelismConfiguration> {
        /**
         * <p>
         * Describes whether the application uses the default parallelism for the Kinesis Data Analytics service. You
         * must set this property to <code>CUSTOM</code> in order to change your application's
         * <code>AutoScalingEnabled</code>, <code>Parallelism</code>, or <code>ParallelismPerKPU</code> properties.
         * </p>
         * 
         * @param configurationType
         *        Describes whether the application uses the default parallelism for the Kinesis Data Analytics service.
         *        You must set this property to <code>CUSTOM</code> in order to change your application's
         *        <code>AutoScalingEnabled</code>, <code>Parallelism</code>, or <code>ParallelismPerKPU</code>
         *        properties.
         * @see ConfigurationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationType
         */
        Builder configurationType(String configurationType);

        /**
         * <p>
         * Describes whether the application uses the default parallelism for the Kinesis Data Analytics service. You
         * must set this property to <code>CUSTOM</code> in order to change your application's
         * <code>AutoScalingEnabled</code>, <code>Parallelism</code>, or <code>ParallelismPerKPU</code> properties.
         * </p>
         * 
         * @param configurationType
         *        Describes whether the application uses the default parallelism for the Kinesis Data Analytics service.
         *        You must set this property to <code>CUSTOM</code> in order to change your application's
         *        <code>AutoScalingEnabled</code>, <code>Parallelism</code>, or <code>ParallelismPerKPU</code>
         *        properties.
         * @see ConfigurationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationType
         */
        Builder configurationType(ConfigurationType configurationType);

        /**
         * <p>
         * Describes the initial number of parallel tasks that a Flink-based Kinesis Data Analytics application can
         * perform. If <code>AutoScalingEnabled</code> is set to True, Kinesis Data Analytics increases the
         * <code>CurrentParallelism</code> value in response to application load. The service can increase the
         * <code>CurrentParallelism</code> value up to the maximum parallelism, which is <code>ParalellismPerKPU</code>
         * times the maximum KPUs for the application. The maximum KPUs for an application is 32 by default, and can be
         * increased by requesting a limit increase. If application load is reduced, the service can reduce the
         * <code>CurrentParallelism</code> value down to the <code>Parallelism</code> setting.
         * </p>
         * 
         * @param parallelism
         *        Describes the initial number of parallel tasks that a Flink-based Kinesis Data Analytics application
         *        can perform. If <code>AutoScalingEnabled</code> is set to True, Kinesis Data Analytics increases the
         *        <code>CurrentParallelism</code> value in response to application load. The service can increase the
         *        <code>CurrentParallelism</code> value up to the maximum parallelism, which is
         *        <code>ParalellismPerKPU</code> times the maximum KPUs for the application. The maximum KPUs for an
         *        application is 32 by default, and can be increased by requesting a limit increase. If application load
         *        is reduced, the service can reduce the <code>CurrentParallelism</code> value down to the
         *        <code>Parallelism</code> setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelism(Integer parallelism);

        /**
         * <p>
         * Describes the number of parallel tasks that a Flink-based Kinesis Data Analytics application can perform per
         * Kinesis Processing Unit (KPU) used by the application. For more information about KPUs, see <a
         * href="http://aws.amazon.com/kinesis/data-analytics/pricing/">Amazon Kinesis Data Analytics Pricing</a>.
         * </p>
         * 
         * @param parallelismPerKPU
         *        Describes the number of parallel tasks that a Flink-based Kinesis Data Analytics application can
         *        perform per Kinesis Processing Unit (KPU) used by the application. For more information about KPUs,
         *        see <a href="http://aws.amazon.com/kinesis/data-analytics/pricing/">Amazon Kinesis Data Analytics
         *        Pricing</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelismPerKPU(Integer parallelismPerKPU);

        /**
         * <p>
         * Describes whether the Kinesis Data Analytics service can increase the parallelism of the application in
         * response to increased throughput.
         * </p>
         * 
         * @param autoScalingEnabled
         *        Describes whether the Kinesis Data Analytics service can increase the parallelism of the application
         *        in response to increased throughput.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingEnabled(Boolean autoScalingEnabled);
    }

    static final class BuilderImpl implements Builder {
        private String configurationType;

        private Integer parallelism;

        private Integer parallelismPerKPU;

        private Boolean autoScalingEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(ParallelismConfiguration model) {
            configurationType(model.configurationType);
            parallelism(model.parallelism);
            parallelismPerKPU(model.parallelismPerKPU);
            autoScalingEnabled(model.autoScalingEnabled);
        }

        public final String getConfigurationType() {
            return configurationType;
        }

        public final void setConfigurationType(String configurationType) {
            this.configurationType = configurationType;
        }

        @Override
        @Transient
        public final Builder configurationType(String configurationType) {
            this.configurationType = configurationType;
            return this;
        }

        @Override
        @Transient
        public final Builder configurationType(ConfigurationType configurationType) {
            this.configurationType(configurationType == null ? null : configurationType.toString());
            return this;
        }

        public final Integer getParallelism() {
            return parallelism;
        }

        public final void setParallelism(Integer parallelism) {
            this.parallelism = parallelism;
        }

        @Override
        @Transient
        public final Builder parallelism(Integer parallelism) {
            this.parallelism = parallelism;
            return this;
        }

        public final Integer getParallelismPerKPU() {
            return parallelismPerKPU;
        }

        public final void setParallelismPerKPU(Integer parallelismPerKPU) {
            this.parallelismPerKPU = parallelismPerKPU;
        }

        @Override
        @Transient
        public final Builder parallelismPerKPU(Integer parallelismPerKPU) {
            this.parallelismPerKPU = parallelismPerKPU;
            return this;
        }

        public final Boolean getAutoScalingEnabled() {
            return autoScalingEnabled;
        }

        public final void setAutoScalingEnabled(Boolean autoScalingEnabled) {
            this.autoScalingEnabled = autoScalingEnabled;
        }

        @Override
        @Transient
        public final Builder autoScalingEnabled(Boolean autoScalingEnabled) {
            this.autoScalingEnabled = autoScalingEnabled;
            return this;
        }

        @Override
        public ParallelismConfiguration build() {
            return new ParallelismConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
