/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideo;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideo.model.AccountStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeviceStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidDeviceException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidResourceFormatException;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagsPerResourceExceededLimitException;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.VersionMismatchException;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetDataEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetDataEndpointResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListStreamsResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagStreamResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateDataRetentionRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateDataRetentionResponseUnmarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateStreamResponseUnmarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisVideoAsyncClient}.
 *
 * @see KinesisVideoAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoAsyncClient implements KinesisVideoAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    protected DefaultKinesisVideoAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.protocolFactory = init(false);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new Kinesis video stream.
     * </p>
     * <p>
     * When you create a new stream, Kinesis Video Streams assigns it a version number. When you change the stream's
     * metadata, Kinesis Video Streams updates the version.
     * </p>
     * <p>
     * <code>CreateStream</code> is an asynchronous operation.
     * </p>
     * <p>
     * For information about how the service works, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html">How it Works</a>.
     * </p>
     * <p>
     * You must have permissions for the <code>KinesisVideo:CreateStream</code> action.
     * </p>
     *
     * @param createStreamRequest
     * @return A Java Future containing the result of the CreateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccountStreamLimitExceededException The number of streams created for the account is too high.</li>
     *         <li>DeviceStreamLimitExceededException Not implemented.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>InvalidDeviceException Not implemented.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.CreateStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamResponse> createStream(CreateStreamRequest createStreamRequest) {
        try {

            HttpResponseHandler<CreateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CreateStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<CreateStreamRequest, CreateStreamResponse>()
                    .withMarshaller(new CreateStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Kinesis video stream and the data contained in the stream.
     * </p>
     * <p>
     * This method marks the stream for deletion, and makes the data in the stream inaccessible immediately.
     * </p>
     * <p>
     * </p>
     * <p>
     * To ensure that you have the latest version of the stream before deleting it, you can specify the stream version.
     * Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams assigns a
     * new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:DeleteStream</code> action.
     * </p>
     *
     * @param deleteStreamRequest
     * @return A Java Future containing the result of the DeleteStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DeleteStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStreamResponse> deleteStream(DeleteStreamRequest deleteStreamRequest) {
        try {

            HttpResponseHandler<DeleteStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DeleteStreamRequest, DeleteStreamResponse>()
                    .withMarshaller(new DeleteStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the specified stream. You must specify either the
     * <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param describeStreamRequest
     * @return A Java Future containing the result of the DescribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStreamResponse> describeStream(DescribeStreamRequest describeStreamRequest) {
        try {

            HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DescribeStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                    .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an endpoint for a specified stream for either reading or writing. Use this endpoint in your application to
     * read from the specified stream (using the <code>GetMedia</code> or <code>GetMediaForFragmentList</code>
     * operations) or write to it (using the <code>PutMedia</code> operation).
     * </p>
     * <note>
     * <p>
     * The returned endpoint does not have the API name appended. The client needs to add the API name to the returned
     * endpoint.
     * </p>
     * </note>
     * <p>
     * In the request, specify the stream either by <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param getDataEndpointRequest
     * @return A Java Future containing the result of the GetDataEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.GetDataEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetDataEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataEndpointResponse> getDataEndpoint(GetDataEndpointRequest getDataEndpointRequest) {
        try {

            HttpResponseHandler<GetDataEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new GetDataEndpointResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<GetDataEndpointRequest, GetDataEndpointResponse>()
                    .withMarshaller(new GetDataEndpointRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDataEndpointRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     *
     * @param listStreamsRequest
     * @return A Java Future containing the result of the ListStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamsResponse> listStreams(ListStreamsRequest listStreamsRequest) {
        try {

            HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListStreamsResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                    .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listStreamsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified stream.
     * </p>
     * <p>
     * In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param listTagsForStreamRequest
     * @return A Java Future containing the result of the ListTagsForStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListTagsForStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForStreamResponse> listTagsForStream(ListTagsForStreamRequest listTagsForStreamRequest) {
        try {

            HttpResponseHandler<ListTagsForStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new ListTagsForStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<ListTagsForStreamRequest, ListTagsForStreamResponse>()
                    .withMarshaller(new ListTagsForStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a stream. A <i>tag</i> is a key-value pair (the value is optional) that you can define
     * and assign to AWS resources. If you specify a tag that already exists, the tag value is replaced with the value
     * that you specify in the request. For more information, see <a
     * href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * You must provide either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:TagStream</code> action.
     * </p>
     * <p>
     * Kinesis video streams support up to 50 tags.
     * </p>
     *
     * @param tagStreamRequest
     * @return A Java Future containing the result of the TagStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. Kinesis video streams support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.TagStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagStreamResponse> tagStream(TagStreamRequest tagStreamRequest) {
        try {

            HttpResponseHandler<TagStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new TagStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<TagStreamRequest, TagStreamResponse>()
                    .withMarshaller(new TagStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a stream. In the request, specify only a tag key or keys; don't specify the value.
     * If you specify a tag key that does not exist, it's ignored.
     * </p>
     * <p>
     * In the request, you must provide the <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param untagStreamRequest
     * @return A Java Future containing the result of the UntagStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UntagStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagStreamResponse> untagStream(UntagStreamRequest untagStreamRequest) {
        try {

            HttpResponseHandler<UntagStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UntagStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<UntagStreamRequest, UntagStreamResponse>()
                    .withMarshaller(new UntagStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Increases or decreases the stream's data retention period by the value that you specify. To indicate whether you
     * want to increase or decrease the data retention period, specify the <code>Operation</code> parameter in the
     * request body. In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <note>
     * <p>
     * The retention period that you specify replaces the current value.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:UpdateDataRetention</code> action.
     * </p>
     * <p>
     * Changing the data retention period affects the data in the stream as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the data retention period is increased, existing data is retained for the new retention period. For example,
     * if the data retention period is increased from one hour to seven hours, all existing data is retained for seven
     * hours.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the data retention period is decreased, existing data is retained for the new retention period. For example,
     * if the data retention period is decreased from seven hours to one hour, all existing data is retained for one
     * hour, and any data older than one hour is deleted immediately.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateDataRetentionRequest
     * @return A Java Future containing the result of the UpdateDataRetention operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateDataRetention
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateDataRetention"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataRetentionResponse> updateDataRetention(
            UpdateDataRetentionRequest updateDataRetentionRequest) {
        try {

            HttpResponseHandler<UpdateDataRetentionResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateDataRetentionResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<UpdateDataRetentionRequest, UpdateDataRetentionResponse>()
                    .withMarshaller(new UpdateDataRetentionRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(updateDataRetentionRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates stream metadata, such as the device name and media type.
     * </p>
     * <p>
     * You must provide the stream name or the Amazon Resource Name (ARN) of the stream.
     * </p>
     * <p>
     * To make sure that you have the latest version of the stream before updating it, you can specify the stream
     * version. Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams
     * assigns a new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * <code>UpdateStream</code> is an asynchronous operation, and takes time to complete.
     * </p>
     *
     * @param updateStreamRequest
     * @return A Java Future containing the result of the UpdateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStreamResponse> updateStream(UpdateStreamRequest updateStreamRequest) {
        try {

            HttpResponseHandler<UpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateStreamResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<UpdateStreamRequest, UpdateStreamResponse>()
                    .withMarshaller(new UpdateStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateStreamRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init(boolean supportsCbor) {
        return new AwsJsonProtocolFactory(new JsonClientMetadata()
                .withSupportsCbor(supportsCbor)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoException.class)
                .withContentTypeOverride("")
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("DeviceStreamLimitExceededException").withModeledClass(
                                DeviceStreamLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("NotAuthorizedException").withModeledClass(
                                NotAuthorizedException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidArgumentException").withModeledClass(
                                InvalidArgumentException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("AccountStreamLimitExceededException").withModeledClass(
                                AccountStreamLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ClientLimitExceededException").withModeledClass(
                                ClientLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceInUseException").withModeledClass(
                                ResourceInUseException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                ResourceNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("VersionMismatchException").withModeledClass(
                                VersionMismatchException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidDeviceException").withModeledClass(
                                InvalidDeviceException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidResourceFormatException").withModeledClass(
                                InvalidResourceFormatException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("TagsPerResourceExceededLimitException").withModeledClass(
                                TagsPerResourceExceededLimitException.class)), AwsJsonProtocolMetadata.builder()
                .protocolVersion("1.1").protocol(AwsJsonProtocol.REST_JSON).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(AwsJsonProtocolFactory protocolFactory) {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }
}
