/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideo;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideo.model.AccessDeniedException;
import software.amazon.awssdk.services.kinesisvideo.model.AccountChannelLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.AccountStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.CreateSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeviceStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.GetSignalingChannelEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetSignalingChannelEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidDeviceException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidResourceFormatException;
import software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoException;
import software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideo.model.TagResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagsPerResourceExceededLimitException;
import software.amazon.awssdk.services.kinesisvideo.model.UntagResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.VersionMismatchException;
import software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsPublisher;
import software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsPublisher;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetDataEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetSignalingChannelEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListSignalingChannelsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateDataRetentionRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateStreamRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisVideoAsyncClient}.
 *
 * @see KinesisVideoAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoAsyncClient implements KinesisVideoAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKinesisVideoAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a signaling channel.
     * </p>
     * <p>
     * <code>CreateSignalingChannel</code> is an asynchronous operation.
     * </p>
     *
     * @param createSignalingChannelRequest
     * @return A Java Future containing the result of the CreateSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>AccountChannelLimitExceededException You have reached the maximum limit of active signaling channels
     *         for this AWS account in this region.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. Kinesis video streams support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.CreateSignalingChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSignalingChannelResponse> createSignalingChannel(
            CreateSignalingChannelRequest createSignalingChannelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSignalingChannelRequest, CreateSignalingChannelResponse>()
                            .withOperationName("CreateSignalingChannel")
                            .withMarshaller(new CreateSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createSignalingChannelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Kinesis video stream.
     * </p>
     * <p>
     * When you create a new stream, Kinesis Video Streams assigns it a version number. When you change the stream's
     * metadata, Kinesis Video Streams updates the version.
     * </p>
     * <p>
     * <code>CreateStream</code> is an asynchronous operation.
     * </p>
     * <p>
     * For information about how the service works, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html">How it Works</a>.
     * </p>
     * <p>
     * You must have permissions for the <code>KinesisVideo:CreateStream</code> action.
     * </p>
     *
     * @param createStreamRequest
     * @return A Java Future containing the result of the CreateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccountStreamLimitExceededException The number of streams created for the account is too high.</li>
     *         <li>DeviceStreamLimitExceededException Not implemented.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>InvalidDeviceException Not implemented.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. Kinesis video streams support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.CreateStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamResponse> createStream(CreateStreamRequest createStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStreamRequest, CreateStreamResponse>()
                            .withOperationName("CreateStream").withMarshaller(new CreateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified signaling channel. <code>DeleteSignalingChannel</code> is an asynchronous operation. If you
     * don't specify the channel's current version, the most recent version is deleted.
     * </p>
     *
     * @param deleteSignalingChannelRequest
     * @return A Java Future containing the result of the DeleteSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DeleteSignalingChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSignalingChannelResponse> deleteSignalingChannel(
            DeleteSignalingChannelRequest deleteSignalingChannelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSignalingChannelRequest, DeleteSignalingChannelResponse>()
                            .withOperationName("DeleteSignalingChannel")
                            .withMarshaller(new DeleteSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSignalingChannelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Kinesis video stream and the data contained in the stream.
     * </p>
     * <p>
     * This method marks the stream for deletion, and makes the data in the stream inaccessible immediately.
     * </p>
     * <p>
     * </p>
     * <p>
     * To ensure that you have the latest version of the stream before deleting it, you can specify the stream version.
     * Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams assigns a
     * new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:DeleteStream</code> action.
     * </p>
     *
     * @param deleteStreamRequest
     * @return A Java Future containing the result of the DeleteStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DeleteStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStreamResponse> deleteStream(DeleteStreamRequest deleteStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStreamRequest, DeleteStreamResponse>()
                            .withOperationName("DeleteStream").withMarshaller(new DeleteStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the signaling channel. You must specify either the name or the ARN of
     * the channel that you want to describe.
     * </p>
     *
     * @param describeSignalingChannelRequest
     * @return A Java Future containing the result of the DescribeSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeSignalingChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSignalingChannelResponse> describeSignalingChannel(
            DescribeSignalingChannelRequest describeSignalingChannelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSignalingChannelRequest, DescribeSignalingChannelResponse>()
                            .withOperationName("DescribeSignalingChannel")
                            .withMarshaller(new DescribeSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeSignalingChannelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the specified stream. You must specify either the
     * <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param describeStreamRequest
     * @return A Java Future containing the result of the DescribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStreamResponse> describeStream(DescribeStreamRequest describeStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                            .withOperationName("DescribeStream")
                            .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an endpoint for a specified stream for either reading or writing. Use this endpoint in your application to
     * read from the specified stream (using the <code>GetMedia</code> or <code>GetMediaForFragmentList</code>
     * operations) or write to it (using the <code>PutMedia</code> operation).
     * </p>
     * <note>
     * <p>
     * The returned endpoint does not have the API name appended. The client needs to add the API name to the returned
     * endpoint.
     * </p>
     * </note>
     * <p>
     * In the request, specify the stream either by <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param getDataEndpointRequest
     * @return A Java Future containing the result of the GetDataEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.GetDataEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetDataEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataEndpointResponse> getDataEndpoint(GetDataEndpointRequest getDataEndpointRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataEndpointRequest, GetDataEndpointResponse>()
                            .withOperationName("GetDataEndpoint")
                            .withMarshaller(new GetDataEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDataEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides an endpoint for the specified signaling channel to send and receive messages. This API uses the
     * <code>SingleMasterChannelEndpointConfiguration</code> input parameter, which consists of the
     * <code>Protocols</code> and <code>Role</code> properties.
     * </p>
     * <p>
     * <code>Protocols</code> is used to determine the communication mechanism. For example, specifying <code>WSS</code>
     * as the protocol, results in this API producing a secure websocket endpoint, and specifying <code>HTTPS</code> as
     * the protocol, results in this API generating an HTTPS endpoint.
     * </p>
     * <p>
     * <code>Role</code> determines the messaging permissions. A <code>MASTER</code> role results in this API generating
     * an endpoint that a client can use to communicate with any of the viewers on the channel. A <code>VIEWER</code>
     * role results in this API generating an endpoint that a client can use to communicate only with a
     * <code>MASTER</code>.
     * </p>
     *
     * @param getSignalingChannelEndpointRequest
     * @return A Java Future containing the result of the GetSignalingChannelEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.GetSignalingChannelEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetSignalingChannelEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSignalingChannelEndpointResponse> getSignalingChannelEndpoint(
            GetSignalingChannelEndpointRequest getSignalingChannelEndpointRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSignalingChannelEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSignalingChannelEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSignalingChannelEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSignalingChannelEndpointRequest, GetSignalingChannelEndpointResponse>()
                            .withOperationName("GetSignalingChannelEndpoint")
                            .withMarshaller(new GetSignalingChannelEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSignalingChannelEndpointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ChannelInfo</code> objects. Each object describes a signaling channel. To retrieve only
     * those channels that satisfy a specific condition, you can specify a <code>ChannelNameCondition</code>.
     * </p>
     *
     * @param listSignalingChannelsRequest
     * @return A Java Future containing the result of the ListSignalingChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListSignalingChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListSignalingChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSignalingChannelsResponse> listSignalingChannels(
            ListSignalingChannelsRequest listSignalingChannelsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSignalingChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSignalingChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSignalingChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSignalingChannelsRequest, ListSignalingChannelsResponse>()
                            .withOperationName("ListSignalingChannels")
                            .withMarshaller(new ListSignalingChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listSignalingChannelsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ChannelInfo</code> objects. Each object describes a signaling channel. To retrieve only
     * those channels that satisfy a specific condition, you can specify a <code>ChannelNameCondition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSignalingChannels(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsPublisher publisher = client.listSignalingChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsPublisher publisher = client.listSignalingChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSignalingChannels(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSignalingChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListSignalingChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListSignalingChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSignalingChannelsPublisher listSignalingChannelsPaginator(ListSignalingChannelsRequest listSignalingChannelsRequest) {
        return new ListSignalingChannelsPublisher(this, applyPaginatorUserAgent(listSignalingChannelsRequest));
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     *
     * @param listStreamsRequest
     * @return A Java Future containing the result of the ListStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamsResponse> listStreams(ListStreamsRequest listStreamsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListStreamsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStreamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                            .withOperationName("ListStreams").withMarshaller(new ListStreamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listStreamsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreams(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    public ListStreamsPublisher listStreamsPaginator(ListStreamsRequest listStreamsRequest) {
        return new ListStreamsPublisher(this, applyPaginatorUserAgent(listStreamsRequest));
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified signaling channel.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified stream.
     * </p>
     * <p>
     * In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param listTagsForStreamRequest
     * @return A Java Future containing the result of the ListTagsForStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListTagsForStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForStreamResponse> listTagsForStream(ListTagsForStreamRequest listTagsForStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForStreamRequest, ListTagsForStreamResponse>()
                            .withOperationName("ListTagsForStream")
                            .withMarshaller(new ListTagsForStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a signaling channel. A <i>tag</i> is a key-value pair (the value is optional) that you
     * can define and assign to AWS resources. If you specify a tag that already exists, the tag value is replaced with
     * the value that you specify in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. Kinesis video streams support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a stream. A <i>tag</i> is a key-value pair (the value is optional) that you can define
     * and assign to AWS resources. If you specify a tag that already exists, the tag value is replaced with the value
     * that you specify in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * You must provide either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:TagStream</code> action.
     * </p>
     * <p>
     * Kinesis video streams support up to 50 tags.
     * </p>
     *
     * @param tagStreamRequest
     * @return A Java Future containing the result of the TagStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. Kinesis video streams support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.TagStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagStreamResponse> tagStream(TagStreamRequest tagStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagStreamRequest, TagStreamResponse>().withOperationName("TagStream")
                            .withMarshaller(new TagStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(tagStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a signaling channel. In the request, specify only a tag key or keys; don't specify
     * the value. If you specify a tag key that does not exist, it's ignored.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a stream. In the request, specify only a tag key or keys; don't specify the value.
     * If you specify a tag key that does not exist, it's ignored.
     * </p>
     * <p>
     * In the request, you must provide the <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param untagStreamRequest
     * @return A Java Future containing the result of the UntagStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UntagStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagStreamResponse> untagStream(UntagStreamRequest untagStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagStreamRequest, UntagStreamResponse>()
                            .withOperationName("UntagStream").withMarshaller(new UntagStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Increases or decreases the stream's data retention period by the value that you specify. To indicate whether you
     * want to increase or decrease the data retention period, specify the <code>Operation</code> parameter in the
     * request body. In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <note>
     * <p>
     * The retention period that you specify replaces the current value.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:UpdateDataRetention</code> action.
     * </p>
     * <p>
     * Changing the data retention period affects the data in the stream as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the data retention period is increased, existing data is retained for the new retention period. For example,
     * if the data retention period is increased from one hour to seven hours, all existing data is retained for seven
     * hours.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the data retention period is decreased, existing data is retained for the new retention period. For example,
     * if the data retention period is decreased from seven hours to one hour, all existing data is retained for one
     * hour, and any data older than one hour is deleted immediately.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateDataRetentionRequest
     * @return A Java Future containing the result of the UpdateDataRetention operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateDataRetention
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateDataRetention"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataRetentionResponse> updateDataRetention(
            UpdateDataRetentionRequest updateDataRetentionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataRetentionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataRetentionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataRetentionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataRetentionRequest, UpdateDataRetentionResponse>()
                            .withOperationName("UpdateDataRetention")
                            .withMarshaller(new UpdateDataRetentionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDataRetentionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the existing signaling channel. This is an asynchronous operation and takes time to complete.
     * </p>
     * <p>
     * If the <code>MessageTtlSeconds</code> value is updated (either increased or reduced), then it only applies to new
     * messages sent via this channel after it's been updated. Existing messages are still expire as per the previous
     * <code>MessageTtlSeconds</code> value.
     * </p>
     *
     * @param updateSignalingChannelRequest
     * @return A Java Future containing the result of the UpdateSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateSignalingChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSignalingChannelResponse> updateSignalingChannel(
            UpdateSignalingChannelRequest updateSignalingChannelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSignalingChannelRequest, UpdateSignalingChannelResponse>()
                            .withOperationName("UpdateSignalingChannel")
                            .withMarshaller(new UpdateSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateSignalingChannelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates stream metadata, such as the device name and media type.
     * </p>
     * <p>
     * You must provide the stream name or the Amazon Resource Name (ARN) of the stream.
     * </p>
     * <p>
     * To make sure that you have the latest version of the stream before updating it, you can specify the stream
     * version. Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams
     * assigns a new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * <code>UpdateStream</code> is an asynchronous operation, and takes time to complete.
     * </p>
     *
     * @param updateStreamRequest
     * @return A Java Future containing the result of the UpdateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException The stream is currently not available for this operation.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStreamResponse> updateStream(UpdateStreamRequest updateStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStreamRequest, UpdateStreamResponse>()
                            .withOperationName("UpdateStream").withMarshaller(new UpdateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountStreamLimitExceededException")
                                .exceptionBuilderSupplier(AccountStreamLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VersionMismatchException")
                                .exceptionBuilderSupplier(VersionMismatchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDeviceException")
                                .exceptionBuilderSupplier(InvalidDeviceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsPerResourceExceededLimitException")
                                .exceptionBuilderSupplier(TagsPerResourceExceededLimitException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountChannelLimitExceededException")
                                .exceptionBuilderSupplier(AccountChannelLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceStreamLimitExceededException")
                                .exceptionBuilderSupplier(DeviceStreamLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientLimitExceededException")
                                .exceptionBuilderSupplier(ClientLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceFormatException")
                                .exceptionBuilderSupplier(InvalidResourceFormatException::builder).httpStatusCode(400).build());
    }

    private <T extends KinesisVideoRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
