/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideo;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideo.model.AccessDeniedException;
import software.amazon.awssdk.services.kinesisvideo.model.AccountChannelLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.AccountStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.CreateSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeEdgeConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeEdgeConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeImageGenerationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeImageGenerationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMediaStorageConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMediaStorageConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeNotificationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeNotificationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeviceStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.GetSignalingChannelEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetSignalingChannelEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidDeviceException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidResourceFormatException;
import software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoException;
import software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.NoDataRetentionException;
import software.amazon.awssdk.services.kinesisvideo.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideo.model.StartEdgeConfigurationUpdateRequest;
import software.amazon.awssdk.services.kinesisvideo.model.StartEdgeConfigurationUpdateResponse;
import software.amazon.awssdk.services.kinesisvideo.model.StreamEdgeConfigurationNotFoundException;
import software.amazon.awssdk.services.kinesisvideo.model.TagResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagsPerResourceExceededLimitException;
import software.amazon.awssdk.services.kinesisvideo.model.UntagResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateImageGenerationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateImageGenerationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateMediaStorageConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateMediaStorageConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateNotificationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateNotificationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.VersionMismatchException;
import software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationIterable;
import software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsIterable;
import software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsIterable;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeEdgeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeImageGenerationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeMappedResourceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeMediaStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetDataEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetSignalingChannelEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListSignalingChannelsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.StartEdgeConfigurationUpdateRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateDataRetentionRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateImageGenerationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateMediaStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateStreamRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link KinesisVideoClient}.
 *
 * @see KinesisVideoClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoClient implements KinesisVideoClient {
    private static final Logger log = Logger.loggerFor(DefaultKinesisVideoClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final KinesisVideoServiceClientConfiguration serviceClientConfiguration;

    protected DefaultKinesisVideoClient(KinesisVideoServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a signaling channel.
     * </p>
     * <p>
     * <code>CreateSignalingChannel</code> is an asynchronous operation.
     * </p>
     *
     * @param createSignalingChannelRequest
     * @return Result of the CreateSignalingChannel operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws AccountChannelLimitExceededException
     *         You have reached the maximum limit of active signaling channels for this Amazon Web Services account in
     *         this region.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws TagsPerResourceExceededLimitException
     *         You have exceeded the limit of tags that you can associate with the resource. A Kinesis video stream can
     *         support up to 50 tags.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.CreateSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSignalingChannelResponse createSignalingChannel(CreateSignalingChannelRequest createSignalingChannelRequest)
            throws InvalidArgumentException, ClientLimitExceededException, AccountChannelLimitExceededException,
            ResourceInUseException, AccessDeniedException, TagsPerResourceExceededLimitException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSignalingChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSignalingChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSignalingChannelRequest, CreateSignalingChannelResponse>()
                            .withOperationName("CreateSignalingChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createSignalingChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSignalingChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Kinesis video stream.
     * </p>
     * <p>
     * When you create a new stream, Kinesis Video Streams assigns it a version number. When you change the stream's
     * metadata, Kinesis Video Streams updates the version.
     * </p>
     * <p>
     * <code>CreateStream</code> is an asynchronous operation.
     * </p>
     * <p>
     * For information about how the service works, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html">How it Works</a>.
     * </p>
     * <p>
     * You must have permissions for the <code>KinesisVideo:CreateStream</code> action.
     * </p>
     *
     * @param createStreamRequest
     * @return Result of the CreateStream operation returned by the service.
     * @throws AccountStreamLimitExceededException
     *         The number of streams created for the account is too high.
     * @throws DeviceStreamLimitExceededException
     *         Not implemented.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws InvalidDeviceException
     *         Not implemented.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws TagsPerResourceExceededLimitException
     *         You have exceeded the limit of tags that you can associate with the resource. A Kinesis video stream can
     *         support up to 50 tags.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.CreateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStreamResponse createStream(CreateStreamRequest createStreamRequest) throws AccountStreamLimitExceededException,
            DeviceStreamLimitExceededException, ResourceInUseException, InvalidDeviceException, InvalidArgumentException,
            ClientLimitExceededException, TagsPerResourceExceededLimitException, AwsServiceException, SdkClientException,
            KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStream");

            return clientHandler.execute(new ClientExecutionParams<CreateStreamRequest, CreateStreamResponse>()
                    .withOperationName("CreateStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified signaling channel. <code>DeleteSignalingChannel</code> is an asynchronous operation. If you
     * don't specify the channel's current version, the most recent version is deleted.
     * </p>
     *
     * @param deleteSignalingChannelRequest
     * @return Result of the DeleteSignalingChannel operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws VersionMismatchException
     *         The stream version that you specified is not the latest version. To get the latest version, use the <a
     *         href
     *         ="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html">DescribeStream</a>
     *         API.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DeleteSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSignalingChannelResponse deleteSignalingChannel(DeleteSignalingChannelRequest deleteSignalingChannelRequest)
            throws InvalidArgumentException, ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException,
            VersionMismatchException, ResourceInUseException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSignalingChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSignalingChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSignalingChannelRequest, DeleteSignalingChannelResponse>()
                            .withOperationName("DeleteSignalingChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteSignalingChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSignalingChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Kinesis video stream and the data contained in the stream.
     * </p>
     * <p>
     * This method marks the stream for deletion, and makes the data in the stream inaccessible immediately.
     * </p>
     * <p>
     * </p>
     * <p>
     * To ensure that you have the latest version of the stream before deleting it, you can specify the stream version.
     * Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams assigns a
     * new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:DeleteStream</code> action.
     * </p>
     *
     * @param deleteStreamRequest
     * @return Result of the DeleteStream operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws VersionMismatchException
     *         The stream version that you specified is not the latest version. To get the latest version, use the <a
     *         href
     *         ="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html">DescribeStream</a>
     *         API.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DeleteStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStreamResponse deleteStream(DeleteStreamRequest deleteStreamRequest) throws ClientLimitExceededException,
            InvalidArgumentException, ResourceNotFoundException, NotAuthorizedException, VersionMismatchException,
            ResourceInUseException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStream");

            return clientHandler.execute(new ClientExecutionParams<DeleteStreamRequest, DeleteStreamResponse>()
                    .withOperationName("DeleteStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a stream’s edge configuration that was set using the <code>StartEdgeConfigurationUpdate</code> API. Use
     * this API to get the status of the configuration if the configuration is in sync with the Edge Agent.
     * </p>
     *
     * @param describeEdgeConfigurationRequest
     * @return Result of the DescribeEdgeConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws StreamEdgeConfigurationNotFoundException
     *         The Exception rendered when the Amazon Kinesis Video Stream can't find a stream's edge configuration that
     *         you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeEdgeConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeEdgeConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEdgeConfigurationResponse describeEdgeConfiguration(
            DescribeEdgeConfigurationRequest describeEdgeConfigurationRequest) throws AccessDeniedException,
            ClientLimitExceededException, InvalidArgumentException, ResourceNotFoundException,
            StreamEdgeConfigurationNotFoundException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEdgeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEdgeConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEdgeConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEdgeConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEdgeConfigurationRequest, DescribeEdgeConfigurationResponse>()
                            .withOperationName("DescribeEdgeConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEdgeConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEdgeConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the <code>ImageGenerationConfiguration</code> for a given Kinesis video stream.
     * </p>
     *
     * @param describeImageGenerationConfigurationRequest
     * @return Result of the DescribeImageGenerationConfiguration operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeImageGenerationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeImageGenerationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeImageGenerationConfigurationResponse describeImageGenerationConfiguration(
            DescribeImageGenerationConfigurationRequest describeImageGenerationConfigurationRequest)
            throws InvalidArgumentException, ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImageGenerationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeImageGenerationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeImageGenerationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageGenerationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeImageGenerationConfigurationRequest, DescribeImageGenerationConfigurationResponse>()
                            .withOperationName("DescribeImageGenerationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeImageGenerationConfigurationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeImageGenerationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the most current information about the stream. Either streamName or streamARN should be provided in the
     * input.
     * </p>
     * <p>
     * Returns the most current information about the stream. The <code>streamName</code> or <code>streamARN</code>
     * should be provided in the input.
     * </p>
     *
     * @param describeMappedResourceConfigurationRequest
     * @return Result of the DescribeMappedResourceConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeMappedResourceConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeMappedResourceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMappedResourceConfigurationResponse describeMappedResourceConfiguration(
            DescribeMappedResourceConfigurationRequest describeMappedResourceConfigurationRequest)
            throws ResourceNotFoundException, InvalidArgumentException, AccessDeniedException, ClientLimitExceededException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMappedResourceConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeMappedResourceConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMappedResourceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMappedResourceConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeMappedResourceConfigurationRequest, DescribeMappedResourceConfigurationResponse>()
                            .withOperationName("DescribeMappedResourceConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeMappedResourceConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeMappedResourceConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the most current information about the stream. Either streamName or streamARN should be provided in the
     * input.
     * </p>
     * <p>
     * Returns the most current information about the stream. The <code>streamName</code> or <code>streamARN</code>
     * should be provided in the input.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeMappedResourceConfiguration(software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationIterable responses = client.describeMappedResourceConfigurationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationIterable responses = client
     *             .describeMappedResourceConfigurationPaginator(request);
     *     for (software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationIterable responses = client.describeMappedResourceConfigurationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMappedResourceConfiguration(software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMappedResourceConfigurationRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeMappedResourceConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeMappedResourceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMappedResourceConfigurationIterable describeMappedResourceConfigurationPaginator(
            DescribeMappedResourceConfigurationRequest describeMappedResourceConfigurationRequest)
            throws ResourceNotFoundException, InvalidArgumentException, AccessDeniedException, ClientLimitExceededException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        return new DescribeMappedResourceConfigurationIterable(this,
                applyPaginatorUserAgent(describeMappedResourceConfigurationRequest));
    }

    /**
     * <p>
     * Returns the most current information about the channel. Specify the <code>ChannelName</code> or
     * <code>ChannelARN</code> in the input.
     * </p>
     *
     * @param describeMediaStorageConfigurationRequest
     * @return Result of the DescribeMediaStorageConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeMediaStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeMediaStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMediaStorageConfigurationResponse describeMediaStorageConfiguration(
            DescribeMediaStorageConfigurationRequest describeMediaStorageConfigurationRequest) throws ResourceNotFoundException,
            InvalidArgumentException, AccessDeniedException, ClientLimitExceededException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMediaStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeMediaStorageConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMediaStorageConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMediaStorageConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeMediaStorageConfigurationRequest, DescribeMediaStorageConfigurationResponse>()
                            .withOperationName("DescribeMediaStorageConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeMediaStorageConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeMediaStorageConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the <code>NotificationConfiguration</code> for a given Kinesis video stream.
     * </p>
     *
     * @param describeNotificationConfigurationRequest
     * @return Result of the DescribeNotificationConfiguration operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotificationConfigurationResponse describeNotificationConfiguration(
            DescribeNotificationConfigurationRequest describeNotificationConfigurationRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationConfigurationRequest, DescribeNotificationConfigurationResponse>()
                            .withOperationName("DescribeNotificationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the most current information about the signaling channel. You must specify either the name or the Amazon
     * Resource Name (ARN) of the channel that you want to describe.
     * </p>
     *
     * @param describeSignalingChannelRequest
     * @return Result of the DescribeSignalingChannel operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSignalingChannelResponse describeSignalingChannel(
            DescribeSignalingChannelRequest describeSignalingChannelRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSignalingChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSignalingChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSignalingChannelRequest, DescribeSignalingChannelResponse>()
                            .withOperationName("DescribeSignalingChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeSignalingChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSignalingChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the most current information about the specified stream. You must specify either the
     * <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param describeStreamRequest
     * @return Result of the DescribeStream operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.DescribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeStreamResponse describeStream(DescribeStreamRequest describeStreamRequest) throws InvalidArgumentException,
            ResourceNotFoundException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStream");

            return clientHandler.execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                    .withOperationName("DescribeStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets an endpoint for a specified stream for either reading or writing. Use this endpoint in your application to
     * read from the specified stream (using the <code>GetMedia</code> or <code>GetMediaForFragmentList</code>
     * operations) or write to it (using the <code>PutMedia</code> operation).
     * </p>
     * <note>
     * <p>
     * The returned endpoint does not have the API name appended. The client needs to add the API name to the returned
     * endpoint.
     * </p>
     * </note>
     * <p>
     * In the request, specify the stream either by <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param getDataEndpointRequest
     * @return Result of the GetDataEndpoint operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.GetDataEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetDataEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDataEndpointResponse getDataEndpoint(GetDataEndpointRequest getDataEndpointRequest)
            throws InvalidArgumentException, ResourceNotFoundException, ClientLimitExceededException, NotAuthorizedException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataEndpointResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDataEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataEndpoint");

            return clientHandler.execute(new ClientExecutionParams<GetDataEndpointRequest, GetDataEndpointResponse>()
                    .withOperationName("GetDataEndpoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDataEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDataEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides an endpoint for the specified signaling channel to send and receive messages. This API uses the
     * <code>SingleMasterChannelEndpointConfiguration</code> input parameter, which consists of the
     * <code>Protocols</code> and <code>Role</code> properties.
     * </p>
     * <p>
     * <code>Protocols</code> is used to determine the communication mechanism. For example, if you specify
     * <code>WSS</code> as the protocol, this API produces a secure websocket endpoint. If you specify
     * <code>HTTPS</code> as the protocol, this API generates an HTTPS endpoint.
     * </p>
     * <p>
     * <code>Role</code> determines the messaging permissions. A <code>MASTER</code> role results in this API generating
     * an endpoint that a client can use to communicate with any of the viewers on the channel. A <code>VIEWER</code>
     * role results in this API generating an endpoint that a client can use to communicate only with a
     * <code>MASTER</code>.
     * </p>
     *
     * @param getSignalingChannelEndpointRequest
     * @return Result of the GetSignalingChannelEndpoint operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.GetSignalingChannelEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetSignalingChannelEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSignalingChannelEndpointResponse getSignalingChannelEndpoint(
            GetSignalingChannelEndpointRequest getSignalingChannelEndpointRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, ResourceInUseException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSignalingChannelEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSignalingChannelEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSignalingChannelEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSignalingChannelEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSignalingChannelEndpointRequest, GetSignalingChannelEndpointResponse>()
                            .withOperationName("GetSignalingChannelEndpoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSignalingChannelEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSignalingChannelEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>ChannelInfo</code> objects. Each object describes a signaling channel. To retrieve only
     * those channels that satisfy a specific condition, you can specify a <code>ChannelNameCondition</code>.
     * </p>
     *
     * @param listSignalingChannelsRequest
     * @return Result of the ListSignalingChannels operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.ListSignalingChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListSignalingChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSignalingChannelsResponse listSignalingChannels(ListSignalingChannelsRequest listSignalingChannelsRequest)
            throws InvalidArgumentException, ClientLimitExceededException, AccessDeniedException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSignalingChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSignalingChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSignalingChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSignalingChannels");

            return clientHandler.execute(new ClientExecutionParams<ListSignalingChannelsRequest, ListSignalingChannelsResponse>()
                    .withOperationName("ListSignalingChannels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSignalingChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSignalingChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>ChannelInfo</code> objects. Each object describes a signaling channel. To retrieve only
     * those channels that satisfy a specific condition, you can specify a <code>ChannelNameCondition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSignalingChannels(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsIterable responses = client.listSignalingChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsIterable responses = client
     *             .listSignalingChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsIterable responses = client.listSignalingChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSignalingChannels(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSignalingChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.ListSignalingChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListSignalingChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSignalingChannelsIterable listSignalingChannelsPaginator(ListSignalingChannelsRequest listSignalingChannelsRequest)
            throws InvalidArgumentException, ClientLimitExceededException, AccessDeniedException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        return new ListSignalingChannelsIterable(this, applyPaginatorUserAgent(listSignalingChannelsRequest));
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     *
     * @param listStreamsRequest
     * @return Result of the ListStreams operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws ClientLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreams");

            return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                    .withOperationName("ListStreams").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreams(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStreamsIterable listStreamsPaginator(ListStreamsRequest listStreamsRequest) throws ClientLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoException {
        return new ListStreamsIterable(this, applyPaginatorUserAgent(listStreamsRequest));
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified signaling channel.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidArgumentException, ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified stream.
     * </p>
     * <p>
     * In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param listTagsForStreamRequest
     * @return Result of the ListTagsForStream operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws InvalidResourceFormatException
     *         The format of the <code>StreamARN</code> is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.ListTagsForStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForStreamResponse listTagsForStream(ListTagsForStreamRequest listTagsForStreamRequest)
            throws ClientLimitExceededException, InvalidArgumentException, ResourceNotFoundException, NotAuthorizedException,
            InvalidResourceFormatException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsForStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForStream");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForStreamRequest, ListTagsForStreamResponse>()
                    .withOperationName("ListTagsForStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * An asynchronous API that updates a stream’s existing edge configuration. The Kinesis Video Stream will sync the
     * stream’s edge configuration with the Edge Agent IoT Greengrass component that runs on an IoT Hub Device, setup at
     * your premise. The time to sync can vary and depends on the connectivity of the Hub Device. The
     * <code>SyncStatus</code> will be updated as the edge configuration is acknowledged, and synced with the Edge
     * Agent.
     * </p>
     * <p>
     * If this API is invoked for the first time, a new edge configuration will be created for the stream, and the sync
     * status will be set to <code>SYNCING</code>. You will have to wait for the sync status to reach a terminal state
     * such as: <code>IN_SYNC</code>, or <code>SYNC_FAILED</code>, before using this API again. If you invoke this API
     * during the syncing process, a <code>ResourceInUseException</code> will be thrown. The connectivity of the
     * stream’s edge configuration and the Edge Agent will be retried for 15 minutes. After 15 minutes, the status will
     * transition into the <code>SYNC_FAILED</code> state.
     * </p>
     *
     * @param startEdgeConfigurationUpdateRequest
     * @return Result of the StartEdgeConfigurationUpdate operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws NoDataRetentionException
     *         The Stream data retention in hours is equal to zero.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.StartEdgeConfigurationUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/StartEdgeConfigurationUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartEdgeConfigurationUpdateResponse startEdgeConfigurationUpdate(
            StartEdgeConfigurationUpdateRequest startEdgeConfigurationUpdateRequest) throws AccessDeniedException,
            ClientLimitExceededException, InvalidArgumentException, NoDataRetentionException, ResourceInUseException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartEdgeConfigurationUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartEdgeConfigurationUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEdgeConfigurationUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEdgeConfigurationUpdate");

            return clientHandler
                    .execute(new ClientExecutionParams<StartEdgeConfigurationUpdateRequest, StartEdgeConfigurationUpdateResponse>()
                            .withOperationName("StartEdgeConfigurationUpdate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startEdgeConfigurationUpdateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartEdgeConfigurationUpdateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to a signaling channel. A <i>tag</i> is a key-value pair (the value is optional) that you
     * can define and assign to Amazon Web Services resources. If you specify a tag that already exists, the tag value
     * is replaced with the value that you specify in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Billing and Cost Management and Cost Management User Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws TagsPerResourceExceededLimitException
     *         You have exceeded the limit of tags that you can associate with the resource. A Kinesis video stream can
     *         support up to 50 tags.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException,
            TagsPerResourceExceededLimitException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to a stream. A <i>tag</i> is a key-value pair (the value is optional) that you can define
     * and assign to Amazon Web Services resources. If you specify a tag that already exists, the tag value is replaced
     * with the value that you specify in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Billing and Cost Management and Cost Management User Guide</i>.
     * </p>
     * <p>
     * You must provide either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:TagStream</code> action.
     * </p>
     * <p>
     * A Kinesis video stream can support up to 50 tags.
     * </p>
     *
     * @param tagStreamRequest
     * @return Result of the TagStream operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws InvalidResourceFormatException
     *         The format of the <code>StreamARN</code> is invalid.
     * @throws TagsPerResourceExceededLimitException
     *         You have exceeded the limit of tags that you can associate with the resource. A Kinesis video stream can
     *         support up to 50 tags.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.TagStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagStreamResponse tagStream(TagStreamRequest tagStreamRequest) throws ClientLimitExceededException,
            InvalidArgumentException, ResourceNotFoundException, NotAuthorizedException, InvalidResourceFormatException,
            TagsPerResourceExceededLimitException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagStream");

            return clientHandler.execute(new ClientExecutionParams<TagStreamRequest, TagStreamResponse>()
                    .withOperationName("TagStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagStreamRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new TagStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a signaling channel. In the request, specify only a tag key or keys; don't specify
     * the value. If you specify a tag key that does not exist, it's ignored.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a stream. In the request, specify only a tag key or keys; don't specify the value.
     * If you specify a tag key that does not exist, it's ignored.
     * </p>
     * <p>
     * In the request, you must provide the <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param untagStreamRequest
     * @return Result of the UntagStream operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws InvalidResourceFormatException
     *         The format of the <code>StreamARN</code> is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UntagStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagStreamResponse untagStream(UntagStreamRequest untagStreamRequest) throws ClientLimitExceededException,
            InvalidArgumentException, ResourceNotFoundException, NotAuthorizedException, InvalidResourceFormatException,
            AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagStream");

            return clientHandler.execute(new ClientExecutionParams<UntagStreamRequest, UntagStreamResponse>()
                    .withOperationName("UntagStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Increases or decreases the stream's data retention period by the value that you specify. To indicate whether you
     * want to increase or decrease the data retention period, specify the <code>Operation</code> parameter in the
     * request body. In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <note>
     * <p>
     * The retention period that you specify replaces the current value.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:UpdateDataRetention</code> action.
     * </p>
     * <p>
     * Changing the data retention period affects the data in the stream as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the data retention period is increased, existing data is retained for the new retention period. For example,
     * if the data retention period is increased from one hour to seven hours, all existing data is retained for seven
     * hours.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the data retention period is decreased, existing data is retained for the new retention period. For example,
     * if the data retention period is decreased from seven hours to one hour, all existing data is retained for one
     * hour, and any data older than one hour is deleted immediately.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateDataRetentionRequest
     * @return Result of the UpdateDataRetention operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws VersionMismatchException
     *         The stream version that you specified is not the latest version. To get the latest version, use the <a
     *         href
     *         ="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html">DescribeStream</a>
     *         API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UpdateDataRetention
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateDataRetention"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDataRetentionResponse updateDataRetention(UpdateDataRetentionRequest updateDataRetentionRequest)
            throws ClientLimitExceededException, InvalidArgumentException, ResourceNotFoundException, ResourceInUseException,
            NotAuthorizedException, VersionMismatchException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataRetentionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDataRetentionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataRetentionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataRetention");

            return clientHandler.execute(new ClientExecutionParams<UpdateDataRetentionRequest, UpdateDataRetentionResponse>()
                    .withOperationName("UpdateDataRetention").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDataRetentionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDataRetentionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>StreamInfo</code> and <code>ImageProcessingConfiguration</code> fields.
     * </p>
     *
     * @param updateImageGenerationConfigurationRequest
     * @return Result of the UpdateImageGenerationConfiguration operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws NoDataRetentionException
     *         The Stream data retention in hours is equal to zero.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UpdateImageGenerationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateImageGenerationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateImageGenerationConfigurationResponse updateImageGenerationConfiguration(
            UpdateImageGenerationConfigurationRequest updateImageGenerationConfigurationRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException, ResourceInUseException,
            NoDataRetentionException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateImageGenerationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateImageGenerationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateImageGenerationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateImageGenerationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateImageGenerationConfigurationRequest, UpdateImageGenerationConfigurationResponse>()
                            .withOperationName("UpdateImageGenerationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateImageGenerationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateImageGenerationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a <code>SignalingChannel</code> to a stream to store the media. There are two signaling modes that can
     * specified :
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>StorageStatus</code> is disabled, no data will be stored, and the <code>StreamARN</code> parameter
     * will not be needed.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>StorageStatus</code> is enabled, the data will be stored in the <code>StreamARN</code> provided.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateMediaStorageConfigurationRequest
     * @return Result of the UpdateMediaStorageConfiguration operation returned by the service.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws NoDataRetentionException
     *         The Stream data retention in hours is equal to zero.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UpdateMediaStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateMediaStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMediaStorageConfigurationResponse updateMediaStorageConfiguration(
            UpdateMediaStorageConfigurationRequest updateMediaStorageConfigurationRequest) throws ResourceInUseException,
            InvalidArgumentException, ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException,
            NoDataRetentionException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMediaStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateMediaStorageConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateMediaStorageConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMediaStorageConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateMediaStorageConfigurationRequest, UpdateMediaStorageConfigurationResponse>()
                            .withOperationName("UpdateMediaStorageConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateMediaStorageConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateMediaStorageConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the notification information for a stream.
     * </p>
     *
     * @param updateNotificationConfigurationRequest
     * @return Result of the UpdateNotificationConfiguration operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws NoDataRetentionException
     *         The Stream data retention in hours is equal to zero.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UpdateNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNotificationConfigurationResponse updateNotificationConfiguration(
            UpdateNotificationConfigurationRequest updateNotificationConfigurationRequest) throws InvalidArgumentException,
            ClientLimitExceededException, ResourceNotFoundException, AccessDeniedException, ResourceInUseException,
            NoDataRetentionException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationConfigurationRequest, UpdateNotificationConfigurationResponse>()
                            .withOperationName("UpdateNotificationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the existing signaling channel. This is an asynchronous operation and takes time to complete.
     * </p>
     * <p>
     * If the <code>MessageTtlSeconds</code> value is updated (either increased or reduced), it only applies to new
     * messages sent via this channel after it's been updated. Existing messages are still expired as per the previous
     * <code>MessageTtlSeconds</code> value.
     * </p>
     *
     * @param updateSignalingChannelRequest
     * @return Result of the UpdateSignalingChannel operation returned by the service.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws AccessDeniedException
     *         You do not have required permissions to perform this operation.
     * @throws VersionMismatchException
     *         The stream version that you specified is not the latest version. To get the latest version, use the <a
     *         href
     *         ="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html">DescribeStream</a>
     *         API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UpdateSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSignalingChannelResponse updateSignalingChannel(UpdateSignalingChannelRequest updateSignalingChannelRequest)
            throws InvalidArgumentException, ClientLimitExceededException, ResourceNotFoundException, ResourceInUseException,
            AccessDeniedException, VersionMismatchException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSignalingChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSignalingChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSignalingChannelRequest, UpdateSignalingChannelResponse>()
                            .withOperationName("UpdateSignalingChannel").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateSignalingChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSignalingChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates stream metadata, such as the device name and media type.
     * </p>
     * <p>
     * You must provide the stream name or the Amazon Resource Name (ARN) of the stream.
     * </p>
     * <p>
     * To make sure that you have the latest version of the stream before updating it, you can specify the stream
     * version. Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams
     * assigns a new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * <code>UpdateStream</code> is an asynchronous operation, and takes time to complete.
     * </p>
     *
     * @param updateStreamRequest
     * @return Result of the UpdateStream operation returned by the service.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws ResourceNotFoundException
     *         Amazon Kinesis Video Streams can't find the stream that you specified.
     * @throws ResourceInUseException
     *         When the input <code>StreamARN</code> or <code>ChannelARN</code> in <code>CLOUD_STORAGE_MODE</code> is
     *         already mapped to a different Kinesis Video Stream resource, or if the provided input
     *         <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the following :
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li>
     * @throws NotAuthorizedException
     *         The caller is not authorized to perform this operation.
     * @throws VersionMismatchException
     *         The stream version that you specified is not the latest version. To get the latest version, use the <a
     *         href
     *         ="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html">DescribeStream</a>
     *         API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoClient.UpdateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateStreamResponse updateStream(UpdateStreamRequest updateStreamRequest) throws ClientLimitExceededException,
            InvalidArgumentException, ResourceNotFoundException, ResourceInUseException, NotAuthorizedException,
            VersionMismatchException, AwsServiceException, SdkClientException, KinesisVideoException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStream");

            return clientHandler.execute(new ClientExecutionParams<UpdateStreamRequest, UpdateStreamResponse>()
                    .withOperationName("UpdateStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends KinesisVideoRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountStreamLimitExceededException")
                                .exceptionBuilderSupplier(AccountStreamLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VersionMismatchException")
                                .exceptionBuilderSupplier(VersionMismatchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountChannelLimitExceededException")
                                .exceptionBuilderSupplier(AccountChannelLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientLimitExceededException")
                                .exceptionBuilderSupplier(ClientLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StreamEdgeConfigurationNotFoundException")
                                .exceptionBuilderSupplier(StreamEdgeConfigurationNotFoundException::builder).httpStatusCode(404)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceFormatException")
                                .exceptionBuilderSupplier(InvalidResourceFormatException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoDataRetentionException")
                                .exceptionBuilderSupplier(NoDataRetentionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDeviceException")
                                .exceptionBuilderSupplier(InvalidDeviceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsPerResourceExceededLimitException")
                                .exceptionBuilderSupplier(TagsPerResourceExceededLimitException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceStreamLimitExceededException")
                                .exceptionBuilderSupplier(DeviceStreamLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(401).build());
    }

    @Override
    public final KinesisVideoServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
