/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideo;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideo.model.AccessDeniedException;
import software.amazon.awssdk.services.kinesisvideo.model.AccountChannelLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.AccountStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.CreateSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.CreateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DeleteStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeEdgeConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeEdgeConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeImageGenerationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeImageGenerationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMediaStorageConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeMediaStorageConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeNotificationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeNotificationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.DescribeStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.DeviceStreamLimitExceededException;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetDataEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.GetSignalingChannelEndpointRequest;
import software.amazon.awssdk.services.kinesisvideo.model.GetSignalingChannelEndpointResponse;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidDeviceException;
import software.amazon.awssdk.services.kinesisvideo.model.InvalidResourceFormatException;
import software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoException;
import software.amazon.awssdk.services.kinesisvideo.model.KinesisVideoRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.ListTagsForStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.NoDataRetentionException;
import software.amazon.awssdk.services.kinesisvideo.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceInUseException;
import software.amazon.awssdk.services.kinesisvideo.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideo.model.StartEdgeConfigurationUpdateRequest;
import software.amazon.awssdk.services.kinesisvideo.model.StartEdgeConfigurationUpdateResponse;
import software.amazon.awssdk.services.kinesisvideo.model.StreamEdgeConfigurationNotFoundException;
import software.amazon.awssdk.services.kinesisvideo.model.TagResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.TagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.TagsPerResourceExceededLimitException;
import software.amazon.awssdk.services.kinesisvideo.model.UntagResourceRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagResourceResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UntagStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateDataRetentionResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateImageGenerationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateImageGenerationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateMediaStorageConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateMediaStorageConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateNotificationConfigurationRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateNotificationConfigurationResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateSignalingChannelRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateSignalingChannelResponse;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamRequest;
import software.amazon.awssdk.services.kinesisvideo.model.UpdateStreamResponse;
import software.amazon.awssdk.services.kinesisvideo.model.VersionMismatchException;
import software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationPublisher;
import software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsPublisher;
import software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsPublisher;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.CreateStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DeleteStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeEdgeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeImageGenerationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeMappedResourceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeMediaStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetDataEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.GetSignalingChannelEndpointRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListSignalingChannelsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.ListTagsForStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.StartEdgeConfigurationUpdateRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.TagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UntagStreamRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateDataRetentionRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateImageGenerationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateMediaStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateSignalingChannelRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideo.transform.UpdateStreamRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisVideoAsyncClient}.
 *
 * @see KinesisVideoAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoAsyncClient implements KinesisVideoAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final KinesisVideoServiceClientConfiguration serviceClientConfiguration;

    protected DefaultKinesisVideoAsyncClient(KinesisVideoServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a signaling channel.
     * </p>
     * <p>
     * <code>CreateSignalingChannel</code> is an asynchronous operation.
     * </p>
     *
     * @param createSignalingChannelRequest
     * @return A Java Future containing the result of the CreateSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>AccountChannelLimitExceededException You have reached the maximum limit of active signaling channels
     *         for this Amazon Web Services account in this region.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. A Kinesis video stream can support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.CreateSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSignalingChannelResponse> createSignalingChannel(
            CreateSignalingChannelRequest createSignalingChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSignalingChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSignalingChannelRequest, CreateSignalingChannelResponse>()
                            .withOperationName("CreateSignalingChannel")
                            .withMarshaller(new CreateSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSignalingChannelRequest));
            CompletableFuture<CreateSignalingChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Kinesis video stream.
     * </p>
     * <p>
     * When you create a new stream, Kinesis Video Streams assigns it a version number. When you change the stream's
     * metadata, Kinesis Video Streams updates the version.
     * </p>
     * <p>
     * <code>CreateStream</code> is an asynchronous operation.
     * </p>
     * <p>
     * For information about how the service works, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html">How it Works</a>.
     * </p>
     * <p>
     * You must have permissions for the <code>KinesisVideo:CreateStream</code> action.
     * </p>
     *
     * @param createStreamRequest
     * @return A Java Future containing the result of the CreateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccountStreamLimitExceededException The number of streams created for the account is too high.</li>
     *         <li>DeviceStreamLimitExceededException Not implemented.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>InvalidDeviceException Not implemented.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. A Kinesis video stream can support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.CreateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/CreateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamResponse> createStream(CreateStreamRequest createStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStreamRequest, CreateStreamResponse>()
                            .withOperationName("CreateStream").withMarshaller(new CreateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createStreamRequest));
            CompletableFuture<CreateStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified signaling channel. <code>DeleteSignalingChannel</code> is an asynchronous operation. If you
     * don't specify the channel's current version, the most recent version is deleted.
     * </p>
     *
     * @param deleteSignalingChannelRequest
     * @return A Java Future containing the result of the DeleteSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DeleteSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSignalingChannelResponse> deleteSignalingChannel(
            DeleteSignalingChannelRequest deleteSignalingChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSignalingChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSignalingChannelRequest, DeleteSignalingChannelResponse>()
                            .withOperationName("DeleteSignalingChannel")
                            .withMarshaller(new DeleteSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSignalingChannelRequest));
            CompletableFuture<DeleteSignalingChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Kinesis video stream and the data contained in the stream.
     * </p>
     * <p>
     * This method marks the stream for deletion, and makes the data in the stream inaccessible immediately.
     * </p>
     * <p>
     * </p>
     * <p>
     * To ensure that you have the latest version of the stream before deleting it, you can specify the stream version.
     * Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams assigns a
     * new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:DeleteStream</code> action.
     * </p>
     *
     * @param deleteStreamRequest
     * @return A Java Future containing the result of the DeleteStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DeleteStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DeleteStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStreamResponse> deleteStream(DeleteStreamRequest deleteStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStreamRequest, DeleteStreamResponse>()
                            .withOperationName("DeleteStream").withMarshaller(new DeleteStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteStreamRequest));
            CompletableFuture<DeleteStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a stream’s edge configuration that was set using the <code>StartEdgeConfigurationUpdate</code> API. Use
     * this API to get the status of the configuration if the configuration is in sync with the Edge Agent.
     * </p>
     *
     * @param describeEdgeConfigurationRequest
     * @return A Java Future containing the result of the DescribeEdgeConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>StreamEdgeConfigurationNotFoundException The Exception rendered when the Amazon Kinesis Video Stream
     *         can't find a stream's edge configuration that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeEdgeConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeEdgeConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEdgeConfigurationResponse> describeEdgeConfiguration(
            DescribeEdgeConfigurationRequest describeEdgeConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEdgeConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEdgeConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEdgeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEdgeConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEdgeConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEdgeConfigurationRequest, DescribeEdgeConfigurationResponse>()
                            .withOperationName("DescribeEdgeConfiguration")
                            .withMarshaller(new DescribeEdgeConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEdgeConfigurationRequest));
            CompletableFuture<DescribeEdgeConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the <code>ImageGenerationConfiguration</code> for a given Kinesis video stream.
     * </p>
     *
     * @param describeImageGenerationConfigurationRequest
     * @return A Java Future containing the result of the DescribeImageGenerationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeImageGenerationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeImageGenerationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeImageGenerationConfigurationResponse> describeImageGenerationConfiguration(
            DescribeImageGenerationConfigurationRequest describeImageGenerationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeImageGenerationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageGenerationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeImageGenerationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeImageGenerationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeImageGenerationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeImageGenerationConfigurationRequest, DescribeImageGenerationConfigurationResponse>()
                            .withOperationName("DescribeImageGenerationConfiguration")
                            .withMarshaller(new DescribeImageGenerationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeImageGenerationConfigurationRequest));
            CompletableFuture<DescribeImageGenerationConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the stream. Either streamName or streamARN should be provided in the
     * input.
     * </p>
     * <p>
     * Returns the most current information about the stream. The <code>streamName</code> or <code>streamARN</code>
     * should be provided in the input.
     * </p>
     *
     * @param describeMappedResourceConfigurationRequest
     * @return A Java Future containing the result of the DescribeMappedResourceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeMappedResourceConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeMappedResourceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMappedResourceConfigurationResponse> describeMappedResourceConfiguration(
            DescribeMappedResourceConfigurationRequest describeMappedResourceConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMappedResourceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMappedResourceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMappedResourceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeMappedResourceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMappedResourceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMappedResourceConfigurationRequest, DescribeMappedResourceConfigurationResponse>()
                            .withOperationName("DescribeMappedResourceConfiguration")
                            .withMarshaller(new DescribeMappedResourceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMappedResourceConfigurationRequest));
            CompletableFuture<DescribeMappedResourceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the stream. Either streamName or streamARN should be provided in the
     * input.
     * </p>
     * <p>
     * Returns the most current information about the stream. The <code>streamName</code> or <code>streamARN</code>
     * should be provided in the input.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeMappedResourceConfiguration(software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationPublisher publisher = client.describeMappedResourceConfigurationPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.DescribeMappedResourceConfigurationPublisher publisher = client.describeMappedResourceConfigurationPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMappedResourceConfiguration(software.amazon.awssdk.services.kinesisvideo.model.DescribeMappedResourceConfigurationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMappedResourceConfigurationRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeMappedResourceConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeMappedResourceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeMappedResourceConfigurationPublisher describeMappedResourceConfigurationPaginator(
            DescribeMappedResourceConfigurationRequest describeMappedResourceConfigurationRequest) {
        return new DescribeMappedResourceConfigurationPublisher(this,
                applyPaginatorUserAgent(describeMappedResourceConfigurationRequest));
    }

    /**
     * <p>
     * Returns the most current information about the channel. Specify the <code>ChannelName</code> or
     * <code>ChannelARN</code> in the input.
     * </p>
     *
     * @param describeMediaStorageConfigurationRequest
     * @return A Java Future containing the result of the DescribeMediaStorageConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeMediaStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeMediaStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMediaStorageConfigurationResponse> describeMediaStorageConfiguration(
            DescribeMediaStorageConfigurationRequest describeMediaStorageConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMediaStorageConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMediaStorageConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMediaStorageConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeMediaStorageConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMediaStorageConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMediaStorageConfigurationRequest, DescribeMediaStorageConfigurationResponse>()
                            .withOperationName("DescribeMediaStorageConfiguration")
                            .withMarshaller(new DescribeMediaStorageConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMediaStorageConfigurationRequest));
            CompletableFuture<DescribeMediaStorageConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the <code>NotificationConfiguration</code> for a given Kinesis video stream.
     * </p>
     *
     * @param describeNotificationConfigurationRequest
     * @return A Java Future containing the result of the DescribeNotificationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNotificationConfigurationResponse> describeNotificationConfiguration(
            DescribeNotificationConfigurationRequest describeNotificationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNotificationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationConfigurationRequest, DescribeNotificationConfigurationResponse>()
                            .withOperationName("DescribeNotificationConfiguration")
                            .withMarshaller(new DescribeNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeNotificationConfigurationRequest));
            CompletableFuture<DescribeNotificationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the signaling channel. You must specify either the name or the Amazon
     * Resource Name (ARN) of the channel that you want to describe.
     * </p>
     *
     * @param describeSignalingChannelRequest
     * @return A Java Future containing the result of the DescribeSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSignalingChannelResponse> describeSignalingChannel(
            DescribeSignalingChannelRequest describeSignalingChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSignalingChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSignalingChannelRequest, DescribeSignalingChannelResponse>()
                            .withOperationName("DescribeSignalingChannel")
                            .withMarshaller(new DescribeSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSignalingChannelRequest));
            CompletableFuture<DescribeSignalingChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most current information about the specified stream. You must specify either the
     * <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param describeStreamRequest
     * @return A Java Future containing the result of the DescribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.DescribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/DescribeStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStreamResponse> describeStream(DescribeStreamRequest describeStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                            .withOperationName("DescribeStream")
                            .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeStreamRequest));
            CompletableFuture<DescribeStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an endpoint for a specified stream for either reading or writing. Use this endpoint in your application to
     * read from the specified stream (using the <code>GetMedia</code> or <code>GetMediaForFragmentList</code>
     * operations) or write to it (using the <code>PutMedia</code> operation).
     * </p>
     * <note>
     * <p>
     * The returned endpoint does not have the API name appended. The client needs to add the API name to the returned
     * endpoint.
     * </p>
     * </note>
     * <p>
     * In the request, specify the stream either by <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param getDataEndpointRequest
     * @return A Java Future containing the result of the GetDataEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.GetDataEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetDataEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataEndpointResponse> getDataEndpoint(GetDataEndpointRequest getDataEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataEndpointRequest, GetDataEndpointResponse>()
                            .withOperationName("GetDataEndpoint")
                            .withMarshaller(new GetDataEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDataEndpointRequest));
            CompletableFuture<GetDataEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides an endpoint for the specified signaling channel to send and receive messages. This API uses the
     * <code>SingleMasterChannelEndpointConfiguration</code> input parameter, which consists of the
     * <code>Protocols</code> and <code>Role</code> properties.
     * </p>
     * <p>
     * <code>Protocols</code> is used to determine the communication mechanism. For example, if you specify
     * <code>WSS</code> as the protocol, this API produces a secure websocket endpoint. If you specify
     * <code>HTTPS</code> as the protocol, this API generates an HTTPS endpoint.
     * </p>
     * <p>
     * <code>Role</code> determines the messaging permissions. A <code>MASTER</code> role results in this API generating
     * an endpoint that a client can use to communicate with any of the viewers on the channel. A <code>VIEWER</code>
     * role results in this API generating an endpoint that a client can use to communicate only with a
     * <code>MASTER</code>.
     * </p>
     *
     * @param getSignalingChannelEndpointRequest
     * @return A Java Future containing the result of the GetSignalingChannelEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.GetSignalingChannelEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/GetSignalingChannelEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSignalingChannelEndpointResponse> getSignalingChannelEndpoint(
            GetSignalingChannelEndpointRequest getSignalingChannelEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSignalingChannelEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSignalingChannelEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSignalingChannelEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSignalingChannelEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSignalingChannelEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSignalingChannelEndpointRequest, GetSignalingChannelEndpointResponse>()
                            .withOperationName("GetSignalingChannelEndpoint")
                            .withMarshaller(new GetSignalingChannelEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSignalingChannelEndpointRequest));
            CompletableFuture<GetSignalingChannelEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ChannelInfo</code> objects. Each object describes a signaling channel. To retrieve only
     * those channels that satisfy a specific condition, you can specify a <code>ChannelNameCondition</code>.
     * </p>
     *
     * @param listSignalingChannelsRequest
     * @return A Java Future containing the result of the ListSignalingChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListSignalingChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListSignalingChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSignalingChannelsResponse> listSignalingChannels(
            ListSignalingChannelsRequest listSignalingChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSignalingChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSignalingChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSignalingChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSignalingChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSignalingChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSignalingChannelsRequest, ListSignalingChannelsResponse>()
                            .withOperationName("ListSignalingChannels")
                            .withMarshaller(new ListSignalingChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSignalingChannelsRequest));
            CompletableFuture<ListSignalingChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ChannelInfo</code> objects. Each object describes a signaling channel. To retrieve only
     * those channels that satisfy a specific condition, you can specify a <code>ChannelNameCondition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSignalingChannels(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsPublisher publisher = client.listSignalingChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListSignalingChannelsPublisher publisher = client.listSignalingChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSignalingChannels(software.amazon.awssdk.services.kinesisvideo.model.ListSignalingChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSignalingChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListSignalingChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListSignalingChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSignalingChannelsPublisher listSignalingChannelsPaginator(ListSignalingChannelsRequest listSignalingChannelsRequest) {
        return new ListSignalingChannelsPublisher(this, applyPaginatorUserAgent(listSignalingChannelsRequest));
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     *
     * @param listStreamsRequest
     * @return A Java Future containing the result of the ListStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamsResponse> listStreams(ListStreamsRequest listStreamsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreams");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListStreamsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStreamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                            .withOperationName("ListStreams").withMarshaller(new ListStreamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listStreamsRequest));
            CompletableFuture<ListStreamsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>StreamInfo</code> objects. Each object describes a stream. To retrieve only streams
     * that satisfy a specific condition, you can specify a <code>StreamNameCondition</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreams(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kinesisvideo.paginators.ListStreamsPublisher publisher = client.listStreamsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.kinesisvideo.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    public ListStreamsPublisher listStreamsPaginator(ListStreamsRequest listStreamsRequest) {
        return new ListStreamsPublisher(this, applyPaginatorUserAgent(listStreamsRequest));
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified signaling channel.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags associated with the specified stream.
     * </p>
     * <p>
     * In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     *
     * @param listTagsForStreamRequest
     * @return A Java Future containing the result of the ListTagsForStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.ListTagsForStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/ListTagsForStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForStreamResponse> listTagsForStream(ListTagsForStreamRequest listTagsForStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForStreamRequest, ListTagsForStreamResponse>()
                            .withOperationName("ListTagsForStream")
                            .withMarshaller(new ListTagsForStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForStreamRequest));
            CompletableFuture<ListTagsForStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An asynchronous API that updates a stream’s existing edge configuration. The Kinesis Video Stream will sync the
     * stream’s edge configuration with the Edge Agent IoT Greengrass component that runs on an IoT Hub Device, setup at
     * your premise. The time to sync can vary and depends on the connectivity of the Hub Device. The
     * <code>SyncStatus</code> will be updated as the edge configuration is acknowledged, and synced with the Edge
     * Agent.
     * </p>
     * <p>
     * If this API is invoked for the first time, a new edge configuration will be created for the stream, and the sync
     * status will be set to <code>SYNCING</code>. You will have to wait for the sync status to reach a terminal state
     * such as: <code>IN_SYNC</code>, or <code>SYNC_FAILED</code>, before using this API again. If you invoke this API
     * during the syncing process, a <code>ResourceInUseException</code> will be thrown. The connectivity of the
     * stream’s edge configuration and the Edge Agent will be retried for 15 minutes. After 15 minutes, the status will
     * transition into the <code>SYNC_FAILED</code> state.
     * </p>
     *
     * @param startEdgeConfigurationUpdateRequest
     * @return A Java Future containing the result of the StartEdgeConfigurationUpdate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>NoDataRetentionException The Stream data retention in hours is equal to zero.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.StartEdgeConfigurationUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/StartEdgeConfigurationUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartEdgeConfigurationUpdateResponse> startEdgeConfigurationUpdate(
            StartEdgeConfigurationUpdateRequest startEdgeConfigurationUpdateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEdgeConfigurationUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEdgeConfigurationUpdate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartEdgeConfigurationUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartEdgeConfigurationUpdateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartEdgeConfigurationUpdateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartEdgeConfigurationUpdateRequest, StartEdgeConfigurationUpdateResponse>()
                            .withOperationName("StartEdgeConfigurationUpdate")
                            .withMarshaller(new StartEdgeConfigurationUpdateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startEdgeConfigurationUpdateRequest));
            CompletableFuture<StartEdgeConfigurationUpdateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a signaling channel. A <i>tag</i> is a key-value pair (the value is optional) that you
     * can define and assign to Amazon Web Services resources. If you specify a tag that already exists, the tag value
     * is replaced with the value that you specify in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Billing and Cost Management and Cost Management User Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. A Kinesis video stream can support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a stream. A <i>tag</i> is a key-value pair (the value is optional) that you can define
     * and assign to Amazon Web Services resources. If you specify a tag that already exists, the tag value is replaced
     * with the value that you specify in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Billing and Cost Management and Cost Management User Guide</i>.
     * </p>
     * <p>
     * You must provide either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:TagStream</code> action.
     * </p>
     * <p>
     * A Kinesis video stream can support up to 50 tags.
     * </p>
     *
     * @param tagStreamRequest
     * @return A Java Future containing the result of the TagStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>TagsPerResourceExceededLimitException You have exceeded the limit of tags that you can associate with
     *         the resource. A Kinesis video stream can support up to 50 tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.TagStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/TagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagStreamResponse> tagStream(TagStreamRequest tagStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagStreamRequest, TagStreamResponse>().withOperationName("TagStream")
                            .withMarshaller(new TagStreamRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagStreamRequest));
            CompletableFuture<TagStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a signaling channel. In the request, specify only a tag key or keys; don't specify
     * the value. If you specify a tag key that does not exist, it's ignored.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a stream. In the request, specify only a tag key or keys; don't specify the value.
     * If you specify a tag key that does not exist, it's ignored.
     * </p>
     * <p>
     * In the request, you must provide the <code>StreamName</code> or <code>StreamARN</code>.
     * </p>
     *
     * @param untagStreamRequest
     * @return A Java Future containing the result of the UntagStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>InvalidResourceFormatException The format of the <code>StreamARN</code> is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UntagStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UntagStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagStreamResponse> untagStream(UntagStreamRequest untagStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagStreamRequest, UntagStreamResponse>()
                            .withOperationName("UntagStream").withMarshaller(new UntagStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagStreamRequest));
            CompletableFuture<UntagStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Increases or decreases the stream's data retention period by the value that you specify. To indicate whether you
     * want to increase or decrease the data retention period, specify the <code>Operation</code> parameter in the
     * request body. In the request, you must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <note>
     * <p>
     * The retention period that you specify replaces the current value.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code>KinesisVideo:UpdateDataRetention</code> action.
     * </p>
     * <p>
     * Changing the data retention period affects the data in the stream as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the data retention period is increased, existing data is retained for the new retention period. For example,
     * if the data retention period is increased from one hour to seven hours, all existing data is retained for seven
     * hours.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the data retention period is decreased, existing data is retained for the new retention period. For example,
     * if the data retention period is decreased from seven hours to one hour, all existing data is retained for one
     * hour, and any data older than one hour is deleted immediately.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateDataRetentionRequest
     * @return A Java Future containing the result of the UpdateDataRetention operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateDataRetention
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateDataRetention"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataRetentionResponse> updateDataRetention(
            UpdateDataRetentionRequest updateDataRetentionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataRetentionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataRetention");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataRetentionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataRetentionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataRetentionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataRetentionRequest, UpdateDataRetentionResponse>()
                            .withOperationName("UpdateDataRetention")
                            .withMarshaller(new UpdateDataRetentionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDataRetentionRequest));
            CompletableFuture<UpdateDataRetentionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the <code>StreamInfo</code> and <code>ImageProcessingConfiguration</code> fields.
     * </p>
     *
     * @param updateImageGenerationConfigurationRequest
     * @return A Java Future containing the result of the UpdateImageGenerationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>NoDataRetentionException The Stream data retention in hours is equal to zero.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateImageGenerationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateImageGenerationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateImageGenerationConfigurationResponse> updateImageGenerationConfiguration(
            UpdateImageGenerationConfigurationRequest updateImageGenerationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateImageGenerationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateImageGenerationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateImageGenerationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateImageGenerationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateImageGenerationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateImageGenerationConfigurationRequest, UpdateImageGenerationConfigurationResponse>()
                            .withOperationName("UpdateImageGenerationConfiguration")
                            .withMarshaller(new UpdateImageGenerationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateImageGenerationConfigurationRequest));
            CompletableFuture<UpdateImageGenerationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a <code>SignalingChannel</code> to a stream to store the media. There are two signaling modes that can
     * specified :
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>StorageStatus</code> is disabled, no data will be stored, and the <code>StreamARN</code> parameter
     * will not be needed.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>StorageStatus</code> is enabled, the data will be stored in the <code>StreamARN</code> provided.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateMediaStorageConfigurationRequest
     * @return A Java Future containing the result of the UpdateMediaStorageConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>NoDataRetentionException The Stream data retention in hours is equal to zero.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateMediaStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateMediaStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMediaStorageConfigurationResponse> updateMediaStorageConfiguration(
            UpdateMediaStorageConfigurationRequest updateMediaStorageConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateMediaStorageConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMediaStorageConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMediaStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMediaStorageConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMediaStorageConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMediaStorageConfigurationRequest, UpdateMediaStorageConfigurationResponse>()
                            .withOperationName("UpdateMediaStorageConfiguration")
                            .withMarshaller(new UpdateMediaStorageConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMediaStorageConfigurationRequest));
            CompletableFuture<UpdateMediaStorageConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the notification information for a stream.
     * </p>
     *
     * @param updateNotificationConfigurationRequest
     * @return A Java Future containing the result of the UpdateNotificationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>NoDataRetentionException The Stream data retention in hours is equal to zero.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNotificationConfigurationResponse> updateNotificationConfiguration(
            UpdateNotificationConfigurationRequest updateNotificationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotificationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationConfigurationRequest, UpdateNotificationConfigurationResponse>()
                            .withOperationName("UpdateNotificationConfiguration")
                            .withMarshaller(new UpdateNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateNotificationConfigurationRequest));
            CompletableFuture<UpdateNotificationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the existing signaling channel. This is an asynchronous operation and takes time to complete.
     * </p>
     * <p>
     * If the <code>MessageTtlSeconds</code> value is updated (either increased or reduced), it only applies to new
     * messages sent via this channel after it's been updated. Existing messages are still expired as per the previous
     * <code>MessageTtlSeconds</code> value.
     * </p>
     *
     * @param updateSignalingChannelRequest
     * @return A Java Future containing the result of the UpdateSignalingChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You do not have required permissions to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateSignalingChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateSignalingChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSignalingChannelResponse> updateSignalingChannel(
            UpdateSignalingChannelRequest updateSignalingChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSignalingChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSignalingChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSignalingChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSignalingChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSignalingChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSignalingChannelRequest, UpdateSignalingChannelResponse>()
                            .withOperationName("UpdateSignalingChannel")
                            .withMarshaller(new UpdateSignalingChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSignalingChannelRequest));
            CompletableFuture<UpdateSignalingChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates stream metadata, such as the device name and media type.
     * </p>
     * <p>
     * You must provide the stream name or the Amazon Resource Name (ARN) of the stream.
     * </p>
     * <p>
     * To make sure that you have the latest version of the stream before updating it, you can specify the stream
     * version. Kinesis Video Streams assigns a version to each stream. When you update a stream, Kinesis Video Streams
     * assigns a new version number. To get the latest stream version, use the <code>DescribeStream</code> API.
     * </p>
     * <p>
     * <code>UpdateStream</code> is an asynchronous operation, and takes time to complete.
     * </p>
     *
     * @param updateStreamRequest
     * @return A Java Future containing the result of the UpdateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>ResourceNotFoundException Amazon Kinesis Video Streams can't find the stream that you specified.</li>
     *         <li>ResourceInUseException When the input <code>StreamARN</code> or <code>ChannelARN</code> in
     *         <code>CLOUD_STORAGE_MODE</code> is already mapped to a different Kinesis Video Stream resource, or if the
     *         provided input <code>StreamARN</code> or <code>ChannelARN</code> is not in Active status, try one of the
     *         following : </p>
     *         <ol>
     *         <li>
     *         <p>
     *         The <code>DescribeMediaStorageConfiguration</code> API to determine what the stream given channel is
     *         mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeMappedResourceConfiguration</code> API to determine the channel that the given stream
     *         is mapped to.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>DescribeStream</code> or <code>DescribeSignalingChannel</code> API to determine the status of
     *         the resource.
     *         </p>
     *         </li></li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>VersionMismatchException The stream version that you specified is not the latest version. To get the
     *         latest version, use the <a
     *         href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html"
     *         >DescribeStream</a> API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoAsyncClient.UpdateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesisvideo-2017-09-30/UpdateStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStreamResponse> updateStream(UpdateStreamRequest updateStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStreamRequest, UpdateStreamResponse>()
                            .withOperationName("UpdateStream").withMarshaller(new UpdateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateStreamRequest));
            CompletableFuture<UpdateStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final KinesisVideoServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountStreamLimitExceededException")
                                .exceptionBuilderSupplier(AccountStreamLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VersionMismatchException")
                                .exceptionBuilderSupplier(VersionMismatchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountChannelLimitExceededException")
                                .exceptionBuilderSupplier(AccountChannelLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientLimitExceededException")
                                .exceptionBuilderSupplier(ClientLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StreamEdgeConfigurationNotFoundException")
                                .exceptionBuilderSupplier(StreamEdgeConfigurationNotFoundException::builder).httpStatusCode(404)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceFormatException")
                                .exceptionBuilderSupplier(InvalidResourceFormatException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoDataRetentionException")
                                .exceptionBuilderSupplier(NoDataRetentionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDeviceException")
                                .exceptionBuilderSupplier(InvalidDeviceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsPerResourceExceededLimitException")
                                .exceptionBuilderSupplier(TagsPerResourceExceededLimitException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceStreamLimitExceededException")
                                .exceptionBuilderSupplier(DeviceStreamLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(401).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends KinesisVideoRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
