/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideo.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure that contains the information required for the KVS images delivery. If null, the configuration will be
 * deleted from the stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageGenerationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ImageGenerationConfiguration.Builder, ImageGenerationConfiguration> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ImageGenerationConfiguration::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> IMAGE_SELECTOR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageSelectorType").getter(getter(ImageGenerationConfiguration::imageSelectorTypeAsString))
            .setter(setter(Builder::imageSelectorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageSelectorType").build()).build();

    private static final SdkField<ImageGenerationDestinationConfig> DESTINATION_CONFIG_FIELD = SdkField
            .<ImageGenerationDestinationConfig> builder(MarshallingType.SDK_POJO).memberName("DestinationConfig")
            .getter(getter(ImageGenerationConfiguration::destinationConfig)).setter(setter(Builder::destinationConfig))
            .constructor(ImageGenerationDestinationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationConfig").build()).build();

    private static final SdkField<Integer> SAMPLING_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SamplingInterval").getter(getter(ImageGenerationConfiguration::samplingInterval))
            .setter(setter(Builder::samplingInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SamplingInterval").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Format")
            .getter(getter(ImageGenerationConfiguration::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final SdkField<Map<String, String>> FORMAT_CONFIG_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("FormatConfig")
            .getter(getter(ImageGenerationConfiguration::formatConfigAsStrings))
            .setter(setter(Builder::formatConfigWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FormatConfig").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integer> WIDTH_PIXELS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("WidthPixels").getter(getter(ImageGenerationConfiguration::widthPixels))
            .setter(setter(Builder::widthPixels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WidthPixels").build()).build();

    private static final SdkField<Integer> HEIGHT_PIXELS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("HeightPixels").getter(getter(ImageGenerationConfiguration::heightPixels))
            .setter(setter(Builder::heightPixels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeightPixels").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            IMAGE_SELECTOR_TYPE_FIELD, DESTINATION_CONFIG_FIELD, SAMPLING_INTERVAL_FIELD, FORMAT_FIELD, FORMAT_CONFIG_FIELD,
            WIDTH_PIXELS_FIELD, HEIGHT_PIXELS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final String imageSelectorType;

    private final ImageGenerationDestinationConfig destinationConfig;

    private final Integer samplingInterval;

    private final String format;

    private final Map<String, String> formatConfig;

    private final Integer widthPixels;

    private final Integer heightPixels;

    private ImageGenerationConfiguration(BuilderImpl builder) {
        this.status = builder.status;
        this.imageSelectorType = builder.imageSelectorType;
        this.destinationConfig = builder.destinationConfig;
        this.samplingInterval = builder.samplingInterval;
        this.format = builder.format;
        this.formatConfig = builder.formatConfig;
        this.widthPixels = builder.widthPixels;
        this.heightPixels = builder.heightPixels;
    }

    /**
     * <p>
     * Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
     * @see ConfigurationStatus
     */
    public final ConfigurationStatus status() {
        return ConfigurationStatus.fromValue(status);
    }

    /**
     * <p>
     * Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
     * @see ConfigurationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The origin of the Server or Producer timestamps to use to generate the images.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #imageSelectorType}
     * will return {@link ImageSelectorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #imageSelectorTypeAsString}.
     * </p>
     * 
     * @return The origin of the Server or Producer timestamps to use to generate the images.
     * @see ImageSelectorType
     */
    public final ImageSelectorType imageSelectorType() {
        return ImageSelectorType.fromValue(imageSelectorType);
    }

    /**
     * <p>
     * The origin of the Server or Producer timestamps to use to generate the images.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #imageSelectorType}
     * will return {@link ImageSelectorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #imageSelectorTypeAsString}.
     * </p>
     * 
     * @return The origin of the Server or Producer timestamps to use to generate the images.
     * @see ImageSelectorType
     */
    public final String imageSelectorTypeAsString() {
        return imageSelectorType;
    }

    /**
     * <p>
     * The structure that contains the information required to deliver images to a customer.
     * </p>
     * 
     * @return The structure that contains the information required to deliver images to a customer.
     */
    public final ImageGenerationDestinationConfig destinationConfig() {
        return destinationConfig;
    }

    /**
     * <p>
     * The time interval in milliseconds (ms) at which the images need to be generated from the stream. The minimum
     * value that can be provided is 200 ms. If the timestamp range is less than the sampling interval, the Image from
     * the <code>StartTimestamp</code> will be returned if available.
     * </p>
     * 
     * @return The time interval in milliseconds (ms) at which the images need to be generated from the stream. The
     *         minimum value that can be provided is 200 ms. If the timestamp range is less than the sampling interval,
     *         the Image from the <code>StartTimestamp</code> will be returned if available.
     */
    public final Integer samplingInterval() {
        return samplingInterval;
    }

    /**
     * <p>
     * The accepted image format.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link Format#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The accepted image format.
     * @see Format
     */
    public final Format format() {
        return Format.fromValue(format);
    }

    /**
     * <p>
     * The accepted image format.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link Format#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The accepted image format.
     * @see Format
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The list of a key-value pair structure that contains extra parameters that can be applied when the image is
     * generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates the JPEG quality
     * key to be used to generate the image. The <code>FormatConfig</code> value accepts ints from 1 to 100. If the
     * value is 1, the image will be generated with less quality and the best compression. If the value is 100, the
     * image will be generated with the best quality and less compression. If no value is provided, the default value of
     * the <code>JPEGQuality</code> key will be set to 80.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFormatConfig} method.
     * </p>
     * 
     * @return The list of a key-value pair structure that contains extra parameters that can be applied when the image
     *         is generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates the JPEG
     *         quality key to be used to generate the image. The <code>FormatConfig</code> value accepts ints from 1 to
     *         100. If the value is 1, the image will be generated with less quality and the best compression. If the
     *         value is 100, the image will be generated with the best quality and less compression. If no value is
     *         provided, the default value of the <code>JPEGQuality</code> key will be set to 80.
     */
    public final Map<FormatConfigKey, String> formatConfig() {
        return FormatConfigCopier.copyStringToEnum(formatConfig);
    }

    /**
     * For responses, this returns true if the service returned a value for the FormatConfig property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFormatConfig() {
        return formatConfig != null && !(formatConfig instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of a key-value pair structure that contains extra parameters that can be applied when the image is
     * generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates the JPEG quality
     * key to be used to generate the image. The <code>FormatConfig</code> value accepts ints from 1 to 100. If the
     * value is 1, the image will be generated with less quality and the best compression. If the value is 100, the
     * image will be generated with the best quality and less compression. If no value is provided, the default value of
     * the <code>JPEGQuality</code> key will be set to 80.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFormatConfig} method.
     * </p>
     * 
     * @return The list of a key-value pair structure that contains extra parameters that can be applied when the image
     *         is generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates the JPEG
     *         quality key to be used to generate the image. The <code>FormatConfig</code> value accepts ints from 1 to
     *         100. If the value is 1, the image will be generated with less quality and the best compression. If the
     *         value is 100, the image will be generated with the best quality and less compression. If no value is
     *         provided, the default value of the <code>JPEGQuality</code> key will be set to 80.
     */
    public final Map<String, String> formatConfigAsStrings() {
        return formatConfig;
    }

    /**
     * <p>
     * The width of the output image that is used in conjunction with the <code>HeightPixels</code> parameter. When both
     * <code>WidthPixels</code> and <code>HeightPixels</code> parameters are provided, the image will be stretched to
     * fit the specified aspect ratio. If only the <code>WidthPixels</code> parameter is provided, its original aspect
     * ratio will be used to calculate the <code>HeightPixels</code> ratio. If neither parameter is provided, the
     * original image size will be returned.
     * </p>
     * 
     * @return The width of the output image that is used in conjunction with the <code>HeightPixels</code> parameter.
     *         When both <code>WidthPixels</code> and <code>HeightPixels</code> parameters are provided, the image will
     *         be stretched to fit the specified aspect ratio. If only the <code>WidthPixels</code> parameter is
     *         provided, its original aspect ratio will be used to calculate the <code>HeightPixels</code> ratio. If
     *         neither parameter is provided, the original image size will be returned.
     */
    public final Integer widthPixels() {
        return widthPixels;
    }

    /**
     * <p>
     * The height of the output image that is used in conjunction with the <code>WidthPixels</code> parameter. When both
     * <code>HeightPixels</code> and <code>WidthPixels</code> parameters are provided, the image will be stretched to
     * fit the specified aspect ratio. If only the <code>HeightPixels</code> parameter is provided, its original aspect
     * ratio will be used to calculate the <code>WidthPixels</code> ratio. If neither parameter is provided, the
     * original image size will be returned.
     * </p>
     * 
     * @return The height of the output image that is used in conjunction with the <code>WidthPixels</code> parameter.
     *         When both <code>HeightPixels</code> and <code>WidthPixels</code> parameters are provided, the image will
     *         be stretched to fit the specified aspect ratio. If only the <code>HeightPixels</code> parameter is
     *         provided, its original aspect ratio will be used to calculate the <code>WidthPixels</code> ratio. If
     *         neither parameter is provided, the original image size will be returned.
     */
    public final Integer heightPixels() {
        return heightPixels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(imageSelectorTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(destinationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(samplingInterval());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFormatConfig() ? formatConfigAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(widthPixels());
        hashCode = 31 * hashCode + Objects.hashCode(heightPixels());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageGenerationConfiguration)) {
            return false;
        }
        ImageGenerationConfiguration other = (ImageGenerationConfiguration) obj;
        return Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(imageSelectorTypeAsString(), other.imageSelectorTypeAsString())
                && Objects.equals(destinationConfig(), other.destinationConfig())
                && Objects.equals(samplingInterval(), other.samplingInterval())
                && Objects.equals(formatAsString(), other.formatAsString()) && hasFormatConfig() == other.hasFormatConfig()
                && Objects.equals(formatConfigAsStrings(), other.formatConfigAsStrings())
                && Objects.equals(widthPixels(), other.widthPixels()) && Objects.equals(heightPixels(), other.heightPixels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageGenerationConfiguration").add("Status", statusAsString())
                .add("ImageSelectorType", imageSelectorTypeAsString()).add("DestinationConfig", destinationConfig())
                .add("SamplingInterval", samplingInterval()).add("Format", formatAsString())
                .add("FormatConfig", hasFormatConfig() ? formatConfigAsStrings() : null).add("WidthPixels", widthPixels())
                .add("HeightPixels", heightPixels()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ImageSelectorType":
            return Optional.ofNullable(clazz.cast(imageSelectorTypeAsString()));
        case "DestinationConfig":
            return Optional.ofNullable(clazz.cast(destinationConfig()));
        case "SamplingInterval":
            return Optional.ofNullable(clazz.cast(samplingInterval()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "FormatConfig":
            return Optional.ofNullable(clazz.cast(formatConfigAsStrings()));
        case "WidthPixels":
            return Optional.ofNullable(clazz.cast(widthPixels()));
        case "HeightPixels":
            return Optional.ofNullable(clazz.cast(heightPixels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Status", STATUS_FIELD);
        map.put("ImageSelectorType", IMAGE_SELECTOR_TYPE_FIELD);
        map.put("DestinationConfig", DESTINATION_CONFIG_FIELD);
        map.put("SamplingInterval", SAMPLING_INTERVAL_FIELD);
        map.put("Format", FORMAT_FIELD);
        map.put("FormatConfig", FORMAT_CONFIG_FIELD);
        map.put("WidthPixels", WIDTH_PIXELS_FIELD);
        map.put("HeightPixels", HEIGHT_PIXELS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ImageGenerationConfiguration, T> g) {
        return obj -> g.apply((ImageGenerationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageGenerationConfiguration> {
        /**
         * <p>
         * Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
         * </p>
         * 
         * @param status
         *        Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
         * @see ConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
         * </p>
         * 
         * @param status
         *        Indicates whether the <code>ContinuousImageGenerationConfigurations</code> API is enabled or disabled.
         * @see ConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationStatus
         */
        Builder status(ConfigurationStatus status);

        /**
         * <p>
         * The origin of the Server or Producer timestamps to use to generate the images.
         * </p>
         * 
         * @param imageSelectorType
         *        The origin of the Server or Producer timestamps to use to generate the images.
         * @see ImageSelectorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageSelectorType
         */
        Builder imageSelectorType(String imageSelectorType);

        /**
         * <p>
         * The origin of the Server or Producer timestamps to use to generate the images.
         * </p>
         * 
         * @param imageSelectorType
         *        The origin of the Server or Producer timestamps to use to generate the images.
         * @see ImageSelectorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageSelectorType
         */
        Builder imageSelectorType(ImageSelectorType imageSelectorType);

        /**
         * <p>
         * The structure that contains the information required to deliver images to a customer.
         * </p>
         * 
         * @param destinationConfig
         *        The structure that contains the information required to deliver images to a customer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationConfig(ImageGenerationDestinationConfig destinationConfig);

        /**
         * <p>
         * The structure that contains the information required to deliver images to a customer.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageGenerationDestinationConfig.Builder}
         * avoiding the need to create one manually via {@link ImageGenerationDestinationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageGenerationDestinationConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #destinationConfig(ImageGenerationDestinationConfig)}.
         * 
         * @param destinationConfig
         *        a consumer that will call methods on {@link ImageGenerationDestinationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationConfig(ImageGenerationDestinationConfig)
         */
        default Builder destinationConfig(Consumer<ImageGenerationDestinationConfig.Builder> destinationConfig) {
            return destinationConfig(ImageGenerationDestinationConfig.builder().applyMutation(destinationConfig).build());
        }

        /**
         * <p>
         * The time interval in milliseconds (ms) at which the images need to be generated from the stream. The minimum
         * value that can be provided is 200 ms. If the timestamp range is less than the sampling interval, the Image
         * from the <code>StartTimestamp</code> will be returned if available.
         * </p>
         * 
         * @param samplingInterval
         *        The time interval in milliseconds (ms) at which the images need to be generated from the stream. The
         *        minimum value that can be provided is 200 ms. If the timestamp range is less than the sampling
         *        interval, the Image from the <code>StartTimestamp</code> will be returned if available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder samplingInterval(Integer samplingInterval);

        /**
         * <p>
         * The accepted image format.
         * </p>
         * 
         * @param format
         *        The accepted image format.
         * @see Format
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Format
         */
        Builder format(String format);

        /**
         * <p>
         * The accepted image format.
         * </p>
         * 
         * @param format
         *        The accepted image format.
         * @see Format
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Format
         */
        Builder format(Format format);

        /**
         * <p>
         * The list of a key-value pair structure that contains extra parameters that can be applied when the image is
         * generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates the JPEG
         * quality key to be used to generate the image. The <code>FormatConfig</code> value accepts ints from 1 to 100.
         * If the value is 1, the image will be generated with less quality and the best compression. If the value is
         * 100, the image will be generated with the best quality and less compression. If no value is provided, the
         * default value of the <code>JPEGQuality</code> key will be set to 80.
         * </p>
         * 
         * @param formatConfig
         *        The list of a key-value pair structure that contains extra parameters that can be applied when the
         *        image is generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates
         *        the JPEG quality key to be used to generate the image. The <code>FormatConfig</code> value accepts
         *        ints from 1 to 100. If the value is 1, the image will be generated with less quality and the best
         *        compression. If the value is 100, the image will be generated with the best quality and less
         *        compression. If no value is provided, the default value of the <code>JPEGQuality</code> key will be
         *        set to 80.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formatConfigWithStrings(Map<String, String> formatConfig);

        /**
         * <p>
         * The list of a key-value pair structure that contains extra parameters that can be applied when the image is
         * generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates the JPEG
         * quality key to be used to generate the image. The <code>FormatConfig</code> value accepts ints from 1 to 100.
         * If the value is 1, the image will be generated with less quality and the best compression. If the value is
         * 100, the image will be generated with the best quality and less compression. If no value is provided, the
         * default value of the <code>JPEGQuality</code> key will be set to 80.
         * </p>
         * 
         * @param formatConfig
         *        The list of a key-value pair structure that contains extra parameters that can be applied when the
         *        image is generated. The <code>FormatConfig</code> key is the <code>JPEGQuality</code>, which indicates
         *        the JPEG quality key to be used to generate the image. The <code>FormatConfig</code> value accepts
         *        ints from 1 to 100. If the value is 1, the image will be generated with less quality and the best
         *        compression. If the value is 100, the image will be generated with the best quality and less
         *        compression. If no value is provided, the default value of the <code>JPEGQuality</code> key will be
         *        set to 80.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formatConfig(Map<FormatConfigKey, String> formatConfig);

        /**
         * <p>
         * The width of the output image that is used in conjunction with the <code>HeightPixels</code> parameter. When
         * both <code>WidthPixels</code> and <code>HeightPixels</code> parameters are provided, the image will be
         * stretched to fit the specified aspect ratio. If only the <code>WidthPixels</code> parameter is provided, its
         * original aspect ratio will be used to calculate the <code>HeightPixels</code> ratio. If neither parameter is
         * provided, the original image size will be returned.
         * </p>
         * 
         * @param widthPixels
         *        The width of the output image that is used in conjunction with the <code>HeightPixels</code>
         *        parameter. When both <code>WidthPixels</code> and <code>HeightPixels</code> parameters are provided,
         *        the image will be stretched to fit the specified aspect ratio. If only the <code>WidthPixels</code>
         *        parameter is provided, its original aspect ratio will be used to calculate the
         *        <code>HeightPixels</code> ratio. If neither parameter is provided, the original image size will be
         *        returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder widthPixels(Integer widthPixels);

        /**
         * <p>
         * The height of the output image that is used in conjunction with the <code>WidthPixels</code> parameter. When
         * both <code>HeightPixels</code> and <code>WidthPixels</code> parameters are provided, the image will be
         * stretched to fit the specified aspect ratio. If only the <code>HeightPixels</code> parameter is provided, its
         * original aspect ratio will be used to calculate the <code>WidthPixels</code> ratio. If neither parameter is
         * provided, the original image size will be returned.
         * </p>
         * 
         * @param heightPixels
         *        The height of the output image that is used in conjunction with the <code>WidthPixels</code>
         *        parameter. When both <code>HeightPixels</code> and <code>WidthPixels</code> parameters are provided,
         *        the image will be stretched to fit the specified aspect ratio. If only the <code>HeightPixels</code>
         *        parameter is provided, its original aspect ratio will be used to calculate the
         *        <code>WidthPixels</code> ratio. If neither parameter is provided, the original image size will be
         *        returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heightPixels(Integer heightPixels);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private String imageSelectorType;

        private ImageGenerationDestinationConfig destinationConfig;

        private Integer samplingInterval;

        private String format;

        private Map<String, String> formatConfig = DefaultSdkAutoConstructMap.getInstance();

        private Integer widthPixels;

        private Integer heightPixels;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageGenerationConfiguration model) {
            status(model.status);
            imageSelectorType(model.imageSelectorType);
            destinationConfig(model.destinationConfig);
            samplingInterval(model.samplingInterval);
            format(model.format);
            formatConfigWithStrings(model.formatConfig);
            widthPixels(model.widthPixels);
            heightPixels(model.heightPixels);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ConfigurationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getImageSelectorType() {
            return imageSelectorType;
        }

        public final void setImageSelectorType(String imageSelectorType) {
            this.imageSelectorType = imageSelectorType;
        }

        @Override
        public final Builder imageSelectorType(String imageSelectorType) {
            this.imageSelectorType = imageSelectorType;
            return this;
        }

        @Override
        public final Builder imageSelectorType(ImageSelectorType imageSelectorType) {
            this.imageSelectorType(imageSelectorType == null ? null : imageSelectorType.toString());
            return this;
        }

        public final ImageGenerationDestinationConfig.Builder getDestinationConfig() {
            return destinationConfig != null ? destinationConfig.toBuilder() : null;
        }

        public final void setDestinationConfig(ImageGenerationDestinationConfig.BuilderImpl destinationConfig) {
            this.destinationConfig = destinationConfig != null ? destinationConfig.build() : null;
        }

        @Override
        public final Builder destinationConfig(ImageGenerationDestinationConfig destinationConfig) {
            this.destinationConfig = destinationConfig;
            return this;
        }

        public final Integer getSamplingInterval() {
            return samplingInterval;
        }

        public final void setSamplingInterval(Integer samplingInterval) {
            this.samplingInterval = samplingInterval;
        }

        @Override
        public final Builder samplingInterval(Integer samplingInterval) {
            this.samplingInterval = samplingInterval;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(Format format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final Map<String, String> getFormatConfig() {
            if (formatConfig instanceof SdkAutoConstructMap) {
                return null;
            }
            return formatConfig;
        }

        public final void setFormatConfig(Map<String, String> formatConfig) {
            this.formatConfig = FormatConfigCopier.copy(formatConfig);
        }

        @Override
        public final Builder formatConfigWithStrings(Map<String, String> formatConfig) {
            this.formatConfig = FormatConfigCopier.copy(formatConfig);
            return this;
        }

        @Override
        public final Builder formatConfig(Map<FormatConfigKey, String> formatConfig) {
            this.formatConfig = FormatConfigCopier.copyEnumToString(formatConfig);
            return this;
        }

        public final Integer getWidthPixels() {
            return widthPixels;
        }

        public final void setWidthPixels(Integer widthPixels) {
            this.widthPixels = widthPixels;
        }

        @Override
        public final Builder widthPixels(Integer widthPixels) {
            this.widthPixels = widthPixels;
            return this;
        }

        public final Integer getHeightPixels() {
            return heightPixels;
        }

        public final void setHeightPixels(Integer heightPixels) {
            this.heightPixels = heightPixels;
        }

        @Override
        public final Builder heightPixels(Integer heightPixels) {
            this.heightPixels = heightPixels;
            return this;
        }

        @Override
        public ImageGenerationConfiguration build() {
            return new ImageGenerationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
