/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetDashStreamingSessionUrlRequest extends KinesisVideoArchivedMediaRequest implements
        ToCopyableBuilder<GetDashStreamingSessionUrlRequest.Builder, GetDashStreamingSessionUrlRequest> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetDashStreamingSessionUrlRequest::streamName)).setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetDashStreamingSessionUrlRequest::streamARN)).setter(setter(Builder::streamARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamARN").build()).build();

    private static final SdkField<String> PLAYBACK_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetDashStreamingSessionUrlRequest::playbackModeAsString)).setter(setter(Builder::playbackMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlaybackMode").build()).build();

    private static final SdkField<String> DISPLAY_FRAGMENT_TIMESTAMP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetDashStreamingSessionUrlRequest::displayFragmentTimestampAsString))
            .setter(setter(Builder::displayFragmentTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayFragmentTimestamp").build())
            .build();

    private static final SdkField<String> DISPLAY_FRAGMENT_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetDashStreamingSessionUrlRequest::displayFragmentNumberAsString))
            .setter(setter(Builder::displayFragmentNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayFragmentNumber").build())
            .build();

    private static final SdkField<DASHFragmentSelector> DASH_FRAGMENT_SELECTOR_FIELD = SdkField
            .<DASHFragmentSelector> builder(MarshallingType.SDK_POJO)
            .getter(getter(GetDashStreamingSessionUrlRequest::dashFragmentSelector))
            .setter(setter(Builder::dashFragmentSelector)).constructor(DASHFragmentSelector::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DASHFragmentSelector").build())
            .build();

    private static final SdkField<Integer> EXPIRES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(GetDashStreamingSessionUrlRequest::expires)).setter(setter(Builder::expires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Expires").build()).build();

    private static final SdkField<Long> MAX_MANIFEST_FRAGMENT_RESULTS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .getter(getter(GetDashStreamingSessionUrlRequest::maxManifestFragmentResults))
            .setter(setter(Builder::maxManifestFragmentResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxManifestFragmentResults").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD,
            STREAM_ARN_FIELD, PLAYBACK_MODE_FIELD, DISPLAY_FRAGMENT_TIMESTAMP_FIELD, DISPLAY_FRAGMENT_NUMBER_FIELD,
            DASH_FRAGMENT_SELECTOR_FIELD, EXPIRES_FIELD, MAX_MANIFEST_FRAGMENT_RESULTS_FIELD));

    private final String streamName;

    private final String streamARN;

    private final String playbackMode;

    private final String displayFragmentTimestamp;

    private final String displayFragmentNumber;

    private final DASHFragmentSelector dashFragmentSelector;

    private final Integer expires;

    private final Long maxManifestFragmentResults;

    private GetDashStreamingSessionUrlRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.streamARN = builder.streamARN;
        this.playbackMode = builder.playbackMode;
        this.displayFragmentTimestamp = builder.displayFragmentTimestamp;
        this.displayFragmentNumber = builder.displayFragmentNumber;
        this.dashFragmentSelector = builder.dashFragmentSelector;
        this.expires = builder.expires;
        this.maxManifestFragmentResults = builder.maxManifestFragmentResults;
    }

    /**
     * <p>
     * The name of the stream for which to retrieve the MPEG-DASH manifest URL.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * 
     * @return The name of the stream for which to retrieve the MPEG-DASH manifest URL.</p>
     *         <p>
     *         You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     */
    public String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the stream for which to retrieve the MPEG-DASH manifest URL.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the stream for which to retrieve the MPEG-DASH manifest URL.</p>
     *         <p>
     *         You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     */
    public String streamARN() {
        return streamARN;
    }

    /**
     * <p>
     * Whether to retrieve live, live replay, or archived, on-demand data.
     * </p>
     * <p>
     * Features of the three types of sessions include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated with the
     * latest fragments as they become available. We recommend that the media player retrieve a new manifest on a
     * one-second interval. When this type of session is played in a media player, the user interface typically displays
     * a "live" notification, with no scrubber control for choosing the position in the playback window to display.
     * </p>
     * <note>
     * <p>
     * In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even if there is
     * a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media player to halt
     * or cause a jump in playback. In this mode, fragments are not added to the MPEG-DASH manifest if they are older
     * than the newest fragment in the playlist. If the missing fragment becomes available after a subsequent fragment
     * is added to the manifest, the older fragment is not added, and the gap is not filled.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated similarly to how
     * it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start time.
     * Instead of fragments being added as they are ingested, fragments are added as the duration of the next fragment
     * elapses. For example, if the fragments in the session are two seconds long, then a new fragment is added to the
     * manifest every two seconds. This mode is useful to be able to start playback from when an event is detected and
     * continue live streaming media that has not yet been ingested as of the time of the session creation. This mode is
     * also useful to stream previously archived media without being limited by the 1,000 fragment limit in the
     * <code>ON_DEMAND</code> mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the fragments for
     * the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The manifest
     * must be retrieved only once for each session. When this type of session is played in a media player, the user
     * interface typically displays a scrubber control for choosing the position in the playback window to display.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there are
     * multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that is, the
     * newer fragment) is included in the MPEG-DASH manifest. The other fragments are not included. Fragments that have
     * different timestamps but have overlapping durations are still included in the MPEG-DASH manifest. This can lead
     * to unexpected behavior in the media player.
     * </p>
     * <p>
     * The default is <code>LIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #playbackMode} will
     * return {@link DASHPlaybackMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #playbackModeAsString}.
     * </p>
     * 
     * @return Whether to retrieve live, live replay, or archived, on-demand data.</p>
     *         <p>
     *         Features of the three types of sessions include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated with
     *         the latest fragments as they become available. We recommend that the media player retrieve a new manifest
     *         on a one-second interval. When this type of session is played in a media player, the user interface
     *         typically displays a "live" notification, with no scrubber control for choosing the position in the
     *         playback window to display.
     *         </p>
     *         <note>
     *         <p>
     *         In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even if
     *         there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
     *         player to halt or cause a jump in playback. In this mode, fragments are not added to the MPEG-DASH
     *         manifest if they are older than the newest fragment in the playlist. If the missing fragment becomes
     *         available after a subsequent fragment is added to the manifest, the older fragment is not added, and the
     *         gap is not filled.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated similarly
     *         to how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given
     *         start time. Instead of fragments being added as they are ingested, fragments are added as the duration of
     *         the next fragment elapses. For example, if the fragments in the session are two seconds long, then a new
     *         fragment is added to the manifest every two seconds. This mode is useful to be able to start playback
     *         from when an event is detected and continue live streaming media that has not yet been ingested as of the
     *         time of the session creation. This mode is also useful to stream previously archived media without being
     *         limited by the 1,000 fragment limit in the <code>ON_DEMAND</code> mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the
     *         fragments for the session, up to the number that is specified in
     *         <code>MaxMediaPlaylistFragmentResults</code>. The manifest must be retrieved only once for each session.
     *         When this type of session is played in a media player, the user interface typically displays a scrubber
     *         control for choosing the position in the playback window to display.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
     *         there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
     *         number (that is, the newer fragment) is included in the MPEG-DASH manifest. The other fragments are not
     *         included. Fragments that have different timestamps but have overlapping durations are still included in
     *         the MPEG-DASH manifest. This can lead to unexpected behavior in the media player.
     *         </p>
     *         <p>
     *         The default is <code>LIVE</code>.
     * @see DASHPlaybackMode
     */
    public DASHPlaybackMode playbackMode() {
        return DASHPlaybackMode.fromValue(playbackMode);
    }

    /**
     * <p>
     * Whether to retrieve live, live replay, or archived, on-demand data.
     * </p>
     * <p>
     * Features of the three types of sessions include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated with the
     * latest fragments as they become available. We recommend that the media player retrieve a new manifest on a
     * one-second interval. When this type of session is played in a media player, the user interface typically displays
     * a "live" notification, with no scrubber control for choosing the position in the playback window to display.
     * </p>
     * <note>
     * <p>
     * In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even if there is
     * a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media player to halt
     * or cause a jump in playback. In this mode, fragments are not added to the MPEG-DASH manifest if they are older
     * than the newest fragment in the playlist. If the missing fragment becomes available after a subsequent fragment
     * is added to the manifest, the older fragment is not added, and the gap is not filled.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated similarly to how
     * it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start time.
     * Instead of fragments being added as they are ingested, fragments are added as the duration of the next fragment
     * elapses. For example, if the fragments in the session are two seconds long, then a new fragment is added to the
     * manifest every two seconds. This mode is useful to be able to start playback from when an event is detected and
     * continue live streaming media that has not yet been ingested as of the time of the session creation. This mode is
     * also useful to stream previously archived media without being limited by the 1,000 fragment limit in the
     * <code>ON_DEMAND</code> mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the fragments for
     * the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The manifest
     * must be retrieved only once for each session. When this type of session is played in a media player, the user
     * interface typically displays a scrubber control for choosing the position in the playback window to display.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there are
     * multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that is, the
     * newer fragment) is included in the MPEG-DASH manifest. The other fragments are not included. Fragments that have
     * different timestamps but have overlapping durations are still included in the MPEG-DASH manifest. This can lead
     * to unexpected behavior in the media player.
     * </p>
     * <p>
     * The default is <code>LIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #playbackMode} will
     * return {@link DASHPlaybackMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #playbackModeAsString}.
     * </p>
     * 
     * @return Whether to retrieve live, live replay, or archived, on-demand data.</p>
     *         <p>
     *         Features of the three types of sessions include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated with
     *         the latest fragments as they become available. We recommend that the media player retrieve a new manifest
     *         on a one-second interval. When this type of session is played in a media player, the user interface
     *         typically displays a "live" notification, with no scrubber control for choosing the position in the
     *         playback window to display.
     *         </p>
     *         <note>
     *         <p>
     *         In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even if
     *         there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
     *         player to halt or cause a jump in playback. In this mode, fragments are not added to the MPEG-DASH
     *         manifest if they are older than the newest fragment in the playlist. If the missing fragment becomes
     *         available after a subsequent fragment is added to the manifest, the older fragment is not added, and the
     *         gap is not filled.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated similarly
     *         to how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given
     *         start time. Instead of fragments being added as they are ingested, fragments are added as the duration of
     *         the next fragment elapses. For example, if the fragments in the session are two seconds long, then a new
     *         fragment is added to the manifest every two seconds. This mode is useful to be able to start playback
     *         from when an event is detected and continue live streaming media that has not yet been ingested as of the
     *         time of the session creation. This mode is also useful to stream previously archived media without being
     *         limited by the 1,000 fragment limit in the <code>ON_DEMAND</code> mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the
     *         fragments for the session, up to the number that is specified in
     *         <code>MaxMediaPlaylistFragmentResults</code>. The manifest must be retrieved only once for each session.
     *         When this type of session is played in a media player, the user interface typically displays a scrubber
     *         control for choosing the position in the playback window to display.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
     *         there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
     *         number (that is, the newer fragment) is included in the MPEG-DASH manifest. The other fragments are not
     *         included. Fragments that have different timestamps but have overlapping durations are still included in
     *         the MPEG-DASH manifest. This can lead to unexpected behavior in the media player.
     *         </p>
     *         <p>
     *         The default is <code>LIVE</code>.
     * @see DASHPlaybackMode
     */
    public String playbackModeAsString() {
        return playbackMode;
    }

    /**
     * <p>
     * Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived using
     * attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not properly handle
     * gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the manifest file to enable
     * playback of media with discontinuities. Therefore, the wall-clock time derived from the manifest file may be
     * inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>, the accurate fragment timestamp is added
     * to each S element in the manifest file with the attribute name “kvs:ts”. A custom MPEG-DASH media player is
     * necessary to leverage this custom attribute.
     * </p>
     * <p>
     * The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
     * timestamps will be the server start timestamps. Similarly, when <a>DASHFragmentSelector</a> is
     * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #displayFragmentTimestamp} will return {@link DASHDisplayFragmentTimestamp#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #displayFragmentTimestampAsString}.
     * </p>
     * 
     * @return Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived
     *         using attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not
     *         properly handle gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the
     *         manifest file to enable playback of media with discontinuities. Therefore, the wall-clock time derived
     *         from the manifest file may be inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>, the
     *         accurate fragment timestamp is added to each S element in the manifest file with the attribute name
     *         “kvs:ts”. A custom MPEG-DASH media player is necessary to leverage this custom attribute.</p>
     *         <p>
     *         The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is
     *         <code>SERVER_TIMESTAMP</code>, the timestamps will be the server start timestamps. Similarly, when
     *         <a>DASHFragmentSelector</a> is <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start
     *         timestamps.
     * @see DASHDisplayFragmentTimestamp
     */
    public DASHDisplayFragmentTimestamp displayFragmentTimestamp() {
        return DASHDisplayFragmentTimestamp.fromValue(displayFragmentTimestamp);
    }

    /**
     * <p>
     * Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived using
     * attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not properly handle
     * gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the manifest file to enable
     * playback of media with discontinuities. Therefore, the wall-clock time derived from the manifest file may be
     * inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>, the accurate fragment timestamp is added
     * to each S element in the manifest file with the attribute name “kvs:ts”. A custom MPEG-DASH media player is
     * necessary to leverage this custom attribute.
     * </p>
     * <p>
     * The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
     * timestamps will be the server start timestamps. Similarly, when <a>DASHFragmentSelector</a> is
     * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #displayFragmentTimestamp} will return {@link DASHDisplayFragmentTimestamp#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #displayFragmentTimestampAsString}.
     * </p>
     * 
     * @return Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived
     *         using attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not
     *         properly handle gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the
     *         manifest file to enable playback of media with discontinuities. Therefore, the wall-clock time derived
     *         from the manifest file may be inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>, the
     *         accurate fragment timestamp is added to each S element in the manifest file with the attribute name
     *         “kvs:ts”. A custom MPEG-DASH media player is necessary to leverage this custom attribute.</p>
     *         <p>
     *         The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is
     *         <code>SERVER_TIMESTAMP</code>, the timestamps will be the server start timestamps. Similarly, when
     *         <a>DASHFragmentSelector</a> is <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start
     *         timestamps.
     * @see DASHDisplayFragmentTimestamp
     */
    public String displayFragmentTimestampAsString() {
        return displayFragmentTimestamp;
    }

    /**
     * <p>
     * Fragments are identified in the manifest file based on their sequence number in the session. If
     * DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is added to each S
     * element in the manifest file with the attribute name “kvs:fn”. These fragment numbers can be used for logging or
     * for use with other APIs (e.g. <code>GetMedia</code> and <code>GetMediaForFragmentList</code>). A custom MPEG-DASH
     * media player is necessary to leverage these this custom attribute.
     * </p>
     * <p>
     * The default value is <code>NEVER</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #displayFragmentNumber} will return {@link DASHDisplayFragmentNumber#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #displayFragmentNumberAsString}.
     * </p>
     * 
     * @return Fragments are identified in the manifest file based on their sequence number in the session. If
     *         DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is added
     *         to each S element in the manifest file with the attribute name “kvs:fn”. These fragment numbers can be
     *         used for logging or for use with other APIs (e.g. <code>GetMedia</code> and
     *         <code>GetMediaForFragmentList</code>). A custom MPEG-DASH media player is necessary to leverage these
     *         this custom attribute.</p>
     *         <p>
     *         The default value is <code>NEVER</code>.
     * @see DASHDisplayFragmentNumber
     */
    public DASHDisplayFragmentNumber displayFragmentNumber() {
        return DASHDisplayFragmentNumber.fromValue(displayFragmentNumber);
    }

    /**
     * <p>
     * Fragments are identified in the manifest file based on their sequence number in the session. If
     * DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is added to each S
     * element in the manifest file with the attribute name “kvs:fn”. These fragment numbers can be used for logging or
     * for use with other APIs (e.g. <code>GetMedia</code> and <code>GetMediaForFragmentList</code>). A custom MPEG-DASH
     * media player is necessary to leverage these this custom attribute.
     * </p>
     * <p>
     * The default value is <code>NEVER</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #displayFragmentNumber} will return {@link DASHDisplayFragmentNumber#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #displayFragmentNumberAsString}.
     * </p>
     * 
     * @return Fragments are identified in the manifest file based on their sequence number in the session. If
     *         DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is added
     *         to each S element in the manifest file with the attribute name “kvs:fn”. These fragment numbers can be
     *         used for logging or for use with other APIs (e.g. <code>GetMedia</code> and
     *         <code>GetMediaForFragmentList</code>). A custom MPEG-DASH media player is necessary to leverage these
     *         this custom attribute.</p>
     *         <p>
     *         The default value is <code>NEVER</code>.
     * @see DASHDisplayFragmentNumber
     */
    public String displayFragmentNumberAsString() {
        return displayFragmentNumber;
    }

    /**
     * <p>
     * The time range of the requested fragment and the source of the timestamps.
     * </p>
     * <p>
     * This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>.
     * This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If <code>PlaybackMode</code> is
     * <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the <code>TimestampRange</code> should
     * not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both
     * <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be set.
     * </p>
     * 
     * @return The time range of the requested fragment and the source of the timestamps.</p>
     *         <p>
     *         This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or
     *         <code>LIVE_REPLAY</code>. This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If
     *         <code>PlaybackMode</code> is <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the
     *         <code>TimestampRange</code> should not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or
     *         <code>LIVE_REPLAY</code>, both <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be
     *         set.
     */
    public DASHFragmentSelector dashFragmentSelector() {
        return dashFragmentSelector;
    }

    /**
     * <p>
     * The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and 43200 (12
     * hours).
     * </p>
     * <p>
     * When a session expires, no new calls to <code>GetDashManifest</code>, <code>GetMP4InitFragment</code>, or
     * <code>GetMP4MediaFragment</code> can be made for that session.
     * </p>
     * <p>
     * The default is 300 (5 minutes).
     * </p>
     * 
     * @return The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and
     *         43200 (12 hours).</p>
     *         <p>
     *         When a session expires, no new calls to <code>GetDashManifest</code>, <code>GetMP4InitFragment</code>, or
     *         <code>GetMP4MediaFragment</code> can be made for that session.
     *         </p>
     *         <p>
     *         The default is 300 (5 minutes).
     */
    public Integer expires() {
        return expires;
    }

    /**
     * <p>
     * The maximum number of fragments that are returned in the MPEG-DASH manifest.
     * </p>
     * <p>
     * When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to this value.
     * When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are returned, up to this
     * maximum number.
     * </p>
     * <p>
     * When there are a higher number of fragments available in a live MPEG-DASH manifest, video players often buffer
     * content before starting playback. Increasing the buffer size increases the playback latency, but it decreases the
     * likelihood that rebuffering will occur during playback. We recommend that a live MPEG-DASH manifest have a
     * minimum of 3 fragments and a maximum of 10 fragments.
     * </p>
     * <p>
     * The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or <code>LIVE_REPLAY</code>, and
     * 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
     * </p>
     * <p>
     * The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with 1-second
     * fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
     * </p>
     * 
     * @return The maximum number of fragments that are returned in the MPEG-DASH manifest.</p>
     *         <p>
     *         When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to
     *         this value. When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are
     *         returned, up to this maximum number.
     *         </p>
     *         <p>
     *         When there are a higher number of fragments available in a live MPEG-DASH manifest, video players often
     *         buffer content before starting playback. Increasing the buffer size increases the playback latency, but
     *         it decreases the likelihood that rebuffering will occur during playback. We recommend that a live
     *         MPEG-DASH manifest have a minimum of 3 fragments and a maximum of 10 fragments.
     *         </p>
     *         <p>
     *         The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or <code>LIVE_REPLAY</code>,
     *         and 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
     *         </p>
     *         <p>
     *         The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with
     *         1-second fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
     */
    public Long maxManifestFragmentResults() {
        return maxManifestFragmentResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(streamARN());
        hashCode = 31 * hashCode + Objects.hashCode(playbackModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(displayFragmentTimestampAsString());
        hashCode = 31 * hashCode + Objects.hashCode(displayFragmentNumberAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dashFragmentSelector());
        hashCode = 31 * hashCode + Objects.hashCode(expires());
        hashCode = 31 * hashCode + Objects.hashCode(maxManifestFragmentResults());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetDashStreamingSessionUrlRequest)) {
            return false;
        }
        GetDashStreamingSessionUrlRequest other = (GetDashStreamingSessionUrlRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(streamARN(), other.streamARN())
                && Objects.equals(playbackModeAsString(), other.playbackModeAsString())
                && Objects.equals(displayFragmentTimestampAsString(), other.displayFragmentTimestampAsString())
                && Objects.equals(displayFragmentNumberAsString(), other.displayFragmentNumberAsString())
                && Objects.equals(dashFragmentSelector(), other.dashFragmentSelector())
                && Objects.equals(expires(), other.expires())
                && Objects.equals(maxManifestFragmentResults(), other.maxManifestFragmentResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetDashStreamingSessionUrlRequest").add("StreamName", streamName())
                .add("StreamARN", streamARN()).add("PlaybackMode", playbackModeAsString())
                .add("DisplayFragmentTimestamp", displayFragmentTimestampAsString())
                .add("DisplayFragmentNumber", displayFragmentNumberAsString())
                .add("DASHFragmentSelector", dashFragmentSelector()).add("Expires", expires())
                .add("MaxManifestFragmentResults", maxManifestFragmentResults()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "StreamARN":
            return Optional.ofNullable(clazz.cast(streamARN()));
        case "PlaybackMode":
            return Optional.ofNullable(clazz.cast(playbackModeAsString()));
        case "DisplayFragmentTimestamp":
            return Optional.ofNullable(clazz.cast(displayFragmentTimestampAsString()));
        case "DisplayFragmentNumber":
            return Optional.ofNullable(clazz.cast(displayFragmentNumberAsString()));
        case "DASHFragmentSelector":
            return Optional.ofNullable(clazz.cast(dashFragmentSelector()));
        case "Expires":
            return Optional.ofNullable(clazz.cast(expires()));
        case "MaxManifestFragmentResults":
            return Optional.ofNullable(clazz.cast(maxManifestFragmentResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetDashStreamingSessionUrlRequest, T> g) {
        return obj -> g.apply((GetDashStreamingSessionUrlRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisVideoArchivedMediaRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetDashStreamingSessionUrlRequest> {
        /**
         * <p>
         * The name of the stream for which to retrieve the MPEG-DASH manifest URL.
         * </p>
         * <p>
         * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * </p>
         * 
         * @param streamName
         *        The name of the stream for which to retrieve the MPEG-DASH manifest URL.</p>
         *        <p>
         *        You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the stream for which to retrieve the MPEG-DASH manifest URL.
         * </p>
         * <p>
         * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * </p>
         * 
         * @param streamARN
         *        The Amazon Resource Name (ARN) of the stream for which to retrieve the MPEG-DASH manifest URL.</p>
         *        <p>
         *        You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamARN(String streamARN);

        /**
         * <p>
         * Whether to retrieve live, live replay, or archived, on-demand data.
         * </p>
         * <p>
         * Features of the three types of sessions include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated with the
         * latest fragments as they become available. We recommend that the media player retrieve a new manifest on a
         * one-second interval. When this type of session is played in a media player, the user interface typically
         * displays a "live" notification, with no scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * <note>
         * <p>
         * In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even if
         * there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
         * player to halt or cause a jump in playback. In this mode, fragments are not added to the MPEG-DASH manifest
         * if they are older than the newest fragment in the playlist. If the missing fragment becomes available after a
         * subsequent fragment is added to the manifest, the older fragment is not added, and the gap is not filled.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated similarly to
         * how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start
         * time. Instead of fragments being added as they are ingested, fragments are added as the duration of the next
         * fragment elapses. For example, if the fragments in the session are two seconds long, then a new fragment is
         * added to the manifest every two seconds. This mode is useful to be able to start playback from when an event
         * is detected and continue live streaming media that has not yet been ingested as of the time of the session
         * creation. This mode is also useful to stream previously archived media without being limited by the 1,000
         * fragment limit in the <code>ON_DEMAND</code> mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the fragments
         * for the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The
         * manifest must be retrieved only once for each session. When this type of session is played in a media player,
         * the user interface typically displays a scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there
         * are multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that
         * is, the newer fragment) is included in the MPEG-DASH manifest. The other fragments are not included.
         * Fragments that have different timestamps but have overlapping durations are still included in the MPEG-DASH
         * manifest. This can lead to unexpected behavior in the media player.
         * </p>
         * <p>
         * The default is <code>LIVE</code>.
         * </p>
         * 
         * @param playbackMode
         *        Whether to retrieve live, live replay, or archived, on-demand data.</p>
         *        <p>
         *        Features of the three types of sessions include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated
         *        with the latest fragments as they become available. We recommend that the media player retrieve a new
         *        manifest on a one-second interval. When this type of session is played in a media player, the user
         *        interface typically displays a "live" notification, with no scrubber control for choosing the position
         *        in the playback window to display.
         *        </p>
         *        <note>
         *        <p>
         *        In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even
         *        if there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a
         *        media player to halt or cause a jump in playback. In this mode, fragments are not added to the
         *        MPEG-DASH manifest if they are older than the newest fragment in the playlist. If the missing fragment
         *        becomes available after a subsequent fragment is added to the manifest, the older fragment is not
         *        added, and the gap is not filled.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated
         *        similarly to how it is updated for <code>LIVE</code> mode except that it starts by including fragments
         *        from a given start time. Instead of fragments being added as they are ingested, fragments are added as
         *        the duration of the next fragment elapses. For example, if the fragments in the session are two
         *        seconds long, then a new fragment is added to the manifest every two seconds. This mode is useful to
         *        be able to start playback from when an event is detected and continue live streaming media that has
         *        not yet been ingested as of the time of the session creation. This mode is also useful to stream
         *        previously archived media without being limited by the 1,000 fragment limit in the
         *        <code>ON_DEMAND</code> mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the
         *        fragments for the session, up to the number that is specified in
         *        <code>MaxMediaPlaylistFragmentResults</code>. The manifest must be retrieved only once for each
         *        session. When this type of session is played in a media player, the user interface typically displays
         *        a scrubber control for choosing the position in the playback window to display.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
         *        there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
         *        number (that is, the newer fragment) is included in the MPEG-DASH manifest. The other fragments are
         *        not included. Fragments that have different timestamps but have overlapping durations are still
         *        included in the MPEG-DASH manifest. This can lead to unexpected behavior in the media player.
         *        </p>
         *        <p>
         *        The default is <code>LIVE</code>.
         * @see DASHPlaybackMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DASHPlaybackMode
         */
        Builder playbackMode(String playbackMode);

        /**
         * <p>
         * Whether to retrieve live, live replay, or archived, on-demand data.
         * </p>
         * <p>
         * Features of the three types of sessions include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated with the
         * latest fragments as they become available. We recommend that the media player retrieve a new manifest on a
         * one-second interval. When this type of session is played in a media player, the user interface typically
         * displays a "live" notification, with no scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * <note>
         * <p>
         * In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even if
         * there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
         * player to halt or cause a jump in playback. In this mode, fragments are not added to the MPEG-DASH manifest
         * if they are older than the newest fragment in the playlist. If the missing fragment becomes available after a
         * subsequent fragment is added to the manifest, the older fragment is not added, and the gap is not filled.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated similarly to
         * how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start
         * time. Instead of fragments being added as they are ingested, fragments are added as the duration of the next
         * fragment elapses. For example, if the fragments in the session are two seconds long, then a new fragment is
         * added to the manifest every two seconds. This mode is useful to be able to start playback from when an event
         * is detected and continue live streaming media that has not yet been ingested as of the time of the session
         * creation. This mode is also useful to stream previously archived media without being limited by the 1,000
         * fragment limit in the <code>ON_DEMAND</code> mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the fragments
         * for the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The
         * manifest must be retrieved only once for each session. When this type of session is played in a media player,
         * the user interface typically displays a scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there
         * are multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that
         * is, the newer fragment) is included in the MPEG-DASH manifest. The other fragments are not included.
         * Fragments that have different timestamps but have overlapping durations are still included in the MPEG-DASH
         * manifest. This can lead to unexpected behavior in the media player.
         * </p>
         * <p>
         * The default is <code>LIVE</code>.
         * </p>
         * 
         * @param playbackMode
         *        Whether to retrieve live, live replay, or archived, on-demand data.</p>
         *        <p>
         *        Features of the three types of sessions include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>LIVE</code> </b>: For sessions of this type, the MPEG-DASH manifest is continually updated
         *        with the latest fragments as they become available. We recommend that the media player retrieve a new
         *        manifest on a one-second interval. When this type of session is played in a media player, the user
         *        interface typically displays a "live" notification, with no scrubber control for choosing the position
         *        in the playback window to display.
         *        </p>
         *        <note>
         *        <p>
         *        In <code>LIVE</code> mode, the newest available fragments are included in an MPEG-DASH manifest, even
         *        if there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a
         *        media player to halt or cause a jump in playback. In this mode, fragments are not added to the
         *        MPEG-DASH manifest if they are older than the newest fragment in the playlist. If the missing fragment
         *        becomes available after a subsequent fragment is added to the manifest, the older fragment is not
         *        added, and the gap is not filled.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the MPEG-DASH manifest is updated
         *        similarly to how it is updated for <code>LIVE</code> mode except that it starts by including fragments
         *        from a given start time. Instead of fragments being added as they are ingested, fragments are added as
         *        the duration of the next fragment elapses. For example, if the fragments in the session are two
         *        seconds long, then a new fragment is added to the manifest every two seconds. This mode is useful to
         *        be able to start playback from when an event is detected and continue live streaming media that has
         *        not yet been ingested as of the time of the session creation. This mode is also useful to stream
         *        previously archived media without being limited by the 1,000 fragment limit in the
         *        <code>ON_DEMAND</code> mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the MPEG-DASH manifest contains all the
         *        fragments for the session, up to the number that is specified in
         *        <code>MaxMediaPlaylistFragmentResults</code>. The manifest must be retrieved only once for each
         *        session. When this type of session is played in a media player, the user interface typically displays
         *        a scrubber control for choosing the position in the playback window to display.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
         *        there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
         *        number (that is, the newer fragment) is included in the MPEG-DASH manifest. The other fragments are
         *        not included. Fragments that have different timestamps but have overlapping durations are still
         *        included in the MPEG-DASH manifest. This can lead to unexpected behavior in the media player.
         *        </p>
         *        <p>
         *        The default is <code>LIVE</code>.
         * @see DASHPlaybackMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DASHPlaybackMode
         */
        Builder playbackMode(DASHPlaybackMode playbackMode);

        /**
         * <p>
         * Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived using
         * attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not properly
         * handle gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the manifest file to
         * enable playback of media with discontinuities. Therefore, the wall-clock time derived from the manifest file
         * may be inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>, the accurate fragment timestamp
         * is added to each S element in the manifest file with the attribute name “kvs:ts”. A custom MPEG-DASH media
         * player is necessary to leverage this custom attribute.
         * </p>
         * <p>
         * The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>,
         * the timestamps will be the server start timestamps. Similarly, when <a>DASHFragmentSelector</a> is
         * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
         * </p>
         * 
         * @param displayFragmentTimestamp
         *        Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived
         *        using attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not
         *        properly handle gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the
         *        manifest file to enable playback of media with discontinuities. Therefore, the wall-clock time derived
         *        from the manifest file may be inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>,
         *        the accurate fragment timestamp is added to each S element in the manifest file with the attribute
         *        name “kvs:ts”. A custom MPEG-DASH media player is necessary to leverage this custom attribute.</p>
         *        <p>
         *        The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is
         *        <code>SERVER_TIMESTAMP</code>, the timestamps will be the server start timestamps. Similarly, when
         *        <a>DASHFragmentSelector</a> is <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer
         *        start timestamps.
         * @see DASHDisplayFragmentTimestamp
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DASHDisplayFragmentTimestamp
         */
        Builder displayFragmentTimestamp(String displayFragmentTimestamp);

        /**
         * <p>
         * Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived using
         * attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not properly
         * handle gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the manifest file to
         * enable playback of media with discontinuities. Therefore, the wall-clock time derived from the manifest file
         * may be inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>, the accurate fragment timestamp
         * is added to each S element in the manifest file with the attribute name “kvs:ts”. A custom MPEG-DASH media
         * player is necessary to leverage this custom attribute.
         * </p>
         * <p>
         * The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>,
         * the timestamps will be the server start timestamps. Similarly, when <a>DASHFragmentSelector</a> is
         * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
         * </p>
         * 
         * @param displayFragmentTimestamp
         *        Per the MPEG-DASH specification, the wall-clock time of fragments in the manifest file can be derived
         *        using attributes in the manifest itself. However, typically, MPEG-DASH compatible media players do not
         *        properly handle gaps in the media timeline. Kinesis Video Streams adjusts the media timeline in the
         *        manifest file to enable playback of media with discontinuities. Therefore, the wall-clock time derived
         *        from the manifest file may be inaccurate. If DisplayFragmentTimestamp is set to <code>ALWAYS</code>,
         *        the accurate fragment timestamp is added to each S element in the manifest file with the attribute
         *        name “kvs:ts”. A custom MPEG-DASH media player is necessary to leverage this custom attribute.</p>
         *        <p>
         *        The default value is <code>NEVER</code>. When <a>DASHFragmentSelector</a> is
         *        <code>SERVER_TIMESTAMP</code>, the timestamps will be the server start timestamps. Similarly, when
         *        <a>DASHFragmentSelector</a> is <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer
         *        start timestamps.
         * @see DASHDisplayFragmentTimestamp
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DASHDisplayFragmentTimestamp
         */
        Builder displayFragmentTimestamp(DASHDisplayFragmentTimestamp displayFragmentTimestamp);

        /**
         * <p>
         * Fragments are identified in the manifest file based on their sequence number in the session. If
         * DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is added to
         * each S element in the manifest file with the attribute name “kvs:fn”. These fragment numbers can be used for
         * logging or for use with other APIs (e.g. <code>GetMedia</code> and <code>GetMediaForFragmentList</code>). A
         * custom MPEG-DASH media player is necessary to leverage these this custom attribute.
         * </p>
         * <p>
         * The default value is <code>NEVER</code>.
         * </p>
         * 
         * @param displayFragmentNumber
         *        Fragments are identified in the manifest file based on their sequence number in the session. If
         *        DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is
         *        added to each S element in the manifest file with the attribute name “kvs:fn”. These fragment numbers
         *        can be used for logging or for use with other APIs (e.g. <code>GetMedia</code> and
         *        <code>GetMediaForFragmentList</code>). A custom MPEG-DASH media player is necessary to leverage these
         *        this custom attribute.</p>
         *        <p>
         *        The default value is <code>NEVER</code>.
         * @see DASHDisplayFragmentNumber
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DASHDisplayFragmentNumber
         */
        Builder displayFragmentNumber(String displayFragmentNumber);

        /**
         * <p>
         * Fragments are identified in the manifest file based on their sequence number in the session. If
         * DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is added to
         * each S element in the manifest file with the attribute name “kvs:fn”. These fragment numbers can be used for
         * logging or for use with other APIs (e.g. <code>GetMedia</code> and <code>GetMediaForFragmentList</code>). A
         * custom MPEG-DASH media player is necessary to leverage these this custom attribute.
         * </p>
         * <p>
         * The default value is <code>NEVER</code>.
         * </p>
         * 
         * @param displayFragmentNumber
         *        Fragments are identified in the manifest file based on their sequence number in the session. If
         *        DisplayFragmentNumber is set to <code>ALWAYS</code>, the Kinesis Video Streams fragment number is
         *        added to each S element in the manifest file with the attribute name “kvs:fn”. These fragment numbers
         *        can be used for logging or for use with other APIs (e.g. <code>GetMedia</code> and
         *        <code>GetMediaForFragmentList</code>). A custom MPEG-DASH media player is necessary to leverage these
         *        this custom attribute.</p>
         *        <p>
         *        The default value is <code>NEVER</code>.
         * @see DASHDisplayFragmentNumber
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DASHDisplayFragmentNumber
         */
        Builder displayFragmentNumber(DASHDisplayFragmentNumber displayFragmentNumber);

        /**
         * <p>
         * The time range of the requested fragment and the source of the timestamps.
         * </p>
         * <p>
         * This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>
         * . This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If <code>PlaybackMode</code> is
         * <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the <code>TimestampRange</code>
         * should not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both
         * <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be set.
         * </p>
         * 
         * @param dashFragmentSelector
         *        The time range of the requested fragment and the source of the timestamps.</p>
         *        <p>
         *        This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or
         *        <code>LIVE_REPLAY</code>. This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If
         *        <code>PlaybackMode</code> is <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but
         *        the <code>TimestampRange</code> should not be set. If <code>PlaybackMode</code> is
         *        <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both <code>FragmentSelectorType</code> and
         *        <code>TimestampRange</code> must be set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dashFragmentSelector(DASHFragmentSelector dashFragmentSelector);

        /**
         * <p>
         * The time range of the requested fragment and the source of the timestamps.
         * </p>
         * <p>
         * This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>
         * . This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If <code>PlaybackMode</code> is
         * <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the <code>TimestampRange</code>
         * should not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both
         * <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be set.
         * </p>
         * This is a convenience that creates an instance of the {@link DASHFragmentSelector.Builder} avoiding the need
         * to create one manually via {@link DASHFragmentSelector#builder()}.
         *
         * When the {@link Consumer} completes, {@link DASHFragmentSelector.Builder#build()} is called immediately and
         * its result is passed to {@link #dashFragmentSelector(DASHFragmentSelector)}.
         * 
         * @param dashFragmentSelector
         *        a consumer that will call methods on {@link DASHFragmentSelector.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dashFragmentSelector(DASHFragmentSelector)
         */
        default Builder dashFragmentSelector(Consumer<DASHFragmentSelector.Builder> dashFragmentSelector) {
            return dashFragmentSelector(DASHFragmentSelector.builder().applyMutation(dashFragmentSelector).build());
        }

        /**
         * <p>
         * The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and 43200
         * (12 hours).
         * </p>
         * <p>
         * When a session expires, no new calls to <code>GetDashManifest</code>, <code>GetMP4InitFragment</code>, or
         * <code>GetMP4MediaFragment</code> can be made for that session.
         * </p>
         * <p>
         * The default is 300 (5 minutes).
         * </p>
         * 
         * @param expires
         *        The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and
         *        43200 (12 hours).</p>
         *        <p>
         *        When a session expires, no new calls to <code>GetDashManifest</code>, <code>GetMP4InitFragment</code>,
         *        or <code>GetMP4MediaFragment</code> can be made for that session.
         *        </p>
         *        <p>
         *        The default is 300 (5 minutes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expires(Integer expires);

        /**
         * <p>
         * The maximum number of fragments that are returned in the MPEG-DASH manifest.
         * </p>
         * <p>
         * When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to this
         * value. When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are returned, up to
         * this maximum number.
         * </p>
         * <p>
         * When there are a higher number of fragments available in a live MPEG-DASH manifest, video players often
         * buffer content before starting playback. Increasing the buffer size increases the playback latency, but it
         * decreases the likelihood that rebuffering will occur during playback. We recommend that a live MPEG-DASH
         * manifest have a minimum of 3 fragments and a maximum of 10 fragments.
         * </p>
         * <p>
         * The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or <code>LIVE_REPLAY</code>, and
         * 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
         * </p>
         * <p>
         * The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with 1-second
         * fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
         * </p>
         * 
         * @param maxManifestFragmentResults
         *        The maximum number of fragments that are returned in the MPEG-DASH manifest.</p>
         *        <p>
         *        When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to
         *        this value. When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are
         *        returned, up to this maximum number.
         *        </p>
         *        <p>
         *        When there are a higher number of fragments available in a live MPEG-DASH manifest, video players
         *        often buffer content before starting playback. Increasing the buffer size increases the playback
         *        latency, but it decreases the likelihood that rebuffering will occur during playback. We recommend
         *        that a live MPEG-DASH manifest have a minimum of 3 fragments and a maximum of 10 fragments.
         *        </p>
         *        <p>
         *        The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or
         *        <code>LIVE_REPLAY</code>, and 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
         *        </p>
         *        <p>
         *        The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with
         *        1-second fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxManifestFragmentResults(Long maxManifestFragmentResults);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KinesisVideoArchivedMediaRequest.BuilderImpl implements Builder {
        private String streamName;

        private String streamARN;

        private String playbackMode;

        private String displayFragmentTimestamp;

        private String displayFragmentNumber;

        private DASHFragmentSelector dashFragmentSelector;

        private Integer expires;

        private Long maxManifestFragmentResults;

        private BuilderImpl() {
        }

        private BuilderImpl(GetDashStreamingSessionUrlRequest model) {
            super(model);
            streamName(model.streamName);
            streamARN(model.streamARN);
            playbackMode(model.playbackMode);
            displayFragmentTimestamp(model.displayFragmentTimestamp);
            displayFragmentNumber(model.displayFragmentNumber);
            dashFragmentSelector(model.dashFragmentSelector);
            expires(model.expires);
            maxManifestFragmentResults(model.maxManifestFragmentResults);
        }

        public final String getStreamName() {
            return streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        public final String getStreamARN() {
            return streamARN;
        }

        @Override
        public final Builder streamARN(String streamARN) {
            this.streamARN = streamARN;
            return this;
        }

        public final void setStreamARN(String streamARN) {
            this.streamARN = streamARN;
        }

        public final String getPlaybackMode() {
            return playbackMode;
        }

        @Override
        public final Builder playbackMode(String playbackMode) {
            this.playbackMode = playbackMode;
            return this;
        }

        @Override
        public final Builder playbackMode(DASHPlaybackMode playbackMode) {
            this.playbackMode(playbackMode == null ? null : playbackMode.toString());
            return this;
        }

        public final void setPlaybackMode(String playbackMode) {
            this.playbackMode = playbackMode;
        }

        public final String getDisplayFragmentTimestamp() {
            return displayFragmentTimestamp;
        }

        @Override
        public final Builder displayFragmentTimestamp(String displayFragmentTimestamp) {
            this.displayFragmentTimestamp = displayFragmentTimestamp;
            return this;
        }

        @Override
        public final Builder displayFragmentTimestamp(DASHDisplayFragmentTimestamp displayFragmentTimestamp) {
            this.displayFragmentTimestamp(displayFragmentTimestamp == null ? null : displayFragmentTimestamp.toString());
            return this;
        }

        public final void setDisplayFragmentTimestamp(String displayFragmentTimestamp) {
            this.displayFragmentTimestamp = displayFragmentTimestamp;
        }

        public final String getDisplayFragmentNumber() {
            return displayFragmentNumber;
        }

        @Override
        public final Builder displayFragmentNumber(String displayFragmentNumber) {
            this.displayFragmentNumber = displayFragmentNumber;
            return this;
        }

        @Override
        public final Builder displayFragmentNumber(DASHDisplayFragmentNumber displayFragmentNumber) {
            this.displayFragmentNumber(displayFragmentNumber == null ? null : displayFragmentNumber.toString());
            return this;
        }

        public final void setDisplayFragmentNumber(String displayFragmentNumber) {
            this.displayFragmentNumber = displayFragmentNumber;
        }

        public final DASHFragmentSelector.Builder getDashFragmentSelector() {
            return dashFragmentSelector != null ? dashFragmentSelector.toBuilder() : null;
        }

        @Override
        public final Builder dashFragmentSelector(DASHFragmentSelector dashFragmentSelector) {
            this.dashFragmentSelector = dashFragmentSelector;
            return this;
        }

        public final void setDashFragmentSelector(DASHFragmentSelector.BuilderImpl dashFragmentSelector) {
            this.dashFragmentSelector = dashFragmentSelector != null ? dashFragmentSelector.build() : null;
        }

        public final Integer getExpires() {
            return expires;
        }

        @Override
        public final Builder expires(Integer expires) {
            this.expires = expires;
            return this;
        }

        public final void setExpires(Integer expires) {
            this.expires = expires;
        }

        public final Long getMaxManifestFragmentResults() {
            return maxManifestFragmentResults;
        }

        @Override
        public final Builder maxManifestFragmentResults(Long maxManifestFragmentResults) {
            this.maxManifestFragmentResults = maxManifestFragmentResults;
            return this;
        }

        public final void setMaxManifestFragmentResults(Long maxManifestFragmentResults) {
            this.maxManifestFragmentResults = maxManifestFragmentResults;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetDashStreamingSessionUrlRequest build() {
            return new GetDashStreamingSessionUrlRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
