/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetHlsStreamingSessionUrlRequest extends KinesisVideoArchivedMediaRequest implements
        ToCopyableBuilder<GetHlsStreamingSessionUrlRequest.Builder, GetHlsStreamingSessionUrlRequest> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(GetHlsStreamingSessionUrlRequest::streamName))
            .setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamARN").getter(getter(GetHlsStreamingSessionUrlRequest::streamARN))
            .setter(setter(Builder::streamARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamARN").build()).build();

    private static final SdkField<String> PLAYBACK_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlaybackMode").getter(getter(GetHlsStreamingSessionUrlRequest::playbackModeAsString))
            .setter(setter(Builder::playbackMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlaybackMode").build()).build();

    private static final SdkField<HLSFragmentSelector> HLS_FRAGMENT_SELECTOR_FIELD = SdkField
            .<HLSFragmentSelector> builder(MarshallingType.SDK_POJO).memberName("HLSFragmentSelector")
            .getter(getter(GetHlsStreamingSessionUrlRequest::hlsFragmentSelector)).setter(setter(Builder::hlsFragmentSelector))
            .constructor(HLSFragmentSelector::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HLSFragmentSelector").build())
            .build();

    private static final SdkField<String> CONTAINER_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerFormat").getter(getter(GetHlsStreamingSessionUrlRequest::containerFormatAsString))
            .setter(setter(Builder::containerFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerFormat").build()).build();

    private static final SdkField<String> DISCONTINUITY_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DiscontinuityMode").getter(getter(GetHlsStreamingSessionUrlRequest::discontinuityModeAsString))
            .setter(setter(Builder::discontinuityMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DiscontinuityMode").build()).build();

    private static final SdkField<String> DISPLAY_FRAGMENT_TIMESTAMP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayFragmentTimestamp")
            .getter(getter(GetHlsStreamingSessionUrlRequest::displayFragmentTimestampAsString))
            .setter(setter(Builder::displayFragmentTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayFragmentTimestamp").build())
            .build();

    private static final SdkField<Integer> EXPIRES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Expires").getter(getter(GetHlsStreamingSessionUrlRequest::expires)).setter(setter(Builder::expires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Expires").build()).build();

    private static final SdkField<Long> MAX_MEDIA_PLAYLIST_FRAGMENT_RESULTS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MaxMediaPlaylistFragmentResults")
            .getter(getter(GetHlsStreamingSessionUrlRequest::maxMediaPlaylistFragmentResults))
            .setter(setter(Builder::maxMediaPlaylistFragmentResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxMediaPlaylistFragmentResults")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD,
            STREAM_ARN_FIELD, PLAYBACK_MODE_FIELD, HLS_FRAGMENT_SELECTOR_FIELD, CONTAINER_FORMAT_FIELD, DISCONTINUITY_MODE_FIELD,
            DISPLAY_FRAGMENT_TIMESTAMP_FIELD, EXPIRES_FIELD, MAX_MEDIA_PLAYLIST_FRAGMENT_RESULTS_FIELD));

    private final String streamName;

    private final String streamARN;

    private final String playbackMode;

    private final HLSFragmentSelector hlsFragmentSelector;

    private final String containerFormat;

    private final String discontinuityMode;

    private final String displayFragmentTimestamp;

    private final Integer expires;

    private final Long maxMediaPlaylistFragmentResults;

    private GetHlsStreamingSessionUrlRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.streamARN = builder.streamARN;
        this.playbackMode = builder.playbackMode;
        this.hlsFragmentSelector = builder.hlsFragmentSelector;
        this.containerFormat = builder.containerFormat;
        this.discontinuityMode = builder.discontinuityMode;
        this.displayFragmentTimestamp = builder.displayFragmentTimestamp;
        this.expires = builder.expires;
        this.maxMediaPlaylistFragmentResults = builder.maxMediaPlaylistFragmentResults;
    }

    /**
     * <p>
     * The name of the stream for which to retrieve the HLS master playlist URL.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * 
     * @return The name of the stream for which to retrieve the HLS master playlist URL.</p>
     *         <p>
     *         You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     */
    public String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the stream for which to retrieve the HLS master playlist URL.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the stream for which to retrieve the HLS master playlist URL.</p>
     *         <p>
     *         You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     */
    public String streamARN() {
        return streamARN;
    }

    /**
     * <p>
     * Whether to retrieve live, live replay, or archived, on-demand data.
     * </p>
     * <p>
     * Features of the three types of sessions include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated with the
     * latest fragments as they become available. We recommend that the media player retrieve a new playlist on a
     * one-second interval. When this type of session is played in a media player, the user interface typically displays
     * a "live" notification, with no scrubber control for choosing the position in the playback window to display.
     * </p>
     * <note>
     * <p>
     * In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even if there is
     * a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media player to halt
     * or cause a jump in playback. In this mode, fragments are not added to the HLS media playlist if they are older
     * than the newest fragment in the playlist. If the missing fragment becomes available after a subsequent fragment
     * is added to the playlist, the older fragment is not added, and the gap is not filled.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated similarly to how
     * it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start time.
     * Instead of fragments being added as they are ingested, fragments are added as the duration of the next fragment
     * elapses. For example, if the fragments in the session are two seconds long, then a new fragment is added to the
     * media playlist every two seconds. This mode is useful to be able to start playback from when an event is detected
     * and continue live streaming media that has not yet been ingested as of the time of the session creation. This
     * mode is also useful to stream previously archived media without being limited by the 1,000 fragment limit in the
     * <code>ON_DEMAND</code> mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the fragments for
     * the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The playlist
     * must be retrieved only once for each session. When this type of session is played in a media player, the user
     * interface typically displays a scrubber control for choosing the position in the playback window to display.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there are
     * multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that is, the
     * newer fragment) is included in the HLS media playlist. The other fragments are not included. Fragments that have
     * different timestamps but have overlapping durations are still included in the HLS media playlist. This can lead
     * to unexpected behavior in the media player.
     * </p>
     * <p>
     * The default is <code>LIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #playbackMode} will
     * return {@link HLSPlaybackMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #playbackModeAsString}.
     * </p>
     * 
     * @return Whether to retrieve live, live replay, or archived, on-demand data.</p>
     *         <p>
     *         Features of the three types of sessions include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated with
     *         the latest fragments as they become available. We recommend that the media player retrieve a new playlist
     *         on a one-second interval. When this type of session is played in a media player, the user interface
     *         typically displays a "live" notification, with no scrubber control for choosing the position in the
     *         playback window to display.
     *         </p>
     *         <note>
     *         <p>
     *         In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even if
     *         there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
     *         player to halt or cause a jump in playback. In this mode, fragments are not added to the HLS media
     *         playlist if they are older than the newest fragment in the playlist. If the missing fragment becomes
     *         available after a subsequent fragment is added to the playlist, the older fragment is not added, and the
     *         gap is not filled.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated similarly
     *         to how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given
     *         start time. Instead of fragments being added as they are ingested, fragments are added as the duration of
     *         the next fragment elapses. For example, if the fragments in the session are two seconds long, then a new
     *         fragment is added to the media playlist every two seconds. This mode is useful to be able to start
     *         playback from when an event is detected and continue live streaming media that has not yet been ingested
     *         as of the time of the session creation. This mode is also useful to stream previously archived media
     *         without being limited by the 1,000 fragment limit in the <code>ON_DEMAND</code> mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the
     *         fragments for the session, up to the number that is specified in
     *         <code>MaxMediaPlaylistFragmentResults</code>. The playlist must be retrieved only once for each session.
     *         When this type of session is played in a media player, the user interface typically displays a scrubber
     *         control for choosing the position in the playback window to display.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
     *         there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
     *         number (that is, the newer fragment) is included in the HLS media playlist. The other fragments are not
     *         included. Fragments that have different timestamps but have overlapping durations are still included in
     *         the HLS media playlist. This can lead to unexpected behavior in the media player.
     *         </p>
     *         <p>
     *         The default is <code>LIVE</code>.
     * @see HLSPlaybackMode
     */
    public HLSPlaybackMode playbackMode() {
        return HLSPlaybackMode.fromValue(playbackMode);
    }

    /**
     * <p>
     * Whether to retrieve live, live replay, or archived, on-demand data.
     * </p>
     * <p>
     * Features of the three types of sessions include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated with the
     * latest fragments as they become available. We recommend that the media player retrieve a new playlist on a
     * one-second interval. When this type of session is played in a media player, the user interface typically displays
     * a "live" notification, with no scrubber control for choosing the position in the playback window to display.
     * </p>
     * <note>
     * <p>
     * In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even if there is
     * a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media player to halt
     * or cause a jump in playback. In this mode, fragments are not added to the HLS media playlist if they are older
     * than the newest fragment in the playlist. If the missing fragment becomes available after a subsequent fragment
     * is added to the playlist, the older fragment is not added, and the gap is not filled.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated similarly to how
     * it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start time.
     * Instead of fragments being added as they are ingested, fragments are added as the duration of the next fragment
     * elapses. For example, if the fragments in the session are two seconds long, then a new fragment is added to the
     * media playlist every two seconds. This mode is useful to be able to start playback from when an event is detected
     * and continue live streaming media that has not yet been ingested as of the time of the session creation. This
     * mode is also useful to stream previously archived media without being limited by the 1,000 fragment limit in the
     * <code>ON_DEMAND</code> mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the fragments for
     * the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The playlist
     * must be retrieved only once for each session. When this type of session is played in a media player, the user
     * interface typically displays a scrubber control for choosing the position in the playback window to display.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there are
     * multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that is, the
     * newer fragment) is included in the HLS media playlist. The other fragments are not included. Fragments that have
     * different timestamps but have overlapping durations are still included in the HLS media playlist. This can lead
     * to unexpected behavior in the media player.
     * </p>
     * <p>
     * The default is <code>LIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #playbackMode} will
     * return {@link HLSPlaybackMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #playbackModeAsString}.
     * </p>
     * 
     * @return Whether to retrieve live, live replay, or archived, on-demand data.</p>
     *         <p>
     *         Features of the three types of sessions include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated with
     *         the latest fragments as they become available. We recommend that the media player retrieve a new playlist
     *         on a one-second interval. When this type of session is played in a media player, the user interface
     *         typically displays a "live" notification, with no scrubber control for choosing the position in the
     *         playback window to display.
     *         </p>
     *         <note>
     *         <p>
     *         In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even if
     *         there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
     *         player to halt or cause a jump in playback. In this mode, fragments are not added to the HLS media
     *         playlist if they are older than the newest fragment in the playlist. If the missing fragment becomes
     *         available after a subsequent fragment is added to the playlist, the older fragment is not added, and the
     *         gap is not filled.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated similarly
     *         to how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given
     *         start time. Instead of fragments being added as they are ingested, fragments are added as the duration of
     *         the next fragment elapses. For example, if the fragments in the session are two seconds long, then a new
     *         fragment is added to the media playlist every two seconds. This mode is useful to be able to start
     *         playback from when an event is detected and continue live streaming media that has not yet been ingested
     *         as of the time of the session creation. This mode is also useful to stream previously archived media
     *         without being limited by the 1,000 fragment limit in the <code>ON_DEMAND</code> mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the
     *         fragments for the session, up to the number that is specified in
     *         <code>MaxMediaPlaylistFragmentResults</code>. The playlist must be retrieved only once for each session.
     *         When this type of session is played in a media player, the user interface typically displays a scrubber
     *         control for choosing the position in the playback window to display.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
     *         there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
     *         number (that is, the newer fragment) is included in the HLS media playlist. The other fragments are not
     *         included. Fragments that have different timestamps but have overlapping durations are still included in
     *         the HLS media playlist. This can lead to unexpected behavior in the media player.
     *         </p>
     *         <p>
     *         The default is <code>LIVE</code>.
     * @see HLSPlaybackMode
     */
    public String playbackModeAsString() {
        return playbackMode;
    }

    /**
     * <p>
     * The time range of the requested fragment and the source of the timestamps.
     * </p>
     * <p>
     * This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>.
     * This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If <code>PlaybackMode</code> is
     * <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the <code>TimestampRange</code> should
     * not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both
     * <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be set.
     * </p>
     * 
     * @return The time range of the requested fragment and the source of the timestamps.</p>
     *         <p>
     *         This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or
     *         <code>LIVE_REPLAY</code>. This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If
     *         <code>PlaybackMode</code> is <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the
     *         <code>TimestampRange</code> should not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or
     *         <code>LIVE_REPLAY</code>, both <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be
     *         set.
     */
    public HLSFragmentSelector hlsFragmentSelector() {
        return hlsFragmentSelector;
    }

    /**
     * <p>
     * Specifies which format should be used for packaging the media. Specifying the <code>FRAGMENTED_MP4</code>
     * container format packages the media into MP4 fragments (fMP4 or CMAF). This is the recommended packaging because
     * there is minimal packaging overhead. The other container format option is <code>MPEG_TS</code>. HLS has supported
     * MPEG TS chunks since it was released and is sometimes the only supported packaging on older HLS players. MPEG TS
     * typically has a 5-25 percent packaging overhead. This means MPEG TS typically requires 5-25 percent more
     * bandwidth and cost than fMP4.
     * </p>
     * <p>
     * The default is <code>FRAGMENTED_MP4</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #containerFormat}
     * will return {@link ContainerFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #containerFormatAsString}.
     * </p>
     * 
     * @return Specifies which format should be used for packaging the media. Specifying the <code>FRAGMENTED_MP4</code>
     *         container format packages the media into MP4 fragments (fMP4 or CMAF). This is the recommended packaging
     *         because there is minimal packaging overhead. The other container format option is <code>MPEG_TS</code>.
     *         HLS has supported MPEG TS chunks since it was released and is sometimes the only supported packaging on
     *         older HLS players. MPEG TS typically has a 5-25 percent packaging overhead. This means MPEG TS typically
     *         requires 5-25 percent more bandwidth and cost than fMP4.</p>
     *         <p>
     *         The default is <code>FRAGMENTED_MP4</code>.
     * @see ContainerFormat
     */
    public ContainerFormat containerFormat() {
        return ContainerFormat.fromValue(containerFormat);
    }

    /**
     * <p>
     * Specifies which format should be used for packaging the media. Specifying the <code>FRAGMENTED_MP4</code>
     * container format packages the media into MP4 fragments (fMP4 or CMAF). This is the recommended packaging because
     * there is minimal packaging overhead. The other container format option is <code>MPEG_TS</code>. HLS has supported
     * MPEG TS chunks since it was released and is sometimes the only supported packaging on older HLS players. MPEG TS
     * typically has a 5-25 percent packaging overhead. This means MPEG TS typically requires 5-25 percent more
     * bandwidth and cost than fMP4.
     * </p>
     * <p>
     * The default is <code>FRAGMENTED_MP4</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #containerFormat}
     * will return {@link ContainerFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #containerFormatAsString}.
     * </p>
     * 
     * @return Specifies which format should be used for packaging the media. Specifying the <code>FRAGMENTED_MP4</code>
     *         container format packages the media into MP4 fragments (fMP4 or CMAF). This is the recommended packaging
     *         because there is minimal packaging overhead. The other container format option is <code>MPEG_TS</code>.
     *         HLS has supported MPEG TS chunks since it was released and is sometimes the only supported packaging on
     *         older HLS players. MPEG TS typically has a 5-25 percent packaging overhead. This means MPEG TS typically
     *         requires 5-25 percent more bandwidth and cost than fMP4.</p>
     *         <p>
     *         The default is <code>FRAGMENTED_MP4</code>.
     * @see ContainerFormat
     */
    public String containerFormatAsString() {
        return containerFormat;
    }

    /**
     * <p>
     * Specifies when flags marking discontinuities between fragments are added to the media playlists.
     * </p>
     * <p>
     * Media players typically build a timeline of media content to play, based on the timestamps of each fragment. This
     * means that if there is any overlap or gap between fragments (as is typical if <a>HLSFragmentSelector</a> is set
     * to <code>SERVER_TIMESTAMP</code>), the media player timeline will also have small gaps between fragments in some
     * places, and will overwrite frames in other places. Gaps in the media player timeline can cause playback to stall
     * and overlaps can cause playback to be jittery. When there are discontinuity flags between fragments, the media
     * player is expected to reset the timeline, resulting in the next fragment being played immediately after the
     * previous fragment.
     * </p>
     * <p>
     * The following modes are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media playlist. It is
     * recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not accurate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
     * <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer timestamps.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or overlap of
     * more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
     * <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a significant issue
     * with the media timeline (e.g. a missing fragment).
     * </p>
     * </li>
     * </ul>
     * <p>
     * The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>, and
     * <code>NEVER</code> when it is set to <code>PRODUCER_TIMESTAMP</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #discontinuityMode}
     * will return {@link HLSDiscontinuityMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #discontinuityModeAsString}.
     * </p>
     * 
     * @return Specifies when flags marking discontinuities between fragments are added to the media playlists.</p>
     *         <p>
     *         Media players typically build a timeline of media content to play, based on the timestamps of each
     *         fragment. This means that if there is any overlap or gap between fragments (as is typical if
     *         <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>), the media player timeline will also
     *         have small gaps between fragments in some places, and will overwrite frames in other places. Gaps in the
     *         media player timeline can cause playback to stall and overlaps can cause playback to be jittery. When
     *         there are discontinuity flags between fragments, the media player is expected to reset the timeline,
     *         resulting in the next fragment being played immediately after the previous fragment.
     *         </p>
     *         <p>
     *         The following modes are supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media playlist.
     *         It is recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not accurate.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
     *         <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer timestamps.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or
     *         overlap of more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
     *         <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a significant
     *         issue with the media timeline (e.g. a missing fragment).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to
     *         <code>SERVER_TIMESTAMP</code>, and <code>NEVER</code> when it is set to <code>PRODUCER_TIMESTAMP</code>.
     * @see HLSDiscontinuityMode
     */
    public HLSDiscontinuityMode discontinuityMode() {
        return HLSDiscontinuityMode.fromValue(discontinuityMode);
    }

    /**
     * <p>
     * Specifies when flags marking discontinuities between fragments are added to the media playlists.
     * </p>
     * <p>
     * Media players typically build a timeline of media content to play, based on the timestamps of each fragment. This
     * means that if there is any overlap or gap between fragments (as is typical if <a>HLSFragmentSelector</a> is set
     * to <code>SERVER_TIMESTAMP</code>), the media player timeline will also have small gaps between fragments in some
     * places, and will overwrite frames in other places. Gaps in the media player timeline can cause playback to stall
     * and overlaps can cause playback to be jittery. When there are discontinuity flags between fragments, the media
     * player is expected to reset the timeline, resulting in the next fragment being played immediately after the
     * previous fragment.
     * </p>
     * <p>
     * The following modes are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media playlist. It is
     * recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not accurate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
     * <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer timestamps.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or overlap of
     * more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
     * <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a significant issue
     * with the media timeline (e.g. a missing fragment).
     * </p>
     * </li>
     * </ul>
     * <p>
     * The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>, and
     * <code>NEVER</code> when it is set to <code>PRODUCER_TIMESTAMP</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #discontinuityMode}
     * will return {@link HLSDiscontinuityMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #discontinuityModeAsString}.
     * </p>
     * 
     * @return Specifies when flags marking discontinuities between fragments are added to the media playlists.</p>
     *         <p>
     *         Media players typically build a timeline of media content to play, based on the timestamps of each
     *         fragment. This means that if there is any overlap or gap between fragments (as is typical if
     *         <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>), the media player timeline will also
     *         have small gaps between fragments in some places, and will overwrite frames in other places. Gaps in the
     *         media player timeline can cause playback to stall and overlaps can cause playback to be jittery. When
     *         there are discontinuity flags between fragments, the media player is expected to reset the timeline,
     *         resulting in the next fragment being played immediately after the previous fragment.
     *         </p>
     *         <p>
     *         The following modes are supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media playlist.
     *         It is recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not accurate.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
     *         <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer timestamps.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or
     *         overlap of more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
     *         <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a significant
     *         issue with the media timeline (e.g. a missing fragment).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to
     *         <code>SERVER_TIMESTAMP</code>, and <code>NEVER</code> when it is set to <code>PRODUCER_TIMESTAMP</code>.
     * @see HLSDiscontinuityMode
     */
    public String discontinuityModeAsString() {
        return discontinuityMode;
    }

    /**
     * <p>
     * Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically, media
     * players report the playhead position as a time relative to the start of the first fragment in the playback
     * session. However, when the start timestamps are included in the HLS media playlist, some media players might
     * report the current playhead as an absolute time based on the fragment timestamps. This can be useful for creating
     * a playback experience that shows viewers the wall-clock time of the media.
     * </p>
     * <p>
     * The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
     * timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
     * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #displayFragmentTimestamp} will return {@link HLSDisplayFragmentTimestamp#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #displayFragmentTimestampAsString}.
     * </p>
     * 
     * @return Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically,
     *         media players report the playhead position as a time relative to the start of the first fragment in the
     *         playback session. However, when the start timestamps are included in the HLS media playlist, some media
     *         players might report the current playhead as an absolute time based on the fragment timestamps. This can
     *         be useful for creating a playback experience that shows viewers the wall-clock time of the media.</p>
     *         <p>
     *         The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
     *         timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
     *         <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
     * @see HLSDisplayFragmentTimestamp
     */
    public HLSDisplayFragmentTimestamp displayFragmentTimestamp() {
        return HLSDisplayFragmentTimestamp.fromValue(displayFragmentTimestamp);
    }

    /**
     * <p>
     * Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically, media
     * players report the playhead position as a time relative to the start of the first fragment in the playback
     * session. However, when the start timestamps are included in the HLS media playlist, some media players might
     * report the current playhead as an absolute time based on the fragment timestamps. This can be useful for creating
     * a playback experience that shows viewers the wall-clock time of the media.
     * </p>
     * <p>
     * The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
     * timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
     * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #displayFragmentTimestamp} will return {@link HLSDisplayFragmentTimestamp#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #displayFragmentTimestampAsString}.
     * </p>
     * 
     * @return Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically,
     *         media players report the playhead position as a time relative to the start of the first fragment in the
     *         playback session. However, when the start timestamps are included in the HLS media playlist, some media
     *         players might report the current playhead as an absolute time based on the fragment timestamps. This can
     *         be useful for creating a playback experience that shows viewers the wall-clock time of the media.</p>
     *         <p>
     *         The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
     *         timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
     *         <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
     * @see HLSDisplayFragmentTimestamp
     */
    public String displayFragmentTimestampAsString() {
        return displayFragmentTimestamp;
    }

    /**
     * <p>
     * The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and 43200 (12
     * hours).
     * </p>
     * <p>
     * When a session expires, no new calls to <code>GetHLSMasterPlaylist</code>, <code>GetHLSMediaPlaylist</code>,
     * <code>GetMP4InitFragment</code>, <code>GetMP4MediaFragment</code>, or <code>GetTSFragment</code> can be made for
     * that session.
     * </p>
     * <p>
     * The default is 300 (5 minutes).
     * </p>
     * 
     * @return The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and
     *         43200 (12 hours).</p>
     *         <p>
     *         When a session expires, no new calls to <code>GetHLSMasterPlaylist</code>,
     *         <code>GetHLSMediaPlaylist</code>, <code>GetMP4InitFragment</code>, <code>GetMP4MediaFragment</code>, or
     *         <code>GetTSFragment</code> can be made for that session.
     *         </p>
     *         <p>
     *         The default is 300 (5 minutes).
     */
    public Integer expires() {
        return expires;
    }

    /**
     * <p>
     * The maximum number of fragments that are returned in the HLS media playlists.
     * </p>
     * <p>
     * When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to this value.
     * When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are returned, up to this
     * maximum number.
     * </p>
     * <p>
     * When there are a higher number of fragments available in a live HLS media playlist, video players often buffer
     * content before starting playback. Increasing the buffer size increases the playback latency, but it decreases the
     * likelihood that rebuffering will occur during playback. We recommend that a live HLS media playlist have a
     * minimum of 3 fragments and a maximum of 10 fragments.
     * </p>
     * <p>
     * The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or <code>LIVE_REPLAY</code>, and
     * 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
     * </p>
     * <p>
     * The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with 1-second
     * fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
     * </p>
     * 
     * @return The maximum number of fragments that are returned in the HLS media playlists.</p>
     *         <p>
     *         When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to
     *         this value. When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are
     *         returned, up to this maximum number.
     *         </p>
     *         <p>
     *         When there are a higher number of fragments available in a live HLS media playlist, video players often
     *         buffer content before starting playback. Increasing the buffer size increases the playback latency, but
     *         it decreases the likelihood that rebuffering will occur during playback. We recommend that a live HLS
     *         media playlist have a minimum of 3 fragments and a maximum of 10 fragments.
     *         </p>
     *         <p>
     *         The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or <code>LIVE_REPLAY</code>,
     *         and 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
     *         </p>
     *         <p>
     *         The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with
     *         1-second fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
     */
    public Long maxMediaPlaylistFragmentResults() {
        return maxMediaPlaylistFragmentResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(streamARN());
        hashCode = 31 * hashCode + Objects.hashCode(playbackModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hlsFragmentSelector());
        hashCode = 31 * hashCode + Objects.hashCode(containerFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(discontinuityModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(displayFragmentTimestampAsString());
        hashCode = 31 * hashCode + Objects.hashCode(expires());
        hashCode = 31 * hashCode + Objects.hashCode(maxMediaPlaylistFragmentResults());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetHlsStreamingSessionUrlRequest)) {
            return false;
        }
        GetHlsStreamingSessionUrlRequest other = (GetHlsStreamingSessionUrlRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(streamARN(), other.streamARN())
                && Objects.equals(playbackModeAsString(), other.playbackModeAsString())
                && Objects.equals(hlsFragmentSelector(), other.hlsFragmentSelector())
                && Objects.equals(containerFormatAsString(), other.containerFormatAsString())
                && Objects.equals(discontinuityModeAsString(), other.discontinuityModeAsString())
                && Objects.equals(displayFragmentTimestampAsString(), other.displayFragmentTimestampAsString())
                && Objects.equals(expires(), other.expires())
                && Objects.equals(maxMediaPlaylistFragmentResults(), other.maxMediaPlaylistFragmentResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetHlsStreamingSessionUrlRequest").add("StreamName", streamName()).add("StreamARN", streamARN())
                .add("PlaybackMode", playbackModeAsString()).add("HLSFragmentSelector", hlsFragmentSelector())
                .add("ContainerFormat", containerFormatAsString()).add("DiscontinuityMode", discontinuityModeAsString())
                .add("DisplayFragmentTimestamp", displayFragmentTimestampAsString()).add("Expires", expires())
                .add("MaxMediaPlaylistFragmentResults", maxMediaPlaylistFragmentResults()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "StreamARN":
            return Optional.ofNullable(clazz.cast(streamARN()));
        case "PlaybackMode":
            return Optional.ofNullable(clazz.cast(playbackModeAsString()));
        case "HLSFragmentSelector":
            return Optional.ofNullable(clazz.cast(hlsFragmentSelector()));
        case "ContainerFormat":
            return Optional.ofNullable(clazz.cast(containerFormatAsString()));
        case "DiscontinuityMode":
            return Optional.ofNullable(clazz.cast(discontinuityModeAsString()));
        case "DisplayFragmentTimestamp":
            return Optional.ofNullable(clazz.cast(displayFragmentTimestampAsString()));
        case "Expires":
            return Optional.ofNullable(clazz.cast(expires()));
        case "MaxMediaPlaylistFragmentResults":
            return Optional.ofNullable(clazz.cast(maxMediaPlaylistFragmentResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetHlsStreamingSessionUrlRequest, T> g) {
        return obj -> g.apply((GetHlsStreamingSessionUrlRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisVideoArchivedMediaRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetHlsStreamingSessionUrlRequest> {
        /**
         * <p>
         * The name of the stream for which to retrieve the HLS master playlist URL.
         * </p>
         * <p>
         * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * </p>
         * 
         * @param streamName
         *        The name of the stream for which to retrieve the HLS master playlist URL.</p>
         *        <p>
         *        You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the stream for which to retrieve the HLS master playlist URL.
         * </p>
         * <p>
         * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * </p>
         * 
         * @param streamARN
         *        The Amazon Resource Name (ARN) of the stream for which to retrieve the HLS master playlist URL.</p>
         *        <p>
         *        You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamARN(String streamARN);

        /**
         * <p>
         * Whether to retrieve live, live replay, or archived, on-demand data.
         * </p>
         * <p>
         * Features of the three types of sessions include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated with the
         * latest fragments as they become available. We recommend that the media player retrieve a new playlist on a
         * one-second interval. When this type of session is played in a media player, the user interface typically
         * displays a "live" notification, with no scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * <note>
         * <p>
         * In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even if
         * there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
         * player to halt or cause a jump in playback. In this mode, fragments are not added to the HLS media playlist
         * if they are older than the newest fragment in the playlist. If the missing fragment becomes available after a
         * subsequent fragment is added to the playlist, the older fragment is not added, and the gap is not filled.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated similarly to
         * how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start
         * time. Instead of fragments being added as they are ingested, fragments are added as the duration of the next
         * fragment elapses. For example, if the fragments in the session are two seconds long, then a new fragment is
         * added to the media playlist every two seconds. This mode is useful to be able to start playback from when an
         * event is detected and continue live streaming media that has not yet been ingested as of the time of the
         * session creation. This mode is also useful to stream previously archived media without being limited by the
         * 1,000 fragment limit in the <code>ON_DEMAND</code> mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the fragments
         * for the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The
         * playlist must be retrieved only once for each session. When this type of session is played in a media player,
         * the user interface typically displays a scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there
         * are multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that
         * is, the newer fragment) is included in the HLS media playlist. The other fragments are not included.
         * Fragments that have different timestamps but have overlapping durations are still included in the HLS media
         * playlist. This can lead to unexpected behavior in the media player.
         * </p>
         * <p>
         * The default is <code>LIVE</code>.
         * </p>
         * 
         * @param playbackMode
         *        Whether to retrieve live, live replay, or archived, on-demand data.</p>
         *        <p>
         *        Features of the three types of sessions include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated
         *        with the latest fragments as they become available. We recommend that the media player retrieve a new
         *        playlist on a one-second interval. When this type of session is played in a media player, the user
         *        interface typically displays a "live" notification, with no scrubber control for choosing the position
         *        in the playback window to display.
         *        </p>
         *        <note>
         *        <p>
         *        In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even
         *        if there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a
         *        media player to halt or cause a jump in playback. In this mode, fragments are not added to the HLS
         *        media playlist if they are older than the newest fragment in the playlist. If the missing fragment
         *        becomes available after a subsequent fragment is added to the playlist, the older fragment is not
         *        added, and the gap is not filled.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated
         *        similarly to how it is updated for <code>LIVE</code> mode except that it starts by including fragments
         *        from a given start time. Instead of fragments being added as they are ingested, fragments are added as
         *        the duration of the next fragment elapses. For example, if the fragments in the session are two
         *        seconds long, then a new fragment is added to the media playlist every two seconds. This mode is
         *        useful to be able to start playback from when an event is detected and continue live streaming media
         *        that has not yet been ingested as of the time of the session creation. This mode is also useful to
         *        stream previously archived media without being limited by the 1,000 fragment limit in the
         *        <code>ON_DEMAND</code> mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the
         *        fragments for the session, up to the number that is specified in
         *        <code>MaxMediaPlaylistFragmentResults</code>. The playlist must be retrieved only once for each
         *        session. When this type of session is played in a media player, the user interface typically displays
         *        a scrubber control for choosing the position in the playback window to display.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
         *        there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
         *        number (that is, the newer fragment) is included in the HLS media playlist. The other fragments are
         *        not included. Fragments that have different timestamps but have overlapping durations are still
         *        included in the HLS media playlist. This can lead to unexpected behavior in the media player.
         *        </p>
         *        <p>
         *        The default is <code>LIVE</code>.
         * @see HLSPlaybackMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HLSPlaybackMode
         */
        Builder playbackMode(String playbackMode);

        /**
         * <p>
         * Whether to retrieve live, live replay, or archived, on-demand data.
         * </p>
         * <p>
         * Features of the three types of sessions include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated with the
         * latest fragments as they become available. We recommend that the media player retrieve a new playlist on a
         * one-second interval. When this type of session is played in a media player, the user interface typically
         * displays a "live" notification, with no scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * <note>
         * <p>
         * In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even if
         * there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a media
         * player to halt or cause a jump in playback. In this mode, fragments are not added to the HLS media playlist
         * if they are older than the newest fragment in the playlist. If the missing fragment becomes available after a
         * subsequent fragment is added to the playlist, the older fragment is not added, and the gap is not filled.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated similarly to
         * how it is updated for <code>LIVE</code> mode except that it starts by including fragments from a given start
         * time. Instead of fragments being added as they are ingested, fragments are added as the duration of the next
         * fragment elapses. For example, if the fragments in the session are two seconds long, then a new fragment is
         * added to the media playlist every two seconds. This mode is useful to be able to start playback from when an
         * event is detected and continue live streaming media that has not yet been ingested as of the time of the
         * session creation. This mode is also useful to stream previously archived media without being limited by the
         * 1,000 fragment limit in the <code>ON_DEMAND</code> mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the fragments
         * for the session, up to the number that is specified in <code>MaxMediaPlaylistFragmentResults</code>. The
         * playlist must be retrieved only once for each session. When this type of session is played in a media player,
         * the user interface typically displays a scrubber control for choosing the position in the playback window to
         * display.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if there
         * are multiple fragments with the same start timestamp, the fragment that has the larger fragment number (that
         * is, the newer fragment) is included in the HLS media playlist. The other fragments are not included.
         * Fragments that have different timestamps but have overlapping durations are still included in the HLS media
         * playlist. This can lead to unexpected behavior in the media player.
         * </p>
         * <p>
         * The default is <code>LIVE</code>.
         * </p>
         * 
         * @param playbackMode
         *        Whether to retrieve live, live replay, or archived, on-demand data.</p>
         *        <p>
         *        Features of the three types of sessions include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>LIVE</code> </b>: For sessions of this type, the HLS media playlist is continually updated
         *        with the latest fragments as they become available. We recommend that the media player retrieve a new
         *        playlist on a one-second interval. When this type of session is played in a media player, the user
         *        interface typically displays a "live" notification, with no scrubber control for choosing the position
         *        in the playback window to display.
         *        </p>
         *        <note>
         *        <p>
         *        In <code>LIVE</code> mode, the newest available fragments are included in an HLS media playlist, even
         *        if there is a gap between fragments (that is, if a fragment is missing). A gap like this might cause a
         *        media player to halt or cause a jump in playback. In this mode, fragments are not added to the HLS
         *        media playlist if they are older than the newest fragment in the playlist. If the missing fragment
         *        becomes available after a subsequent fragment is added to the playlist, the older fragment is not
         *        added, and the gap is not filled.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>LIVE_REPLAY</code> </b>: For sessions of this type, the HLS media playlist is updated
         *        similarly to how it is updated for <code>LIVE</code> mode except that it starts by including fragments
         *        from a given start time. Instead of fragments being added as they are ingested, fragments are added as
         *        the duration of the next fragment elapses. For example, if the fragments in the session are two
         *        seconds long, then a new fragment is added to the media playlist every two seconds. This mode is
         *        useful to be able to start playback from when an event is detected and continue live streaming media
         *        that has not yet been ingested as of the time of the session creation. This mode is also useful to
         *        stream previously archived media without being limited by the 1,000 fragment limit in the
         *        <code>ON_DEMAND</code> mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>ON_DEMAND</code> </b>: For sessions of this type, the HLS media playlist contains all the
         *        fragments for the session, up to the number that is specified in
         *        <code>MaxMediaPlaylistFragmentResults</code>. The playlist must be retrieved only once for each
         *        session. When this type of session is played in a media player, the user interface typically displays
         *        a scrubber control for choosing the position in the playback window to display.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In all playback modes, if <code>FragmentSelectorType</code> is <code>PRODUCER_TIMESTAMP</code>, and if
         *        there are multiple fragments with the same start timestamp, the fragment that has the larger fragment
         *        number (that is, the newer fragment) is included in the HLS media playlist. The other fragments are
         *        not included. Fragments that have different timestamps but have overlapping durations are still
         *        included in the HLS media playlist. This can lead to unexpected behavior in the media player.
         *        </p>
         *        <p>
         *        The default is <code>LIVE</code>.
         * @see HLSPlaybackMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HLSPlaybackMode
         */
        Builder playbackMode(HLSPlaybackMode playbackMode);

        /**
         * <p>
         * The time range of the requested fragment and the source of the timestamps.
         * </p>
         * <p>
         * This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>
         * . This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If <code>PlaybackMode</code> is
         * <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the <code>TimestampRange</code>
         * should not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both
         * <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be set.
         * </p>
         * 
         * @param hlsFragmentSelector
         *        The time range of the requested fragment and the source of the timestamps.</p>
         *        <p>
         *        This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or
         *        <code>LIVE_REPLAY</code>. This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If
         *        <code>PlaybackMode</code> is <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but
         *        the <code>TimestampRange</code> should not be set. If <code>PlaybackMode</code> is
         *        <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both <code>FragmentSelectorType</code> and
         *        <code>TimestampRange</code> must be set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hlsFragmentSelector(HLSFragmentSelector hlsFragmentSelector);

        /**
         * <p>
         * The time range of the requested fragment and the source of the timestamps.
         * </p>
         * <p>
         * This parameter is required if <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>
         * . This parameter is optional if PlaybackMode is<code/> <code>LIVE</code>. If <code>PlaybackMode</code> is
         * <code>LIVE</code>, the <code>FragmentSelectorType</code> can be set, but the <code>TimestampRange</code>
         * should not be set. If <code>PlaybackMode</code> is <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>, both
         * <code>FragmentSelectorType</code> and <code>TimestampRange</code> must be set.
         * </p>
         * This is a convenience that creates an instance of the {@link HLSFragmentSelector.Builder} avoiding the need
         * to create one manually via {@link HLSFragmentSelector#builder()}.
         *
         * When the {@link Consumer} completes, {@link HLSFragmentSelector.Builder#build()} is called immediately and
         * its result is passed to {@link #hlsFragmentSelector(HLSFragmentSelector)}.
         * 
         * @param hlsFragmentSelector
         *        a consumer that will call methods on {@link HLSFragmentSelector.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hlsFragmentSelector(HLSFragmentSelector)
         */
        default Builder hlsFragmentSelector(Consumer<HLSFragmentSelector.Builder> hlsFragmentSelector) {
            return hlsFragmentSelector(HLSFragmentSelector.builder().applyMutation(hlsFragmentSelector).build());
        }

        /**
         * <p>
         * Specifies which format should be used for packaging the media. Specifying the <code>FRAGMENTED_MP4</code>
         * container format packages the media into MP4 fragments (fMP4 or CMAF). This is the recommended packaging
         * because there is minimal packaging overhead. The other container format option is <code>MPEG_TS</code>. HLS
         * has supported MPEG TS chunks since it was released and is sometimes the only supported packaging on older HLS
         * players. MPEG TS typically has a 5-25 percent packaging overhead. This means MPEG TS typically requires 5-25
         * percent more bandwidth and cost than fMP4.
         * </p>
         * <p>
         * The default is <code>FRAGMENTED_MP4</code>.
         * </p>
         * 
         * @param containerFormat
         *        Specifies which format should be used for packaging the media. Specifying the
         *        <code>FRAGMENTED_MP4</code> container format packages the media into MP4 fragments (fMP4 or CMAF).
         *        This is the recommended packaging because there is minimal packaging overhead. The other container
         *        format option is <code>MPEG_TS</code>. HLS has supported MPEG TS chunks since it was released and is
         *        sometimes the only supported packaging on older HLS players. MPEG TS typically has a 5-25 percent
         *        packaging overhead. This means MPEG TS typically requires 5-25 percent more bandwidth and cost than
         *        fMP4.</p>
         *        <p>
         *        The default is <code>FRAGMENTED_MP4</code>.
         * @see ContainerFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerFormat
         */
        Builder containerFormat(String containerFormat);

        /**
         * <p>
         * Specifies which format should be used for packaging the media. Specifying the <code>FRAGMENTED_MP4</code>
         * container format packages the media into MP4 fragments (fMP4 or CMAF). This is the recommended packaging
         * because there is minimal packaging overhead. The other container format option is <code>MPEG_TS</code>. HLS
         * has supported MPEG TS chunks since it was released and is sometimes the only supported packaging on older HLS
         * players. MPEG TS typically has a 5-25 percent packaging overhead. This means MPEG TS typically requires 5-25
         * percent more bandwidth and cost than fMP4.
         * </p>
         * <p>
         * The default is <code>FRAGMENTED_MP4</code>.
         * </p>
         * 
         * @param containerFormat
         *        Specifies which format should be used for packaging the media. Specifying the
         *        <code>FRAGMENTED_MP4</code> container format packages the media into MP4 fragments (fMP4 or CMAF).
         *        This is the recommended packaging because there is minimal packaging overhead. The other container
         *        format option is <code>MPEG_TS</code>. HLS has supported MPEG TS chunks since it was released and is
         *        sometimes the only supported packaging on older HLS players. MPEG TS typically has a 5-25 percent
         *        packaging overhead. This means MPEG TS typically requires 5-25 percent more bandwidth and cost than
         *        fMP4.</p>
         *        <p>
         *        The default is <code>FRAGMENTED_MP4</code>.
         * @see ContainerFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerFormat
         */
        Builder containerFormat(ContainerFormat containerFormat);

        /**
         * <p>
         * Specifies when flags marking discontinuities between fragments are added to the media playlists.
         * </p>
         * <p>
         * Media players typically build a timeline of media content to play, based on the timestamps of each fragment.
         * This means that if there is any overlap or gap between fragments (as is typical if <a>HLSFragmentSelector</a>
         * is set to <code>SERVER_TIMESTAMP</code>), the media player timeline will also have small gaps between
         * fragments in some places, and will overwrite frames in other places. Gaps in the media player timeline can
         * cause playback to stall and overlaps can cause playback to be jittery. When there are discontinuity flags
         * between fragments, the media player is expected to reset the timeline, resulting in the next fragment being
         * played immediately after the previous fragment.
         * </p>
         * <p>
         * The following modes are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media playlist. It is
         * recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not accurate.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
         * <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer timestamps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or overlap
         * of more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
         * <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a significant
         * issue with the media timeline (e.g. a missing fragment).
         * </p>
         * </li>
         * </ul>
         * <p>
         * The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>,
         * and <code>NEVER</code> when it is set to <code>PRODUCER_TIMESTAMP</code>.
         * </p>
         * 
         * @param discontinuityMode
         *        Specifies when flags marking discontinuities between fragments are added to the media playlists.</p>
         *        <p>
         *        Media players typically build a timeline of media content to play, based on the timestamps of each
         *        fragment. This means that if there is any overlap or gap between fragments (as is typical if
         *        <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>), the media player timeline will
         *        also have small gaps between fragments in some places, and will overwrite frames in other places. Gaps
         *        in the media player timeline can cause playback to stall and overlaps can cause playback to be
         *        jittery. When there are discontinuity flags between fragments, the media player is expected to reset
         *        the timeline, resulting in the next fragment being played immediately after the previous fragment.
         *        </p>
         *        <p>
         *        The following modes are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media
         *        playlist. It is recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not
         *        accurate.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
         *        <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer
         *        timestamps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or
         *        overlap of more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
         *        <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a
         *        significant issue with the media timeline (e.g. a missing fragment).
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to
         *        <code>SERVER_TIMESTAMP</code>, and <code>NEVER</code> when it is set to
         *        <code>PRODUCER_TIMESTAMP</code>.
         * @see HLSDiscontinuityMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HLSDiscontinuityMode
         */
        Builder discontinuityMode(String discontinuityMode);

        /**
         * <p>
         * Specifies when flags marking discontinuities between fragments are added to the media playlists.
         * </p>
         * <p>
         * Media players typically build a timeline of media content to play, based on the timestamps of each fragment.
         * This means that if there is any overlap or gap between fragments (as is typical if <a>HLSFragmentSelector</a>
         * is set to <code>SERVER_TIMESTAMP</code>), the media player timeline will also have small gaps between
         * fragments in some places, and will overwrite frames in other places. Gaps in the media player timeline can
         * cause playback to stall and overlaps can cause playback to be jittery. When there are discontinuity flags
         * between fragments, the media player is expected to reset the timeline, resulting in the next fragment being
         * played immediately after the previous fragment.
         * </p>
         * <p>
         * The following modes are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media playlist. It is
         * recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not accurate.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
         * <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer timestamps.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or overlap
         * of more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
         * <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a significant
         * issue with the media timeline (e.g. a missing fragment).
         * </p>
         * </li>
         * </ul>
         * <p>
         * The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>,
         * and <code>NEVER</code> when it is set to <code>PRODUCER_TIMESTAMP</code>.
         * </p>
         * 
         * @param discontinuityMode
         *        Specifies when flags marking discontinuities between fragments are added to the media playlists.</p>
         *        <p>
         *        Media players typically build a timeline of media content to play, based on the timestamps of each
         *        fragment. This means that if there is any overlap or gap between fragments (as is typical if
         *        <a>HLSFragmentSelector</a> is set to <code>SERVER_TIMESTAMP</code>), the media player timeline will
         *        also have small gaps between fragments in some places, and will overwrite frames in other places. Gaps
         *        in the media player timeline can cause playback to stall and overlaps can cause playback to be
         *        jittery. When there are discontinuity flags between fragments, the media player is expected to reset
         *        the timeline, resulting in the next fragment being played immediately after the previous fragment.
         *        </p>
         *        <p>
         *        The following modes are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALWAYS</code>: a discontinuity marker is placed between every fragment in the HLS media
         *        playlist. It is recommended to use a value of <code>ALWAYS</code> if the fragment timestamps are not
         *        accurate.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEVER</code>: no discontinuity markers are placed anywhere. It is recommended to use a value of
         *        <code>NEVER</code> to ensure the media player timeline most accurately maps to the producer
         *        timestamps.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ON_DISCONTIUNITY</code>: a discontinuity marker is placed between fragments that have a gap or
         *        overlap of more than 50 milliseconds. For most playback scenarios, it is recommended to use a value of
         *        <code>ON_DISCONTINUITY</code> so that the media player timeline is only reset when there is a
         *        significant issue with the media timeline (e.g. a missing fragment).
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The default is <code>ALWAYS</code> when <a>HLSFragmentSelector</a> is set to
         *        <code>SERVER_TIMESTAMP</code>, and <code>NEVER</code> when it is set to
         *        <code>PRODUCER_TIMESTAMP</code>.
         * @see HLSDiscontinuityMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HLSDiscontinuityMode
         */
        Builder discontinuityMode(HLSDiscontinuityMode discontinuityMode);

        /**
         * <p>
         * Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically, media
         * players report the playhead position as a time relative to the start of the first fragment in the playback
         * session. However, when the start timestamps are included in the HLS media playlist, some media players might
         * report the current playhead as an absolute time based on the fragment timestamps. This can be useful for
         * creating a playback experience that shows viewers the wall-clock time of the media.
         * </p>
         * <p>
         * The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
         * timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
         * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
         * </p>
         * 
         * @param displayFragmentTimestamp
         *        Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically,
         *        media players report the playhead position as a time relative to the start of the first fragment in
         *        the playback session. However, when the start timestamps are included in the HLS media playlist, some
         *        media players might report the current playhead as an absolute time based on the fragment timestamps.
         *        This can be useful for creating a playback experience that shows viewers the wall-clock time of the
         *        media.</p>
         *        <p>
         *        The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>,
         *        the timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
         *        <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
         * @see HLSDisplayFragmentTimestamp
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HLSDisplayFragmentTimestamp
         */
        Builder displayFragmentTimestamp(String displayFragmentTimestamp);

        /**
         * <p>
         * Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically, media
         * players report the playhead position as a time relative to the start of the first fragment in the playback
         * session. However, when the start timestamps are included in the HLS media playlist, some media players might
         * report the current playhead as an absolute time based on the fragment timestamps. This can be useful for
         * creating a playback experience that shows viewers the wall-clock time of the media.
         * </p>
         * <p>
         * The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>, the
         * timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
         * <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
         * </p>
         * 
         * @param displayFragmentTimestamp
         *        Specifies when the fragment start timestamps should be included in the HLS media playlist. Typically,
         *        media players report the playhead position as a time relative to the start of the first fragment in
         *        the playback session. However, when the start timestamps are included in the HLS media playlist, some
         *        media players might report the current playhead as an absolute time based on the fragment timestamps.
         *        This can be useful for creating a playback experience that shows viewers the wall-clock time of the
         *        media.</p>
         *        <p>
         *        The default is <code>NEVER</code>. When <a>HLSFragmentSelector</a> is <code>SERVER_TIMESTAMP</code>,
         *        the timestamps will be the server start timestamps. Similarly, when <a>HLSFragmentSelector</a> is
         *        <code>PRODUCER_TIMESTAMP</code>, the timestamps will be the producer start timestamps.
         * @see HLSDisplayFragmentTimestamp
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HLSDisplayFragmentTimestamp
         */
        Builder displayFragmentTimestamp(HLSDisplayFragmentTimestamp displayFragmentTimestamp);

        /**
         * <p>
         * The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and 43200
         * (12 hours).
         * </p>
         * <p>
         * When a session expires, no new calls to <code>GetHLSMasterPlaylist</code>, <code>GetHLSMediaPlaylist</code>,
         * <code>GetMP4InitFragment</code>, <code>GetMP4MediaFragment</code>, or <code>GetTSFragment</code> can be made
         * for that session.
         * </p>
         * <p>
         * The default is 300 (5 minutes).
         * </p>
         * 
         * @param expires
         *        The time in seconds until the requested session expires. This value can be between 300 (5 minutes) and
         *        43200 (12 hours).</p>
         *        <p>
         *        When a session expires, no new calls to <code>GetHLSMasterPlaylist</code>,
         *        <code>GetHLSMediaPlaylist</code>, <code>GetMP4InitFragment</code>, <code>GetMP4MediaFragment</code>,
         *        or <code>GetTSFragment</code> can be made for that session.
         *        </p>
         *        <p>
         *        The default is 300 (5 minutes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expires(Integer expires);

        /**
         * <p>
         * The maximum number of fragments that are returned in the HLS media playlists.
         * </p>
         * <p>
         * When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to this
         * value. When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are returned, up to
         * this maximum number.
         * </p>
         * <p>
         * When there are a higher number of fragments available in a live HLS media playlist, video players often
         * buffer content before starting playback. Increasing the buffer size increases the playback latency, but it
         * decreases the likelihood that rebuffering will occur during playback. We recommend that a live HLS media
         * playlist have a minimum of 3 fragments and a maximum of 10 fragments.
         * </p>
         * <p>
         * The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or <code>LIVE_REPLAY</code>, and
         * 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
         * </p>
         * <p>
         * The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with 1-second
         * fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
         * </p>
         * 
         * @param maxMediaPlaylistFragmentResults
         *        The maximum number of fragments that are returned in the HLS media playlists.</p>
         *        <p>
         *        When the <code>PlaybackMode</code> is <code>LIVE</code>, the most recent fragments are returned up to
         *        this value. When the <code>PlaybackMode</code> is <code>ON_DEMAND</code>, the oldest fragments are
         *        returned, up to this maximum number.
         *        </p>
         *        <p>
         *        When there are a higher number of fragments available in a live HLS media playlist, video players
         *        often buffer content before starting playback. Increasing the buffer size increases the playback
         *        latency, but it decreases the likelihood that rebuffering will occur during playback. We recommend
         *        that a live HLS media playlist have a minimum of 3 fragments and a maximum of 10 fragments.
         *        </p>
         *        <p>
         *        The default is 5 fragments if <code>PlaybackMode</code> is <code>LIVE</code> or
         *        <code>LIVE_REPLAY</code>, and 1,000 if <code>PlaybackMode</code> is <code>ON_DEMAND</code>.
         *        </p>
         *        <p>
         *        The maximum value of 1,000 fragments corresponds to more than 16 minutes of video on streams with
         *        1-second fragments, and more than 2 1/2 hours of video on streams with 10-second fragments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxMediaPlaylistFragmentResults(Long maxMediaPlaylistFragmentResults);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KinesisVideoArchivedMediaRequest.BuilderImpl implements Builder {
        private String streamName;

        private String streamARN;

        private String playbackMode;

        private HLSFragmentSelector hlsFragmentSelector;

        private String containerFormat;

        private String discontinuityMode;

        private String displayFragmentTimestamp;

        private Integer expires;

        private Long maxMediaPlaylistFragmentResults;

        private BuilderImpl() {
        }

        private BuilderImpl(GetHlsStreamingSessionUrlRequest model) {
            super(model);
            streamName(model.streamName);
            streamARN(model.streamARN);
            playbackMode(model.playbackMode);
            hlsFragmentSelector(model.hlsFragmentSelector);
            containerFormat(model.containerFormat);
            discontinuityMode(model.discontinuityMode);
            displayFragmentTimestamp(model.displayFragmentTimestamp);
            expires(model.expires);
            maxMediaPlaylistFragmentResults(model.maxMediaPlaylistFragmentResults);
        }

        public final String getStreamName() {
            return streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        public final String getStreamARN() {
            return streamARN;
        }

        @Override
        public final Builder streamARN(String streamARN) {
            this.streamARN = streamARN;
            return this;
        }

        public final void setStreamARN(String streamARN) {
            this.streamARN = streamARN;
        }

        public final String getPlaybackMode() {
            return playbackMode;
        }

        @Override
        public final Builder playbackMode(String playbackMode) {
            this.playbackMode = playbackMode;
            return this;
        }

        @Override
        public final Builder playbackMode(HLSPlaybackMode playbackMode) {
            this.playbackMode(playbackMode == null ? null : playbackMode.toString());
            return this;
        }

        public final void setPlaybackMode(String playbackMode) {
            this.playbackMode = playbackMode;
        }

        public final HLSFragmentSelector.Builder getHlsFragmentSelector() {
            return hlsFragmentSelector != null ? hlsFragmentSelector.toBuilder() : null;
        }

        @Override
        public final Builder hlsFragmentSelector(HLSFragmentSelector hlsFragmentSelector) {
            this.hlsFragmentSelector = hlsFragmentSelector;
            return this;
        }

        public final void setHlsFragmentSelector(HLSFragmentSelector.BuilderImpl hlsFragmentSelector) {
            this.hlsFragmentSelector = hlsFragmentSelector != null ? hlsFragmentSelector.build() : null;
        }

        public final String getContainerFormat() {
            return containerFormat;
        }

        @Override
        public final Builder containerFormat(String containerFormat) {
            this.containerFormat = containerFormat;
            return this;
        }

        @Override
        public final Builder containerFormat(ContainerFormat containerFormat) {
            this.containerFormat(containerFormat == null ? null : containerFormat.toString());
            return this;
        }

        public final void setContainerFormat(String containerFormat) {
            this.containerFormat = containerFormat;
        }

        public final String getDiscontinuityMode() {
            return discontinuityMode;
        }

        @Override
        public final Builder discontinuityMode(String discontinuityMode) {
            this.discontinuityMode = discontinuityMode;
            return this;
        }

        @Override
        public final Builder discontinuityMode(HLSDiscontinuityMode discontinuityMode) {
            this.discontinuityMode(discontinuityMode == null ? null : discontinuityMode.toString());
            return this;
        }

        public final void setDiscontinuityMode(String discontinuityMode) {
            this.discontinuityMode = discontinuityMode;
        }

        public final String getDisplayFragmentTimestamp() {
            return displayFragmentTimestamp;
        }

        @Override
        public final Builder displayFragmentTimestamp(String displayFragmentTimestamp) {
            this.displayFragmentTimestamp = displayFragmentTimestamp;
            return this;
        }

        @Override
        public final Builder displayFragmentTimestamp(HLSDisplayFragmentTimestamp displayFragmentTimestamp) {
            this.displayFragmentTimestamp(displayFragmentTimestamp == null ? null : displayFragmentTimestamp.toString());
            return this;
        }

        public final void setDisplayFragmentTimestamp(String displayFragmentTimestamp) {
            this.displayFragmentTimestamp = displayFragmentTimestamp;
        }

        public final Integer getExpires() {
            return expires;
        }

        @Override
        public final Builder expires(Integer expires) {
            this.expires = expires;
            return this;
        }

        public final void setExpires(Integer expires) {
            this.expires = expires;
        }

        public final Long getMaxMediaPlaylistFragmentResults() {
            return maxMediaPlaylistFragmentResults;
        }

        @Override
        public final Builder maxMediaPlaylistFragmentResults(Long maxMediaPlaylistFragmentResults) {
            this.maxMediaPlaylistFragmentResults = maxMediaPlaylistFragmentResults;
            return this;
        }

        public final void setMaxMediaPlaylistFragmentResults(Long maxMediaPlaylistFragmentResults) {
            this.maxMediaPlaylistFragmentResults = maxMediaPlaylistFragmentResults;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetHlsStreamingSessionUrlRequest build() {
            return new GetHlsStreamingSessionUrlRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
