/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetMediaForFragmentListRequest extends KinesisVideoArchivedMediaRequest implements
        ToCopyableBuilder<GetMediaForFragmentListRequest.Builder, GetMediaForFragmentListRequest> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(GetMediaForFragmentListRequest::streamName))
            .setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<List<String>> FRAGMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Fragments")
            .getter(getter(GetMediaForFragmentListRequest::fragments))
            .setter(setter(Builder::fragments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Fragments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD,
            FRAGMENTS_FIELD));

    private final String streamName;

    private final List<String> fragments;

    private GetMediaForFragmentListRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.fragments = builder.fragments;
    }

    /**
     * <p>
     * The name of the stream from which to retrieve fragment media.
     * </p>
     * 
     * @return The name of the stream from which to retrieve fragment media.
     */
    public String streamName() {
        return streamName;
    }

    /**
     * Returns true if the Fragments property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasFragments() {
        return fragments != null && !(fragments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the numbers of fragments for which to retrieve media. You retrieve these values with
     * <a>ListFragments</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFragments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the numbers of fragments for which to retrieve media. You retrieve these values with
     *         <a>ListFragments</a>.
     */
    public List<String> fragments() {
        return fragments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(hasFragments() ? fragments() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetMediaForFragmentListRequest)) {
            return false;
        }
        GetMediaForFragmentListRequest other = (GetMediaForFragmentListRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && hasFragments() == other.hasFragments()
                && Objects.equals(fragments(), other.fragments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetMediaForFragmentListRequest").add("StreamName", streamName())
                .add("Fragments", hasFragments() ? fragments() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "Fragments":
            return Optional.ofNullable(clazz.cast(fragments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetMediaForFragmentListRequest, T> g) {
        return obj -> g.apply((GetMediaForFragmentListRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisVideoArchivedMediaRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetMediaForFragmentListRequest> {
        /**
         * <p>
         * The name of the stream from which to retrieve fragment media.
         * </p>
         * 
         * @param streamName
         *        The name of the stream from which to retrieve fragment media.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * A list of the numbers of fragments for which to retrieve media. You retrieve these values with
         * <a>ListFragments</a>.
         * </p>
         * 
         * @param fragments
         *        A list of the numbers of fragments for which to retrieve media. You retrieve these values with
         *        <a>ListFragments</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fragments(Collection<String> fragments);

        /**
         * <p>
         * A list of the numbers of fragments for which to retrieve media. You retrieve these values with
         * <a>ListFragments</a>.
         * </p>
         * 
         * @param fragments
         *        A list of the numbers of fragments for which to retrieve media. You retrieve these values with
         *        <a>ListFragments</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fragments(String... fragments);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KinesisVideoArchivedMediaRequest.BuilderImpl implements Builder {
        private String streamName;

        private List<String> fragments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetMediaForFragmentListRequest model) {
            super(model);
            streamName(model.streamName);
            fragments(model.fragments);
        }

        public final String getStreamName() {
            return streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        public final Collection<String> getFragments() {
            if (fragments instanceof SdkAutoConstructList) {
                return null;
            }
            return fragments;
        }

        @Override
        public final Builder fragments(Collection<String> fragments) {
            this.fragments = FragmentNumberListCopier.copy(fragments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fragments(String... fragments) {
            fragments(Arrays.asList(fragments));
            return this;
        }

        public final void setFragments(Collection<String> fragments) {
            this.fragments = FragmentNumberListCopier.copy(fragments);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetMediaForFragmentListRequest build() {
            return new GetMediaForFragmentListRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
