/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The start and end of the timestamp range for the requested media.
 * </p>
 * <p>
 * This value should not be present if <code>PlaybackType</code> is <code>LIVE</code>.
 * </p>
 * <p>
 * The values in <code>DASHimestampRange</code> are inclusive. Fragments that start exactly at or after the start time
 * are included in the session. Fragments that start before the start time and continue past it are not included in the
 * session.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DASHTimestampRange implements SdkPojo, Serializable,
        ToCopyableBuilder<DASHTimestampRange.Builder, DASHTimestampRange> {
    private static final SdkField<Instant> START_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTimestamp").getter(getter(DASHTimestampRange::startTimestamp))
            .setter(setter(Builder::startTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTimestamp").build()).build();

    private static final SdkField<Instant> END_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTimestamp").getter(getter(DASHTimestampRange::endTimestamp)).setter(setter(Builder::endTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTimestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(START_TIMESTAMP_FIELD,
            END_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant startTimestamp;

    private final Instant endTimestamp;

    private DASHTimestampRange(BuilderImpl builder) {
        this.startTimestamp = builder.startTimestamp;
        this.endTimestamp = builder.endTimestamp;
    }

    /**
     * <p>
     * The start of the timestamp range for the requested media.
     * </p>
     * <p>
     * If the <code>DASHTimestampRange</code> value is specified, the <code>StartTimestamp</code> value is required.
     * </p>
     * <p>
     * Only fragments that start exactly at or after <code>StartTimestamp</code> are included in the session. Fragments
     * that start before <code>StartTimestamp</code> and continue past it aren't included in the session. If
     * <code>FragmentSelectorType</code> is <code>SERVER_TIMESTAMP</code>, the <code>StartTimestamp</code> must be later
     * than the stream head.
     * </p>
     * 
     * @return The start of the timestamp range for the requested media.</p>
     *         <p>
     *         If the <code>DASHTimestampRange</code> value is specified, the <code>StartTimestamp</code> value is
     *         required.
     *         </p>
     *         <p>
     *         Only fragments that start exactly at or after <code>StartTimestamp</code> are included in the session.
     *         Fragments that start before <code>StartTimestamp</code> and continue past it aren't included in the
     *         session. If <code>FragmentSelectorType</code> is <code>SERVER_TIMESTAMP</code>, the
     *         <code>StartTimestamp</code> must be later than the stream head.
     */
    public final Instant startTimestamp() {
        return startTimestamp;
    }

    /**
     * <p>
     * The end of the timestamp range for the requested media. This value must be within 24 hours of the specified
     * <code>StartTimestamp</code>, and it must be later than the <code>StartTimestamp</code> value.
     * </p>
     * <p>
     * If <code>FragmentSelectorType</code> for the request is <code>SERVER_TIMESTAMP</code>, this value must be in the
     * past.
     * </p>
     * <p>
     * The <code>EndTimestamp</code> value is required for <code>ON_DEMAND</code> mode, but optional for
     * <code>LIVE_REPLAY</code> mode. If the <code>EndTimestamp</code> is not set for <code>LIVE_REPLAY</code> mode then
     * the session will continue to include newly ingested fragments until the session expires.
     * </p>
     * <note>
     * <p>
     * This value is inclusive. The <code>EndTimestamp</code> is compared to the (starting) timestamp of the fragment.
     * Fragments that start before the <code>EndTimestamp</code> value and continue past it are included in the session.
     * </p>
     * </note>
     * 
     * @return The end of the timestamp range for the requested media. This value must be within 24 hours of the
     *         specified <code>StartTimestamp</code>, and it must be later than the <code>StartTimestamp</code>
     *         value.</p>
     *         <p>
     *         If <code>FragmentSelectorType</code> for the request is <code>SERVER_TIMESTAMP</code>, this value must be
     *         in the past.
     *         </p>
     *         <p>
     *         The <code>EndTimestamp</code> value is required for <code>ON_DEMAND</code> mode, but optional for
     *         <code>LIVE_REPLAY</code> mode. If the <code>EndTimestamp</code> is not set for <code>LIVE_REPLAY</code>
     *         mode then the session will continue to include newly ingested fragments until the session expires.
     *         </p>
     *         <note>
     *         <p>
     *         This value is inclusive. The <code>EndTimestamp</code> is compared to the (starting) timestamp of the
     *         fragment. Fragments that start before the <code>EndTimestamp</code> value and continue past it are
     *         included in the session.
     *         </p>
     */
    public final Instant endTimestamp() {
        return endTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(startTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(endTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DASHTimestampRange)) {
            return false;
        }
        DASHTimestampRange other = (DASHTimestampRange) obj;
        return Objects.equals(startTimestamp(), other.startTimestamp()) && Objects.equals(endTimestamp(), other.endTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DASHTimestampRange").add("StartTimestamp", startTimestamp()).add("EndTimestamp", endTimestamp())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StartTimestamp":
            return Optional.ofNullable(clazz.cast(startTimestamp()));
        case "EndTimestamp":
            return Optional.ofNullable(clazz.cast(endTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DASHTimestampRange, T> g) {
        return obj -> g.apply((DASHTimestampRange) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DASHTimestampRange> {
        /**
         * <p>
         * The start of the timestamp range for the requested media.
         * </p>
         * <p>
         * If the <code>DASHTimestampRange</code> value is specified, the <code>StartTimestamp</code> value is required.
         * </p>
         * <p>
         * Only fragments that start exactly at or after <code>StartTimestamp</code> are included in the session.
         * Fragments that start before <code>StartTimestamp</code> and continue past it aren't included in the session.
         * If <code>FragmentSelectorType</code> is <code>SERVER_TIMESTAMP</code>, the <code>StartTimestamp</code> must
         * be later than the stream head.
         * </p>
         * 
         * @param startTimestamp
         *        The start of the timestamp range for the requested media.</p>
         *        <p>
         *        If the <code>DASHTimestampRange</code> value is specified, the <code>StartTimestamp</code> value is
         *        required.
         *        </p>
         *        <p>
         *        Only fragments that start exactly at or after <code>StartTimestamp</code> are included in the session.
         *        Fragments that start before <code>StartTimestamp</code> and continue past it aren't included in the
         *        session. If <code>FragmentSelectorType</code> is <code>SERVER_TIMESTAMP</code>, the
         *        <code>StartTimestamp</code> must be later than the stream head.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTimestamp(Instant startTimestamp);

        /**
         * <p>
         * The end of the timestamp range for the requested media. This value must be within 24 hours of the specified
         * <code>StartTimestamp</code>, and it must be later than the <code>StartTimestamp</code> value.
         * </p>
         * <p>
         * If <code>FragmentSelectorType</code> for the request is <code>SERVER_TIMESTAMP</code>, this value must be in
         * the past.
         * </p>
         * <p>
         * The <code>EndTimestamp</code> value is required for <code>ON_DEMAND</code> mode, but optional for
         * <code>LIVE_REPLAY</code> mode. If the <code>EndTimestamp</code> is not set for <code>LIVE_REPLAY</code> mode
         * then the session will continue to include newly ingested fragments until the session expires.
         * </p>
         * <note>
         * <p>
         * This value is inclusive. The <code>EndTimestamp</code> is compared to the (starting) timestamp of the
         * fragment. Fragments that start before the <code>EndTimestamp</code> value and continue past it are included
         * in the session.
         * </p>
         * </note>
         * 
         * @param endTimestamp
         *        The end of the timestamp range for the requested media. This value must be within 24 hours of the
         *        specified <code>StartTimestamp</code>, and it must be later than the <code>StartTimestamp</code>
         *        value.</p>
         *        <p>
         *        If <code>FragmentSelectorType</code> for the request is <code>SERVER_TIMESTAMP</code>, this value must
         *        be in the past.
         *        </p>
         *        <p>
         *        The <code>EndTimestamp</code> value is required for <code>ON_DEMAND</code> mode, but optional for
         *        <code>LIVE_REPLAY</code> mode. If the <code>EndTimestamp</code> is not set for
         *        <code>LIVE_REPLAY</code> mode then the session will continue to include newly ingested fragments until
         *        the session expires.
         *        </p>
         *        <note>
         *        <p>
         *        This value is inclusive. The <code>EndTimestamp</code> is compared to the (starting) timestamp of the
         *        fragment. Fragments that start before the <code>EndTimestamp</code> value and continue past it are
         *        included in the session.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTimestamp(Instant endTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private Instant startTimestamp;

        private Instant endTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(DASHTimestampRange model) {
            startTimestamp(model.startTimestamp);
            endTimestamp(model.endTimestamp);
        }

        public final Instant getStartTimestamp() {
            return startTimestamp;
        }

        @Override
        public final Builder startTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
            return this;
        }

        public final void setStartTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
        }

        public final Instant getEndTimestamp() {
            return endTimestamp;
        }

        @Override
        public final Builder endTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
            return this;
        }

        public final void setEndTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
        }

        @Override
        public DASHTimestampRange build() {
            return new DASHTimestampRange(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
