/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetHlsStreamingSessionUrlRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetHlsStreamingSessionUrlResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetMediaForFragmentListRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetMediaForFragmentListResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidCodecPrivateDataException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.KinesisVideoArchivedMediaException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ListFragmentsRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ListFragmentsResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.MissingCodecPrivateDataException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.NoDataRetentionException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.UnsupportedStreamMediaTypeException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetHlsStreamingSessionUrlRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetMediaForFragmentListRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.ListFragmentsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisVideoArchivedMediaAsyncClient}.
 *
 * @see KinesisVideoArchivedMediaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoArchivedMediaAsyncClient implements KinesisVideoArchivedMediaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoArchivedMediaAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKinesisVideoArchivedMediaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves an HTTP Live Streaming (HLS) URL for the stream. You can then open the URL in a browser or media player
     * to view the stream contents.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * An Amazon Kinesis video stream has the following requirements for providing data through HLS:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The media type must be <code>video/h264</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data retention must be greater than 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * The fragments must contain codec private data in the AVC (Advanced Video Coding) for H.264 format (<a
     * href="https://www.iso.org/standard/55980.html">MPEG-4 specification ISO/IEC 14496-15</a>). For information about
     * adapting stream data to a given format, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/latest/dg/producer-reference-nal.html">NAL
     * Adaptation Flags</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Kinesis Video Streams HLS sessions contain fragments in the fragmented MPEG-4 form (also called fMP4 or CMAF),
     * rather than the MPEG-2 form (also called TS chunks, which the HLS specification also supports). For more
     * information about HLS fragment types, see the <a
     * href="https://tools.ietf.org/html/draft-pantos-http-live-streaming-23">HLS specification</a>.
     * </p>
     * <p>
     * The following procedure shows how to use HLS with Kinesis Video Streams:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Get an endpoint using <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html">GetDataEndpoint</a>,
     * specifying <code>GET_HLS_STREAMING_SESSION_URL</code> for the <code>APIName</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Retrieve the HLS URL using <code>GetHLSStreamingSessionURL</code>. Kinesis Video Streams creates an HLS streaming
     * session to be used for accessing content in a stream using the HLS protocol.
     * <code>GetHLSStreamingSessionURL</code> returns an authenticated URL (that includes an encrypted session token)
     * for the session's HLS <i>master playlist</i> (the root resource needed for streaming with HLS).
     * </p>
     * <note>
     * <p>
     * Don't share or store this token where an unauthorized entity could access it. The token provides access to the
     * content of the stream. Safeguard the token with the same measures that you would use with your AWS credentials.
     * </p>
     * </note>
     * <p>
     * The media that is made available through the playlist consists only of the requested stream, time range, and
     * format. No other media data (such as frames outside the requested window or alternate bit rates) is made
     * available.
     * </p>
     * </li>
     * <li>
     * <p>
     * Provide the URL (containing the encrypted session token) for the HLS master playlist to a media player that
     * supports the HLS protocol. Kinesis Video Streams makes the HLS media playlist, initialization fragment, and media
     * fragments available through the master playlist URL. The initialization fragment contains the codec private data
     * for the stream, and other data needed to set up the video decoder and renderer. The media fragments contain
     * H.264-encoded video frames and time stamps.
     * </p>
     * </li>
     * <li>
     * <p>
     * The media player receives the authenticated URL and requests stream metadata and media data normally. When the
     * media player requests data, it calls the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>GetHLSMasterPlaylist:</b> Retrieves an HLS master playlist, which contains a URL for the
     * <code>GetHLSMediaPlaylist</code> action, and additional metadata for the media player, including estimated bit
     * rate and resolution.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetHLSMediaPlaylist:</b> Retrieves an HLS media playlist, which contains a URL to access the MP4
     * initialization fragment with the <code>GetMP4InitFragment</code> action, and URLs to access the MP4 media
     * fragments with the <code>GetMP4MediaFragment</code> actions. The HLS media playlist also contains metadata about
     * the stream that the player needs to play it, such as whether the <code>PlaybackMode</code> is <code>LIVE</code>
     * or <code>ON_DEMAND</code>. The HLS media playlist is typically static for sessions with a
     * <code>PlaybackType</code> of <code>ON_DEMAND</code>. The HLS media playlist is continually updated with new
     * fragments for sessions with a <code>PlaybackType</code> of <code>LIVE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4InitFragment:</b> Retrieves the MP4 initialization fragment. The media player typically loads the
     * initialization fragment before loading any media fragments. This fragment contains the "<code>fytp</code>" and "
     * <code>moov</code>" MP4 atoms, and the child atoms that are needed to initialize the media player decoder.
     * </p>
     * <p>
     * The initialization fragment does not correspond to a fragment in a Kinesis video stream. It contains only the
     * codec private data for the stream, which the media player needs to decode video frames.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4MediaFragment:</b> Retrieves MP4 media fragments. These fragments contain the "<code>moof</code>" and "
     * <code>mdat</code>" MP4 atoms and their child atoms, containing the encoded fragment's video frames and their time
     * stamps.
     * </p>
     * <note>
     * <p>
     * After the first media fragment is made available in a streaming session, any fragments that don't contain the
     * same codec private data are excluded in the HLS media playlist. Therefore, the codec private data does not change
     * between fragments in a session.
     * </p>
     * </note>
     * <p>
     * Data retrieved with this action is billable. See <a
     * href="aws.amazon.comkinesis/video-streams/pricing/">Pricing</a> for details.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <note>
     * <p>
     * The following restrictions apply to HLS sessions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A streaming session URL should not be shared between players. The service might throttle a session if multiple
     * media players are sharing it. For connection limits, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams Limits</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A Kinesis video stream can have a maximum of five active HLS streaming sessions. If a new session is created when
     * the maximum number of sessions is already active, the oldest (earliest created) session is closed. The number of
     * active <code>GetMedia</code> connections on a Kinesis video stream does not count against this limit, and the
     * number of active HLS sessions does not count against the active <code>GetMedia</code> connection limit.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * You can monitor the amount of data that the media player consumes by monitoring the
     * <code>GetMP4MediaFragment.OutgoingBytes</code> Amazon CloudWatch metric. For information about using CloudWatch
     * to monitor Kinesis Video Streams, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/monitoring.html">Monitoring Kinesis Video
     * Streams</a>. For pricing information, see <a href="https://aws.amazon.com/kinesis/video-streams/pricing/">Amazon
     * Kinesis Video Streams Pricing</a> and <a href="https://aws.amazon.com/pricing/">AWS Pricing</a>. Charges for both
     * HLS sessions and outgoing AWS data apply.
     * </p>
     * <p>
     * For more information about HLS, see <a href="https://developer.apple.com/streaming/">HTTP Live Streaming</a> on
     * the <a href="https://developer.apple.com">Apple Developer site</a>.
     * </p>
     *
     * @param getHlsStreamingSessionUrlRequest
     * @return A Java Future containing the result of the GetHLSStreamingSessionURL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetMedia</code> throws this error when Kinesis Video Streams can't
     *         find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>UnsupportedStreamMediaTypeException An HLS streaming session was requested for a stream with a media
     *         type that is not <code>video/h264</code>.</li>
     *         <li>NoDataRetentionException A <code>PlaybackMode</code> of <code>ON_DEMAND</code> was requested for a
     *         stream that does not retain data (that is, has a <code>DataRetentionInHours</code> of 0).</li>
     *         <li>MissingCodecPrivateDataException No Codec Private Data was found in the video stream.</li>
     *         <li>InvalidCodecPrivateDataException The Codec Private Data in the video stream is not valid for this
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetHLSStreamingSessionURL
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetHLSStreamingSessionURL"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHlsStreamingSessionUrlResponse> getHLSStreamingSessionURL(
            GetHlsStreamingSessionUrlRequest getHlsStreamingSessionUrlRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHlsStreamingSessionUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetHlsStreamingSessionUrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<GetHlsStreamingSessionUrlRequest, GetHlsStreamingSessionUrlResponse>()
                            .withOperationName("GetHLSStreamingSessionURL")
                            .withMarshaller(new GetHlsStreamingSessionUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getHlsStreamingSessionUrlRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMediaForFragmentListRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see <a
     *        href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *        The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *        include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be included if an exception occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetMedia</code> throws this error when Kinesis Video Streams can't
     *         find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetMediaForFragmentList
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getMediaForFragmentList(
            GetMediaForFragmentListRequest getMediaForFragmentListRequest,
            AsyncResponseTransformer<GetMediaForFragmentListResponse, ReturnT> asyncResponseTransformer) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMediaForFragmentListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMediaForFragmentListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(
                    new ClientExecutionParams<GetMediaForFragmentListRequest, GetMediaForFragmentListResponse>()
                            .withOperationName("GetMediaForFragmentList")
                            .withMarshaller(new GetMediaForFragmentListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getMediaForFragmentListRequest), asyncResponseTransformer).whenComplete((r, e) -> {
                if (e != null) {
                    asyncResponseTransformer.exceptionOccurred(e);
                }
            });
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseTransformer.exceptionOccurred(t));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a>Fragment</a> objects from the specified stream and start location within the archived data.
     * </p>
     *
     * @param listFragmentsRequest
     * @return A Java Future containing the result of the ListFragments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetMedia</code> throws this error when Kinesis Video Streams can't
     *         find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.ListFragments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/ListFragments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFragmentsResponse> listFragments(ListFragmentsRequest listFragmentsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFragmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFragmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListFragmentsRequest, ListFragmentsResponse>()
                    .withOperationName("ListFragments").withMarshaller(new ListFragmentsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listFragmentsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoArchivedMediaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                                .exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                                .exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientLimitExceededException")
                                .exceptionBuilderSupplier(ClientLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoDataRetentionException")
                                .exceptionBuilderSupplier(NoDataRetentionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                                .exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
