/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetHlsStreamingSessionUrlRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetHlsStreamingSessionUrlResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetMediaForFragmentListRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetMediaForFragmentListResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidCodecPrivateDataException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.KinesisVideoArchivedMediaException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ListFragmentsRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ListFragmentsResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.MissingCodecPrivateDataException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.NoDataRetentionException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.UnsupportedStreamMediaTypeException;

/**
 * Service client for accessing Kinesis Video Archived Media. This can be created using the static {@link #builder()}
 * method.
 *
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public interface KinesisVideoArchivedMediaClient extends SdkClient {
    String SERVICE_NAME = "kinesisvideo";

    /**
     * Create a {@link KinesisVideoArchivedMediaClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static KinesisVideoArchivedMediaClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link KinesisVideoArchivedMediaClient}.
     */
    static KinesisVideoArchivedMediaClientBuilder builder() {
        return new DefaultKinesisVideoArchivedMediaClientBuilder();
    }

    /**
     * <p>
     * Retrieves an HTTP Live Streaming (HLS) URL for the stream. You can then open the URL in a browser or media player
     * to view the stream contents.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * An Amazon Kinesis video stream has the following requirements for providing data through HLS:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The media type must be <code>video/h264</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data retention must be greater than 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * The fragments must contain codec private data in the AVC (Advanced Video Coding) for H.264 format (<a
     * href="https://www.iso.org/standard/55980.html">MPEG-4 specification ISO/IEC 14496-15</a>). For information about
     * adapting stream data to a given format, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/latest/dg/producer-reference-nal.html">NAL
     * Adaptation Flags</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Kinesis Video Streams HLS sessions contain fragments in the fragmented MPEG-4 form (also called fMP4 or CMAF),
     * rather than the MPEG-2 form (also called TS chunks, which the HLS specification also supports). For more
     * information about HLS fragment types, see the <a
     * href="https://tools.ietf.org/html/draft-pantos-http-live-streaming-23">HLS specification</a>.
     * </p>
     * <p>
     * The following procedure shows how to use HLS with Kinesis Video Streams:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Get an endpoint using <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html">GetDataEndpoint</a>,
     * specifying <code>GET_HLS_STREAMING_SESSION_URL</code> for the <code>APIName</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Retrieve the HLS URL using <code>GetHLSStreamingSessionURL</code>. Kinesis Video Streams creates an HLS streaming
     * session to be used for accessing content in a stream using the HLS protocol.
     * <code>GetHLSStreamingSessionURL</code> returns an authenticated URL (that includes an encrypted session token)
     * for the session's HLS <i>master playlist</i> (the root resource needed for streaming with HLS).
     * </p>
     * <note>
     * <p>
     * Don't share or store this token where an unauthorized entity could access it. The token provides access to the
     * content of the stream. Safeguard the token with the same measures that you would use with your AWS credentials.
     * </p>
     * </note>
     * <p>
     * The media that is made available through the playlist consists only of the requested stream, time range, and
     * format. No other media data (such as frames outside the requested window or alternate bit rates) is made
     * available.
     * </p>
     * </li>
     * <li>
     * <p>
     * Provide the URL (containing the encrypted session token) for the HLS master playlist to a media player that
     * supports the HLS protocol. Kinesis Video Streams makes the HLS media playlist, initialization fragment, and media
     * fragments available through the master playlist URL. The initialization fragment contains the codec private data
     * for the stream, and other data needed to set up the video decoder and renderer. The media fragments contain
     * H.264-encoded video frames and time stamps.
     * </p>
     * </li>
     * <li>
     * <p>
     * The media player receives the authenticated URL and requests stream metadata and media data normally. When the
     * media player requests data, it calls the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>GetHLSMasterPlaylist:</b> Retrieves an HLS master playlist, which contains a URL for the
     * <code>GetHLSMediaPlaylist</code> action, and additional metadata for the media player, including estimated bit
     * rate and resolution.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetHLSMediaPlaylist:</b> Retrieves an HLS media playlist, which contains a URL to access the MP4
     * initialization fragment with the <code>GetMP4InitFragment</code> action, and URLs to access the MP4 media
     * fragments with the <code>GetMP4MediaFragment</code> actions. The HLS media playlist also contains metadata about
     * the stream that the player needs to play it, such as whether the <code>PlaybackMode</code> is <code>LIVE</code>
     * or <code>ON_DEMAND</code>. The HLS media playlist is typically static for sessions with a
     * <code>PlaybackType</code> of <code>ON_DEMAND</code>. The HLS media playlist is continually updated with new
     * fragments for sessions with a <code>PlaybackType</code> of <code>LIVE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4InitFragment:</b> Retrieves the MP4 initialization fragment. The media player typically loads the
     * initialization fragment before loading any media fragments. This fragment contains the "<code>fytp</code>" and "
     * <code>moov</code>" MP4 atoms, and the child atoms that are needed to initialize the media player decoder.
     * </p>
     * <p>
     * The initialization fragment does not correspond to a fragment in a Kinesis video stream. It contains only the
     * codec private data for the stream, which the media player needs to decode video frames.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4MediaFragment:</b> Retrieves MP4 media fragments. These fragments contain the "<code>moof</code>" and "
     * <code>mdat</code>" MP4 atoms and their child atoms, containing the encoded fragment's video frames and their time
     * stamps.
     * </p>
     * <note>
     * <p>
     * After the first media fragment is made available in a streaming session, any fragments that don't contain the
     * same codec private data are excluded in the HLS media playlist. Therefore, the codec private data does not change
     * between fragments in a session.
     * </p>
     * </note>
     * <p>
     * Data retrieved with this action is billable. See <a
     * href="aws.amazon.comkinesis/video-streams/pricing/">Pricing</a> for details.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <note>
     * <p>
     * The following restrictions apply to HLS sessions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A streaming session URL should not be shared between players. The service might throttle a session if multiple
     * media players are sharing it. For connection limits, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams Limits</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A Kinesis video stream can have a maximum of five active HLS streaming sessions. If a new session is created when
     * the maximum number of sessions is already active, the oldest (earliest created) session is closed. The number of
     * active <code>GetMedia</code> connections on a Kinesis video stream does not count against this limit, and the
     * number of active HLS sessions does not count against the active <code>GetMedia</code> connection limit.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * You can monitor the amount of data that the media player consumes by monitoring the
     * <code>GetMP4MediaFragment.OutgoingBytes</code> Amazon CloudWatch metric. For information about using CloudWatch
     * to monitor Kinesis Video Streams, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/monitoring.html">Monitoring Kinesis Video
     * Streams</a>. For pricing information, see <a href="https://aws.amazon.com/kinesis/video-streams/pricing/">Amazon
     * Kinesis Video Streams Pricing</a> and <a href="https://aws.amazon.com/pricing/">AWS Pricing</a>. Charges for both
     * HLS sessions and outgoing AWS data apply.
     * </p>
     * <p>
     * For more information about HLS, see <a href="https://developer.apple.com/streaming/">HTTP Live Streaming</a> on
     * the <a href="https://developer.apple.com">Apple Developer site</a>.
     * </p>
     *
     * @param getHlsStreamingSessionUrlRequest
     * @return Result of the GetHLSStreamingSessionURL operation returned by the service.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws UnsupportedStreamMediaTypeException
     *         An HLS streaming session was requested for a stream with a media type that is not <code>video/h264</code>
     *         .
     * @throws NoDataRetentionException
     *         A <code>PlaybackMode</code> of <code>ON_DEMAND</code> was requested for a stream that does not retain
     *         data (that is, has a <code>DataRetentionInHours</code> of 0).
     * @throws MissingCodecPrivateDataException
     *         No Codec Private Data was found in the video stream.
     * @throws InvalidCodecPrivateDataException
     *         The Codec Private Data in the video stream is not valid for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetHLSStreamingSessionURL
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetHLSStreamingSessionURL"
     *      target="_top">AWS API Documentation</a>
     */
    default GetHlsStreamingSessionUrlResponse getHLSStreamingSessionURL(
            GetHlsStreamingSessionUrlRequest getHlsStreamingSessionUrlRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, UnsupportedStreamMediaTypeException,
            NoDataRetentionException, MissingCodecPrivateDataException, InvalidCodecPrivateDataException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves an HTTP Live Streaming (HLS) URL for the stream. You can then open the URL in a browser or media player
     * to view the stream contents.
     * </p>
     * <p>
     * You must specify either the <code>StreamName</code> or the <code>StreamARN</code>.
     * </p>
     * <p>
     * An Amazon Kinesis video stream has the following requirements for providing data through HLS:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The media type must be <code>video/h264</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data retention must be greater than 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * The fragments must contain codec private data in the AVC (Advanced Video Coding) for H.264 format (<a
     * href="https://www.iso.org/standard/55980.html">MPEG-4 specification ISO/IEC 14496-15</a>). For information about
     * adapting stream data to a given format, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/latest/dg/producer-reference-nal.html">NAL
     * Adaptation Flags</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Kinesis Video Streams HLS sessions contain fragments in the fragmented MPEG-4 form (also called fMP4 or CMAF),
     * rather than the MPEG-2 form (also called TS chunks, which the HLS specification also supports). For more
     * information about HLS fragment types, see the <a
     * href="https://tools.ietf.org/html/draft-pantos-http-live-streaming-23">HLS specification</a>.
     * </p>
     * <p>
     * The following procedure shows how to use HLS with Kinesis Video Streams:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Get an endpoint using <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html">GetDataEndpoint</a>,
     * specifying <code>GET_HLS_STREAMING_SESSION_URL</code> for the <code>APIName</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Retrieve the HLS URL using <code>GetHLSStreamingSessionURL</code>. Kinesis Video Streams creates an HLS streaming
     * session to be used for accessing content in a stream using the HLS protocol.
     * <code>GetHLSStreamingSessionURL</code> returns an authenticated URL (that includes an encrypted session token)
     * for the session's HLS <i>master playlist</i> (the root resource needed for streaming with HLS).
     * </p>
     * <note>
     * <p>
     * Don't share or store this token where an unauthorized entity could access it. The token provides access to the
     * content of the stream. Safeguard the token with the same measures that you would use with your AWS credentials.
     * </p>
     * </note>
     * <p>
     * The media that is made available through the playlist consists only of the requested stream, time range, and
     * format. No other media data (such as frames outside the requested window or alternate bit rates) is made
     * available.
     * </p>
     * </li>
     * <li>
     * <p>
     * Provide the URL (containing the encrypted session token) for the HLS master playlist to a media player that
     * supports the HLS protocol. Kinesis Video Streams makes the HLS media playlist, initialization fragment, and media
     * fragments available through the master playlist URL. The initialization fragment contains the codec private data
     * for the stream, and other data needed to set up the video decoder and renderer. The media fragments contain
     * H.264-encoded video frames and time stamps.
     * </p>
     * </li>
     * <li>
     * <p>
     * The media player receives the authenticated URL and requests stream metadata and media data normally. When the
     * media player requests data, it calls the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>GetHLSMasterPlaylist:</b> Retrieves an HLS master playlist, which contains a URL for the
     * <code>GetHLSMediaPlaylist</code> action, and additional metadata for the media player, including estimated bit
     * rate and resolution.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetHLSMediaPlaylist:</b> Retrieves an HLS media playlist, which contains a URL to access the MP4
     * initialization fragment with the <code>GetMP4InitFragment</code> action, and URLs to access the MP4 media
     * fragments with the <code>GetMP4MediaFragment</code> actions. The HLS media playlist also contains metadata about
     * the stream that the player needs to play it, such as whether the <code>PlaybackMode</code> is <code>LIVE</code>
     * or <code>ON_DEMAND</code>. The HLS media playlist is typically static for sessions with a
     * <code>PlaybackType</code> of <code>ON_DEMAND</code>. The HLS media playlist is continually updated with new
     * fragments for sessions with a <code>PlaybackType</code> of <code>LIVE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4InitFragment:</b> Retrieves the MP4 initialization fragment. The media player typically loads the
     * initialization fragment before loading any media fragments. This fragment contains the "<code>fytp</code>" and "
     * <code>moov</code>" MP4 atoms, and the child atoms that are needed to initialize the media player decoder.
     * </p>
     * <p>
     * The initialization fragment does not correspond to a fragment in a Kinesis video stream. It contains only the
     * codec private data for the stream, which the media player needs to decode video frames.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4MediaFragment:</b> Retrieves MP4 media fragments. These fragments contain the "<code>moof</code>" and "
     * <code>mdat</code>" MP4 atoms and their child atoms, containing the encoded fragment's video frames and their time
     * stamps.
     * </p>
     * <note>
     * <p>
     * After the first media fragment is made available in a streaming session, any fragments that don't contain the
     * same codec private data are excluded in the HLS media playlist. Therefore, the codec private data does not change
     * between fragments in a session.
     * </p>
     * </note>
     * <p>
     * Data retrieved with this action is billable. See <a
     * href="aws.amazon.comkinesis/video-streams/pricing/">Pricing</a> for details.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <note>
     * <p>
     * The following restrictions apply to HLS sessions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A streaming session URL should not be shared between players. The service might throttle a session if multiple
     * media players are sharing it. For connection limits, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams Limits</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A Kinesis video stream can have a maximum of five active HLS streaming sessions. If a new session is created when
     * the maximum number of sessions is already active, the oldest (earliest created) session is closed. The number of
     * active <code>GetMedia</code> connections on a Kinesis video stream does not count against this limit, and the
     * number of active HLS sessions does not count against the active <code>GetMedia</code> connection limit.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * You can monitor the amount of data that the media player consumes by monitoring the
     * <code>GetMP4MediaFragment.OutgoingBytes</code> Amazon CloudWatch metric. For information about using CloudWatch
     * to monitor Kinesis Video Streams, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/monitoring.html">Monitoring Kinesis Video
     * Streams</a>. For pricing information, see <a href="https://aws.amazon.com/kinesis/video-streams/pricing/">Amazon
     * Kinesis Video Streams Pricing</a> and <a href="https://aws.amazon.com/pricing/">AWS Pricing</a>. Charges for both
     * HLS sessions and outgoing AWS data apply.
     * </p>
     * <p>
     * For more information about HLS, see <a href="https://developer.apple.com/streaming/">HTTP Live Streaming</a> on
     * the <a href="https://developer.apple.com">Apple Developer site</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetHlsStreamingSessionUrlRequest.Builder} avoiding
     * the need to create one manually via {@link GetHlsStreamingSessionUrlRequest#builder()}
     * </p>
     *
     * @param getHlsStreamingSessionUrlRequest
     *        A {@link Consumer} that will call methods on {@link GetHLSStreamingSessionURLInput.Builder} to create a
     *        request.
     * @return Result of the GetHLSStreamingSessionURL operation returned by the service.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws UnsupportedStreamMediaTypeException
     *         An HLS streaming session was requested for a stream with a media type that is not <code>video/h264</code>
     *         .
     * @throws NoDataRetentionException
     *         A <code>PlaybackMode</code> of <code>ON_DEMAND</code> was requested for a stream that does not retain
     *         data (that is, has a <code>DataRetentionInHours</code> of 0).
     * @throws MissingCodecPrivateDataException
     *         No Codec Private Data was found in the video stream.
     * @throws InvalidCodecPrivateDataException
     *         The Codec Private Data in the video stream is not valid for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetHLSStreamingSessionURL
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetHLSStreamingSessionURL"
     *      target="_top">AWS API Documentation</a>
     */
    default GetHlsStreamingSessionUrlResponse getHLSStreamingSessionURL(
            Consumer<GetHlsStreamingSessionUrlRequest.Builder> getHlsStreamingSessionUrlRequest)
            throws ResourceNotFoundException, InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException,
            UnsupportedStreamMediaTypeException, NoDataRetentionException, MissingCodecPrivateDataException,
            InvalidCodecPrivateDataException, AwsServiceException, SdkClientException, KinesisVideoArchivedMediaException {
        return getHLSStreamingSessionURL(GetHlsStreamingSessionUrlRequest.builder()
                .applyMutation(getHlsStreamingSessionUrlRequest).build());
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMediaForFragmentListRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetMediaForFragmentListResponse and an InputStream to the response content are provided as parameters to
     *        the callback. The callback may return a transformed type which will be the return value of this method.
     *        See {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this
     *        interface and for links to pre-canned implementations for common scenarios like downloading to a file. The
     *        service documentation for the response content is as follows '
     *        <p>
     *        The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see <a
     *        href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *        The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *        include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be included if an exception occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT getMediaForFragmentList(GetMediaForFragmentListRequest getMediaForFragmentListRequest,
            ResponseTransformer<GetMediaForFragmentListResponse, ReturnT> responseTransformer) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaForFragmentListRequest.Builder} avoiding
     * the need to create one manually via {@link GetMediaForFragmentListRequest#builder()}
     * </p>
     *
     * @param getMediaForFragmentListRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaForFragmentListInput.Builder} to create a
     *        request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetMediaForFragmentListResponse and an InputStream to the response content are provided as parameters to
     *        the callback. The callback may return a transformed type which will be the return value of this method.
     *        See {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this
     *        interface and for links to pre-canned implementations for common scenarios like downloading to a file. The
     *        service documentation for the response content is as follows '
     *        <p>
     *        The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see <a
     *        href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *        The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *        include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be included if an exception occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> ReturnT getMediaForFragmentList(
            Consumer<GetMediaForFragmentListRequest.Builder> getMediaForFragmentListRequest,
            ResponseTransformer<GetMediaForFragmentListResponse, ReturnT> responseTransformer) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentList(GetMediaForFragmentListRequest.builder().applyMutation(getMediaForFragmentListRequest)
                .build(), responseTransformer);
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMediaForFragmentListRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see <a
     *        href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *        The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *        include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be included if an exception occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see #getMediaForFragmentList(GetMediaForFragmentListRequest, ResponseTransformer)
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default GetMediaForFragmentListResponse getMediaForFragmentList(
            GetMediaForFragmentListRequest getMediaForFragmentListRequest, Path filePath) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentList(getMediaForFragmentListRequest, ResponseTransformer.toFile(filePath));
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaForFragmentListRequest.Builder} avoiding
     * the need to create one manually via {@link GetMediaForFragmentListRequest#builder()}
     * </p>
     *
     * @param getMediaForFragmentListRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaForFragmentListInput.Builder} to create a
     *        request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see <a
     *        href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *        The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *        include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be included if an exception occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see #getMediaForFragmentList(GetMediaForFragmentListRequest, ResponseTransformer)
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default GetMediaForFragmentListResponse getMediaForFragmentList(
            Consumer<GetMediaForFragmentListRequest.Builder> getMediaForFragmentListRequest, Path filePath)
            throws ResourceNotFoundException, InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException,
            AwsServiceException, SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentList(GetMediaForFragmentListRequest.builder().applyMutation(getMediaForFragmentListRequest)
                .build(), filePath);
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMediaForFragmentListRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *         The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *         include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be included if an exception occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see #getObject(getMediaForFragmentList, ResponseTransformer)
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseInputStream<GetMediaForFragmentListResponse> getMediaForFragmentList(
            GetMediaForFragmentListRequest getMediaForFragmentListRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentList(getMediaForFragmentListRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaForFragmentListRequest.Builder} avoiding
     * the need to create one manually via {@link GetMediaForFragmentListRequest#builder()}
     * </p>
     *
     * @param getMediaForFragmentListRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaForFragmentListInput.Builder} to create a
     *        request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *         The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *         include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be included if an exception occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see #getObject(getMediaForFragmentList, ResponseTransformer)
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseInputStream<GetMediaForFragmentListResponse> getMediaForFragmentList(
            Consumer<GetMediaForFragmentListRequest.Builder> getMediaForFragmentListRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentList(GetMediaForFragmentListRequest.builder().applyMutation(getMediaForFragmentListRequest)
                .build());
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMediaForFragmentListRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *         The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *         include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be included if an exception occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see #getObject(getMediaForFragmentList, ResponseTransformer)
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseBytes<GetMediaForFragmentListResponse> getMediaForFragmentListAsBytes(
            GetMediaForFragmentListRequest getMediaForFragmentListRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentList(getMediaForFragmentListRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMediaForFragmentList</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMediaForFragmentList</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMediaForFragmentList</code> session.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaForFragmentListRequest.Builder} avoiding
     * the need to create one manually via {@link GetMediaForFragmentListRequest#builder()}
     * </p>
     *
     * @param getMediaForFragmentListRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaForFragmentListInput.Builder} to create a
     *        request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *         The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *         include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side time stamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be included if an exception occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the exception
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.GetMediaForFragmentList
     * @see #getObject(getMediaForFragmentList, ResponseTransformer)
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    default ResponseBytes<GetMediaForFragmentListResponse> getMediaForFragmentListAsBytes(
            Consumer<GetMediaForFragmentListRequest.Builder> getMediaForFragmentListRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        return getMediaForFragmentListAsBytes(GetMediaForFragmentListRequest.builder()
                .applyMutation(getMediaForFragmentListRequest).build());
    }

    /**
     * <p>
     * Returns a list of <a>Fragment</a> objects from the specified stream and start location within the archived data.
     * </p>
     *
     * @param listFragmentsRequest
     * @return Result of the ListFragments operation returned by the service.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.ListFragments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/ListFragments"
     *      target="_top">AWS API Documentation</a>
     */
    default ListFragmentsResponse listFragments(ListFragmentsRequest listFragmentsRequest) throws ResourceNotFoundException,
            InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException, AwsServiceException,
            SdkClientException, KinesisVideoArchivedMediaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of <a>Fragment</a> objects from the specified stream and start location within the archived data.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListFragmentsRequest.Builder} avoiding the need to
     * create one manually via {@link ListFragmentsRequest#builder()}
     * </p>
     *
     * @param listFragmentsRequest
     *        A {@link Consumer} that will call methods on {@link ListFragmentsInput.Builder} to create a request.
     * @return Result of the ListFragments operation returned by the service.
     * @throws ResourceNotFoundException
     *         <code>GetMedia</code> throws this error when Kinesis Video Streams can't find the stream that you
     *         specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> throws this error if a session with a <code>PlaybackMode</code> of
     *         <code>ON_DEMAND</code> is requested for a stream that has no fragments within the requested time range,
     *         or if a session with a <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has
     *         no fragments within the last 30 seconds.
     * @throws InvalidArgumentException
     *         A specified parameter exceeds its restrictions, is not supported, or can't be used.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoArchivedMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoArchivedMediaClient.ListFragments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/ListFragments"
     *      target="_top">AWS API Documentation</a>
     */
    default ListFragmentsResponse listFragments(Consumer<ListFragmentsRequest.Builder> listFragmentsRequest)
            throws ResourceNotFoundException, InvalidArgumentException, ClientLimitExceededException, NotAuthorizedException,
            AwsServiceException, SdkClientException, KinesisVideoArchivedMediaException {
        return listFragments(ListFragmentsRequest.builder().applyMutation(listFragmentsRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("kinesisvideo");
    }
}
