/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideoarchivedmedia;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.internal.KinesisVideoArchivedMediaServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetClipRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetClipResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetDashStreamingSessionUrlRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetDashStreamingSessionUrlResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetHlsStreamingSessionUrlRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetHlsStreamingSessionUrlResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetImagesRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetImagesResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetMediaForFragmentListRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.GetMediaForFragmentListResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidCodecPrivateDataException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.InvalidMediaFrameException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.KinesisVideoArchivedMediaException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ListFragmentsRequest;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ListFragmentsResponse;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.MissingCodecPrivateDataException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.NoDataRetentionException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.model.UnsupportedStreamMediaTypeException;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetClipRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetDashStreamingSessionUrlRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetHlsStreamingSessionUrlRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetImagesRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.GetMediaForFragmentListRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideoarchivedmedia.transform.ListFragmentsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link KinesisVideoArchivedMediaAsyncClient}.
 *
 * @see KinesisVideoArchivedMediaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoArchivedMediaAsyncClient implements KinesisVideoArchivedMediaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoArchivedMediaAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKinesisVideoArchivedMediaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Kinesis_Video_Archived_Media" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Downloads an MP4 file (clip) containing the archived, on-demand media from the specified video stream over the
     * specified time range.
     * </p>
     * <p>
     * Both the StreamName and the StreamARN parameters are optional, but you must specify either the StreamName or the
     * StreamARN when invoking this API operation.
     * </p>
     * <p>
     * As a prerequisite to using GetCLip API, you must obtain an endpoint using <code>GetDataEndpoint</code>,
     * specifying GET_CLIP for<code/> the <code>APIName</code> parameter.
     * </p>
     * <p>
     * An Amazon Kinesis video stream has the following requirements for providing data through MP4:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The media must contain h.264 or h.265 encoded video and, optionally, AAC or G.711 encoded audio. Specifically,
     * the codec ID of track 1 should be <code>V_MPEG/ISO/AVC</code> (for h.264) or V_MPEGH/ISO/HEVC (for H.265).
     * Optionally, the codec ID of track 2 should be <code>A_AAC</code> (for AAC) or A_MS/ACM (for G.711).
     * </p>
     * </li>
     * <li>
     * <p>
     * Data retention must be greater than 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * The video track of each fragment must contain codec private data in the Advanced Video Coding (AVC) for H.264
     * format and HEVC for H.265 format. For more information, see <a
     * href="https://www.iso.org/standard/55980.html">MPEG-4 specification ISO/IEC 14496-15</a>. For information about
     * adapting stream data to a given format, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/producer-reference-nal.html">NAL Adaptation
     * Flags</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The audio track (if present) of each fragment must contain codec private data in the AAC format (<a
     * href="https://www.iso.org/standard/43345.html">AAC specification ISO/IEC 13818-7</a>) or the <a
     * href="http://www-mmsp.ece.mcgill.ca/Documents/AudioFormats/WAVE/WAVE.html">MS Wave format</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can monitor the amount of outgoing data by monitoring the <code>GetClip.OutgoingBytes</code> Amazon
     * CloudWatch metric. For information about using CloudWatch to monitor Kinesis Video Streams, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/monitoring.html">Monitoring Kinesis Video
     * Streams</a>. For pricing information, see <a href="https://aws.amazon.com/kinesis/video-streams/pricing/">Amazon
     * Kinesis Video Streams Pricing</a> and <a href="https://aws.amazon.com/pricing/"> Amazon Web Services Pricing</a>.
     * Charges for outgoing Amazon Web Services data apply.
     * </p>
     *
     * @param getClipRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        Traditional MP4 file that contains the media clip from the specified video stream. The output will contain
     *        the first 100 MB or the first 200 fragments from the specified start timestamp. For more information, see
     *        <a href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *        Limits</a>.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetImages</code> will throw this error when Kinesis Video Streams
     *         can't find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> and <code>GetDASHStreamingSessionURL</code> throw this error if a
     *         session with a <code>PlaybackMode</code> of <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>is
     *         requested for a stream that has no fragments within the requested time range, or if a session with a
     *         <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has no fragments within the
     *         last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded a limit. Try making the call later. For information about limits, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *         Limits</a>.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>UnsupportedStreamMediaTypeException The type of the media (for example, h.264 or h.265 video or ACC
     *         or G.711 audio) could not be determined from the codec IDs of the tracks in the first fragment for a
     *         playback session. The codec ID for track 1 should be <code>V_MPEG/ISO/AVC</code> and, optionally, the
     *         codec ID for track 2 should be <code>A_AAC</code>.</li>
     *         <li>MissingCodecPrivateDataException No codec private data was found in at least one of tracks of the
     *         video stream.</li>
     *         <li>InvalidCodecPrivateDataException The codec private data in at least one of the tracks of the video
     *         stream is not valid for this operation.</li>
     *         <li>InvalidMediaFrameException One or more frames in the requested clip could not be parsed based on the
     *         specified codec.</li>
     *         <li>NoDataRetentionException <code>GetImages</code> was requested for a stream that does not retain data
     *         (that is, has a <code>DataRetentionInHours</code> of 0).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetClip
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetClip"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getClip(GetClipRequest getClipRequest,
            AsyncResponseTransformer<GetClipResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getClipRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getClipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Archived Media");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetClip");
            Pair<AsyncResponseTransformer<GetClipResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetClipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetClipResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "NotAuthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
                case "InvalidCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).build());
                case "InvalidArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidArgumentException::builder).build());
                case "ClientLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ClientLimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ClientLimitExceededException::builder).build());
                case "NoDataRetentionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NoDataRetentionException").httpStatusCode(400)
                            .exceptionBuilderSupplier(NoDataRetentionException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedStreamMediaTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).build());
                case "InvalidMediaFrameException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMediaFrameException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMediaFrameException::builder).build());
                case "MissingCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetClipRequest, GetClipResponse>().withOperationName("GetClip")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetClipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncResponseTransformer(asyncResponseTransformer).withInput(getClipRequest),
                    asyncResponseTransformer);
            AsyncResponseTransformer<GetClipResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetClipResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an MPEG Dynamic Adaptive Streaming over HTTP (DASH) URL for the stream. You can then open the URL in a
     * media player to view the stream contents.
     * </p>
     * <p>
     * Both the <code>StreamName</code> and the <code>StreamARN</code> parameters are optional, but you must specify
     * either the <code>StreamName</code> or the <code>StreamARN</code> when invoking this API operation.
     * </p>
     * <p>
     * An Amazon Kinesis video stream has the following requirements for providing data through MPEG-DASH:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The media must contain h.264 or h.265 encoded video and, optionally, AAC or G.711 encoded audio. Specifically,
     * the codec ID of track 1 should be <code>V_MPEG/ISO/AVC</code> (for h.264) or V_MPEGH/ISO/HEVC (for H.265).
     * Optionally, the codec ID of track 2 should be <code>A_AAC</code> (for AAC) or A_MS/ACM (for G.711).
     * </p>
     * </li>
     * <li>
     * <p>
     * Data retention must be greater than 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * The video track of each fragment must contain codec private data in the Advanced Video Coding (AVC) for H.264
     * format and HEVC for H.265 format. For more information, see <a
     * href="https://www.iso.org/standard/55980.html">MPEG-4 specification ISO/IEC 14496-15</a>. For information about
     * adapting stream data to a given format, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/producer-reference-nal.html">NAL Adaptation
     * Flags</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The audio track (if present) of each fragment must contain codec private data in the AAC format (<a
     * href="https://www.iso.org/standard/43345.html">AAC specification ISO/IEC 13818-7</a>) or the <a
     * href="http://www-mmsp.ece.mcgill.ca/Documents/AudioFormats/WAVE/WAVE.html">MS Wave format</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following procedure shows how to use MPEG-DASH with Kinesis Video Streams:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Get an endpoint using <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html">GetDataEndpoint</a>,
     * specifying <code>GET_DASH_STREAMING_SESSION_URL</code> for the <code>APIName</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Retrieve the MPEG-DASH URL using <code>GetDASHStreamingSessionURL</code>. Kinesis Video Streams creates an
     * MPEG-DASH streaming session to be used for accessing content in a stream using the MPEG-DASH protocol.
     * <code>GetDASHStreamingSessionURL</code> returns an authenticated URL (that includes an encrypted session token)
     * for the session's MPEG-DASH <i>manifest</i> (the root resource needed for streaming with MPEG-DASH).
     * </p>
     * <note>
     * <p>
     * Don't share or store this token where an unauthorized entity can access it. The token provides access to the
     * content of the stream. Safeguard the token with the same measures that you use with your Amazon Web Services
     * credentials.
     * </p>
     * </note>
     * <p>
     * The media that is made available through the manifest consists only of the requested stream, time range, and
     * format. No other media data (such as frames outside the requested window or alternate bitrates) is made
     * available.
     * </p>
     * </li>
     * <li>
     * <p>
     * Provide the URL (containing the encrypted session token) for the MPEG-DASH manifest to a media player that
     * supports the MPEG-DASH protocol. Kinesis Video Streams makes the initialization fragment and media fragments
     * available through the manifest URL. The initialization fragment contains the codec private data for the stream,
     * and other data needed to set up the video or audio decoder and renderer. The media fragments contain encoded
     * video frames or encoded audio samples.
     * </p>
     * </li>
     * <li>
     * <p>
     * The media player receives the authenticated URL and requests stream metadata and media data normally. When the
     * media player requests data, it calls the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>GetDASHManifest:</b> Retrieves an MPEG DASH manifest, which contains the metadata for the media that you want
     * to playback.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4InitFragment:</b> Retrieves the MP4 initialization fragment. The media player typically loads the
     * initialization fragment before loading any media fragments. This fragment contains the "<code>fytp</code>" and "
     * <code>moov</code>" MP4 atoms, and the child atoms that are needed to initialize the media player decoder.
     * </p>
     * <p>
     * The initialization fragment does not correspond to a fragment in a Kinesis video stream. It contains only the
     * codec private data for the stream and respective track, which the media player needs to decode the media frames.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4MediaFragment:</b> Retrieves MP4 media fragments. These fragments contain the "<code>moof</code>" and "
     * <code>mdat</code>" MP4 atoms and their child atoms, containing the encoded fragment's media frames and their
     * timestamps.
     * </p>
     * <note>
     * <p>
     * After the first media fragment is made available in a streaming session, any fragments that don't contain the
     * same codec private data cause an error to be returned when those different media fragments are loaded. Therefore,
     * the codec private data should not change between fragments in a session. This also means that the session fails
     * if the fragments in a stream change from having only video to having both audio and video.
     * </p>
     * </note>
     * <p>
     * Data retrieved with this action is billable. See <a
     * href="https://aws.amazon.com/kinesis/video-streams/pricing/">Pricing</a> for details.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <note>
     * <p>
     * For restrictions that apply to MPEG-DASH sessions, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams Limits</a>.
     * </p>
     * </note>
     * <p>
     * You can monitor the amount of data that the media player consumes by monitoring the
     * <code>GetMP4MediaFragment.OutgoingBytes</code> Amazon CloudWatch metric. For information about using CloudWatch
     * to monitor Kinesis Video Streams, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/monitoring.html">Monitoring Kinesis Video
     * Streams</a>. For pricing information, see <a href="https://aws.amazon.com/kinesis/video-streams/pricing/">Amazon
     * Kinesis Video Streams Pricing</a> and <a href="https://aws.amazon.com/pricing/">Amazon Web Services Pricing</a>.
     * Charges for both HLS sessions and outgoing Amazon Web Services data apply.
     * </p>
     * <p>
     * For more information about HLS, see <a href="https://developer.apple.com/streaming/">HTTP Live Streaming</a> on
     * the <a href="https://developer.apple.com">Apple Developer site</a>.
     * </p>
     * <important>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams archived media API, in addition to the HTTP status
     * code and the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to Amazon Web Services the support team
     * can better diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </important>
     *
     * @param getDashStreamingSessionUrlRequest
     * @return A Java Future containing the result of the GetDASHStreamingSessionURL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetImages</code> will throw this error when Kinesis Video Streams
     *         can't find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> and <code>GetDASHStreamingSessionURL</code> throw this error if a
     *         session with a <code>PlaybackMode</code> of <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>is
     *         requested for a stream that has no fragments within the requested time range, or if a session with a
     *         <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has no fragments within the
     *         last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded a limit. Try making the call later. For information about limits, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *         Limits</a>.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>UnsupportedStreamMediaTypeException The type of the media (for example, h.264 or h.265 video or ACC
     *         or G.711 audio) could not be determined from the codec IDs of the tracks in the first fragment for a
     *         playback session. The codec ID for track 1 should be <code>V_MPEG/ISO/AVC</code> and, optionally, the
     *         codec ID for track 2 should be <code>A_AAC</code>.</li>
     *         <li>NoDataRetentionException <code>GetImages</code> was requested for a stream that does not retain data
     *         (that is, has a <code>DataRetentionInHours</code> of 0).</li>
     *         <li>MissingCodecPrivateDataException No codec private data was found in at least one of tracks of the
     *         video stream.</li>
     *         <li>InvalidCodecPrivateDataException The codec private data in at least one of the tracks of the video
     *         stream is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetDASHStreamingSessionURL
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetDASHStreamingSessionURL"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDashStreamingSessionUrlResponse> getDASHStreamingSessionURL(
            GetDashStreamingSessionUrlRequest getDashStreamingSessionUrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDashStreamingSessionUrlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDashStreamingSessionUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Archived Media");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDASHStreamingSessionURL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDashStreamingSessionUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDashStreamingSessionUrlResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "NotAuthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
                case "InvalidCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).build());
                case "InvalidArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidArgumentException::builder).build());
                case "ClientLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ClientLimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ClientLimitExceededException::builder).build());
                case "NoDataRetentionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NoDataRetentionException").httpStatusCode(400)
                            .exceptionBuilderSupplier(NoDataRetentionException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedStreamMediaTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).build());
                case "InvalidMediaFrameException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMediaFrameException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMediaFrameException::builder).build());
                case "MissingCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDashStreamingSessionUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDashStreamingSessionUrlRequest, GetDashStreamingSessionUrlResponse>()
                            .withOperationName("GetDASHStreamingSessionURL").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDashStreamingSessionUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDashStreamingSessionUrlRequest));
            CompletableFuture<GetDashStreamingSessionUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an HTTP Live Streaming (HLS) URL for the stream. You can then open the URL in a browser or media player
     * to view the stream contents.
     * </p>
     * <p>
     * Both the <code>StreamName</code> and the <code>StreamARN</code> parameters are optional, but you must specify
     * either the <code>StreamName</code> or the <code>StreamARN</code> when invoking this API operation.
     * </p>
     * <p>
     * An Amazon Kinesis video stream has the following requirements for providing data through HLS:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For streaming video, the media must contain H.264 or H.265 encoded video and, optionally, AAC encoded audio.
     * Specifically, the codec ID of track 1 should be <code>V_MPEG/ISO/AVC</code> (for H.264) or
     * <code>V_MPEG/ISO/HEVC</code> (for H.265). Optionally, the codec ID of track 2 should be <code>A_AAC</code>. For
     * audio only streaming, the codec ID of track 1 should be <code>A_AAC</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data retention must be greater than 0.
     * </p>
     * </li>
     * <li>
     * <p>
     * The video track of each fragment must contain codec private data in the Advanced Video Coding (AVC) for H.264
     * format or HEVC for H.265 format (<a href="https://www.iso.org/standard/55980.html">MPEG-4 specification ISO/IEC
     * 14496-15</a>). For information about adapting stream data to a given format, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/producer-reference-nal.html">NAL Adaptation
     * Flags</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The audio track (if present) of each fragment must contain codec private data in the AAC format (<a
     * href="https://www.iso.org/standard/43345.html">AAC specification ISO/IEC 13818-7</a>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Kinesis Video Streams HLS sessions contain fragments in the fragmented MPEG-4 form (also called fMP4 or CMAF) or
     * the MPEG-2 form (also called TS chunks, which the HLS specification also supports). For more information about
     * HLS fragment types, see the <a href="https://tools.ietf.org/html/draft-pantos-http-live-streaming-23">HLS
     * specification</a>.
     * </p>
     * <p>
     * The following procedure shows how to use HLS with Kinesis Video Streams:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Get an endpoint using <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html">GetDataEndpoint</a>,
     * specifying <code>GET_HLS_STREAMING_SESSION_URL</code> for the <code>APIName</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Retrieve the HLS URL using <code>GetHLSStreamingSessionURL</code>. Kinesis Video Streams creates an HLS streaming
     * session to be used for accessing content in a stream using the HLS protocol.
     * <code>GetHLSStreamingSessionURL</code> returns an authenticated URL (that includes an encrypted session token)
     * for the session's HLS <i>master playlist</i> (the root resource needed for streaming with HLS).
     * </p>
     * <note>
     * <p>
     * Don't share or store this token where an unauthorized entity could access it. The token provides access to the
     * content of the stream. Safeguard the token with the same measures that you would use with your Amazon Web
     * Services credentials.
     * </p>
     * </note>
     * <p>
     * The media that is made available through the playlist consists only of the requested stream, time range, and
     * format. No other media data (such as frames outside the requested window or alternate bitrates) is made
     * available.
     * </p>
     * </li>
     * <li>
     * <p>
     * Provide the URL (containing the encrypted session token) for the HLS master playlist to a media player that
     * supports the HLS protocol. Kinesis Video Streams makes the HLS media playlist, initialization fragment, and media
     * fragments available through the master playlist URL. The initialization fragment contains the codec private data
     * for the stream, and other data needed to set up the video or audio decoder and renderer. The media fragments
     * contain H.264-encoded video frames or AAC-encoded audio samples.
     * </p>
     * </li>
     * <li>
     * <p>
     * The media player receives the authenticated URL and requests stream metadata and media data normally. When the
     * media player requests data, it calls the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>GetHLSMasterPlaylist:</b> Retrieves an HLS master playlist, which contains a URL for the
     * <code>GetHLSMediaPlaylist</code> action for each track, and additional metadata for the media player, including
     * estimated bitrate and resolution.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetHLSMediaPlaylist:</b> Retrieves an HLS media playlist, which contains a URL to access the MP4
     * initialization fragment with the <code>GetMP4InitFragment</code> action, and URLs to access the MP4 media
     * fragments with the <code>GetMP4MediaFragment</code> actions. The HLS media playlist also contains metadata about
     * the stream that the player needs to play it, such as whether the <code>PlaybackMode</code> is <code>LIVE</code>
     * or <code>ON_DEMAND</code>. The HLS media playlist is typically static for sessions with a
     * <code>PlaybackType</code> of <code>ON_DEMAND</code>. The HLS media playlist is continually updated with new
     * fragments for sessions with a <code>PlaybackType</code> of <code>LIVE</code>. There is a distinct HLS media
     * playlist for the video track and the audio track (if applicable) that contains MP4 media URLs for the specific
     * track.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4InitFragment:</b> Retrieves the MP4 initialization fragment. The media player typically loads the
     * initialization fragment before loading any media fragments. This fragment contains the "<code>fytp</code>" and "
     * <code>moov</code>" MP4 atoms, and the child atoms that are needed to initialize the media player decoder.
     * </p>
     * <p>
     * The initialization fragment does not correspond to a fragment in a Kinesis video stream. It contains only the
     * codec private data for the stream and respective track, which the media player needs to decode the media frames.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetMP4MediaFragment:</b> Retrieves MP4 media fragments. These fragments contain the "<code>moof</code>" and "
     * <code>mdat</code>" MP4 atoms and their child atoms, containing the encoded fragment's media frames and their
     * timestamps.
     * </p>
     * <note>
     * <p>
     * For the HLS streaming session, in-track codec private data (CPD) changes are supported. After the first media
     * fragment is made available in a streaming session, fragments can contain CPD changes for each track. Therefore,
     * the fragments in a session can have a different resolution, bit rate, or other information in the CPD without
     * interrupting playback. However, any change made in the track number or track codec format can return an error
     * when those different media fragments are loaded. For example, streaming will fail if the fragments in the stream
     * change from having only video to having both audio and video, or if an AAC audio track is changed to an ALAW
     * audio track. For each streaming session, only 500 CPD changes are allowed.
     * </p>
     * </note>
     * <p>
     * Data retrieved with this action is billable. For information, see <a
     * href="https://aws.amazon.com/kinesis/video-streams/pricing/">Pricing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>GetTSFragment:</b> Retrieves MPEG TS fragments containing both initialization and media data for all tracks in
     * the stream.
     * </p>
     * <note>
     * <p>
     * If the <code>ContainerFormat</code> is <code>MPEG_TS</code>, this API is used instead of
     * <code>GetMP4InitFragment</code> and <code>GetMP4MediaFragment</code> to retrieve stream media.
     * </p>
     * </note>
     * <p>
     * Data retrieved with this action is billable. For more information, see <a
     * href="https://aws.amazon.com/kinesis/video-streams/pricing/">Kinesis Video Streams pricing</a>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <p>
     * A streaming session URL must not be shared between players. The service might throttle a session if multiple
     * media players are sharing it. For connection limits, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams Limits</a>.
     * </p>
     * <p>
     * You can monitor the amount of data that the media player consumes by monitoring the
     * <code>GetMP4MediaFragment.OutgoingBytes</code> Amazon CloudWatch metric. For information about using CloudWatch
     * to monitor Kinesis Video Streams, see <a
     * href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/monitoring.html">Monitoring Kinesis Video
     * Streams</a>. For pricing information, see <a href="https://aws.amazon.com/kinesis/video-streams/pricing/">Amazon
     * Kinesis Video Streams Pricing</a> and <a href="https://aws.amazon.com/pricing/">Amazon Web Services Pricing</a>.
     * Charges for both HLS sessions and outgoing Amazon Web Services data apply.
     * </p>
     * <p>
     * For more information about HLS, see <a href="https://developer.apple.com/streaming/">HTTP Live Streaming</a> on
     * the <a href="https://developer.apple.com">Apple Developer site</a>.
     * </p>
     * <important>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams archived media API, in addition to the HTTP status
     * code and the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to Amazon Web Services, the support
     * team can better diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </important>
     *
     * @param getHlsStreamingSessionUrlRequest
     * @return A Java Future containing the result of the GetHLSStreamingSessionURL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetImages</code> will throw this error when Kinesis Video Streams
     *         can't find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> and <code>GetDASHStreamingSessionURL</code> throw this error if a
     *         session with a <code>PlaybackMode</code> of <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>is
     *         requested for a stream that has no fragments within the requested time range, or if a session with a
     *         <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has no fragments within the
     *         last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded a limit. Try making the call later. For information about limits, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *         Limits</a>.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>UnsupportedStreamMediaTypeException The type of the media (for example, h.264 or h.265 video or ACC
     *         or G.711 audio) could not be determined from the codec IDs of the tracks in the first fragment for a
     *         playback session. The codec ID for track 1 should be <code>V_MPEG/ISO/AVC</code> and, optionally, the
     *         codec ID for track 2 should be <code>A_AAC</code>.</li>
     *         <li>NoDataRetentionException <code>GetImages</code> was requested for a stream that does not retain data
     *         (that is, has a <code>DataRetentionInHours</code> of 0).</li>
     *         <li>MissingCodecPrivateDataException No codec private data was found in at least one of tracks of the
     *         video stream.</li>
     *         <li>InvalidCodecPrivateDataException The codec private data in at least one of the tracks of the video
     *         stream is not valid for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetHLSStreamingSessionURL
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetHLSStreamingSessionURL"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHlsStreamingSessionUrlResponse> getHLSStreamingSessionURL(
            GetHlsStreamingSessionUrlRequest getHlsStreamingSessionUrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHlsStreamingSessionUrlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHlsStreamingSessionUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Archived Media");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHLSStreamingSessionURL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHlsStreamingSessionUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetHlsStreamingSessionUrlResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "NotAuthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
                case "InvalidCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).build());
                case "InvalidArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidArgumentException::builder).build());
                case "ClientLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ClientLimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ClientLimitExceededException::builder).build());
                case "NoDataRetentionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NoDataRetentionException").httpStatusCode(400)
                            .exceptionBuilderSupplier(NoDataRetentionException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedStreamMediaTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).build());
                case "InvalidMediaFrameException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMediaFrameException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMediaFrameException::builder).build());
                case "MissingCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetHlsStreamingSessionUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHlsStreamingSessionUrlRequest, GetHlsStreamingSessionUrlResponse>()
                            .withOperationName("GetHLSStreamingSessionURL").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetHlsStreamingSessionUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHlsStreamingSessionUrlRequest));
            CompletableFuture<GetHlsStreamingSessionUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of images corresponding to each timestamp for a given time range, sampling interval, and image
     * format configuration.
     * </p>
     *
     * @param getImagesRequest
     * @return A Java Future containing the result of the GetImages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetImages</code> will throw this error when Kinesis Video Streams
     *         can't find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> and <code>GetDASHStreamingSessionURL</code> throw this error if a
     *         session with a <code>PlaybackMode</code> of <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>is
     *         requested for a stream that has no fragments within the requested time range, or if a session with a
     *         <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has no fragments within the
     *         last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded a limit. Try making the call later. For information about limits, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *         Limits</a>.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>NoDataRetentionException <code>GetImages</code> was requested for a stream that does not retain data
     *         (that is, has a <code>DataRetentionInHours</code> of 0).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetImages"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetImagesResponse> getImages(GetImagesRequest getImagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Archived Media");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetImagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetImagesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "NotAuthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
                case "InvalidCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).build());
                case "InvalidArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidArgumentException::builder).build());
                case "ClientLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ClientLimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ClientLimitExceededException::builder).build());
                case "NoDataRetentionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NoDataRetentionException").httpStatusCode(400)
                            .exceptionBuilderSupplier(NoDataRetentionException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedStreamMediaTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).build());
                case "InvalidMediaFrameException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMediaFrameException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMediaFrameException::builder).build());
                case "MissingCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetImagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetImagesRequest, GetImagesResponse>().withOperationName("GetImages")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetImagesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getImagesRequest));
            CompletableFuture<GetImagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets media for a list of fragments (specified by fragment number) from the archived data in an Amazon Kinesis
     * video stream.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the
     * <code>GetMediaForFragmentList</code> requests to this endpoint using the <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url parameter</a>.
     * </p>
     * </note>
     * <p>
     * For limits, see <a href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video
     * Streams Limits</a>.
     * </p>
     * <important>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams archived media API, in addition to the HTTP status
     * code and the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to Amazon Web Services, the support
     * team can better diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </important>
     *
     * @param getMediaForFragmentListRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The payload that Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see <a
     *        href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>.
     *        The chunks that Kinesis Video Streams returns in the <code>GetMediaForFragmentList</code> call also
     *        include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_SIDE_TIMESTAMP - Server-side timestamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_SIDE_TIMESTAMP - Producer-side timestamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be included if an exception occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - The number of the fragment that threw the exception
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_ERROR_CODE - The integer code of the
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_EXCEPTION_MESSAGE - A text description of the exception
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetImages</code> will throw this error when Kinesis Video Streams
     *         can't find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> and <code>GetDASHStreamingSessionURL</code> throw this error if a
     *         session with a <code>PlaybackMode</code> of <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>is
     *         requested for a stream that has no fragments within the requested time range, or if a session with a
     *         <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has no fragments within the
     *         last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded a limit. Try making the call later. For information about limits, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *         Limits</a>.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.GetMediaForFragmentList
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/GetMediaForFragmentList"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getMediaForFragmentList(
            GetMediaForFragmentListRequest getMediaForFragmentListRequest,
            AsyncResponseTransformer<GetMediaForFragmentListResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMediaForFragmentListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMediaForFragmentListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Archived Media");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMediaForFragmentList");
            Pair<AsyncResponseTransformer<GetMediaForFragmentListResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMediaForFragmentListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMediaForFragmentListResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "NotAuthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
                case "InvalidCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).build());
                case "InvalidArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidArgumentException::builder).build());
                case "ClientLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ClientLimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ClientLimitExceededException::builder).build());
                case "NoDataRetentionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NoDataRetentionException").httpStatusCode(400)
                            .exceptionBuilderSupplier(NoDataRetentionException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedStreamMediaTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).build());
                case "InvalidMediaFrameException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMediaFrameException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMediaFrameException::builder).build());
                case "MissingCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetMediaForFragmentListRequest, GetMediaForFragmentListResponse>()
                            .withOperationName("GetMediaForFragmentList").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMediaForFragmentListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncResponseTransformer(asyncResponseTransformer).withInput(getMediaForFragmentListRequest),
                    asyncResponseTransformer);
            AsyncResponseTransformer<GetMediaForFragmentListResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetMediaForFragmentListResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a>Fragment</a> objects from the specified stream and timestamp range within the archived data.
     * </p>
     * <p>
     * Listing fragments is eventually consistent. This means that even if the producer receives an acknowledgment that
     * a fragment is persisted, the result might not be returned immediately from a request to
     * <code>ListFragments</code>. However, results are typically available in less than one second.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the
     * <code>ListFragments</code> requests to this endpoint using the <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url parameter</a>.
     * </p>
     * </note> <important>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams archived media API, in addition to the HTTP status
     * code and the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to Amazon Web Services, the support
     * team can better diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </important>
     *
     * @param listFragmentsRequest
     * @return A Java Future containing the result of the ListFragments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException <code>GetImages</code> will throw this error when Kinesis Video Streams
     *         can't find the stream that you specified.</p>
     *         <p>
     *         <code>GetHLSStreamingSessionURL</code> and <code>GetDASHStreamingSessionURL</code> throw this error if a
     *         session with a <code>PlaybackMode</code> of <code>ON_DEMAND</code> or <code>LIVE_REPLAY</code>is
     *         requested for a stream that has no fragments within the requested time range, or if a session with a
     *         <code>PlaybackMode</code> of <code>LIVE</code> is requested for a stream that has no fragments within the
     *         last 30 seconds.</li>
     *         <li>InvalidArgumentException A specified parameter exceeds its restrictions, is not supported, or can't
     *         be used.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded a limit. Try making the call later. For information about limits, see <a
     *         href="http://docs.aws.amazon.com/kinesisvideostreams/latest/dg/limits.html">Kinesis Video Streams
     *         Limits</a>.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoArchivedMediaException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoArchivedMediaAsyncClient.ListFragments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kinesis-video-archived-media-2017-09-30/ListFragments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFragmentsResponse> listFragments(ListFragmentsRequest listFragmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFragmentsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFragmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Archived Media");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFragments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFragmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFragmentsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "NotAuthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
                case "InvalidCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidCodecPrivateDataException::builder).build());
                case "InvalidArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidArgumentException::builder).build());
                case "ClientLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ClientLimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ClientLimitExceededException::builder).build());
                case "NoDataRetentionException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NoDataRetentionException").httpStatusCode(400)
                            .exceptionBuilderSupplier(NoDataRetentionException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedStreamMediaTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedStreamMediaTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedStreamMediaTypeException::builder).build());
                case "InvalidMediaFrameException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMediaFrameException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMediaFrameException::builder).build());
                case "MissingCodecPrivateDataException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MissingCodecPrivateDataException")
                            .httpStatusCode(400).exceptionBuilderSupplier(MissingCodecPrivateDataException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListFragmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFragmentsRequest, ListFragmentsResponse>()
                            .withOperationName("ListFragments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFragmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFragmentsRequest));
            CompletableFuture<ListFragmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final KinesisVideoArchivedMediaServiceClientConfiguration serviceClientConfiguration() {
        return new KinesisVideoArchivedMediaServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoArchivedMediaException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        KinesisVideoArchivedMediaServiceClientConfigurationBuilder serviceConfigBuilder = new KinesisVideoArchivedMediaServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
