/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideomedia;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideomedia.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideomedia.model.ConnectionLimitExceededException;
import software.amazon.awssdk.services.kinesisvideomedia.model.GetMediaRequest;
import software.amazon.awssdk.services.kinesisvideomedia.model.GetMediaResponse;
import software.amazon.awssdk.services.kinesisvideomedia.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideomedia.model.InvalidEndpointException;
import software.amazon.awssdk.services.kinesisvideomedia.model.KinesisVideoMediaException;
import software.amazon.awssdk.services.kinesisvideomedia.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideomedia.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideomedia.transform.GetMediaRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisVideoMediaAsyncClient}.
 *
 * @see KinesisVideoMediaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoMediaAsyncClient implements KinesisVideoMediaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoMediaAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKinesisVideoMediaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify stream name or
     * stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of chunks
     * in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint to which you can then send the
     * <code>GetMedia</code> requests.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see . The <code>GetMedia</code> API returns a stream
     * of these chunks starting from the chunk that you specify in the request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMediaRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *        <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *        terminates, you can use this continuation token in your next request to get the next chunk where the last
     *        request terminated.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *        determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server time stamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer time stamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be present if an error occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The error codes are as follows:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        3002 - Error writing to the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4000 - Requested fragment is not found
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4500 - Access denied for the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4501 - Stream's KMS key is disabled
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4502 - Validation error on the Stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4503 - KMS key specified in the stream is unavailable
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4504 - Invalid usage of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4505 - Invalid state of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4506 - Unable to find the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        5000 - Internal error
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Status Code: 404, The stream with the given name does not exist.</li>
     *         <li>NotAuthorizedException Status Code: 403, The caller is not authorized to perform an operation on the
     *         given stream, or the token has expired.</li>
     *         <li>InvalidEndpointException Status Code: 400, Caller used wrong endpoint to write data to a stream. On
     *         receiving such an exception, the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code>
     *         set to "READ" and use the endpoint Kinesis Video returns in the next <code>GetMedia</code> call.</li>
     *         <li>ClientLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client calls. Try making the call later.</li>
     *         <li>ConnectionLimitExceededException Kinesis Video Streams has throttled the request because you have
     *         exceeded the limit of allowed client connections.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoMediaException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoMediaAsyncClient.GetMedia
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getMedia(GetMediaRequest getMediaRequest,
            AsyncResponseTransformer<GetMediaResponse, ReturnT> asyncResponseTransformer) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMediaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMediaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(
                    new ClientExecutionParams<GetMediaRequest, GetMediaResponse>()
                            .withMarshaller(new GetMediaRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getMediaRequest), asyncResponseTransformer)
                    .whenComplete((r, e) -> {
                        if (e != null) {
                            asyncResponseTransformer.exceptionOccurred(e);
                        }
                    });
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseTransformer.exceptionOccurred(t));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoMediaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientLimitExceededException")
                                .exceptionBuilderSupplier(ClientLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConnectionLimitExceededException")
                                .exceptionBuilderSupplier(ConnectionLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointException")
                                .exceptionBuilderSupplier(InvalidEndpointException::builder).httpStatusCode(400).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
