/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideomedia;

import java.nio.file.Path;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.kinesisvideomedia.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideomedia.model.ConnectionLimitExceededException;
import software.amazon.awssdk.services.kinesisvideomedia.model.GetMediaRequest;
import software.amazon.awssdk.services.kinesisvideomedia.model.GetMediaResponse;
import software.amazon.awssdk.services.kinesisvideomedia.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideomedia.model.InvalidEndpointException;
import software.amazon.awssdk.services.kinesisvideomedia.model.KinesisVideoMediaException;
import software.amazon.awssdk.services.kinesisvideomedia.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideomedia.model.ResourceNotFoundException;

/**
 * Service client for accessing Kinesis Video Media. This can be created using the static {@link #builder()} method.
 *
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public interface KinesisVideoMediaClient extends SdkClient {
    String SERVICE_NAME = "kinesisvideo";

    /**
     * Create a {@link KinesisVideoMediaClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static KinesisVideoMediaClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link KinesisVideoMediaClient}.
     */
    static KinesisVideoMediaClientBuilder builder() {
        return new DefaultKinesisVideoMediaClientBuilder();
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note>
     *
     * @param getMediaRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetMediaResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *        <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *        terminates, you can use this continuation token in your next request to get the next chunk where the last
     *        request terminated.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *        determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be present if an error occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The error codes are as follows:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        3002 - Error writing to the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4000 - Requested fragment is not found
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4500 - Access denied for the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4501 - Stream's KMS key is disabled
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4502 - Validation error on the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4503 - KMS key specified in the stream is unavailable
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4504 - Invalid usage of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4505 - Invalid state of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4506 - Unable to find the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        5000 - Internal error
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default <ReturnT> ReturnT getMedia(GetMediaRequest getMediaRequest,
            ResponseTransformer<GetMediaResponse, ReturnT> responseTransformer) throws ResourceNotFoundException,
            NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException, ConnectionLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoMediaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaRequest.Builder} avoiding the need to
     * create one manually via {@link GetMediaRequest#builder()}
     * </p>
     *
     * @param getMediaRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaInput.Builder} to create a request.
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetMediaResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *        <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *        terminates, you can use this continuation token in your next request to get the next chunk where the last
     *        request terminated.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *        determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be present if an error occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The error codes are as follows:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        3002 - Error writing to the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4000 - Requested fragment is not found
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4500 - Access denied for the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4501 - Stream's KMS key is disabled
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4502 - Validation error on the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4503 - KMS key specified in the stream is unavailable
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4504 - Invalid usage of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4505 - Invalid state of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4506 - Unable to find the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        5000 - Internal error
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default <ReturnT> ReturnT getMedia(Consumer<GetMediaRequest.Builder> getMediaRequest,
            ResponseTransformer<GetMediaResponse, ReturnT> responseTransformer) throws ResourceNotFoundException,
            NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException, ConnectionLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoMediaException {
        return getMedia(GetMediaRequest.builder().applyMutation(getMediaRequest).build(), responseTransformer);
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note>
     *
     * @param getMediaRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *        <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *        terminates, you can use this continuation token in your next request to get the next chunk where the last
     *        request terminated.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *        determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be present if an error occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The error codes are as follows:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        3002 - Error writing to the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4000 - Requested fragment is not found
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4500 - Access denied for the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4501 - Stream's KMS key is disabled
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4502 - Validation error on the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4503 - KMS key specified in the stream is unavailable
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4504 - Invalid usage of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4505 - Invalid state of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4506 - Unable to find the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        5000 - Internal error
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see #getMedia(GetMediaRequest, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default GetMediaResponse getMedia(GetMediaRequest getMediaRequest, Path filePath) throws ResourceNotFoundException,
            NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException, ConnectionLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoMediaException {
        return getMedia(getMediaRequest, ResponseTransformer.toFile(filePath));
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaRequest.Builder} avoiding the need to
     * create one manually via {@link GetMediaRequest#builder()}
     * </p>
     *
     * @param getMediaRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaInput.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *        information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *        <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *        terminates, you can use this continuation token in your next request to get the next chunk where the last
     *        request terminated.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *        determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The following tags will be present if an error occurs:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *        </p>
     *        </li>
     *        </ul>
     *        <p>
     *        The error codes are as follows:
     *        </p>
     *        <ul>
     *        <li>
     *        <p>
     *        3002 - Error writing to the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4000 - Requested fragment is not found
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4500 - Access denied for the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4501 - Stream's KMS key is disabled
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4502 - Validation error on the stream's KMS key
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4503 - KMS key specified in the stream is unavailable
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4504 - Invalid usage of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4505 - Invalid state of the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        4506 - Unable to find the KMS key specified in the stream
     *        </p>
     *        </li>
     *        <li>
     *        <p>
     *        5000 - Internal error
     *        </p>
     *        </li>
     *        </ul>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see #getMedia(GetMediaRequest, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default GetMediaResponse getMedia(Consumer<GetMediaRequest.Builder> getMediaRequest, Path filePath)
            throws ResourceNotFoundException, NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException,
            ConnectionLimitExceededException, InvalidArgumentException, AwsServiceException, SdkClientException,
            KinesisVideoMediaException {
        return getMedia(GetMediaRequest.builder().applyMutation(getMediaRequest).build(), filePath);
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note>
     *
     * @param getMediaRequest
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *         <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *         terminates, you can use this continuation token in your next request to get the next chunk where the last
     *         request terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *         determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be present if an error occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The error codes are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         3002 - Error writing to the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4000 - Requested fragment is not found
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4500 - Access denied for the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4501 - Stream's KMS key is disabled
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4502 - Validation error on the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4503 - KMS key specified in the stream is unavailable
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4504 - Invalid usage of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4505 - Invalid state of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4506 - Unable to find the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         5000 - Internal error
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see #getObject(getMedia, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseInputStream<GetMediaResponse> getMedia(GetMediaRequest getMediaRequest) throws ResourceNotFoundException,
            NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException, ConnectionLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoMediaException {
        return getMedia(getMediaRequest, ResponseTransformer.toInputStream());
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaRequest.Builder} avoiding the need to
     * create one manually via {@link GetMediaRequest#builder()}
     * </p>
     *
     * @param getMediaRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaInput.Builder} to create a request.
     * @return A {@link ResponseInputStream} containing data streamed from service. Note that this is an unmanaged
     *         reference to the underlying HTTP connection so great care must be taken to ensure all data if fully read
     *         from the input stream and that it is properly closed. Failure to do so may result in sub-optimal behavior
     *         and exhausting connections in the connection pool. The unmarshalled response object can be obtained via
     *         {@link ResponseInputStream#response()}. The service documentation for the response content is as follows
     *         '
     *         <p>
     *         The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *         <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *         terminates, you can use this continuation token in your next request to get the next chunk where the last
     *         request terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *         determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be present if an error occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The error codes are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         3002 - Error writing to the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4000 - Requested fragment is not found
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4500 - Access denied for the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4501 - Stream's KMS key is disabled
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4502 - Validation error on the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4503 - KMS key specified in the stream is unavailable
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4504 - Invalid usage of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4505 - Invalid state of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4506 - Unable to find the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         5000 - Internal error
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see #getObject(getMedia, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseInputStream<GetMediaResponse> getMedia(Consumer<GetMediaRequest.Builder> getMediaRequest)
            throws ResourceNotFoundException, NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException,
            ConnectionLimitExceededException, InvalidArgumentException, AwsServiceException, SdkClientException,
            KinesisVideoMediaException {
        return getMedia(GetMediaRequest.builder().applyMutation(getMediaRequest).build());
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note>
     *
     * @param getMediaRequest
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *         <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *         terminates, you can use this continuation token in your next request to get the next chunk where the last
     *         request terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *         determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be present if an error occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The error codes are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         3002 - Error writing to the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4000 - Requested fragment is not found
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4500 - Access denied for the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4501 - Stream's KMS key is disabled
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4502 - Validation error on the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4503 - KMS key specified in the stream is unavailable
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4504 - Invalid usage of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4505 - Invalid state of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4506 - Unable to find the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         5000 - Internal error
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see #getObject(getMedia, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseBytes<GetMediaResponse> getMediaAsBytes(GetMediaRequest getMediaRequest) throws ResourceNotFoundException,
            NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException, ConnectionLimitExceededException,
            InvalidArgumentException, AwsServiceException, SdkClientException, KinesisVideoMediaException {
        return getMedia(getMediaRequest, ResponseTransformer.toBytes());
    }

    /**
     * <p>
     * Use this API to retrieve media content from a Kinesis video stream. In the request, you identify the stream name
     * or stream Amazon Resource Name (ARN), and the starting chunk. Kinesis Video Streams then returns a stream of
     * chunks in order by fragment number.
     * </p>
     * <note>
     * <p>
     * You must first call the <code>GetDataEndpoint</code> API to get an endpoint. Then send the <code>GetMedia</code>
     * requests to this endpoint using the <a href="https://docs.aws.amazon.com/cli/latest/reference/">--endpoint-url
     * parameter</a>.
     * </p>
     * </note>
     * <p>
     * When you put media data (fragments) on a stream, Kinesis Video Streams stores each incoming fragment and related
     * metadata in what is called a "chunk." For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_dataplane_PutMedia.html">PutMedia</a>. The
     * <code>GetMedia</code> API returns a stream of these chunks starting from the chunk that you specify in the
     * request.
     * </p>
     * <p>
     * The following limits apply when using the <code>GetMedia</code> API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A client can call <code>GetMedia</code> up to five times per second per stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis Video Streams sends media data at a rate of up to 25 megabytes per second (or 200 megabits per second)
     * during a <code>GetMedia</code> session.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an error is thrown after invoking a Kinesis Video Streams media API, in addition to the HTTP status code and
     * the response body, it includes the following pieces of information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-ErrorType</code> HTTP header – contains a more specific error type in addition to what the HTTP
     * status code provides.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-RequestId</code> HTTP header – if you want to report an issue to AWS, the support team can better
     * diagnose the problem if given the Request Id.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Both the HTTP status code and the ErrorType header can be utilized to make programmatic decisions about whether
     * errors are retry-able and under what conditions, as well as provide information on what actions the client
     * programmer might need to take in order to successfully try again.
     * </p>
     * <p>
     * For more information, see the <b>Errors</b> section at the bottom of this topic, as well as <a
     * href="https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/CommonErrors.html">Common Errors</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMediaRequest.Builder} avoiding the need to
     * create one manually via {@link GetMediaRequest#builder()}
     * </p>
     *
     * @param getMediaRequest
     *        A {@link Consumer} that will call methods on {@link GetMediaInput.Builder} to create a request.
     * @return A {@link ResponseBytes} that loads the data streamed from the service into memory and exposes it in
     *         convenient in-memory representations like a byte buffer or string. The unmarshalled response object can
     *         be obtained via {@link ResponseBytes#response()}. The service documentation for the response content is
     *         as follows '
     *         <p>
     *         The payload Kinesis Video Streams returns is a sequence of chunks from the specified stream. For
     *         information about the chunks, see . The chunks that Kinesis Video Streams returns in the
     *         <code>GetMedia</code> call also include the following additional Matroska (MKV) tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_CONTINUATION_TOKEN (UTF-8 string) - In the event your <code>GetMedia</code> call
     *         terminates, you can use this continuation token in your next request to get the next chunk where the last
     *         request terminated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_MILLIS_BEHIND_NOW (UTF-8 string) - Client applications can use this tag value to
     *         determine how far behind the chunk returned in the response is from the latest chunk on the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_FRAGMENT_NUMBER - Fragment number returned in the chunk.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_SERVER_TIMESTAMP - Server timestamp of the fragment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_PRODUCER_TIMESTAMP - Producer timestamp of the fragment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following tags will be present if an error occurs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_CODE - String description of an error that caused GetMedia to stop.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS_KINESISVIDEO_ERROR_ID: Integer code of the error.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The error codes are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         3002 - Error writing to the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4000 - Requested fragment is not found
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4500 - Access denied for the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4501 - Stream's KMS key is disabled
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4502 - Validation error on the stream's KMS key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4503 - KMS key specified in the stream is unavailable
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4504 - Invalid usage of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4505 - Invalid state of the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4506 - Unable to find the KMS key specified in the stream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         5000 - Internal error
     *         </p>
     *         </li>
     *         </ul>
     *         '.
     * @throws ResourceNotFoundException
     *         Status Code: 404, The stream with the given name does not exist.
     * @throws NotAuthorizedException
     *         Status Code: 403, The caller is not authorized to perform an operation on the given stream, or the token
     *         has expired.
     * @throws InvalidEndpointException
     *         Status Code: 400, Caller used wrong endpoint to write data to a stream. On receiving such an exception,
     *         the user must call <code>GetDataEndpoint</code> with <code>AccessMode</code> set to "READ" and use the
     *         endpoint Kinesis Video returns in the next <code>GetMedia</code> call.
     * @throws ClientLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         calls. Try making the call later.
     * @throws ConnectionLimitExceededException
     *         Kinesis Video Streams has throttled the request because you have exceeded the limit of allowed client
     *         connections.
     * @throws InvalidArgumentException
     *         The value for this input parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KinesisVideoMediaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KinesisVideoMediaClient.GetMedia
     * @see #getObject(getMedia, ResponseTransformer)
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-media-2017-09-30/GetMedia" target="_top">AWS
     *      API Documentation</a>
     */
    default ResponseBytes<GetMediaResponse> getMediaAsBytes(Consumer<GetMediaRequest.Builder> getMediaRequest)
            throws ResourceNotFoundException, NotAuthorizedException, InvalidEndpointException, ClientLimitExceededException,
            ConnectionLimitExceededException, InvalidArgumentException, AwsServiceException, SdkClientException,
            KinesisVideoMediaException {
        return getMediaAsBytes(GetMediaRequest.builder().applyMutation(getMediaRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("kinesisvideo");
    }
}
