/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisvideosignaling;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kinesisvideosignaling.model.ClientLimitExceededException;
import software.amazon.awssdk.services.kinesisvideosignaling.model.GetIceServerConfigRequest;
import software.amazon.awssdk.services.kinesisvideosignaling.model.GetIceServerConfigResponse;
import software.amazon.awssdk.services.kinesisvideosignaling.model.InvalidArgumentException;
import software.amazon.awssdk.services.kinesisvideosignaling.model.InvalidClientException;
import software.amazon.awssdk.services.kinesisvideosignaling.model.KinesisVideoSignalingException;
import software.amazon.awssdk.services.kinesisvideosignaling.model.NotAuthorizedException;
import software.amazon.awssdk.services.kinesisvideosignaling.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kinesisvideosignaling.model.SendAlexaOfferToMasterRequest;
import software.amazon.awssdk.services.kinesisvideosignaling.model.SendAlexaOfferToMasterResponse;
import software.amazon.awssdk.services.kinesisvideosignaling.model.SessionExpiredException;
import software.amazon.awssdk.services.kinesisvideosignaling.transform.GetIceServerConfigRequestMarshaller;
import software.amazon.awssdk.services.kinesisvideosignaling.transform.SendAlexaOfferToMasterRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KinesisVideoSignalingAsyncClient}.
 *
 * @see KinesisVideoSignalingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKinesisVideoSignalingAsyncClient implements KinesisVideoSignalingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKinesisVideoSignalingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKinesisVideoSignalingAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Gets the Interactive Connectivity Establishment (ICE) server configuration information, including URIs, username,
     * and password which can be used to configure the WebRTC connection. The ICE component uses this configuration
     * information to setup the WebRTC connection, including authenticating with the Traversal Using Relays around NAT
     * (TURN) relay server.
     * </p>
     * <p>
     * TURN is a protocol that is used to improve the connectivity of peer-to-peer applications. By providing a
     * cloud-based relay service, TURN ensures that a connection can be established even when one or more peers are
     * incapable of a direct peer-to-peer connection. For more information, see <a
     * href="https://tools.ietf.org/html/draft-uberti-rtcweb-turn-rest-00">A REST API For Access To TURN Services</a>.
     * </p>
     * <p>
     * You can invoke this API to establish a fallback mechanism in case either of the peers is unable to establish a
     * direct peer-to-peer connection over a signaling channel. You must specify either a signaling channel ARN or the
     * client ID in order to invoke this API.
     * </p>
     *
     * @param getIceServerConfigRequest
     * @return A Java Future containing the result of the GetIceServerConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidClientException The specified client is invalid.</li>
     *         <li>SessionExpiredException If the client session is expired. Once the client is connected, the session
     *         is valid for 45 minutes. Client should reconnect to the channel to continue sending/receiving messages.</li>
     *         <li>ClientLimitExceededException Your request was throttled because you have exceeded the limit of
     *         allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException The specified resource is not found.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoSignalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoSignalingAsyncClient.GetIceServerConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-signaling-2019-12-04/GetIceServerConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIceServerConfigResponse> getIceServerConfig(GetIceServerConfigRequest getIceServerConfigRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Signaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIceServerConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIceServerConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIceServerConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIceServerConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIceServerConfigRequest, GetIceServerConfigResponse>()
                            .withOperationName("GetIceServerConfig")
                            .withMarshaller(new GetIceServerConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIceServerConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIceServerConfigRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetIceServerConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIceServerConfigRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API allows you to connect WebRTC-enabled devices with Alexa display devices. When invoked, it sends the
     * Alexa Session Description Protocol (SDP) offer to the master peer. The offer is delivered as soon as the master
     * is connected to the specified signaling channel. This API returns the SDP answer from the connected master. If
     * the master is not connected to the signaling channel, redelivery requests are made until the message expires.
     * </p>
     *
     * @param sendAlexaOfferToMasterRequest
     * @return A Java Future containing the result of the SendAlexaOfferToMaster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientLimitExceededException Your request was throttled because you have exceeded the limit of
     *         allowed client calls. Try making the call later.</li>
     *         <li>ResourceNotFoundException The specified resource is not found.</li>
     *         <li>InvalidArgumentException The value for this input parameter is invalid.</li>
     *         <li>NotAuthorizedException The caller is not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KinesisVideoSignalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample KinesisVideoSignalingAsyncClient.SendAlexaOfferToMaster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-video-signaling-2019-12-04/SendAlexaOfferToMaster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendAlexaOfferToMasterResponse> sendAlexaOfferToMaster(
            SendAlexaOfferToMasterRequest sendAlexaOfferToMasterRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kinesis Video Signaling");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendAlexaOfferToMaster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendAlexaOfferToMasterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendAlexaOfferToMasterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendAlexaOfferToMasterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendAlexaOfferToMasterRequest, SendAlexaOfferToMasterResponse>()
                            .withOperationName("SendAlexaOfferToMaster")
                            .withMarshaller(new SendAlexaOfferToMasterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendAlexaOfferToMasterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = sendAlexaOfferToMasterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<SendAlexaOfferToMasterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendAlexaOfferToMasterRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KinesisVideoSignalingException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientLimitExceededException")
                                .exceptionBuilderSupplier(ClientLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientException")
                                .exceptionBuilderSupplier(InvalidClientException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SessionExpiredException")
                                .exceptionBuilderSupplier(SessionExpiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
