/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about each custom key store in the custom key store list.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomKeyStoresListEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomKeyStoresListEntry.Builder, CustomKeyStoresListEntry> {
    private static final SdkField<String> CUSTOM_KEY_STORE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomKeyStoresListEntry::customKeyStoreId)).setter(setter(Builder::customKeyStoreId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomKeyStoreId").build()).build();

    private static final SdkField<String> CUSTOM_KEY_STORE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomKeyStoresListEntry::customKeyStoreName)).setter(setter(Builder::customKeyStoreName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomKeyStoreName").build())
            .build();

    private static final SdkField<String> CLOUD_HSM_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomKeyStoresListEntry::cloudHsmClusterId)).setter(setter(Builder::cloudHsmClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudHsmClusterId").build()).build();

    private static final SdkField<String> TRUST_ANCHOR_CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomKeyStoresListEntry::trustAnchorCertificate)).setter(setter(Builder::trustAnchorCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrustAnchorCertificate").build())
            .build();

    private static final SdkField<String> CONNECTION_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomKeyStoresListEntry::connectionStateAsString)).setter(setter(Builder::connectionState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionState").build()).build();

    private static final SdkField<String> CONNECTION_ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CustomKeyStoresListEntry::connectionErrorCodeAsString)).setter(setter(Builder::connectionErrorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionErrorCode").build())
            .build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(CustomKeyStoresListEntry::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CUSTOM_KEY_STORE_ID_FIELD,
            CUSTOM_KEY_STORE_NAME_FIELD, CLOUD_HSM_CLUSTER_ID_FIELD, TRUST_ANCHOR_CERTIFICATE_FIELD, CONNECTION_STATE_FIELD,
            CONNECTION_ERROR_CODE_FIELD, CREATION_DATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String customKeyStoreId;

    private final String customKeyStoreName;

    private final String cloudHsmClusterId;

    private final String trustAnchorCertificate;

    private final String connectionState;

    private final String connectionErrorCode;

    private final Instant creationDate;

    private CustomKeyStoresListEntry(BuilderImpl builder) {
        this.customKeyStoreId = builder.customKeyStoreId;
        this.customKeyStoreName = builder.customKeyStoreName;
        this.cloudHsmClusterId = builder.cloudHsmClusterId;
        this.trustAnchorCertificate = builder.trustAnchorCertificate;
        this.connectionState = builder.connectionState;
        this.connectionErrorCode = builder.connectionErrorCode;
        this.creationDate = builder.creationDate;
    }

    /**
     * <p>
     * A unique identifier for the custom key store.
     * </p>
     * 
     * @return A unique identifier for the custom key store.
     */
    public String customKeyStoreId() {
        return customKeyStoreId;
    }

    /**
     * <p>
     * The user-specified friendly name for the custom key store.
     * </p>
     * 
     * @return The user-specified friendly name for the custom key store.
     */
    public String customKeyStoreName() {
        return customKeyStoreName;
    }

    /**
     * <p>
     * A unique identifier for the AWS CloudHSM cluster that is associated with the custom key store.
     * </p>
     * 
     * @return A unique identifier for the AWS CloudHSM cluster that is associated with the custom key store.
     */
    public String cloudHsmClusterId() {
        return cloudHsmClusterId;
    }

    /**
     * <p>
     * The trust anchor certificate of the associated AWS CloudHSM cluster. When you <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/initialize-cluster.html#sign-csr">initialize the
     * cluster</a>, you create this certificate and save it in the <code>customerCA.crt</code> file.
     * </p>
     * 
     * @return The trust anchor certificate of the associated AWS CloudHSM cluster. When you <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/initialize-cluster.html#sign-csr">initialize
     *         the cluster</a>, you create this certificate and save it in the <code>customerCA.crt</code> file.
     */
    public String trustAnchorCertificate() {
        return trustAnchorCertificate;
    }

    /**
     * <p>
     * Indicates whether the custom key store is connected to its AWS CloudHSM cluster.
     * </p>
     * <p>
     * You can create and use CMKs in your custom key stores only when its connection state is <code>CONNECTED</code>.
     * </p>
     * <p>
     * The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
     * <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but you are
     * having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster is active and
     * contains at least one active HSM.
     * </p>
     * <p>
     * A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
     * <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help resolving a
     * connection failure, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom Key
     * Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionState}
     * will return {@link ConnectionStateType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #connectionStateAsString}.
     * </p>
     * 
     * @return Indicates whether the custom key store is connected to its AWS CloudHSM cluster.</p>
     *         <p>
     *         You can create and use CMKs in your custom key stores only when its connection state is
     *         <code>CONNECTED</code>.
     *         </p>
     *         <p>
     *         The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
     *         <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but
     *         you are having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster is
     *         active and contains at least one active HSM.
     *         </p>
     *         <p>
     *         A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
     *         <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help
     *         resolving a connection failure, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom
     *         Key Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @see ConnectionStateType
     */
    public ConnectionStateType connectionState() {
        return ConnectionStateType.fromValue(connectionState);
    }

    /**
     * <p>
     * Indicates whether the custom key store is connected to its AWS CloudHSM cluster.
     * </p>
     * <p>
     * You can create and use CMKs in your custom key stores only when its connection state is <code>CONNECTED</code>.
     * </p>
     * <p>
     * The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
     * <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but you are
     * having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster is active and
     * contains at least one active HSM.
     * </p>
     * <p>
     * A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
     * <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help resolving a
     * connection failure, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom Key
     * Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionState}
     * will return {@link ConnectionStateType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #connectionStateAsString}.
     * </p>
     * 
     * @return Indicates whether the custom key store is connected to its AWS CloudHSM cluster.</p>
     *         <p>
     *         You can create and use CMKs in your custom key stores only when its connection state is
     *         <code>CONNECTED</code>.
     *         </p>
     *         <p>
     *         The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
     *         <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but
     *         you are having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster is
     *         active and contains at least one active HSM.
     *         </p>
     *         <p>
     *         A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
     *         <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help
     *         resolving a connection failure, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom
     *         Key Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @see ConnectionStateType
     */
    public String connectionStateAsString() {
        return connectionState;
    }

    /**
     * <p>
     * Describes the connection error. This field appears in the response only when the <code>ConnectionState</code> is
     * <code>FAILED</code>. For help resolving these errors, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix a
     * Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any active HSMs.
     * To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at least one active HSM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry the request.
     * For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before trying to connect again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the <code>kmsuser</code> crypto
     * user in the AWS CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you
     * must change the <code>kmsuser</code> account password and update the key store password value for the custom key
     * store.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key store.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS KMS cannot
     * find all of the subnets that were configured for the cluster when the custom key store was created, attempts to
     * connect fail. To fix this error, create a cluster from a backup and associate it with your custom key store. This
     * process includes selecting a VPC and subnets. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix a
     * Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS CloudHSM
     * cluster due to too many failed password attempts. Before you can connect your custom key store to its AWS
     * CloudHSM cluster, you must change the <code>kmsuser</code> account password and update the key store password
     * value for the custom key store.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated AWS CloudHSM
     * cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password and logging into the
     * cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must log the
     * <code>kmsuser</code> CU out of the cluster. If you changed the <code>kmsuser</code> password to log into the
     * cluster, you must also and update the key store password value for the custom key store. For help, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to Log Out and
     * Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated AWS
     * CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must create a
     * <code>kmsuser</code> CU account in the cluster, and then update the key store password value for the custom key
     * store.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #connectionErrorCode} will return {@link ConnectionErrorCodeType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #connectionErrorCodeAsString}.
     * </p>
     * 
     * @return Describes the connection error. This field appears in the response only when the
     *         <code>ConnectionState</code> is <code>FAILED</code>. For help resolving these errors, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to
     *         Fix a Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.</p>
     *         <p>
     *         Valid values are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster
     *         ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any active
     *         HSMs. To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at least one
     *         active HSM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry the
     *         request. For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before trying
     *         to connect again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the
     *         <code>kmsuser</code> crypto user in the AWS CloudHSM cluster. Before you can connect your custom key
     *         store to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and update
     *         the key store password value for the custom key store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key
     *         store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS
     *         KMS cannot find all of the subnets that were configured for the cluster when the custom key store was
     *         created, attempts to connect fail. To fix this error, create a cluster from a backup and associate it
     *         with your custom key store. This process includes selecting a VPC and subnets. For details, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to
     *         Fix a Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS
     *         CloudHSM cluster due to too many failed password attempts. Before you can connect your custom key store
     *         to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and update the key
     *         store password value for the custom key store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated AWS
     *         CloudHSM cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password and
     *         logging into the cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you
     *         must log the <code>kmsuser</code> CU out of the cluster. If you changed the <code>kmsuser</code> password
     *         to log into the cluster, you must also and update the key store password value for the custom key store.
     *         For help, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to Log
     *         Out and Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated AWS
     *         CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must
     *         create a <code>kmsuser</code> CU account in the cluster, and then update the key store password value for
     *         the custom key store.
     *         </p>
     *         </li>
     * @see ConnectionErrorCodeType
     */
    public ConnectionErrorCodeType connectionErrorCode() {
        return ConnectionErrorCodeType.fromValue(connectionErrorCode);
    }

    /**
     * <p>
     * Describes the connection error. This field appears in the response only when the <code>ConnectionState</code> is
     * <code>FAILED</code>. For help resolving these errors, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix a
     * Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any active HSMs.
     * To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at least one active HSM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry the request.
     * For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before trying to connect again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the <code>kmsuser</code> crypto
     * user in the AWS CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you
     * must change the <code>kmsuser</code> account password and update the key store password value for the custom key
     * store.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key store.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS KMS cannot
     * find all of the subnets that were configured for the cluster when the custom key store was created, attempts to
     * connect fail. To fix this error, create a cluster from a backup and associate it with your custom key store. This
     * process includes selecting a VPC and subnets. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix a
     * Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS CloudHSM
     * cluster due to too many failed password attempts. Before you can connect your custom key store to its AWS
     * CloudHSM cluster, you must change the <code>kmsuser</code> account password and update the key store password
     * value for the custom key store.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated AWS CloudHSM
     * cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password and logging into the
     * cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must log the
     * <code>kmsuser</code> CU out of the cluster. If you changed the <code>kmsuser</code> password to log into the
     * cluster, you must also and update the key store password value for the custom key store. For help, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to Log Out and
     * Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated AWS
     * CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must create a
     * <code>kmsuser</code> CU account in the cluster, and then update the key store password value for the custom key
     * store.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #connectionErrorCode} will return {@link ConnectionErrorCodeType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #connectionErrorCodeAsString}.
     * </p>
     * 
     * @return Describes the connection error. This field appears in the response only when the
     *         <code>ConnectionState</code> is <code>FAILED</code>. For help resolving these errors, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to
     *         Fix a Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.</p>
     *         <p>
     *         Valid values are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster
     *         ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any active
     *         HSMs. To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at least one
     *         active HSM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry the
     *         request. For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before trying
     *         to connect again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the
     *         <code>kmsuser</code> crypto user in the AWS CloudHSM cluster. Before you can connect your custom key
     *         store to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and update
     *         the key store password value for the custom key store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key
     *         store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS
     *         KMS cannot find all of the subnets that were configured for the cluster when the custom key store was
     *         created, attempts to connect fail. To fix this error, create a cluster from a backup and associate it
     *         with your custom key store. This process includes selecting a VPC and subnets. For details, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to
     *         Fix a Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS
     *         CloudHSM cluster due to too many failed password attempts. Before you can connect your custom key store
     *         to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and update the key
     *         store password value for the custom key store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated AWS
     *         CloudHSM cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password and
     *         logging into the cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you
     *         must log the <code>kmsuser</code> CU out of the cluster. If you changed the <code>kmsuser</code> password
     *         to log into the cluster, you must also and update the key store password value for the custom key store.
     *         For help, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to Log
     *         Out and Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated AWS
     *         CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must
     *         create a <code>kmsuser</code> CU account in the cluster, and then update the key store password value for
     *         the custom key store.
     *         </p>
     *         </li>
     * @see ConnectionErrorCodeType
     */
    public String connectionErrorCodeAsString() {
        return connectionErrorCode;
    }

    /**
     * <p>
     * The date and time when the custom key store was created.
     * </p>
     * 
     * @return The date and time when the custom key store was created.
     */
    public Instant creationDate() {
        return creationDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(customKeyStoreId());
        hashCode = 31 * hashCode + Objects.hashCode(customKeyStoreName());
        hashCode = 31 * hashCode + Objects.hashCode(cloudHsmClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(trustAnchorCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(connectionStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(connectionErrorCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomKeyStoresListEntry)) {
            return false;
        }
        CustomKeyStoresListEntry other = (CustomKeyStoresListEntry) obj;
        return Objects.equals(customKeyStoreId(), other.customKeyStoreId())
                && Objects.equals(customKeyStoreName(), other.customKeyStoreName())
                && Objects.equals(cloudHsmClusterId(), other.cloudHsmClusterId())
                && Objects.equals(trustAnchorCertificate(), other.trustAnchorCertificate())
                && Objects.equals(connectionStateAsString(), other.connectionStateAsString())
                && Objects.equals(connectionErrorCodeAsString(), other.connectionErrorCodeAsString())
                && Objects.equals(creationDate(), other.creationDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CustomKeyStoresListEntry").add("CustomKeyStoreId", customKeyStoreId())
                .add("CustomKeyStoreName", customKeyStoreName()).add("CloudHsmClusterId", cloudHsmClusterId())
                .add("TrustAnchorCertificate", trustAnchorCertificate()).add("ConnectionState", connectionStateAsString())
                .add("ConnectionErrorCode", connectionErrorCodeAsString()).add("CreationDate", creationDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CustomKeyStoreId":
            return Optional.ofNullable(clazz.cast(customKeyStoreId()));
        case "CustomKeyStoreName":
            return Optional.ofNullable(clazz.cast(customKeyStoreName()));
        case "CloudHsmClusterId":
            return Optional.ofNullable(clazz.cast(cloudHsmClusterId()));
        case "TrustAnchorCertificate":
            return Optional.ofNullable(clazz.cast(trustAnchorCertificate()));
        case "ConnectionState":
            return Optional.ofNullable(clazz.cast(connectionStateAsString()));
        case "ConnectionErrorCode":
            return Optional.ofNullable(clazz.cast(connectionErrorCodeAsString()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomKeyStoresListEntry, T> g) {
        return obj -> g.apply((CustomKeyStoresListEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomKeyStoresListEntry> {
        /**
         * <p>
         * A unique identifier for the custom key store.
         * </p>
         * 
         * @param customKeyStoreId
         *        A unique identifier for the custom key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeyStoreId(String customKeyStoreId);

        /**
         * <p>
         * The user-specified friendly name for the custom key store.
         * </p>
         * 
         * @param customKeyStoreName
         *        The user-specified friendly name for the custom key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeyStoreName(String customKeyStoreName);

        /**
         * <p>
         * A unique identifier for the AWS CloudHSM cluster that is associated with the custom key store.
         * </p>
         * 
         * @param cloudHsmClusterId
         *        A unique identifier for the AWS CloudHSM cluster that is associated with the custom key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudHsmClusterId(String cloudHsmClusterId);

        /**
         * <p>
         * The trust anchor certificate of the associated AWS CloudHSM cluster. When you <a
         * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/initialize-cluster.html#sign-csr">initialize the
         * cluster</a>, you create this certificate and save it in the <code>customerCA.crt</code> file.
         * </p>
         * 
         * @param trustAnchorCertificate
         *        The trust anchor certificate of the associated AWS CloudHSM cluster. When you <a
         *        href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/initialize-cluster.html#sign-csr"
         *        >initialize the cluster</a>, you create this certificate and save it in the
         *        <code>customerCA.crt</code> file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trustAnchorCertificate(String trustAnchorCertificate);

        /**
         * <p>
         * Indicates whether the custom key store is connected to its AWS CloudHSM cluster.
         * </p>
         * <p>
         * You can create and use CMKs in your custom key stores only when its connection state is
         * <code>CONNECTED</code>.
         * </p>
         * <p>
         * The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
         * <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but you
         * are having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster is active
         * and contains at least one active HSM.
         * </p>
         * <p>
         * A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
         * <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help resolving
         * a connection failure, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom Key
         * Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param connectionState
         *        Indicates whether the custom key store is connected to its AWS CloudHSM cluster.</p>
         *        <p>
         *        You can create and use CMKs in your custom key stores only when its connection state is
         *        <code>CONNECTED</code>.
         *        </p>
         *        <p>
         *        The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
         *        <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but
         *        you are having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster
         *        is active and contains at least one active HSM.
         *        </p>
         *        <p>
         *        A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
         *        <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help
         *        resolving a connection failure, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a
         *        Custom Key Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @see ConnectionStateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionStateType
         */
        Builder connectionState(String connectionState);

        /**
         * <p>
         * Indicates whether the custom key store is connected to its AWS CloudHSM cluster.
         * </p>
         * <p>
         * You can create and use CMKs in your custom key stores only when its connection state is
         * <code>CONNECTED</code>.
         * </p>
         * <p>
         * The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
         * <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but you
         * are having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster is active
         * and contains at least one active HSM.
         * </p>
         * <p>
         * A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
         * <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help resolving
         * a connection failure, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom Key
         * Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param connectionState
         *        Indicates whether the custom key store is connected to its AWS CloudHSM cluster.</p>
         *        <p>
         *        You can create and use CMKs in your custom key stores only when its connection state is
         *        <code>CONNECTED</code>.
         *        </p>
         *        <p>
         *        The value is <code>DISCONNECTED</code> if the key store has never been connected or you use the
         *        <a>DisconnectCustomKeyStore</a> operation to disconnect it. If the value is <code>CONNECTED</code> but
         *        you are having trouble using the custom key store, make sure that its associated AWS CloudHSM cluster
         *        is active and contains at least one active HSM.
         *        </p>
         *        <p>
         *        A value of <code>FAILED</code> indicates that an attempt to connect was unsuccessful. The
         *        <code>ConnectionErrorCode</code> field in the response indicates the cause of the failure. For help
         *        resolving a connection failure, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a
         *        Custom Key Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @see ConnectionStateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionStateType
         */
        Builder connectionState(ConnectionStateType connectionState);

        /**
         * <p>
         * Describes the connection error. This field appears in the response only when the <code>ConnectionState</code>
         * is <code>FAILED</code>. For help resolving these errors, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix
         * a Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any active
         * HSMs. To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at least one active
         * HSM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry the
         * request. For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before trying to
         * connect again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the <code>kmsuser</code>
         * crypto user in the AWS CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM
         * cluster, you must change the <code>kmsuser</code> account password and update the key store password value
         * for the custom key store.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key store.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS KMS
         * cannot find all of the subnets that were configured for the cluster when the custom key store was created,
         * attempts to connect fail. To fix this error, create a cluster from a backup and associate it with your custom
         * key store. This process includes selecting a VPC and subnets. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix
         * a Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS
         * CloudHSM cluster due to too many failed password attempts. Before you can connect your custom key store to
         * its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and update the key store
         * password value for the custom key store.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated AWS
         * CloudHSM cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password and logging
         * into the cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must log the
         * <code>kmsuser</code> CU out of the cluster. If you changed the <code>kmsuser</code> password to log into the
         * cluster, you must also and update the key store password value for the custom key store. For help, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to Log Out
         * and Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated AWS
         * CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must create a
         * <code>kmsuser</code> CU account in the cluster, and then update the key store password value for the custom
         * key store.
         * </p>
         * </li>
         * </ul>
         * 
         * @param connectionErrorCode
         *        Describes the connection error. This field appears in the response only when the
         *        <code>ConnectionState</code> is <code>FAILED</code>. For help resolving these errors, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How
         *        to Fix a Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.</p>
         *        <p>
         *        Valid values are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified
         *        cluster ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any
         *        active HSMs. To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at
         *        least one active HSM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry
         *        the request. For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before
         *        trying to connect again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the
         *        <code>kmsuser</code> crypto user in the AWS CloudHSM cluster. Before you can connect your custom key
         *        store to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and
         *        update the key store password value for the custom key store.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key
         *        store.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS
         *        KMS cannot find all of the subnets that were configured for the cluster when the custom key store was
         *        created, attempts to connect fail. To fix this error, create a cluster from a backup and associate it
         *        with your custom key store. This process includes selecting a VPC and subnets. For details, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How
         *        to Fix a Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS
         *        CloudHSM cluster due to too many failed password attempts. Before you can connect your custom key
         *        store to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and
         *        update the key store password value for the custom key store.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated
         *        AWS CloudHSM cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password
         *        and logging into the cluster. Before you can connect your custom key store to its AWS CloudHSM
         *        cluster, you must log the <code>kmsuser</code> CU out of the cluster. If you changed the
         *        <code>kmsuser</code> password to log into the cluster, you must also and update the key store password
         *        value for the custom key store. For help, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to
         *        Log Out and Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated
         *        AWS CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you
         *        must create a <code>kmsuser</code> CU account in the cluster, and then update the key store password
         *        value for the custom key store.
         *        </p>
         *        </li>
         * @see ConnectionErrorCodeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionErrorCodeType
         */
        Builder connectionErrorCode(String connectionErrorCode);

        /**
         * <p>
         * Describes the connection error. This field appears in the response only when the <code>ConnectionState</code>
         * is <code>FAILED</code>. For help resolving these errors, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix
         * a Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any active
         * HSMs. To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at least one active
         * HSM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry the
         * request. For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before trying to
         * connect again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the <code>kmsuser</code>
         * crypto user in the AWS CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM
         * cluster, you must change the <code>kmsuser</code> account password and update the key store password value
         * for the custom key store.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key store.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS KMS
         * cannot find all of the subnets that were configured for the cluster when the custom key store was created,
         * attempts to connect fail. To fix this error, create a cluster from a backup and associate it with your custom
         * key store. This process includes selecting a VPC and subnets. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How to Fix
         * a Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS
         * CloudHSM cluster due to too many failed password attempts. Before you can connect your custom key store to
         * its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and update the key store
         * password value for the custom key store.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated AWS
         * CloudHSM cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password and logging
         * into the cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must log the
         * <code>kmsuser</code> CU out of the cluster. If you changed the <code>kmsuser</code> password to log into the
         * cluster, you must also and update the key store password value for the custom key store. For help, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to Log Out
         * and Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated AWS
         * CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you must create a
         * <code>kmsuser</code> CU account in the cluster, and then update the key store password value for the custom
         * key store.
         * </p>
         * </li>
         * </ul>
         * 
         * @param connectionErrorCode
         *        Describes the connection error. This field appears in the response only when the
         *        <code>ConnectionState</code> is <code>FAILED</code>. For help resolving these errors, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How
         *        to Fix a Connection Failure</a> in <i>AWS Key Management Service Developer Guide</i>.</p>
         *        <p>
         *        Valid values are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CLUSTER_NOT_FOUND</code> - AWS KMS cannot find the AWS CloudHSM cluster with the specified
         *        cluster ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSUFFICIENT_CLOUDHSM_HSMS</code> - The associated AWS CloudHSM cluster does not contain any
         *        active HSMs. To connect a custom key store to its AWS CloudHSM cluster, the cluster must contain at
         *        least one active HSM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INTERNAL_ERROR</code> - AWS KMS could not complete the request due to an internal error. Retry
         *        the request. For <code>ConnectCustomKeyStore</code> requests, disconnect the custom key store before
         *        trying to connect again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_CREDENTIALS</code> - AWS KMS does not have the correct password for the
         *        <code>kmsuser</code> crypto user in the AWS CloudHSM cluster. Before you can connect your custom key
         *        store to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and
         *        update the key store password value for the custom key store.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NETWORK_ERRORS</code> - Network errors are preventing AWS KMS from connecting to the custom key
         *        store.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUBNET_NOT_FOUND</code> - A subnet in the AWS CloudHSM cluster configuration was deleted. If AWS
         *        KMS cannot find all of the subnets that were configured for the cluster when the custom key store was
         *        created, attempts to connect fail. To fix this error, create a cluster from a backup and associate it
         *        with your custom key store. This process includes selecting a VPC and subnets. For details, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-failed">How
         *        to Fix a Connection Failure</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_LOCKED_OUT</code> - The <code>kmsuser</code> CU account is locked out of the associated AWS
         *        CloudHSM cluster due to too many failed password attempts. Before you can connect your custom key
         *        store to its AWS CloudHSM cluster, you must change the <code>kmsuser</code> account password and
         *        update the key store password value for the custom key store.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_LOGGED_IN</code> - The <code>kmsuser</code> CU account is logged into the the associated
         *        AWS CloudHSM cluster. This prevents AWS KMS from rotating the <code>kmsuser</code> account password
         *        and logging into the cluster. Before you can connect your custom key store to its AWS CloudHSM
         *        cluster, you must log the <code>kmsuser</code> CU out of the cluster. If you changed the
         *        <code>kmsuser</code> password to log into the cluster, you must also and update the key store password
         *        value for the custom key store. For help, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#login-kmsuser-2">How to
         *        Log Out and Reconnect</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_NOT_FOUND</code> - AWS KMS cannot find a <code>kmsuser</code> CU account in the associated
         *        AWS CloudHSM cluster. Before you can connect your custom key store to its AWS CloudHSM cluster, you
         *        must create a <code>kmsuser</code> CU account in the cluster, and then update the key store password
         *        value for the custom key store.
         *        </p>
         *        </li>
         * @see ConnectionErrorCodeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionErrorCodeType
         */
        Builder connectionErrorCode(ConnectionErrorCodeType connectionErrorCode);

        /**
         * <p>
         * The date and time when the custom key store was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the custom key store was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);
    }

    static final class BuilderImpl implements Builder {
        private String customKeyStoreId;

        private String customKeyStoreName;

        private String cloudHsmClusterId;

        private String trustAnchorCertificate;

        private String connectionState;

        private String connectionErrorCode;

        private Instant creationDate;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomKeyStoresListEntry model) {
            customKeyStoreId(model.customKeyStoreId);
            customKeyStoreName(model.customKeyStoreName);
            cloudHsmClusterId(model.cloudHsmClusterId);
            trustAnchorCertificate(model.trustAnchorCertificate);
            connectionState(model.connectionState);
            connectionErrorCode(model.connectionErrorCode);
            creationDate(model.creationDate);
        }

        public final String getCustomKeyStoreId() {
            return customKeyStoreId;
        }

        @Override
        public final Builder customKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
            return this;
        }

        public final void setCustomKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
        }

        public final String getCustomKeyStoreName() {
            return customKeyStoreName;
        }

        @Override
        public final Builder customKeyStoreName(String customKeyStoreName) {
            this.customKeyStoreName = customKeyStoreName;
            return this;
        }

        public final void setCustomKeyStoreName(String customKeyStoreName) {
            this.customKeyStoreName = customKeyStoreName;
        }

        public final String getCloudHsmClusterId() {
            return cloudHsmClusterId;
        }

        @Override
        public final Builder cloudHsmClusterId(String cloudHsmClusterId) {
            this.cloudHsmClusterId = cloudHsmClusterId;
            return this;
        }

        public final void setCloudHsmClusterId(String cloudHsmClusterId) {
            this.cloudHsmClusterId = cloudHsmClusterId;
        }

        public final String getTrustAnchorCertificate() {
            return trustAnchorCertificate;
        }

        @Override
        public final Builder trustAnchorCertificate(String trustAnchorCertificate) {
            this.trustAnchorCertificate = trustAnchorCertificate;
            return this;
        }

        public final void setTrustAnchorCertificate(String trustAnchorCertificate) {
            this.trustAnchorCertificate = trustAnchorCertificate;
        }

        public final String getConnectionStateAsString() {
            return connectionState;
        }

        @Override
        public final Builder connectionState(String connectionState) {
            this.connectionState = connectionState;
            return this;
        }

        @Override
        public final Builder connectionState(ConnectionStateType connectionState) {
            this.connectionState(connectionState == null ? null : connectionState.toString());
            return this;
        }

        public final void setConnectionState(String connectionState) {
            this.connectionState = connectionState;
        }

        public final String getConnectionErrorCodeAsString() {
            return connectionErrorCode;
        }

        @Override
        public final Builder connectionErrorCode(String connectionErrorCode) {
            this.connectionErrorCode = connectionErrorCode;
            return this;
        }

        @Override
        public final Builder connectionErrorCode(ConnectionErrorCodeType connectionErrorCode) {
            this.connectionErrorCode(connectionErrorCode == null ? null : connectionErrorCode.toString());
            return this;
        }

        public final void setConnectionErrorCode(String connectionErrorCode) {
            this.connectionErrorCode = connectionErrorCode;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public CustomKeyStoresListEntry build() {
            return new CustomKeyStoresListEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
