/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class DecryptRequest extends KmsRequest implements ToCopyableBuilder<DecryptRequest.Builder, DecryptRequest> {
    private static final SdkField<SdkBytes> CIPHERTEXT_BLOB_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(DecryptRequest::ciphertextBlob)).setter(setter(Builder::ciphertextBlob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CiphertextBlob").build()).build();

    private static final SdkField<Map<String, String>> ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(DecryptRequest::encryptionContext))
            .setter(setter(Builder::encryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionContext").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> GRANT_TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DecryptRequest::grantTokens))
            .setter(setter(Builder::grantTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GrantTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DecryptRequest::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<String> ENCRYPTION_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DecryptRequest::encryptionAlgorithmAsString)).setter(setter(Builder::encryptionAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAlgorithm").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CIPHERTEXT_BLOB_FIELD,
            ENCRYPTION_CONTEXT_FIELD, GRANT_TOKENS_FIELD, KEY_ID_FIELD, ENCRYPTION_ALGORITHM_FIELD));

    private final SdkBytes ciphertextBlob;

    private final Map<String, String> encryptionContext;

    private final List<String> grantTokens;

    private final String keyId;

    private final String encryptionAlgorithm;

    private DecryptRequest(BuilderImpl builder) {
        super(builder);
        this.ciphertextBlob = builder.ciphertextBlob;
        this.encryptionContext = builder.encryptionContext;
        this.grantTokens = builder.grantTokens;
        this.keyId = builder.keyId;
        this.encryptionAlgorithm = builder.encryptionAlgorithm;
    }

    /**
     * <p>
     * Ciphertext to be decrypted. The blob includes metadata.
     * </p>
     * 
     * @return Ciphertext to be decrypted. The blob includes metadata.
     */
    public SdkBytes ciphertextBlob() {
        return ciphertextBlob;
    }

    /**
     * Returns true if the EncryptionContext property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEncryptionContext() {
        return encryptionContext != null && !(encryptionContext instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the encryption context to use when decrypting the data. An encryption context is valid only for
     * cryptographic operations with a symmetric CMK. The standard asymmetric encryption algorithms that AWS KMS uses do
     * not support an encryption context.
     * </p>
     * <p>
     * An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
     * authenticated data. When you use an encryption context to encrypt data, you must specify the same (an exact
     * case-sensitive match) encryption context to decrypt the data. An encryption context is optional when encrypting
     * with a symmetric CMK, but it is highly recommended.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption Context</a>
     * in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEncryptionContext()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the encryption context to use when decrypting the data. An encryption context is valid only for
     *         cryptographic operations with a symmetric CMK. The standard asymmetric encryption algorithms that AWS KMS
     *         uses do not support an encryption context.</p>
     *         <p>
     *         An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
     *         authenticated data. When you use an encryption context to encrypt data, you must specify the same (an
     *         exact case-sensitive match) encryption context to decrypt the data. An encryption context is optional
     *         when encrypting with a symmetric CMK, but it is highly recommended.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
     *         Context</a> in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public Map<String, String> encryptionContext() {
        return encryptionContext;
    }

    /**
     * Returns true if the GrantTokens property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGrantTokens() {
        return grantTokens != null && !(grantTokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in the
     * <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGrantTokens()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of grant tokens.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a>
     *         in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public List<String> grantTokens() {
        return grantTokens;
    }

    /**
     * <p>
     * Specifies the customer master key (CMK) that AWS KMS will use to decrypt the ciphertext. Enter a key ID of the
     * CMK that was used to encrypt the ciphertext.
     * </p>
     * <p>
     * If you specify a <code>KeyId</code> value, the <code>Decrypt</code> operation succeeds only if the specified CMK
     * was used to encrypt the ciphertext.
     * </p>
     * <p>
     * This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. Otherwise, AWS KMS
     * uses the metadata that it adds to the ciphertext blob to determine which CMK was used to encrypt the ciphertext.
     * However, you can use this parameter to ensure that a particular CMK (of any kind) is used to decrypt the
     * ciphertext.
     * </p>
     * <p>
     * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias name,
     * prefix it with <code>"alias/"</code>.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
     * alias ARN, use <a>ListAliases</a>.
     * </p>
     * 
     * @return Specifies the customer master key (CMK) that AWS KMS will use to decrypt the ciphertext. Enter a key ID
     *         of the CMK that was used to encrypt the ciphertext.</p>
     *         <p>
     *         If you specify a <code>KeyId</code> value, the <code>Decrypt</code> operation succeeds only if the
     *         specified CMK was used to encrypt the ciphertext.
     *         </p>
     *         <p>
     *         This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. Otherwise, AWS
     *         KMS uses the metadata that it adds to the ciphertext blob to determine which CMK was used to encrypt the
     *         ciphertext. However, you can use this parameter to ensure that a particular CMK (of any kind) is used to
     *         decrypt the ciphertext.
     *         </p>
     *         <p>
     *         To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
     *         alias name, prefix it with <code>"alias/"</code>.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name
     *         and alias ARN, use <a>ListAliases</a>.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same algorithm that
     * was used to encrypt the data. If you specify a different algorithm, the <code>Decrypt</code> operation fails.
     * </p>
     * <p>
     * This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default value,
     * <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for symmetric CMKs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #encryptionAlgorithm} will return {@link EncryptionAlgorithmSpec#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #encryptionAlgorithmAsString}.
     * </p>
     * 
     * @return Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same
     *         algorithm that was used to encrypt the data. If you specify a different algorithm, the
     *         <code>Decrypt</code> operation fails.</p>
     *         <p>
     *         This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default
     *         value, <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for
     *         symmetric CMKs.
     * @see EncryptionAlgorithmSpec
     */
    public EncryptionAlgorithmSpec encryptionAlgorithm() {
        return EncryptionAlgorithmSpec.fromValue(encryptionAlgorithm);
    }

    /**
     * <p>
     * Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same algorithm that
     * was used to encrypt the data. If you specify a different algorithm, the <code>Decrypt</code> operation fails.
     * </p>
     * <p>
     * This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default value,
     * <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for symmetric CMKs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #encryptionAlgorithm} will return {@link EncryptionAlgorithmSpec#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #encryptionAlgorithmAsString}.
     * </p>
     * 
     * @return Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same
     *         algorithm that was used to encrypt the data. If you specify a different algorithm, the
     *         <code>Decrypt</code> operation fails.</p>
     *         <p>
     *         This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default
     *         value, <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for
     *         symmetric CMKs.
     * @see EncryptionAlgorithmSpec
     */
    public String encryptionAlgorithmAsString() {
        return encryptionAlgorithm;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(ciphertextBlob());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionContext());
        hashCode = 31 * hashCode + Objects.hashCode(grantTokens());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAlgorithmAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DecryptRequest)) {
            return false;
        }
        DecryptRequest other = (DecryptRequest) obj;
        return Objects.equals(ciphertextBlob(), other.ciphertextBlob())
                && Objects.equals(encryptionContext(), other.encryptionContext())
                && Objects.equals(grantTokens(), other.grantTokens()) && Objects.equals(keyId(), other.keyId())
                && Objects.equals(encryptionAlgorithmAsString(), other.encryptionAlgorithmAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DecryptRequest").add("CiphertextBlob", ciphertextBlob())
                .add("EncryptionContext", encryptionContext()).add("GrantTokens", grantTokens()).add("KeyId", keyId())
                .add("EncryptionAlgorithm", encryptionAlgorithmAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CiphertextBlob":
            return Optional.ofNullable(clazz.cast(ciphertextBlob()));
        case "EncryptionContext":
            return Optional.ofNullable(clazz.cast(encryptionContext()));
        case "GrantTokens":
            return Optional.ofNullable(clazz.cast(grantTokens()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "EncryptionAlgorithm":
            return Optional.ofNullable(clazz.cast(encryptionAlgorithmAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DecryptRequest, T> g) {
        return obj -> g.apply((DecryptRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, DecryptRequest> {
        /**
         * <p>
         * Ciphertext to be decrypted. The blob includes metadata.
         * </p>
         * 
         * @param ciphertextBlob
         *        Ciphertext to be decrypted. The blob includes metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ciphertextBlob(SdkBytes ciphertextBlob);

        /**
         * <p>
         * Specifies the encryption context to use when decrypting the data. An encryption context is valid only for
         * cryptographic operations with a symmetric CMK. The standard asymmetric encryption algorithms that AWS KMS
         * uses do not support an encryption context.
         * </p>
         * <p>
         * An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
         * authenticated data. When you use an encryption context to encrypt data, you must specify the same (an exact
         * case-sensitive match) encryption context to decrypt the data. An encryption context is optional when
         * encrypting with a symmetric CMK, but it is highly recommended.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         * Context</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param encryptionContext
         *        Specifies the encryption context to use when decrypting the data. An encryption context is valid only
         *        for cryptographic operations with a symmetric CMK. The standard asymmetric encryption algorithms that
         *        AWS KMS uses do not support an encryption context.</p>
         *        <p>
         *        An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
         *        authenticated data. When you use an encryption context to encrypt data, you must specify the same (an
         *        exact case-sensitive match) encryption context to decrypt the data. An encryption context is optional
         *        when encrypting with a symmetric CMK, but it is highly recommended.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         *        Context</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContext(Map<String, String> encryptionContext);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(Collection<String> grantTokens);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(String... grantTokens);

        /**
         * <p>
         * Specifies the customer master key (CMK) that AWS KMS will use to decrypt the ciphertext. Enter a key ID of
         * the CMK that was used to encrypt the ciphertext.
         * </p>
         * <p>
         * If you specify a <code>KeyId</code> value, the <code>Decrypt</code> operation succeeds only if the specified
         * CMK was used to encrypt the ciphertext.
         * </p>
         * <p>
         * This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. Otherwise, AWS KMS
         * uses the metadata that it adds to the ciphertext blob to determine which CMK was used to encrypt the
         * ciphertext. However, you can use this parameter to ensure that a particular CMK (of any kind) is used to
         * decrypt the ciphertext.
         * </p>
         * <p>
         * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias
         * name, prefix it with <code>"alias/"</code>.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
         * alias ARN, use <a>ListAliases</a>.
         * </p>
         * 
         * @param keyId
         *        Specifies the customer master key (CMK) that AWS KMS will use to decrypt the ciphertext. Enter a key
         *        ID of the CMK that was used to encrypt the ciphertext.</p>
         *        <p>
         *        If you specify a <code>KeyId</code> value, the <code>Decrypt</code> operation succeeds only if the
         *        specified CMK was used to encrypt the ciphertext.
         *        </p>
         *        <p>
         *        This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. Otherwise,
         *        AWS KMS uses the metadata that it adds to the ciphertext blob to determine which CMK was used to
         *        encrypt the ciphertext. However, you can use this parameter to ensure that a particular CMK (of any
         *        kind) is used to decrypt the ciphertext.
         *        </p>
         *        <p>
         *        To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
         *        alias name, prefix it with <code>"alias/"</code>.
         *        </p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias
         *        name and alias ARN, use <a>ListAliases</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same algorithm
         * that was used to encrypt the data. If you specify a different algorithm, the <code>Decrypt</code> operation
         * fails.
         * </p>
         * <p>
         * This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default value,
         * <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for symmetric CMKs.
         * </p>
         * 
         * @param encryptionAlgorithm
         *        Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same
         *        algorithm that was used to encrypt the data. If you specify a different algorithm, the
         *        <code>Decrypt</code> operation fails.</p>
         *        <p>
         *        This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default
         *        value, <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for
         *        symmetric CMKs.
         * @see EncryptionAlgorithmSpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionAlgorithmSpec
         */
        Builder encryptionAlgorithm(String encryptionAlgorithm);

        /**
         * <p>
         * Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same algorithm
         * that was used to encrypt the data. If you specify a different algorithm, the <code>Decrypt</code> operation
         * fails.
         * </p>
         * <p>
         * This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default value,
         * <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for symmetric CMKs.
         * </p>
         * 
         * @param encryptionAlgorithm
         *        Specifies the encryption algorithm that will be used to decrypt the ciphertext. Specify the same
         *        algorithm that was used to encrypt the data. If you specify a different algorithm, the
         *        <code>Decrypt</code> operation fails.</p>
         *        <p>
         *        This parameter is required only when the ciphertext was encrypted under an asymmetric CMK. The default
         *        value, <code>SYMMETRIC_DEFAULT</code>, represents the only supported algorithm that is valid for
         *        symmetric CMKs.
         * @see EncryptionAlgorithmSpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionAlgorithmSpec
         */
        Builder encryptionAlgorithm(EncryptionAlgorithmSpec encryptionAlgorithm);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private SdkBytes ciphertextBlob;

        private Map<String, String> encryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private List<String> grantTokens = DefaultSdkAutoConstructList.getInstance();

        private String keyId;

        private String encryptionAlgorithm;

        private BuilderImpl() {
        }

        private BuilderImpl(DecryptRequest model) {
            super(model);
            ciphertextBlob(model.ciphertextBlob);
            encryptionContext(model.encryptionContext);
            grantTokens(model.grantTokens);
            keyId(model.keyId);
            encryptionAlgorithm(model.encryptionAlgorithm);
        }

        public final ByteBuffer getCiphertextBlob() {
            return ciphertextBlob == null ? null : ciphertextBlob.asByteBuffer();
        }

        @Override
        public final Builder ciphertextBlob(SdkBytes ciphertextBlob) {
            this.ciphertextBlob = StandardMemberCopier.copy(ciphertextBlob);
            return this;
        }

        public final void setCiphertextBlob(ByteBuffer ciphertextBlob) {
            ciphertextBlob(ciphertextBlob == null ? null : SdkBytes.fromByteBuffer(ciphertextBlob));
        }

        public final Map<String, String> getEncryptionContext() {
            return encryptionContext;
        }

        @Override
        public final Builder encryptionContext(Map<String, String> encryptionContext) {
            this.encryptionContext = EncryptionContextTypeCopier.copy(encryptionContext);
            return this;
        }

        public final void setEncryptionContext(Map<String, String> encryptionContext) {
            this.encryptionContext = EncryptionContextTypeCopier.copy(encryptionContext);
        }

        public final Collection<String> getGrantTokens() {
            return grantTokens;
        }

        @Override
        public final Builder grantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder grantTokens(String... grantTokens) {
            grantTokens(Arrays.asList(grantTokens));
            return this;
        }

        public final void setGrantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final String getEncryptionAlgorithmAsString() {
            return encryptionAlgorithm;
        }

        @Override
        public final Builder encryptionAlgorithm(String encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
            return this;
        }

        @Override
        public final Builder encryptionAlgorithm(EncryptionAlgorithmSpec encryptionAlgorithm) {
            this.encryptionAlgorithm(encryptionAlgorithm == null ? null : encryptionAlgorithm.toString());
            return this;
        }

        public final void setEncryptionAlgorithm(String encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DecryptRequest build() {
            return new DecryptRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
