/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GenerateDataKeyRequest extends KmsRequest implements
        ToCopyableBuilder<GenerateDataKeyRequest.Builder, GenerateDataKeyRequest> {
    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyId")
            .getter(getter(GenerateDataKeyRequest::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<Map<String, String>> ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("EncryptionContext")
            .getter(getter(GenerateDataKeyRequest::encryptionContext))
            .setter(setter(Builder::encryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionContext").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integer> NUMBER_OF_BYTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfBytes").getter(getter(GenerateDataKeyRequest::numberOfBytes))
            .setter(setter(Builder::numberOfBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfBytes").build()).build();

    private static final SdkField<String> KEY_SPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeySpec").getter(getter(GenerateDataKeyRequest::keySpecAsString)).setter(setter(Builder::keySpec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeySpec").build()).build();

    private static final SdkField<List<String>> GRANT_TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GrantTokens")
            .getter(getter(GenerateDataKeyRequest::grantTokens))
            .setter(setter(Builder::grantTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GrantTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_ID_FIELD,
            ENCRYPTION_CONTEXT_FIELD, NUMBER_OF_BYTES_FIELD, KEY_SPEC_FIELD, GRANT_TOKENS_FIELD));

    private final String keyId;

    private final Map<String, String> encryptionContext;

    private final Integer numberOfBytes;

    private final String keySpec;

    private final List<String> grantTokens;

    private GenerateDataKeyRequest(BuilderImpl builder) {
        super(builder);
        this.keyId = builder.keyId;
        this.encryptionContext = builder.encryptionContext;
        this.numberOfBytes = builder.numberOfBytes;
        this.keySpec = builder.keySpec;
        this.grantTokens = builder.grantTokens;
    }

    /**
     * <p>
     * Identifies the symmetric CMK that encrypts the data key.
     * </p>
     * <p>
     * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias name,
     * prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must use the key ARN or
     * alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
     * alias ARN, use <a>ListAliases</a>.
     * </p>
     * 
     * @return Identifies the symmetric CMK that encrypts the data key.</p>
     *         <p>
     *         To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
     *         alias name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must
     *         use the key ARN or alias ARN.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name
     *         and alias ARN, use <a>ListAliases</a>.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * Returns true if the EncryptionContext property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEncryptionContext() {
        return encryptionContext != null && !(encryptionContext instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the encryption context that will be used when encrypting the data key.
     * </p>
     * <p>
     * An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
     * authenticated data. When you use an encryption context to encrypt data, you must specify the same (an exact
     * case-sensitive match) encryption context to decrypt the data. An encryption context is optional when encrypting
     * with a symmetric CMK, but it is highly recommended.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption Context</a>
     * in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEncryptionContext()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the encryption context that will be used when encrypting the data key.</p>
     *         <p>
     *         An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
     *         authenticated data. When you use an encryption context to encrypt data, you must specify the same (an
     *         exact case-sensitive match) encryption context to decrypt the data. An encryption context is optional
     *         when encrypting with a symmetric CMK, but it is highly recommended.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
     *         Context</a> in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public Map<String, String> encryptionContext() {
        return encryptionContext;
    }

    /**
     * <p>
     * Specifies the length of the data key in bytes. For example, use the value 64 to generate a 512-bit data key (64
     * bytes is 512 bits). For 128-bit (16-byte) and 256-bit (32-byte) data keys, use the <code>KeySpec</code>
     * parameter.
     * </p>
     * <p>
     * You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not both) in
     * every <code>GenerateDataKey</code> request.
     * </p>
     * 
     * @return Specifies the length of the data key in bytes. For example, use the value 64 to generate a 512-bit data
     *         key (64 bytes is 512 bits). For 128-bit (16-byte) and 256-bit (32-byte) data keys, use the
     *         <code>KeySpec</code> parameter.</p>
     *         <p>
     *         You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not
     *         both) in every <code>GenerateDataKey</code> request.
     */
    public Integer numberOfBytes() {
        return numberOfBytes;
    }

    /**
     * <p>
     * Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
     * <code>AES_256</code> to generate a 256-bit symmetric key.
     * </p>
     * <p>
     * You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not both) in
     * every <code>GenerateDataKey</code> request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keySpec} will
     * return {@link DataKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keySpecAsString}.
     * </p>
     * 
     * @return Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
     *         <code>AES_256</code> to generate a 256-bit symmetric key.</p>
     *         <p>
     *         You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not
     *         both) in every <code>GenerateDataKey</code> request.
     * @see DataKeySpec
     */
    public DataKeySpec keySpec() {
        return DataKeySpec.fromValue(keySpec);
    }

    /**
     * <p>
     * Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
     * <code>AES_256</code> to generate a 256-bit symmetric key.
     * </p>
     * <p>
     * You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not both) in
     * every <code>GenerateDataKey</code> request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keySpec} will
     * return {@link DataKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keySpecAsString}.
     * </p>
     * 
     * @return Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
     *         <code>AES_256</code> to generate a 256-bit symmetric key.</p>
     *         <p>
     *         You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not
     *         both) in every <code>GenerateDataKey</code> request.
     * @see DataKeySpec
     */
    public String keySpecAsString() {
        return keySpec;
    }

    /**
     * Returns true if the GrantTokens property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGrantTokens() {
        return grantTokens != null && !(grantTokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in the
     * <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGrantTokens()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of grant tokens.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a>
     *         in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public List<String> grantTokens() {
        return grantTokens;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionContext());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfBytes());
        hashCode = 31 * hashCode + Objects.hashCode(keySpecAsString());
        hashCode = 31 * hashCode + Objects.hashCode(grantTokens());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GenerateDataKeyRequest)) {
            return false;
        }
        GenerateDataKeyRequest other = (GenerateDataKeyRequest) obj;
        return Objects.equals(keyId(), other.keyId()) && Objects.equals(encryptionContext(), other.encryptionContext())
                && Objects.equals(numberOfBytes(), other.numberOfBytes())
                && Objects.equals(keySpecAsString(), other.keySpecAsString())
                && Objects.equals(grantTokens(), other.grantTokens());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GenerateDataKeyRequest").add("KeyId", keyId()).add("EncryptionContext", encryptionContext())
                .add("NumberOfBytes", numberOfBytes()).add("KeySpec", keySpecAsString()).add("GrantTokens", grantTokens())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "EncryptionContext":
            return Optional.ofNullable(clazz.cast(encryptionContext()));
        case "NumberOfBytes":
            return Optional.ofNullable(clazz.cast(numberOfBytes()));
        case "KeySpec":
            return Optional.ofNullable(clazz.cast(keySpecAsString()));
        case "GrantTokens":
            return Optional.ofNullable(clazz.cast(grantTokens()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GenerateDataKeyRequest, T> g) {
        return obj -> g.apply((GenerateDataKeyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, GenerateDataKeyRequest> {
        /**
         * <p>
         * Identifies the symmetric CMK that encrypts the data key.
         * </p>
         * <p>
         * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias
         * name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must use the key
         * ARN or alias ARN.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
         * alias ARN, use <a>ListAliases</a>.
         * </p>
         * 
         * @param keyId
         *        Identifies the symmetric CMK that encrypts the data key.</p>
         *        <p>
         *        To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
         *        alias name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you
         *        must use the key ARN or alias ARN.
         *        </p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias
         *        name and alias ARN, use <a>ListAliases</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * Specifies the encryption context that will be used when encrypting the data key.
         * </p>
         * <p>
         * An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
         * authenticated data. When you use an encryption context to encrypt data, you must specify the same (an exact
         * case-sensitive match) encryption context to decrypt the data. An encryption context is optional when
         * encrypting with a symmetric CMK, but it is highly recommended.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         * Context</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param encryptionContext
         *        Specifies the encryption context that will be used when encrypting the data key.</p>
         *        <p>
         *        An <i>encryption context</i> is a collection of non-secret key-value pairs that represents additional
         *        authenticated data. When you use an encryption context to encrypt data, you must specify the same (an
         *        exact case-sensitive match) encryption context to decrypt the data. An encryption context is optional
         *        when encrypting with a symmetric CMK, but it is highly recommended.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         *        Context</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContext(Map<String, String> encryptionContext);

        /**
         * <p>
         * Specifies the length of the data key in bytes. For example, use the value 64 to generate a 512-bit data key
         * (64 bytes is 512 bits). For 128-bit (16-byte) and 256-bit (32-byte) data keys, use the <code>KeySpec</code>
         * parameter.
         * </p>
         * <p>
         * You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not both)
         * in every <code>GenerateDataKey</code> request.
         * </p>
         * 
         * @param numberOfBytes
         *        Specifies the length of the data key in bytes. For example, use the value 64 to generate a 512-bit
         *        data key (64 bytes is 512 bits). For 128-bit (16-byte) and 256-bit (32-byte) data keys, use the
         *        <code>KeySpec</code> parameter.</p>
         *        <p>
         *        You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not
         *        both) in every <code>GenerateDataKey</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfBytes(Integer numberOfBytes);

        /**
         * <p>
         * Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
         * <code>AES_256</code> to generate a 256-bit symmetric key.
         * </p>
         * <p>
         * You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not both)
         * in every <code>GenerateDataKey</code> request.
         * </p>
         * 
         * @param keySpec
         *        Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
         *        <code>AES_256</code> to generate a 256-bit symmetric key.</p>
         *        <p>
         *        You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not
         *        both) in every <code>GenerateDataKey</code> request.
         * @see DataKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataKeySpec
         */
        Builder keySpec(String keySpec);

        /**
         * <p>
         * Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
         * <code>AES_256</code> to generate a 256-bit symmetric key.
         * </p>
         * <p>
         * You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not both)
         * in every <code>GenerateDataKey</code> request.
         * </p>
         * 
         * @param keySpec
         *        Specifies the length of the data key. Use <code>AES_128</code> to generate a 128-bit symmetric key, or
         *        <code>AES_256</code> to generate a 256-bit symmetric key.</p>
         *        <p>
         *        You must specify either the <code>KeySpec</code> or the <code>NumberOfBytes</code> parameter (but not
         *        both) in every <code>GenerateDataKey</code> request.
         * @see DataKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataKeySpec
         */
        Builder keySpec(DataKeySpec keySpec);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(Collection<String> grantTokens);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(String... grantTokens);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private String keyId;

        private Map<String, String> encryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private Integer numberOfBytes;

        private String keySpec;

        private List<String> grantTokens = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GenerateDataKeyRequest model) {
            super(model);
            keyId(model.keyId);
            encryptionContext(model.encryptionContext);
            numberOfBytes(model.numberOfBytes);
            keySpec(model.keySpec);
            grantTokens(model.grantTokens);
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final Map<String, String> getEncryptionContext() {
            return encryptionContext;
        }

        @Override
        public final Builder encryptionContext(Map<String, String> encryptionContext) {
            this.encryptionContext = EncryptionContextTypeCopier.copy(encryptionContext);
            return this;
        }

        public final void setEncryptionContext(Map<String, String> encryptionContext) {
            this.encryptionContext = EncryptionContextTypeCopier.copy(encryptionContext);
        }

        public final Integer getNumberOfBytes() {
            return numberOfBytes;
        }

        @Override
        public final Builder numberOfBytes(Integer numberOfBytes) {
            this.numberOfBytes = numberOfBytes;
            return this;
        }

        public final void setNumberOfBytes(Integer numberOfBytes) {
            this.numberOfBytes = numberOfBytes;
        }

        public final String getKeySpec() {
            return keySpec;
        }

        @Override
        public final Builder keySpec(String keySpec) {
            this.keySpec = keySpec;
            return this;
        }

        @Override
        public final Builder keySpec(DataKeySpec keySpec) {
            this.keySpec(keySpec == null ? null : keySpec.toString());
            return this;
        }

        public final void setKeySpec(String keySpec) {
            this.keySpec = keySpec;
        }

        public final Collection<String> getGrantTokens() {
            return grantTokens;
        }

        @Override
        public final Builder grantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder grantTokens(String... grantTokens) {
            grantTokens(Arrays.asList(grantTokens));
            return this;
        }

        public final void setGrantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GenerateDataKeyRequest build() {
            return new GenerateDataKeyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
