/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains metadata about a customer master key (CMK).
 * </p>
 * <p>
 * This data type is used as a response element for the <a>CreateKey</a> and <a>DescribeKey</a> operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeyMetadata implements SdkPojo, Serializable, ToCopyableBuilder<KeyMetadata.Builder, KeyMetadata> {
    private static final SdkField<String> AWS_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AWSAccountId").getter(getter(KeyMetadata::awsAccountId)).setter(setter(Builder::awsAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AWSAccountId").build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyId")
            .getter(getter(KeyMetadata::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(KeyMetadata::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(KeyMetadata::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(KeyMetadata::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(KeyMetadata::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> KEY_USAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyUsage").getter(getter(KeyMetadata::keyUsageAsString)).setter(setter(Builder::keyUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyUsage").build()).build();

    private static final SdkField<String> KEY_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyState").getter(getter(KeyMetadata::keyStateAsString)).setter(setter(Builder::keyState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyState").build()).build();

    private static final SdkField<Instant> DELETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeletionDate").getter(getter(KeyMetadata::deletionDate)).setter(setter(Builder::deletionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionDate").build()).build();

    private static final SdkField<Instant> VALID_TO_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ValidTo").getter(getter(KeyMetadata::validTo)).setter(setter(Builder::validTo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidTo").build()).build();

    private static final SdkField<String> ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Origin")
            .getter(getter(KeyMetadata::originAsString)).setter(setter(Builder::origin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Origin").build()).build();

    private static final SdkField<String> CUSTOM_KEY_STORE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomKeyStoreId").getter(getter(KeyMetadata::customKeyStoreId))
            .setter(setter(Builder::customKeyStoreId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomKeyStoreId").build()).build();

    private static final SdkField<String> CLOUD_HSM_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CloudHsmClusterId").getter(getter(KeyMetadata::cloudHsmClusterId))
            .setter(setter(Builder::cloudHsmClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudHsmClusterId").build()).build();

    private static final SdkField<String> EXPIRATION_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExpirationModel").getter(getter(KeyMetadata::expirationModelAsString))
            .setter(setter(Builder::expirationModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationModel").build()).build();

    private static final SdkField<String> KEY_MANAGER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyManager").getter(getter(KeyMetadata::keyManagerAsString)).setter(setter(Builder::keyManager))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyManager").build()).build();

    private static final SdkField<String> CUSTOMER_MASTER_KEY_SPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerMasterKeySpec").getter(getter(KeyMetadata::customerMasterKeySpecAsString))
            .setter(setter(Builder::customerMasterKeySpec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerMasterKeySpec").build())
            .build();

    private static final SdkField<List<String>> ENCRYPTION_ALGORITHMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EncryptionAlgorithms")
            .getter(getter(KeyMetadata::encryptionAlgorithmsAsStrings))
            .setter(setter(Builder::encryptionAlgorithmsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAlgorithms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SIGNING_ALGORITHMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SigningAlgorithms")
            .getter(getter(KeyMetadata::signingAlgorithmsAsStrings))
            .setter(setter(Builder::signingAlgorithmsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningAlgorithms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AWS_ACCOUNT_ID_FIELD,
            KEY_ID_FIELD, ARN_FIELD, CREATION_DATE_FIELD, ENABLED_FIELD, DESCRIPTION_FIELD, KEY_USAGE_FIELD, KEY_STATE_FIELD,
            DELETION_DATE_FIELD, VALID_TO_FIELD, ORIGIN_FIELD, CUSTOM_KEY_STORE_ID_FIELD, CLOUD_HSM_CLUSTER_ID_FIELD,
            EXPIRATION_MODEL_FIELD, KEY_MANAGER_FIELD, CUSTOMER_MASTER_KEY_SPEC_FIELD, ENCRYPTION_ALGORITHMS_FIELD,
            SIGNING_ALGORITHMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String awsAccountId;

    private final String keyId;

    private final String arn;

    private final Instant creationDate;

    private final Boolean enabled;

    private final String description;

    private final String keyUsage;

    private final String keyState;

    private final Instant deletionDate;

    private final Instant validTo;

    private final String origin;

    private final String customKeyStoreId;

    private final String cloudHsmClusterId;

    private final String expirationModel;

    private final String keyManager;

    private final String customerMasterKeySpec;

    private final List<String> encryptionAlgorithms;

    private final List<String> signingAlgorithms;

    private KeyMetadata(BuilderImpl builder) {
        this.awsAccountId = builder.awsAccountId;
        this.keyId = builder.keyId;
        this.arn = builder.arn;
        this.creationDate = builder.creationDate;
        this.enabled = builder.enabled;
        this.description = builder.description;
        this.keyUsage = builder.keyUsage;
        this.keyState = builder.keyState;
        this.deletionDate = builder.deletionDate;
        this.validTo = builder.validTo;
        this.origin = builder.origin;
        this.customKeyStoreId = builder.customKeyStoreId;
        this.cloudHsmClusterId = builder.cloudHsmClusterId;
        this.expirationModel = builder.expirationModel;
        this.keyManager = builder.keyManager;
        this.customerMasterKeySpec = builder.customerMasterKeySpec;
        this.encryptionAlgorithms = builder.encryptionAlgorithms;
        this.signingAlgorithms = builder.signingAlgorithms;
    }

    /**
     * <p>
     * The twelve-digit account ID of the AWS account that owns the CMK.
     * </p>
     * 
     * @return The twelve-digit account ID of the AWS account that owns the CMK.
     */
    public String awsAccountId() {
        return awsAccountId;
    }

    /**
     * <p>
     * The globally unique identifier for the CMK.
     * </p>
     * 
     * @return The globally unique identifier for the CMK.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the CMK. For examples, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-kms">AWS Key
     * Management Service (AWS KMS)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the CMK. For examples, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-kms">AWS Key
     *         Management Service (AWS KMS)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The date and time when the CMK was created.
     * </p>
     * 
     * @return The date and time when the CMK was created.
     */
    public Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * Specifies whether the CMK is enabled. When <code>KeyState</code> is <code>Enabled</code> this value is true,
     * otherwise it is false.
     * </p>
     * 
     * @return Specifies whether the CMK is enabled. When <code>KeyState</code> is <code>Enabled</code> this value is
     *         true, otherwise it is false.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The description of the CMK.
     * </p>
     * 
     * @return The description of the CMK.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">
     * cryptographic operations</a> for which you can use the CMK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyUsage} will
     * return {@link KeyUsageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyUsageAsString}.
     * </p>
     * 
     * @return The <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
     *         >cryptographic operations</a> for which you can use the CMK.
     * @see KeyUsageType
     */
    public KeyUsageType keyUsage() {
        return KeyUsageType.fromValue(keyUsage);
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">
     * cryptographic operations</a> for which you can use the CMK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyUsage} will
     * return {@link KeyUsageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyUsageAsString}.
     * </p>
     * 
     * @return The <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
     *         >cryptographic operations</a> for which you can use the CMK.
     * @see KeyUsageType
     */
    public String keyUsageAsString() {
        return keyUsage;
    }

    /**
     * <p>
     * The current status of the CMK.
     * </p>
     * <p>
     * For more information about how key state affects the use of a CMK, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your CMK</a> in
     * the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyState} will
     * return {@link KeyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyStateAsString}.
     * </p>
     * 
     * @return The current status of the CMK.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your
     *         CMK</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @see KeyState
     */
    public KeyState keyState() {
        return KeyState.fromValue(keyState);
    }

    /**
     * <p>
     * The current status of the CMK.
     * </p>
     * <p>
     * For more information about how key state affects the use of a CMK, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your CMK</a> in
     * the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyState} will
     * return {@link KeyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyStateAsString}.
     * </p>
     * 
     * @return The current status of the CMK.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your
     *         CMK</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @see KeyState
     */
    public String keyStateAsString() {
        return keyState;
    }

    /**
     * <p>
     * The date and time after which AWS KMS deletes the CMK. This value is present only when <code>KeyState</code> is
     * <code>PendingDeletion</code>.
     * </p>
     * 
     * @return The date and time after which AWS KMS deletes the CMK. This value is present only when
     *         <code>KeyState</code> is <code>PendingDeletion</code>.
     */
    public Instant deletionDate() {
        return deletionDate;
    }

    /**
     * <p>
     * The time at which the imported key material expires. When the key material expires, AWS KMS deletes the key
     * material and the CMK becomes unusable. This value is present only for CMKs whose <code>Origin</code> is
     * <code>EXTERNAL</code> and whose <code>ExpirationModel</code> is <code>KEY_MATERIAL_EXPIRES</code>, otherwise this
     * value is omitted.
     * </p>
     * 
     * @return The time at which the imported key material expires. When the key material expires, AWS KMS deletes the
     *         key material and the CMK becomes unusable. This value is present only for CMKs whose <code>Origin</code>
     *         is <code>EXTERNAL</code> and whose <code>ExpirationModel</code> is <code>KEY_MATERIAL_EXPIRES</code>,
     *         otherwise this value is omitted.
     */
    public Instant validTo() {
        return validTo;
    }

    /**
     * <p>
     * The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key material.
     * When this value is <code>EXTERNAL</code>, the key material was imported from your existing key management
     * infrastructure or the CMK lacks key material. When this value is <code>AWS_CLOUDHSM</code>, the key material was
     * created in the AWS CloudHSM cluster associated with a custom key store.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key
     *         material. When this value is <code>EXTERNAL</code>, the key material was imported from your existing key
     *         management infrastructure or the CMK lacks key material. When this value is <code>AWS_CLOUDHSM</code>,
     *         the key material was created in the AWS CloudHSM cluster associated with a custom key store.
     * @see OriginType
     */
    public OriginType origin() {
        return OriginType.fromValue(origin);
    }

    /**
     * <p>
     * The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key material.
     * When this value is <code>EXTERNAL</code>, the key material was imported from your existing key management
     * infrastructure or the CMK lacks key material. When this value is <code>AWS_CLOUDHSM</code>, the key material was
     * created in the AWS CloudHSM cluster associated with a custom key store.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key
     *         material. When this value is <code>EXTERNAL</code>, the key material was imported from your existing key
     *         management infrastructure or the CMK lacks key material. When this value is <code>AWS_CLOUDHSM</code>,
     *         the key material was created in the AWS CloudHSM cluster associated with a custom key store.
     * @see OriginType
     */
    public String originAsString() {
        return origin;
    }

    /**
     * <p>
     * A unique identifier for the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * that contains the CMK. This value is present only when the CMK is created in a custom key store.
     * </p>
     * 
     * @return A unique identifier for the <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> that contains the CMK. This value is present only when the CMK is created in a custom key
     *         store.
     */
    public String customKeyStoreId() {
        return customKeyStoreId;
    }

    /**
     * <p>
     * The cluster ID of the AWS CloudHSM cluster that contains the key material for the CMK. When you create a CMK in a
     * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     * store</a>, AWS KMS creates the key material for the CMK in the associated AWS CloudHSM cluster. This value is
     * present only when the CMK is created in a custom key store.
     * </p>
     * 
     * @return The cluster ID of the AWS CloudHSM cluster that contains the key material for the CMK. When you create a
     *         CMK in a <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a>, AWS KMS creates the key material for the CMK in the associated AWS CloudHSM cluster. This
     *         value is present only when the CMK is created in a custom key store.
     */
    public String cloudHsmClusterId() {
        return cloudHsmClusterId;
    }

    /**
     * <p>
     * Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code> is
     * <code>EXTERNAL</code>, otherwise this value is omitted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #expirationModel}
     * will return {@link ExpirationModelType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #expirationModelAsString}.
     * </p>
     * 
     * @return Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code> is
     *         <code>EXTERNAL</code>, otherwise this value is omitted.
     * @see ExpirationModelType
     */
    public ExpirationModelType expirationModel() {
        return ExpirationModelType.fromValue(expirationModel);
    }

    /**
     * <p>
     * Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code> is
     * <code>EXTERNAL</code>, otherwise this value is omitted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #expirationModel}
     * will return {@link ExpirationModelType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #expirationModelAsString}.
     * </p>
     * 
     * @return Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code> is
     *         <code>EXTERNAL</code>, otherwise this value is omitted.
     * @see ExpirationModelType
     */
    public String expirationModelAsString() {
        return expirationModel;
    }

    /**
     * <p>
     * The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more information
     * about the difference, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master Keys</a>
     * in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyManager} will
     * return {@link KeyManagerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyManagerAsString}.
     * </p>
     * 
     * @return The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more
     *         information about the difference, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master
     *         Keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @see KeyManagerType
     */
    public KeyManagerType keyManager() {
        return KeyManagerType.fromValue(keyManager);
    }

    /**
     * <p>
     * The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more information
     * about the difference, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master Keys</a>
     * in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyManager} will
     * return {@link KeyManagerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyManagerAsString}.
     * </p>
     * 
     * @return The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more
     *         information about the difference, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master
     *         Keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @see KeyManagerType
     */
    public String keyManagerAsString() {
        return keyManager;
    }

    /**
     * <p>
     * Describes the type of key material in the CMK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #customerMasterKeySpec} will return {@link CustomerMasterKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #customerMasterKeySpecAsString}.
     * </p>
     * 
     * @return Describes the type of key material in the CMK.
     * @see CustomerMasterKeySpec
     */
    public CustomerMasterKeySpec customerMasterKeySpec() {
        return CustomerMasterKeySpec.fromValue(customerMasterKeySpec);
    }

    /**
     * <p>
     * Describes the type of key material in the CMK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #customerMasterKeySpec} will return {@link CustomerMasterKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #customerMasterKeySpecAsString}.
     * </p>
     * 
     * @return Describes the type of key material in the CMK.
     * @see CustomerMasterKeySpec
     */
    public String customerMasterKeySpecAsString() {
        return customerMasterKeySpec;
    }

    /**
     * <p>
     * The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms within
     * AWS KMS.
     * </p>
     * <p>
     * This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEncryptionAlgorithms()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms
     *         within AWS KMS.</p>
     *         <p>
     *         This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
     */
    public List<EncryptionAlgorithmSpec> encryptionAlgorithms() {
        return EncryptionAlgorithmSpecListCopier.copyStringToEnum(encryptionAlgorithms);
    }

    /**
     * Returns true if the EncryptionAlgorithms property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEncryptionAlgorithms() {
        return encryptionAlgorithms != null && !(encryptionAlgorithms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms within
     * AWS KMS.
     * </p>
     * <p>
     * This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEncryptionAlgorithms()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms
     *         within AWS KMS.</p>
     *         <p>
     *         This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
     */
    public List<String> encryptionAlgorithmsAsStrings() {
        return encryptionAlgorithms;
    }

    /**
     * <p>
     * The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within AWS
     * KMS.
     * </p>
     * <p>
     * This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSigningAlgorithms()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within
     *         AWS KMS.</p>
     *         <p>
     *         This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
     */
    public List<SigningAlgorithmSpec> signingAlgorithms() {
        return SigningAlgorithmSpecListCopier.copyStringToEnum(signingAlgorithms);
    }

    /**
     * Returns true if the SigningAlgorithms property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSigningAlgorithms() {
        return signingAlgorithms != null && !(signingAlgorithms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within AWS
     * KMS.
     * </p>
     * <p>
     * This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSigningAlgorithms()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within
     *         AWS KMS.</p>
     *         <p>
     *         This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
     */
    public List<String> signingAlgorithmsAsStrings() {
        return signingAlgorithms;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(awsAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(keyUsageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deletionDate());
        hashCode = 31 * hashCode + Objects.hashCode(validTo());
        hashCode = 31 * hashCode + Objects.hashCode(originAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customKeyStoreId());
        hashCode = 31 * hashCode + Objects.hashCode(cloudHsmClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(expirationModelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyManagerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customerMasterKeySpecAsString());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAlgorithmsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(signingAlgorithmsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeyMetadata)) {
            return false;
        }
        KeyMetadata other = (KeyMetadata) obj;
        return Objects.equals(awsAccountId(), other.awsAccountId()) && Objects.equals(keyId(), other.keyId())
                && Objects.equals(arn(), other.arn()) && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(enabled(), other.enabled()) && Objects.equals(description(), other.description())
                && Objects.equals(keyUsageAsString(), other.keyUsageAsString())
                && Objects.equals(keyStateAsString(), other.keyStateAsString())
                && Objects.equals(deletionDate(), other.deletionDate()) && Objects.equals(validTo(), other.validTo())
                && Objects.equals(originAsString(), other.originAsString())
                && Objects.equals(customKeyStoreId(), other.customKeyStoreId())
                && Objects.equals(cloudHsmClusterId(), other.cloudHsmClusterId())
                && Objects.equals(expirationModelAsString(), other.expirationModelAsString())
                && Objects.equals(keyManagerAsString(), other.keyManagerAsString())
                && Objects.equals(customerMasterKeySpecAsString(), other.customerMasterKeySpecAsString())
                && Objects.equals(encryptionAlgorithmsAsStrings(), other.encryptionAlgorithmsAsStrings())
                && Objects.equals(signingAlgorithmsAsStrings(), other.signingAlgorithmsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("KeyMetadata").add("AWSAccountId", awsAccountId()).add("KeyId", keyId()).add("Arn", arn())
                .add("CreationDate", creationDate()).add("Enabled", enabled()).add("Description", description())
                .add("KeyUsage", keyUsageAsString()).add("KeyState", keyStateAsString()).add("DeletionDate", deletionDate())
                .add("ValidTo", validTo()).add("Origin", originAsString()).add("CustomKeyStoreId", customKeyStoreId())
                .add("CloudHsmClusterId", cloudHsmClusterId()).add("ExpirationModel", expirationModelAsString())
                .add("KeyManager", keyManagerAsString()).add("CustomerMasterKeySpec", customerMasterKeySpecAsString())
                .add("EncryptionAlgorithms", encryptionAlgorithmsAsStrings())
                .add("SigningAlgorithms", signingAlgorithmsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AWSAccountId":
            return Optional.ofNullable(clazz.cast(awsAccountId()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "KeyUsage":
            return Optional.ofNullable(clazz.cast(keyUsageAsString()));
        case "KeyState":
            return Optional.ofNullable(clazz.cast(keyStateAsString()));
        case "DeletionDate":
            return Optional.ofNullable(clazz.cast(deletionDate()));
        case "ValidTo":
            return Optional.ofNullable(clazz.cast(validTo()));
        case "Origin":
            return Optional.ofNullable(clazz.cast(originAsString()));
        case "CustomKeyStoreId":
            return Optional.ofNullable(clazz.cast(customKeyStoreId()));
        case "CloudHsmClusterId":
            return Optional.ofNullable(clazz.cast(cloudHsmClusterId()));
        case "ExpirationModel":
            return Optional.ofNullable(clazz.cast(expirationModelAsString()));
        case "KeyManager":
            return Optional.ofNullable(clazz.cast(keyManagerAsString()));
        case "CustomerMasterKeySpec":
            return Optional.ofNullable(clazz.cast(customerMasterKeySpecAsString()));
        case "EncryptionAlgorithms":
            return Optional.ofNullable(clazz.cast(encryptionAlgorithmsAsStrings()));
        case "SigningAlgorithms":
            return Optional.ofNullable(clazz.cast(signingAlgorithmsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KeyMetadata, T> g) {
        return obj -> g.apply((KeyMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeyMetadata> {
        /**
         * <p>
         * The twelve-digit account ID of the AWS account that owns the CMK.
         * </p>
         * 
         * @param awsAccountId
         *        The twelve-digit account ID of the AWS account that owns the CMK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsAccountId(String awsAccountId);

        /**
         * <p>
         * The globally unique identifier for the CMK.
         * </p>
         * 
         * @param keyId
         *        The globally unique identifier for the CMK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the CMK. For examples, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-kms">AWS Key
         * Management Service (AWS KMS)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the CMK. For examples, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-kms">AWS
         *        Key Management Service (AWS KMS)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date and time when the CMK was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the CMK was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * Specifies whether the CMK is enabled. When <code>KeyState</code> is <code>Enabled</code> this value is true,
         * otherwise it is false.
         * </p>
         * 
         * @param enabled
         *        Specifies whether the CMK is enabled. When <code>KeyState</code> is <code>Enabled</code> this value is
         *        true, otherwise it is false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The description of the CMK.
         * </p>
         * 
         * @param description
         *        The description of the CMK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">
         * cryptographic operations</a> for which you can use the CMK.
         * </p>
         * 
         * @param keyUsage
         *        The <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         *        >cryptographic operations</a> for which you can use the CMK.
         * @see KeyUsageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyUsageType
         */
        Builder keyUsage(String keyUsage);

        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">
         * cryptographic operations</a> for which you can use the CMK.
         * </p>
         * 
         * @param keyUsage
         *        The <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         *        >cryptographic operations</a> for which you can use the CMK.
         * @see KeyUsageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyUsageType
         */
        Builder keyUsage(KeyUsageType keyUsage);

        /**
         * <p>
         * The current status of the CMK.
         * </p>
         * <p>
         * For more information about how key state affects the use of a CMK, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your CMK</a>
         * in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param keyState
         *        The current status of the CMK.</p>
         *        <p>
         *        For more information about how key state affects the use of a CMK, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your
         *        CMK</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @see KeyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyState
         */
        Builder keyState(String keyState);

        /**
         * <p>
         * The current status of the CMK.
         * </p>
         * <p>
         * For more information about how key state affects the use of a CMK, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your CMK</a>
         * in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param keyState
         *        The current status of the CMK.</p>
         *        <p>
         *        For more information about how key state affects the use of a CMK, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on your
         *        CMK</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @see KeyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyState
         */
        Builder keyState(KeyState keyState);

        /**
         * <p>
         * The date and time after which AWS KMS deletes the CMK. This value is present only when <code>KeyState</code>
         * is <code>PendingDeletion</code>.
         * </p>
         * 
         * @param deletionDate
         *        The date and time after which AWS KMS deletes the CMK. This value is present only when
         *        <code>KeyState</code> is <code>PendingDeletion</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionDate(Instant deletionDate);

        /**
         * <p>
         * The time at which the imported key material expires. When the key material expires, AWS KMS deletes the key
         * material and the CMK becomes unusable. This value is present only for CMKs whose <code>Origin</code> is
         * <code>EXTERNAL</code> and whose <code>ExpirationModel</code> is <code>KEY_MATERIAL_EXPIRES</code>, otherwise
         * this value is omitted.
         * </p>
         * 
         * @param validTo
         *        The time at which the imported key material expires. When the key material expires, AWS KMS deletes
         *        the key material and the CMK becomes unusable. This value is present only for CMKs whose
         *        <code>Origin</code> is <code>EXTERNAL</code> and whose <code>ExpirationModel</code> is
         *        <code>KEY_MATERIAL_EXPIRES</code>, otherwise this value is omitted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validTo(Instant validTo);

        /**
         * <p>
         * The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key
         * material. When this value is <code>EXTERNAL</code>, the key material was imported from your existing key
         * management infrastructure or the CMK lacks key material. When this value is <code>AWS_CLOUDHSM</code>, the
         * key material was created in the AWS CloudHSM cluster associated with a custom key store.
         * </p>
         * 
         * @param origin
         *        The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key
         *        material. When this value is <code>EXTERNAL</code>, the key material was imported from your existing
         *        key management infrastructure or the CMK lacks key material. When this value is
         *        <code>AWS_CLOUDHSM</code>, the key material was created in the AWS CloudHSM cluster associated with a
         *        custom key store.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(String origin);

        /**
         * <p>
         * The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key
         * material. When this value is <code>EXTERNAL</code>, the key material was imported from your existing key
         * management infrastructure or the CMK lacks key material. When this value is <code>AWS_CLOUDHSM</code>, the
         * key material was created in the AWS CloudHSM cluster associated with a custom key store.
         * </p>
         * 
         * @param origin
         *        The source of the CMK's key material. When this value is <code>AWS_KMS</code>, AWS KMS created the key
         *        material. When this value is <code>EXTERNAL</code>, the key material was imported from your existing
         *        key management infrastructure or the CMK lacks key material. When this value is
         *        <code>AWS_CLOUDHSM</code>, the key material was created in the AWS CloudHSM cluster associated with a
         *        custom key store.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(OriginType origin);

        /**
         * <p>
         * A unique identifier for the <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> that contains the CMK. This value is present only when the CMK is created in a custom key store.
         * </p>
         * 
         * @param customKeyStoreId
         *        A unique identifier for the <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> that contains the CMK. This value is present only when the CMK is created in a custom key
         *        store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeyStoreId(String customKeyStoreId);

        /**
         * <p>
         * The cluster ID of the AWS CloudHSM cluster that contains the key material for the CMK. When you create a CMK
         * in a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom
         * key store</a>, AWS KMS creates the key material for the CMK in the associated AWS CloudHSM cluster. This
         * value is present only when the CMK is created in a custom key store.
         * </p>
         * 
         * @param cloudHsmClusterId
         *        The cluster ID of the AWS CloudHSM cluster that contains the key material for the CMK. When you create
         *        a CMK in a <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a>, AWS KMS creates the key material for the CMK in the associated AWS CloudHSM cluster. This
         *        value is present only when the CMK is created in a custom key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudHsmClusterId(String cloudHsmClusterId);

        /**
         * <p>
         * Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code> is
         * <code>EXTERNAL</code>, otherwise this value is omitted.
         * </p>
         * 
         * @param expirationModel
         *        Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code>
         *        is <code>EXTERNAL</code>, otherwise this value is omitted.
         * @see ExpirationModelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpirationModelType
         */
        Builder expirationModel(String expirationModel);

        /**
         * <p>
         * Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code> is
         * <code>EXTERNAL</code>, otherwise this value is omitted.
         * </p>
         * 
         * @param expirationModel
         *        Specifies whether the CMK's key material expires. This value is present only when <code>Origin</code>
         *        is <code>EXTERNAL</code>, otherwise this value is omitted.
         * @see ExpirationModelType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExpirationModelType
         */
        Builder expirationModel(ExpirationModelType expirationModel);

        /**
         * <p>
         * The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more
         * information about the difference, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master
         * Keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param keyManager
         *        The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more
         *        information about the difference, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master
         *        Keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @see KeyManagerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyManagerType
         */
        Builder keyManager(String keyManager);

        /**
         * <p>
         * The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more
         * information about the difference, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master
         * Keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param keyManager
         *        The manager of the CMK. CMKs in your AWS account are either customer managed or AWS managed. For more
         *        information about the difference, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">Customer Master
         *        Keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @see KeyManagerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyManagerType
         */
        Builder keyManager(KeyManagerType keyManager);

        /**
         * <p>
         * Describes the type of key material in the CMK.
         * </p>
         * 
         * @param customerMasterKeySpec
         *        Describes the type of key material in the CMK.
         * @see CustomerMasterKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomerMasterKeySpec
         */
        Builder customerMasterKeySpec(String customerMasterKeySpec);

        /**
         * <p>
         * Describes the type of key material in the CMK.
         * </p>
         * 
         * @param customerMasterKeySpec
         *        Describes the type of key material in the CMK.
         * @see CustomerMasterKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomerMasterKeySpec
         */
        Builder customerMasterKeySpec(CustomerMasterKeySpec customerMasterKeySpec);

        /**
         * <p>
         * The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms
         * within AWS KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * </p>
         * 
         * @param encryptionAlgorithms
         *        The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption
         *        algorithms within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAlgorithmsWithStrings(Collection<String> encryptionAlgorithms);

        /**
         * <p>
         * The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms
         * within AWS KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * </p>
         * 
         * @param encryptionAlgorithms
         *        The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption
         *        algorithms within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAlgorithmsWithStrings(String... encryptionAlgorithms);

        /**
         * <p>
         * The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms
         * within AWS KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * </p>
         * 
         * @param encryptionAlgorithms
         *        The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption
         *        algorithms within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAlgorithms(Collection<EncryptionAlgorithmSpec> encryptionAlgorithms);

        /**
         * <p>
         * The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption algorithms
         * within AWS KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * </p>
         * 
         * @param encryptionAlgorithms
         *        The encryption algorithms that the CMK supports. You cannot use the CMK with other encryption
         *        algorithms within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>ENCRYPT_DECRYPT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAlgorithms(EncryptionAlgorithmSpec... encryptionAlgorithms);

        /**
         * <p>
         * The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within AWS
         * KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * </p>
         * 
         * @param signingAlgorithms
         *        The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms
         *        within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingAlgorithmsWithStrings(Collection<String> signingAlgorithms);

        /**
         * <p>
         * The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within AWS
         * KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * </p>
         * 
         * @param signingAlgorithms
         *        The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms
         *        within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingAlgorithmsWithStrings(String... signingAlgorithms);

        /**
         * <p>
         * The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within AWS
         * KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * </p>
         * 
         * @param signingAlgorithms
         *        The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms
         *        within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingAlgorithms(Collection<SigningAlgorithmSpec> signingAlgorithms);

        /**
         * <p>
         * The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms within AWS
         * KMS.
         * </p>
         * <p>
         * This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * </p>
         * 
         * @param signingAlgorithms
         *        The signing algorithms that the CMK supports. You cannot use the CMK with other signing algorithms
         *        within AWS KMS.</p>
         *        <p>
         *        This field appears only when the <code>KeyUsage</code> of the CMK is <code>SIGN_VERIFY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingAlgorithms(SigningAlgorithmSpec... signingAlgorithms);
    }

    static final class BuilderImpl implements Builder {
        private String awsAccountId;

        private String keyId;

        private String arn;

        private Instant creationDate;

        private Boolean enabled;

        private String description;

        private String keyUsage;

        private String keyState;

        private Instant deletionDate;

        private Instant validTo;

        private String origin;

        private String customKeyStoreId;

        private String cloudHsmClusterId;

        private String expirationModel;

        private String keyManager;

        private String customerMasterKeySpec;

        private List<String> encryptionAlgorithms = DefaultSdkAutoConstructList.getInstance();

        private List<String> signingAlgorithms = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(KeyMetadata model) {
            awsAccountId(model.awsAccountId);
            keyId(model.keyId);
            arn(model.arn);
            creationDate(model.creationDate);
            enabled(model.enabled);
            description(model.description);
            keyUsage(model.keyUsage);
            keyState(model.keyState);
            deletionDate(model.deletionDate);
            validTo(model.validTo);
            origin(model.origin);
            customKeyStoreId(model.customKeyStoreId);
            cloudHsmClusterId(model.cloudHsmClusterId);
            expirationModel(model.expirationModel);
            keyManager(model.keyManager);
            customerMasterKeySpec(model.customerMasterKeySpec);
            encryptionAlgorithmsWithStrings(model.encryptionAlgorithms);
            signingAlgorithmsWithStrings(model.signingAlgorithms);
        }

        public final String getAwsAccountId() {
            return awsAccountId;
        }

        @Override
        public final Builder awsAccountId(String awsAccountId) {
            this.awsAccountId = awsAccountId;
            return this;
        }

        public final void setAwsAccountId(String awsAccountId) {
            this.awsAccountId = awsAccountId;
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKeyUsage() {
            return keyUsage;
        }

        @Override
        public final Builder keyUsage(String keyUsage) {
            this.keyUsage = keyUsage;
            return this;
        }

        @Override
        public final Builder keyUsage(KeyUsageType keyUsage) {
            this.keyUsage(keyUsage == null ? null : keyUsage.toString());
            return this;
        }

        public final void setKeyUsage(String keyUsage) {
            this.keyUsage = keyUsage;
        }

        public final String getKeyState() {
            return keyState;
        }

        @Override
        public final Builder keyState(String keyState) {
            this.keyState = keyState;
            return this;
        }

        @Override
        public final Builder keyState(KeyState keyState) {
            this.keyState(keyState == null ? null : keyState.toString());
            return this;
        }

        public final void setKeyState(String keyState) {
            this.keyState = keyState;
        }

        public final Instant getDeletionDate() {
            return deletionDate;
        }

        @Override
        public final Builder deletionDate(Instant deletionDate) {
            this.deletionDate = deletionDate;
            return this;
        }

        public final void setDeletionDate(Instant deletionDate) {
            this.deletionDate = deletionDate;
        }

        public final Instant getValidTo() {
            return validTo;
        }

        @Override
        public final Builder validTo(Instant validTo) {
            this.validTo = validTo;
            return this;
        }

        public final void setValidTo(Instant validTo) {
            this.validTo = validTo;
        }

        public final String getOrigin() {
            return origin;
        }

        @Override
        public final Builder origin(String origin) {
            this.origin = origin;
            return this;
        }

        @Override
        public final Builder origin(OriginType origin) {
            this.origin(origin == null ? null : origin.toString());
            return this;
        }

        public final void setOrigin(String origin) {
            this.origin = origin;
        }

        public final String getCustomKeyStoreId() {
            return customKeyStoreId;
        }

        @Override
        public final Builder customKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
            return this;
        }

        public final void setCustomKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
        }

        public final String getCloudHsmClusterId() {
            return cloudHsmClusterId;
        }

        @Override
        public final Builder cloudHsmClusterId(String cloudHsmClusterId) {
            this.cloudHsmClusterId = cloudHsmClusterId;
            return this;
        }

        public final void setCloudHsmClusterId(String cloudHsmClusterId) {
            this.cloudHsmClusterId = cloudHsmClusterId;
        }

        public final String getExpirationModel() {
            return expirationModel;
        }

        @Override
        public final Builder expirationModel(String expirationModel) {
            this.expirationModel = expirationModel;
            return this;
        }

        @Override
        public final Builder expirationModel(ExpirationModelType expirationModel) {
            this.expirationModel(expirationModel == null ? null : expirationModel.toString());
            return this;
        }

        public final void setExpirationModel(String expirationModel) {
            this.expirationModel = expirationModel;
        }

        public final String getKeyManager() {
            return keyManager;
        }

        @Override
        public final Builder keyManager(String keyManager) {
            this.keyManager = keyManager;
            return this;
        }

        @Override
        public final Builder keyManager(KeyManagerType keyManager) {
            this.keyManager(keyManager == null ? null : keyManager.toString());
            return this;
        }

        public final void setKeyManager(String keyManager) {
            this.keyManager = keyManager;
        }

        public final String getCustomerMasterKeySpec() {
            return customerMasterKeySpec;
        }

        @Override
        public final Builder customerMasterKeySpec(String customerMasterKeySpec) {
            this.customerMasterKeySpec = customerMasterKeySpec;
            return this;
        }

        @Override
        public final Builder customerMasterKeySpec(CustomerMasterKeySpec customerMasterKeySpec) {
            this.customerMasterKeySpec(customerMasterKeySpec == null ? null : customerMasterKeySpec.toString());
            return this;
        }

        public final void setCustomerMasterKeySpec(String customerMasterKeySpec) {
            this.customerMasterKeySpec = customerMasterKeySpec;
        }

        public final Collection<String> getEncryptionAlgorithms() {
            return encryptionAlgorithms;
        }

        @Override
        public final Builder encryptionAlgorithmsWithStrings(Collection<String> encryptionAlgorithms) {
            this.encryptionAlgorithms = EncryptionAlgorithmSpecListCopier.copy(encryptionAlgorithms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionAlgorithmsWithStrings(String... encryptionAlgorithms) {
            encryptionAlgorithmsWithStrings(Arrays.asList(encryptionAlgorithms));
            return this;
        }

        @Override
        public final Builder encryptionAlgorithms(Collection<EncryptionAlgorithmSpec> encryptionAlgorithms) {
            this.encryptionAlgorithms = EncryptionAlgorithmSpecListCopier.copyEnumToString(encryptionAlgorithms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionAlgorithms(EncryptionAlgorithmSpec... encryptionAlgorithms) {
            encryptionAlgorithms(Arrays.asList(encryptionAlgorithms));
            return this;
        }

        public final void setEncryptionAlgorithms(Collection<String> encryptionAlgorithms) {
            this.encryptionAlgorithms = EncryptionAlgorithmSpecListCopier.copy(encryptionAlgorithms);
        }

        public final Collection<String> getSigningAlgorithms() {
            return signingAlgorithms;
        }

        @Override
        public final Builder signingAlgorithmsWithStrings(Collection<String> signingAlgorithms) {
            this.signingAlgorithms = SigningAlgorithmSpecListCopier.copy(signingAlgorithms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signingAlgorithmsWithStrings(String... signingAlgorithms) {
            signingAlgorithmsWithStrings(Arrays.asList(signingAlgorithms));
            return this;
        }

        @Override
        public final Builder signingAlgorithms(Collection<SigningAlgorithmSpec> signingAlgorithms) {
            this.signingAlgorithms = SigningAlgorithmSpecListCopier.copyEnumToString(signingAlgorithms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signingAlgorithms(SigningAlgorithmSpec... signingAlgorithms) {
            signingAlgorithms(Arrays.asList(signingAlgorithms));
            return this;
        }

        public final void setSigningAlgorithms(Collection<String> signingAlgorithms) {
            this.signingAlgorithms = SigningAlgorithmSpecListCopier.copy(signingAlgorithms);
        }

        @Override
        public KeyMetadata build() {
            return new KeyMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
