/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kms.model.AlreadyExistsException;
import software.amazon.awssdk.services.kms.model.CancelKeyDeletionRequest;
import software.amazon.awssdk.services.kms.model.CancelKeyDeletionResponse;
import software.amazon.awssdk.services.kms.model.CloudHsmClusterInUseException;
import software.amazon.awssdk.services.kms.model.CloudHsmClusterInvalidConfigurationException;
import software.amazon.awssdk.services.kms.model.CloudHsmClusterNotActiveException;
import software.amazon.awssdk.services.kms.model.CloudHsmClusterNotFoundException;
import software.amazon.awssdk.services.kms.model.CloudHsmClusterNotRelatedException;
import software.amazon.awssdk.services.kms.model.ConnectCustomKeyStoreRequest;
import software.amazon.awssdk.services.kms.model.ConnectCustomKeyStoreResponse;
import software.amazon.awssdk.services.kms.model.CreateAliasRequest;
import software.amazon.awssdk.services.kms.model.CreateAliasResponse;
import software.amazon.awssdk.services.kms.model.CreateCustomKeyStoreRequest;
import software.amazon.awssdk.services.kms.model.CreateCustomKeyStoreResponse;
import software.amazon.awssdk.services.kms.model.CreateGrantRequest;
import software.amazon.awssdk.services.kms.model.CreateGrantResponse;
import software.amazon.awssdk.services.kms.model.CreateKeyRequest;
import software.amazon.awssdk.services.kms.model.CreateKeyResponse;
import software.amazon.awssdk.services.kms.model.CustomKeyStoreHasCmKsException;
import software.amazon.awssdk.services.kms.model.CustomKeyStoreInvalidStateException;
import software.amazon.awssdk.services.kms.model.CustomKeyStoreNameInUseException;
import software.amazon.awssdk.services.kms.model.CustomKeyStoreNotFoundException;
import software.amazon.awssdk.services.kms.model.DecryptRequest;
import software.amazon.awssdk.services.kms.model.DecryptResponse;
import software.amazon.awssdk.services.kms.model.DeleteAliasRequest;
import software.amazon.awssdk.services.kms.model.DeleteAliasResponse;
import software.amazon.awssdk.services.kms.model.DeleteCustomKeyStoreRequest;
import software.amazon.awssdk.services.kms.model.DeleteCustomKeyStoreResponse;
import software.amazon.awssdk.services.kms.model.DeleteImportedKeyMaterialRequest;
import software.amazon.awssdk.services.kms.model.DeleteImportedKeyMaterialResponse;
import software.amazon.awssdk.services.kms.model.DependencyTimeoutException;
import software.amazon.awssdk.services.kms.model.DescribeCustomKeyStoresRequest;
import software.amazon.awssdk.services.kms.model.DescribeCustomKeyStoresResponse;
import software.amazon.awssdk.services.kms.model.DescribeKeyRequest;
import software.amazon.awssdk.services.kms.model.DescribeKeyResponse;
import software.amazon.awssdk.services.kms.model.DisableKeyRequest;
import software.amazon.awssdk.services.kms.model.DisableKeyResponse;
import software.amazon.awssdk.services.kms.model.DisableKeyRotationRequest;
import software.amazon.awssdk.services.kms.model.DisableKeyRotationResponse;
import software.amazon.awssdk.services.kms.model.DisabledException;
import software.amazon.awssdk.services.kms.model.DisconnectCustomKeyStoreRequest;
import software.amazon.awssdk.services.kms.model.DisconnectCustomKeyStoreResponse;
import software.amazon.awssdk.services.kms.model.EnableKeyRequest;
import software.amazon.awssdk.services.kms.model.EnableKeyResponse;
import software.amazon.awssdk.services.kms.model.EnableKeyRotationRequest;
import software.amazon.awssdk.services.kms.model.EnableKeyRotationResponse;
import software.amazon.awssdk.services.kms.model.EncryptRequest;
import software.amazon.awssdk.services.kms.model.EncryptResponse;
import software.amazon.awssdk.services.kms.model.ExpiredImportTokenException;
import software.amazon.awssdk.services.kms.model.GenerateDataKeyRequest;
import software.amazon.awssdk.services.kms.model.GenerateDataKeyResponse;
import software.amazon.awssdk.services.kms.model.GenerateDataKeyWithoutPlaintextRequest;
import software.amazon.awssdk.services.kms.model.GenerateDataKeyWithoutPlaintextResponse;
import software.amazon.awssdk.services.kms.model.GenerateRandomRequest;
import software.amazon.awssdk.services.kms.model.GenerateRandomResponse;
import software.amazon.awssdk.services.kms.model.GetKeyPolicyRequest;
import software.amazon.awssdk.services.kms.model.GetKeyPolicyResponse;
import software.amazon.awssdk.services.kms.model.GetKeyRotationStatusRequest;
import software.amazon.awssdk.services.kms.model.GetKeyRotationStatusResponse;
import software.amazon.awssdk.services.kms.model.GetParametersForImportRequest;
import software.amazon.awssdk.services.kms.model.GetParametersForImportResponse;
import software.amazon.awssdk.services.kms.model.ImportKeyMaterialRequest;
import software.amazon.awssdk.services.kms.model.ImportKeyMaterialResponse;
import software.amazon.awssdk.services.kms.model.IncorrectKeyMaterialException;
import software.amazon.awssdk.services.kms.model.IncorrectTrustAnchorException;
import software.amazon.awssdk.services.kms.model.InvalidAliasNameException;
import software.amazon.awssdk.services.kms.model.InvalidArnException;
import software.amazon.awssdk.services.kms.model.InvalidCiphertextException;
import software.amazon.awssdk.services.kms.model.InvalidGrantIdException;
import software.amazon.awssdk.services.kms.model.InvalidGrantTokenException;
import software.amazon.awssdk.services.kms.model.InvalidImportTokenException;
import software.amazon.awssdk.services.kms.model.InvalidKeyUsageException;
import software.amazon.awssdk.services.kms.model.InvalidMarkerException;
import software.amazon.awssdk.services.kms.model.KeyUnavailableException;
import software.amazon.awssdk.services.kms.model.KmsException;
import software.amazon.awssdk.services.kms.model.KmsInternalException;
import software.amazon.awssdk.services.kms.model.KmsInvalidStateException;
import software.amazon.awssdk.services.kms.model.KmsRequest;
import software.amazon.awssdk.services.kms.model.LimitExceededException;
import software.amazon.awssdk.services.kms.model.ListAliasesRequest;
import software.amazon.awssdk.services.kms.model.ListAliasesResponse;
import software.amazon.awssdk.services.kms.model.ListGrantsRequest;
import software.amazon.awssdk.services.kms.model.ListGrantsResponse;
import software.amazon.awssdk.services.kms.model.ListKeyPoliciesRequest;
import software.amazon.awssdk.services.kms.model.ListKeyPoliciesResponse;
import software.amazon.awssdk.services.kms.model.ListKeysRequest;
import software.amazon.awssdk.services.kms.model.ListKeysResponse;
import software.amazon.awssdk.services.kms.model.ListResourceTagsRequest;
import software.amazon.awssdk.services.kms.model.ListResourceTagsResponse;
import software.amazon.awssdk.services.kms.model.ListRetirableGrantsRequest;
import software.amazon.awssdk.services.kms.model.ListRetirableGrantsResponse;
import software.amazon.awssdk.services.kms.model.MalformedPolicyDocumentException;
import software.amazon.awssdk.services.kms.model.NotFoundException;
import software.amazon.awssdk.services.kms.model.PutKeyPolicyRequest;
import software.amazon.awssdk.services.kms.model.PutKeyPolicyResponse;
import software.amazon.awssdk.services.kms.model.ReEncryptRequest;
import software.amazon.awssdk.services.kms.model.ReEncryptResponse;
import software.amazon.awssdk.services.kms.model.RetireGrantRequest;
import software.amazon.awssdk.services.kms.model.RetireGrantResponse;
import software.amazon.awssdk.services.kms.model.RevokeGrantRequest;
import software.amazon.awssdk.services.kms.model.RevokeGrantResponse;
import software.amazon.awssdk.services.kms.model.ScheduleKeyDeletionRequest;
import software.amazon.awssdk.services.kms.model.ScheduleKeyDeletionResponse;
import software.amazon.awssdk.services.kms.model.TagException;
import software.amazon.awssdk.services.kms.model.TagResourceRequest;
import software.amazon.awssdk.services.kms.model.TagResourceResponse;
import software.amazon.awssdk.services.kms.model.UnsupportedOperationException;
import software.amazon.awssdk.services.kms.model.UntagResourceRequest;
import software.amazon.awssdk.services.kms.model.UntagResourceResponse;
import software.amazon.awssdk.services.kms.model.UpdateAliasRequest;
import software.amazon.awssdk.services.kms.model.UpdateAliasResponse;
import software.amazon.awssdk.services.kms.model.UpdateCustomKeyStoreRequest;
import software.amazon.awssdk.services.kms.model.UpdateCustomKeyStoreResponse;
import software.amazon.awssdk.services.kms.model.UpdateKeyDescriptionRequest;
import software.amazon.awssdk.services.kms.model.UpdateKeyDescriptionResponse;
import software.amazon.awssdk.services.kms.paginators.ListAliasesIterable;
import software.amazon.awssdk.services.kms.paginators.ListGrantsIterable;
import software.amazon.awssdk.services.kms.paginators.ListKeyPoliciesIterable;
import software.amazon.awssdk.services.kms.paginators.ListKeysIterable;
import software.amazon.awssdk.services.kms.transform.CancelKeyDeletionRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ConnectCustomKeyStoreRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.CreateCustomKeyStoreRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.CreateGrantRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.CreateKeyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DecryptRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DeleteAliasRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DeleteCustomKeyStoreRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DeleteImportedKeyMaterialRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DescribeCustomKeyStoresRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DescribeKeyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DisableKeyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DisableKeyRotationRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.DisconnectCustomKeyStoreRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.EnableKeyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.EnableKeyRotationRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.EncryptRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.GenerateDataKeyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.GenerateDataKeyWithoutPlaintextRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.GenerateRandomRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.GetKeyPolicyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.GetKeyRotationStatusRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.GetParametersForImportRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ImportKeyMaterialRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ListAliasesRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ListGrantsRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ListKeyPoliciesRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ListKeysRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ListResourceTagsRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ListRetirableGrantsRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.PutKeyPolicyRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ReEncryptRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.RetireGrantRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.RevokeGrantRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.ScheduleKeyDeletionRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.UpdateAliasRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.UpdateCustomKeyStoreRequestMarshaller;
import software.amazon.awssdk.services.kms.transform.UpdateKeyDescriptionRequestMarshaller;

/**
 * Internal implementation of {@link KmsClient}.
 *
 * @see KmsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKmsClient implements KmsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKmsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels the deletion of a customer master key (CMK). When this operation is successful, the CMK is set to the
     * <code>Disabled</code> state. To enable a CMK, use <a>EnableKey</a>. You cannot perform this operation on a CMK in
     * a different AWS account.
     * </p>
     * <p>
     * For more information about scheduling and canceling deletion of a CMK, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/deleting-keys.html">Deleting Customer Master Keys</a>
     * in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param cancelKeyDeletionRequest
     * @return Result of the CancelKeyDeletion operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.CancelKeyDeletion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/CancelKeyDeletion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelKeyDeletionResponse cancelKeyDeletion(CancelKeyDeletionRequest cancelKeyDeletionRequest)
            throws NotFoundException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelKeyDeletionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelKeyDeletionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CancelKeyDeletionRequest, CancelKeyDeletionResponse>()
                .withOperationName("CancelKeyDeletion").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(cancelKeyDeletionRequest)
                .withMarshaller(new CancelKeyDeletionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Connects or reconnects a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * to its associated AWS CloudHSM cluster.
     * </p>
     * <p>
     * The custom key store must be connected before you can create customer master keys (CMKs) in the key store or use
     * the CMKs it contains. You can disconnect and reconnect a custom key store at any time.
     * </p>
     * <p>
     * To connect a custom key store, its associated AWS CloudHSM cluster must have at least one active HSM. To get the
     * number of active HSMs in a cluster, use the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeClusters.html">DescribeClusters</a>
     * operation. To add HSMs to the cluster, use the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html">CreateHsm</a> operation.
     * </p>
     * <p>
     * The connection process can take an extended amount of time to complete; up to 20 minutes. This operation starts
     * the connection process, but it does not wait for it to complete. When it succeeds, this operation quickly returns
     * an HTTP 200 response and a JSON object with no properties. However, this response does not indicate that the
     * custom key store is connected. To get the connection state of the custom key store, use the
     * <a>DescribeCustomKeyStores</a> operation.
     * </p>
     * <p>
     * During the connection process, AWS KMS finds the AWS CloudHSM cluster that is associated with the custom key
     * store, creates the connection infrastructure, connects to the cluster, logs into the AWS CloudHSM client as the
     * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-store-concepts.html#concept-kmsuser">
     * <code>kmsuser</code> crypto user</a> (CU), and rotates its password.
     * </p>
     * <p>
     * The <code>ConnectCustomKeyStore</code> operation might fail for various reasons. To find the reason, use the
     * <a>DescribeCustomKeyStores</a> operation and see the <code>ConnectionErrorCode</code> in the response. For help
     * interpreting the <code>ConnectionErrorCode</code>, see <a>CustomKeyStoresListEntry</a>.
     * </p>
     * <p>
     * To fix the failure, use the <a>DisconnectCustomKeyStore</a> operation to disconnect the custom key store, correct
     * the error, use the <a>UpdateCustomKeyStore</a> operation if necessary, and then use
     * <code>ConnectCustomKeyStore</code> again.
     * </p>
     * <p>
     * If you are having trouble connecting or disconnecting a custom key store, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom Key
     * Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param connectCustomKeyStoreRequest
     * @return Result of the ConnectCustomKeyStore operation returned by the service.
     * @throws CloudHsmClusterNotActiveException
     *         The request was rejected because the AWS CloudHSM cluster that is associated with the custom key store is
     *         not active. Initialize and activate the cluster and try the command again. For detailed instructions, see
     *         <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/getting-started.html">Getting Started</a>
     *         in the <i>AWS CloudHSM User Guide</i>.
     * @throws CustomKeyStoreInvalidStateException
     *         The request was rejected because of the <code>ConnectionState</code> of the custom key store. To get the
     *         <code>ConnectionState</code> of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.</p>
     *         <p>
     *         This exception is thrown under the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You requested the <a>CreateKey</a> or <a>GenerateRandom</a> operation in a custom key store that is not
     *         connected. These operations are valid only when the custom key store <code>ConnectionState</code> is
     *         <code>CONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>UpdateCustomKeyStore</a> or <a>DeleteCustomKeyStore</a> operation on a custom key
     *         store that is not disconnected. This operation is valid only when the custom key store
     *         <code>ConnectionState</code> is <code>DISCONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>ConnectCustomKeyStore</a> operation on a custom key store with a
     *         <code>ConnectionState</code> of <code>DISCONNECTING</code> or <code>FAILED</code>. This operation is
     *         valid for all other <code>ConnectionState</code> values.
     *         </p>
     *         </li>
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws CloudHsmClusterInvalidConfigurationException
     *         The request was rejected because the associated AWS CloudHSM cluster did not meet the configuration
     *         requirements for a custom key store.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The cluster must be configured with private subnets in at least two different Availability Zones in the
     *         Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">security group for
     *         the cluster</a> (cloudhsm-cluster-<i>&lt;cluster-id&gt;</i>-sg) must include inbound rules and outbound
     *         rules that allow TCP traffic on ports 2223-2225. The <b>Source</b> in the inbound rules and the
     *         <b>Destination</b> in the outbound rules must match the security group ID. These rules are set by default
     *         when you create the cluster. Do not delete or change them. To get information about a particular security
     *         group, use the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html"
     *         >DescribeSecurityGroups</a> operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster must contain at least as many HSMs as the operation requires. To add HSMs, use the AWS
     *         CloudHSM <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html">CreateHsm</a>
     *         operation.
     *         </p>
     *         <p>
     *         For the <a>CreateCustomKeyStore</a>, <a>UpdateCustomKeyStore</a>, and <a>CreateKey</a> operations, the
     *         AWS CloudHSM cluster must have at least two active HSMs, each in a different Availability Zone. For the
     *         <a>ConnectCustomKeyStore</a> operation, the AWS CloudHSM must contain at least one active HSM.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the requirements for an AWS CloudHSM cluster that is associated with a custom key
     *         store, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/create-keystore.html#before-keystore"
     *         >Assemble the Prerequisites</a> in the <i>AWS Key Management Service Developer Guide</i>. For information
     *         about creating a private subnet for an AWS CloudHSM cluster, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/create-subnets.html">Create a Private
     *         Subnet</a> in the <i>AWS CloudHSM User Guide</i>. For information about cluster security groups, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">Configure a Default
     *         Security Group</a> in the <i> <i>AWS CloudHSM User Guide</i> </i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ConnectCustomKeyStore
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ConnectCustomKeyStore" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ConnectCustomKeyStoreResponse connectCustomKeyStore(ConnectCustomKeyStoreRequest connectCustomKeyStoreRequest)
            throws CloudHsmClusterNotActiveException, CustomKeyStoreInvalidStateException, CustomKeyStoreNotFoundException,
            KmsInternalException, CloudHsmClusterInvalidConfigurationException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ConnectCustomKeyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ConnectCustomKeyStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ConnectCustomKeyStoreRequest, ConnectCustomKeyStoreResponse>()
                .withOperationName("ConnectCustomKeyStore").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(connectCustomKeyStoreRequest)
                .withMarshaller(new ConnectCustomKeyStoreRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a display name for a customer managed customer master key (CMK). You can use an alias to identify a CMK
     * in selected operations, such as <a>Encrypt</a> and <a>GenerateDataKey</a>.
     * </p>
     * <p>
     * Each CMK can have multiple aliases, but each alias points to only one CMK. The alias name must be unique in the
     * AWS account and region. To simplify code that runs in multiple regions, use the same alias name, but point it to
     * a different CMK in each region.
     * </p>
     * <p>
     * Because an alias is not a property of a CMK, you can delete and change the aliases of a CMK without affecting the
     * CMK. Also, aliases do not appear in the response from the <a>DescribeKey</a> operation. To get the aliases of all
     * CMKs, use the <a>ListAliases</a> operation.
     * </p>
     * <p>
     * The alias name must begin with <code>alias/</code> followed by a name, such as <code>alias/ExampleAlias</code>.
     * It can contain only alphanumeric characters, forward slashes (/), underscores (_), and dashes (-). The alias name
     * cannot begin with <code>alias/aws/</code>. The <code>alias/aws/</code> prefix is reserved for <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk">AWS managed CMKs</a>.
     * </p>
     * <p>
     * The alias and the CMK it is mapped to must be in the same AWS account and the same region. You cannot perform
     * this operation on an alias in a different AWS account.
     * </p>
     * <p>
     * To map an existing alias to a different CMK, call <a>UpdateAlias</a>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param createAliasRequest
     * @return Result of the CreateAlias operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws AlreadyExistsException
     *         The request was rejected because it attempted to create a resource that already exists.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidAliasNameException
     *         The request was rejected because the specified alias name is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws LimitExceededException
     *         The request was rejected because a limit was exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html">Limits</a> in the <i>AWS Key
     *         Management Service Developer Guide</i>.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.CreateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/CreateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAliasResponse createAlias(CreateAliasRequest createAliasRequest) throws DependencyTimeoutException,
            AlreadyExistsException, NotFoundException, InvalidAliasNameException, KmsInternalException, LimitExceededException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                .withOperationName("CreateAlias").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createAliasRequest)
                .withMarshaller(new CreateAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom
     * key store</a> that is associated with an <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/clusters.html">AWS CloudHSM cluster</a> that you own
     * and manage.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     * <p>
     * Before you create the custom key store, you must assemble the required elements, including an AWS CloudHSM
     * cluster that fulfills the requirements for a custom key store. For details about the required elements, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/create-keystore.html#before-keystore">Assemble the
     * Prerequisites</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * When the operation completes successfully, it returns the ID of the new custom key store. Before you can use your
     * new custom key store, you need to use the <a>ConnectCustomKeyStore</a> operation to connect the new key store to
     * its AWS CloudHSM cluster. Even if you are not going to use your custom key store immediately, you might want to
     * connect it to verify that all settings are correct and then disconnect it until you are ready to use it.
     * </p>
     * <p>
     * For help with failures, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting a Custom Key
     * Store</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param createCustomKeyStoreRequest
     * @return Result of the CreateCustomKeyStore operation returned by the service.
     * @throws CloudHsmClusterInUseException
     *         The request was rejected because the specified AWS CloudHSM cluster is already associated with a custom
     *         key store or it shares a backup history with a cluster that is associated with a custom key store. Each
     *         custom key store must be associated with a different AWS CloudHSM cluster.</p>
     *         <p>
     *         Clusters that share a backup history have the same cluster certificate. To view the cluster certificate
     *         of a cluster, use the <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeClusters.html"
     *         >DescribeClusters</a> operation.
     * @throws CustomKeyStoreNameInUseException
     *         The request was rejected because the specified custom key store name is already assigned to another
     *         custom key store in the account. Try again with a custom key store name that is unique in the account.
     * @throws CloudHsmClusterNotFoundException
     *         The request was rejected because AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster
     *         ID. Retry the request with a different cluster ID.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws CloudHsmClusterNotActiveException
     *         The request was rejected because the AWS CloudHSM cluster that is associated with the custom key store is
     *         not active. Initialize and activate the cluster and try the command again. For detailed instructions, see
     *         <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/getting-started.html">Getting Started</a>
     *         in the <i>AWS CloudHSM User Guide</i>.
     * @throws IncorrectTrustAnchorException
     *         The request was rejected because the trust anchor certificate in the request is not the trust anchor
     *         certificate for the specified AWS CloudHSM cluster.
     *         </p>
     *         <p>
     *         When you <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/initialize-cluster.html#sign-csr">initialize
     *         the cluster</a>, you create the trust anchor certificate and save it in the <code>customerCA.crt</code>
     *         file.
     * @throws CloudHsmClusterInvalidConfigurationException
     *         The request was rejected because the associated AWS CloudHSM cluster did not meet the configuration
     *         requirements for a custom key store.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The cluster must be configured with private subnets in at least two different Availability Zones in the
     *         Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">security group for
     *         the cluster</a> (cloudhsm-cluster-<i>&lt;cluster-id&gt;</i>-sg) must include inbound rules and outbound
     *         rules that allow TCP traffic on ports 2223-2225. The <b>Source</b> in the inbound rules and the
     *         <b>Destination</b> in the outbound rules must match the security group ID. These rules are set by default
     *         when you create the cluster. Do not delete or change them. To get information about a particular security
     *         group, use the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html"
     *         >DescribeSecurityGroups</a> operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster must contain at least as many HSMs as the operation requires. To add HSMs, use the AWS
     *         CloudHSM <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html">CreateHsm</a>
     *         operation.
     *         </p>
     *         <p>
     *         For the <a>CreateCustomKeyStore</a>, <a>UpdateCustomKeyStore</a>, and <a>CreateKey</a> operations, the
     *         AWS CloudHSM cluster must have at least two active HSMs, each in a different Availability Zone. For the
     *         <a>ConnectCustomKeyStore</a> operation, the AWS CloudHSM must contain at least one active HSM.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the requirements for an AWS CloudHSM cluster that is associated with a custom key
     *         store, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/create-keystore.html#before-keystore"
     *         >Assemble the Prerequisites</a> in the <i>AWS Key Management Service Developer Guide</i>. For information
     *         about creating a private subnet for an AWS CloudHSM cluster, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/create-subnets.html">Create a Private
     *         Subnet</a> in the <i>AWS CloudHSM User Guide</i>. For information about cluster security groups, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">Configure a Default
     *         Security Group</a> in the <i> <i>AWS CloudHSM User Guide</i> </i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.CreateCustomKeyStore
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/CreateCustomKeyStore" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCustomKeyStoreResponse createCustomKeyStore(CreateCustomKeyStoreRequest createCustomKeyStoreRequest)
            throws CloudHsmClusterInUseException, CustomKeyStoreNameInUseException, CloudHsmClusterNotFoundException,
            KmsInternalException, CloudHsmClusterNotActiveException, IncorrectTrustAnchorException,
            CloudHsmClusterInvalidConfigurationException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCustomKeyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCustomKeyStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateCustomKeyStoreRequest, CreateCustomKeyStoreResponse>()
                .withOperationName("CreateCustomKeyStore").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createCustomKeyStoreRequest)
                .withMarshaller(new CreateCustomKeyStoreRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds a grant to a customer master key (CMK). The grant allows the grantee principal to use the CMK when the
     * conditions specified in the grant are met. When setting permissions, grants are an alternative to key policies.
     * </p>
     * <p>
     * To create a grant that allows a cryptographic operation only when the encryption context in the operation request
     * matches or includes a specified encryption context, use the <code>Constraints</code> parameter. For details, see
     * <a>GrantConstraints</a>.
     * </p>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN in the value of the
     * <code>KeyId</code> parameter. For more information about grants, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/grants.html">Grants</a> in the <i> <i>AWS Key
     * Management Service Developer Guide</i> </i>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param createGrantRequest
     * @return Result of the CreateGrant operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws LimitExceededException
     *         The request was rejected because a limit was exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html">Limits</a> in the <i>AWS Key
     *         Management Service Developer Guide</i>.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.CreateGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/CreateGrant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGrantResponse createGrant(CreateGrantRequest createGrantRequest) throws NotFoundException, DisabledException,
            DependencyTimeoutException, InvalidArnException, KmsInternalException, InvalidGrantTokenException,
            LimitExceededException, KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGrantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateGrantRequest, CreateGrantResponse>()
                .withOperationName("CreateGrant").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createGrantRequest)
                .withMarshaller(new CreateGrantRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a customer managed <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">customer master key</a>
     * (CMK) in your AWS account.
     * </p>
     * <p>
     * You can use a CMK to encrypt small amounts of data (up to 4096 bytes) directly. But CMKs are more commonly used
     * to encrypt the <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#data-keys">data
     * keys</a> that are used to encrypt data.
     * </p>
     * <p>
     * To create a CMK for imported key material, use the <code>Origin</code> parameter with a value of
     * <code>EXTERNAL</code>.
     * </p>
     * <p>
     * To create a CMK in a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>,
     * use the <code>CustomKeyStoreId</code> parameter to specify the custom key store. You must also use the
     * <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>. The AWS CloudHSM cluster that is
     * associated with the custom key store must have at least two active HSMs in different Availability Zones in the
     * AWS Region.
     * </p>
     * <p>
     * You cannot use this operation to create a CMK in a different AWS account.
     * </p>
     *
     * @param createKeyRequest
     * @return Result of the CreateKey operation returned by the service.
     * @throws MalformedPolicyDocumentException
     *         The request was rejected because the specified policy is not syntactically or semantically correct.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws LimitExceededException
     *         The request was rejected because a limit was exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html">Limits</a> in the <i>AWS Key
     *         Management Service Developer Guide</i>.
     * @throws TagException
     *         The request was rejected because one or more tags are not valid.
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws CustomKeyStoreInvalidStateException
     *         The request was rejected because of the <code>ConnectionState</code> of the custom key store. To get the
     *         <code>ConnectionState</code> of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.</p>
     *         <p>
     *         This exception is thrown under the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You requested the <a>CreateKey</a> or <a>GenerateRandom</a> operation in a custom key store that is not
     *         connected. These operations are valid only when the custom key store <code>ConnectionState</code> is
     *         <code>CONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>UpdateCustomKeyStore</a> or <a>DeleteCustomKeyStore</a> operation on a custom key
     *         store that is not disconnected. This operation is valid only when the custom key store
     *         <code>ConnectionState</code> is <code>DISCONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>ConnectCustomKeyStore</a> operation on a custom key store with a
     *         <code>ConnectionState</code> of <code>DISCONNECTING</code> or <code>FAILED</code>. This operation is
     *         valid for all other <code>ConnectionState</code> values.
     *         </p>
     *         </li>
     * @throws CloudHsmClusterInvalidConfigurationException
     *         The request was rejected because the associated AWS CloudHSM cluster did not meet the configuration
     *         requirements for a custom key store.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The cluster must be configured with private subnets in at least two different Availability Zones in the
     *         Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">security group for
     *         the cluster</a> (cloudhsm-cluster-<i>&lt;cluster-id&gt;</i>-sg) must include inbound rules and outbound
     *         rules that allow TCP traffic on ports 2223-2225. The <b>Source</b> in the inbound rules and the
     *         <b>Destination</b> in the outbound rules must match the security group ID. These rules are set by default
     *         when you create the cluster. Do not delete or change them. To get information about a particular security
     *         group, use the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html"
     *         >DescribeSecurityGroups</a> operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster must contain at least as many HSMs as the operation requires. To add HSMs, use the AWS
     *         CloudHSM <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html">CreateHsm</a>
     *         operation.
     *         </p>
     *         <p>
     *         For the <a>CreateCustomKeyStore</a>, <a>UpdateCustomKeyStore</a>, and <a>CreateKey</a> operations, the
     *         AWS CloudHSM cluster must have at least two active HSMs, each in a different Availability Zone. For the
     *         <a>ConnectCustomKeyStore</a> operation, the AWS CloudHSM must contain at least one active HSM.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the requirements for an AWS CloudHSM cluster that is associated with a custom key
     *         store, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/create-keystore.html#before-keystore"
     *         >Assemble the Prerequisites</a> in the <i>AWS Key Management Service Developer Guide</i>. For information
     *         about creating a private subnet for an AWS CloudHSM cluster, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/create-subnets.html">Create a Private
     *         Subnet</a> in the <i>AWS CloudHSM User Guide</i>. For information about cluster security groups, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">Configure a Default
     *         Security Group</a> in the <i> <i>AWS CloudHSM User Guide</i> </i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.CreateKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/CreateKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateKeyResponse createKey(CreateKeyRequest createKeyRequest) throws MalformedPolicyDocumentException,
            DependencyTimeoutException, InvalidArnException, UnsupportedOperationException, KmsInternalException,
            LimitExceededException, TagException, CustomKeyStoreNotFoundException, CustomKeyStoreInvalidStateException,
            CloudHsmClusterInvalidConfigurationException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateKeyRequest, CreateKeyResponse>()
                .withOperationName("CreateKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createKeyRequest)
                .withMarshaller(new CreateKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Decrypts ciphertext. Ciphertext is plaintext that has been previously encrypted by using any of the following
     * operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GenerateDataKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GenerateDataKeyWithoutPlaintext</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>Encrypt</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Whenever possible, use key policies to give users permission to call the Decrypt operation on the CMK, instead of
     * IAM policies. Otherwise, you might create an IAM user policy that gives the user Decrypt permission on all CMKs.
     * This user could decrypt ciphertext that was encrypted by CMKs in other accounts if the key policy for the
     * cross-account CMK permits it. If you must use an IAM policy for <code>Decrypt</code> permissions, limit the user
     * to particular CMKs or particular trusted accounts.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param decryptRequest
     * @return Result of the Decrypt operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws InvalidCiphertextException
     *         The request was rejected because the specified ciphertext, or additional authenticated data incorporated
     *         into the ciphertext, such as the encryption context, is corrupted, missing, or otherwise invalid.
     * @throws KeyUnavailableException
     *         The request was rejected because the specified CMK was not available. The request can be retried.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.Decrypt
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/Decrypt" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DecryptResponse decrypt(DecryptRequest decryptRequest) throws NotFoundException, DisabledException,
            InvalidCiphertextException, KeyUnavailableException, DependencyTimeoutException, InvalidGrantTokenException,
            KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DecryptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DecryptResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DecryptRequest, DecryptResponse>().withOperationName("Decrypt")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(decryptRequest)
                .withMarshaller(new DecryptRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified alias. You cannot perform this operation on an alias in a different AWS account.
     * </p>
     * <p>
     * Because an alias is not a property of a CMK, you can delete and change the aliases of a CMK without affecting the
     * CMK. Also, aliases do not appear in the response from the <a>DescribeKey</a> operation. To get the aliases of all
     * CMKs, use the <a>ListAliases</a> operation.
     * </p>
     * <p>
     * Each CMK can have multiple aliases. To change the alias of a CMK, use <a>DeleteAlias</a> to delete the current
     * alias and <a>CreateAlias</a> to create a new alias. To associate an existing alias with a different customer
     * master key (CMK), call <a>UpdateAlias</a>.
     * </p>
     *
     * @param deleteAliasRequest
     * @return Result of the DeleteAlias operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DeleteAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DeleteAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAliasResponse deleteAlias(DeleteAliasRequest deleteAliasRequest) throws DependencyTimeoutException,
            NotFoundException, KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteAliasRequest, DeleteAliasResponse>()
                .withOperationName("DeleteAlias").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteAliasRequest)
                .withMarshaller(new DeleteAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a <a href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom
     * key store</a>. This operation does not delete the AWS CloudHSM cluster that is associated with the custom key
     * store, or affect any users or keys in the cluster.
     * </p>
     * <p>
     * The custom key store that you delete cannot contain any AWS KMS <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">customer master keys
     * (CMKs)</a>. Before deleting the key store, verify that you will never need to use any of the CMKs in the key
     * store for any cryptographic operations. Then, use <a>ScheduleKeyDeletion</a> to delete the AWS KMS customer
     * master keys (CMKs) from the key store. When the scheduled waiting period expires, the
     * <code>ScheduleKeyDeletion</code> operation deletes the CMKs. Then it makes a best effort to delete the key
     * material from the associated cluster. However, you might need to manually <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-orphaned-key">delete
     * the orphaned key material</a> from the cluster and its backups.
     * </p>
     * <p>
     * After all CMKs are deleted from AWS KMS, use <a>DisconnectCustomKeyStore</a> to disconnect the key store from AWS
     * KMS. Then, you can delete the custom key store.
     * </p>
     * <p>
     * Instead of deleting the custom key store, consider using <a>DisconnectCustomKeyStore</a> to disconnect it from
     * AWS KMS. While the key store is disconnected, you cannot create or use the CMKs in the key store. But, you do not
     * need to delete CMKs and you can reconnect a disconnected custom key store at any time.
     * </p>
     * <p>
     * If the operation succeeds, it returns a JSON object with no properties.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     *
     * @param deleteCustomKeyStoreRequest
     * @return Result of the DeleteCustomKeyStore operation returned by the service.
     * @throws CustomKeyStoreHasCmKsException
     *         The request was rejected because the custom key store contains AWS KMS customer master keys (CMKs). After
     *         verifying that you do not need to use the CMKs, use the <a>ScheduleKeyDeletion</a> operation to delete
     *         the CMKs. After they are deleted, you can delete the custom key store.
     * @throws CustomKeyStoreInvalidStateException
     *         The request was rejected because of the <code>ConnectionState</code> of the custom key store. To get the
     *         <code>ConnectionState</code> of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.</p>
     *         <p>
     *         This exception is thrown under the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You requested the <a>CreateKey</a> or <a>GenerateRandom</a> operation in a custom key store that is not
     *         connected. These operations are valid only when the custom key store <code>ConnectionState</code> is
     *         <code>CONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>UpdateCustomKeyStore</a> or <a>DeleteCustomKeyStore</a> operation on a custom key
     *         store that is not disconnected. This operation is valid only when the custom key store
     *         <code>ConnectionState</code> is <code>DISCONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>ConnectCustomKeyStore</a> operation on a custom key store with a
     *         <code>ConnectionState</code> of <code>DISCONNECTING</code> or <code>FAILED</code>. This operation is
     *         valid for all other <code>ConnectionState</code> values.
     *         </p>
     *         </li>
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DeleteCustomKeyStore
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DeleteCustomKeyStore" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteCustomKeyStoreResponse deleteCustomKeyStore(DeleteCustomKeyStoreRequest deleteCustomKeyStoreRequest)
            throws CustomKeyStoreHasCmKsException, CustomKeyStoreInvalidStateException, CustomKeyStoreNotFoundException,
            KmsInternalException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCustomKeyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCustomKeyStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteCustomKeyStoreRequest, DeleteCustomKeyStoreResponse>()
                .withOperationName("DeleteCustomKeyStore").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteCustomKeyStoreRequest)
                .withMarshaller(new DeleteCustomKeyStoreRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes key material that you previously imported. This operation makes the specified customer master key (CMK)
     * unusable. For more information about importing key material into AWS KMS, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>. You cannot perform this operation on a CMK in a different
     * AWS account.
     * </p>
     * <p>
     * When the specified CMK is in the <code>PendingDeletion</code> state, this operation does not change the CMK's
     * state. Otherwise, it changes the CMK's state to <code>PendingImport</code>.
     * </p>
     * <p>
     * After you delete key material, you can use <a>ImportKeyMaterial</a> to reimport the same key material into the
     * CMK.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param deleteImportedKeyMaterialRequest
     * @return Result of the DeleteImportedKeyMaterial operation returned by the service.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DeleteImportedKeyMaterial
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DeleteImportedKeyMaterial" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteImportedKeyMaterialResponse deleteImportedKeyMaterial(
            DeleteImportedKeyMaterialRequest deleteImportedKeyMaterialRequest) throws InvalidArnException,
            UnsupportedOperationException, DependencyTimeoutException, NotFoundException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteImportedKeyMaterialResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteImportedKeyMaterialResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteImportedKeyMaterialRequest, DeleteImportedKeyMaterialResponse>()
                        .withOperationName("DeleteImportedKeyMaterial").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteImportedKeyMaterialRequest)
                        .withMarshaller(new DeleteImportedKeyMaterialRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key stores</a>
     * in the account and region.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     * <p>
     * By default, this operation returns information about all custom key stores in the account and region. To get only
     * information about a particular custom key store, use either the <code>CustomKeyStoreName</code> or
     * <code>CustomKeyStoreId</code> parameter (but not both).
     * </p>
     * <p>
     * To determine whether the custom key store is connected to its AWS CloudHSM cluster, use the
     * <code>ConnectionState</code> element in the response. If an attempt to connect the custom key store failed, the
     * <code>ConnectionState</code> value is <code>FAILED</code> and the <code>ConnectionErrorCode</code> element in the
     * response indicates the cause of the failure. For help interpreting the <code>ConnectionErrorCode</code>, see
     * <a>CustomKeyStoresListEntry</a>.
     * </p>
     * <p>
     * Custom key stores have a <code>DISCONNECTED</code> connection state if the key store has never been connected or
     * you use the <a>DisconnectCustomKeyStore</a> operation to disconnect it. If your custom key store state is
     * <code>CONNECTED</code> but you are having trouble using it, make sure that its associated AWS CloudHSM cluster is
     * active and contains the minimum number of HSMs required for the operation, if any.
     * </p>
     * <p>
     * For help repairing your custom key store, see the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html">Troubleshooting Custom Key
     * Stores</a> topic in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param describeCustomKeyStoresRequest
     * @return Result of the DescribeCustomKeyStores operation returned by the service.
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DescribeCustomKeyStores
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DescribeCustomKeyStores" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeCustomKeyStoresResponse describeCustomKeyStores(DescribeCustomKeyStoresRequest describeCustomKeyStoresRequest)
            throws CustomKeyStoreNotFoundException, KmsInternalException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCustomKeyStoresResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCustomKeyStoresResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeCustomKeyStoresRequest, DescribeCustomKeyStoresResponse>()
                .withOperationName("DescribeCustomKeyStores").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeCustomKeyStoresRequest)
                .withMarshaller(new DescribeCustomKeyStoresRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides detailed information about the specified customer master key (CMK).
     * </p>
     * <p>
     * You can use <code>DescribeKey</code> on a predefined AWS alias, that is, an AWS alias with no key ID. When you
     * do, AWS KMS associates the alias with an <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys">AWS managed CMK</a> and
     * returns its <code>KeyId</code> and <code>Arn</code> in the response.
     * </p>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN or alias ARN in the value of
     * the KeyId parameter.
     * </p>
     *
     * @param describeKeyRequest
     * @return Result of the DescribeKey operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DescribeKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DescribeKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeKeyResponse describeKey(DescribeKeyRequest describeKeyRequest) throws NotFoundException, InvalidArnException,
            DependencyTimeoutException, KmsInternalException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeKeyRequest, DescribeKeyResponse>()
                .withOperationName("DescribeKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeKeyRequest)
                .withMarshaller(new DescribeKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the state of a customer master key (CMK) to disabled, thereby preventing its use for cryptographic
     * operations. You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     * <p>
     * For more information about how key state affects the use of a CMK, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects the Use of a
     * Customer Master Key</a> in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param disableKeyRequest
     * @return Result of the DisableKey operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DisableKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DisableKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DisableKeyResponse disableKey(DisableKeyRequest disableKeyRequest) throws NotFoundException, InvalidArnException,
            DependencyTimeoutException, KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisableKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisableKeyRequest, DisableKeyResponse>()
                .withOperationName("DisableKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disableKeyRequest)
                .withMarshaller(new DisableKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disables <a href="https://docs.aws.amazon.com/kms/latest/developerguide/rotate-keys.html">automatic rotation of
     * the key material</a> for the specified customer master key (CMK). You cannot perform this operation on a CMK in a
     * different AWS account.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param disableKeyRotationRequest
     * @return Result of the DisableKeyRotation operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DisableKeyRotation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DisableKeyRotation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DisableKeyRotationResponse disableKeyRotation(DisableKeyRotationRequest disableKeyRotationRequest)
            throws NotFoundException, DisabledException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, UnsupportedOperationException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableKeyRotationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableKeyRotationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisableKeyRotationRequest, DisableKeyRotationResponse>()
                .withOperationName("DisableKeyRotation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disableKeyRotationRequest)
                .withMarshaller(new DisableKeyRotationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disconnects the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * from its associated AWS CloudHSM cluster. While a custom key store is disconnected, you can manage the custom key
     * store and its customer master keys (CMKs), but you cannot create or use CMKs in the custom key store. You can
     * reconnect the custom key store at any time.
     * </p>
     * <note>
     * <p>
     * While a custom key store is disconnected, all attempts to create customer master keys (CMKs) in the custom key
     * store or to use existing CMKs in cryptographic operations will fail. This action can prevent users from storing
     * and accessing sensitive data.
     * </p>
     * </note>
     * <p/>
     * <p>
     * To find the connection state of a custom key store, use the <a>DescribeCustomKeyStores</a> operation. To
     * reconnect a custom key store, use the <a>ConnectCustomKeyStore</a> operation.
     * </p>
     * <p>
     * If the operation succeeds, it returns a JSON object with no properties.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     *
     * @param disconnectCustomKeyStoreRequest
     * @return Result of the DisconnectCustomKeyStore operation returned by the service.
     * @throws CustomKeyStoreInvalidStateException
     *         The request was rejected because of the <code>ConnectionState</code> of the custom key store. To get the
     *         <code>ConnectionState</code> of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.</p>
     *         <p>
     *         This exception is thrown under the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You requested the <a>CreateKey</a> or <a>GenerateRandom</a> operation in a custom key store that is not
     *         connected. These operations are valid only when the custom key store <code>ConnectionState</code> is
     *         <code>CONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>UpdateCustomKeyStore</a> or <a>DeleteCustomKeyStore</a> operation on a custom key
     *         store that is not disconnected. This operation is valid only when the custom key store
     *         <code>ConnectionState</code> is <code>DISCONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>ConnectCustomKeyStore</a> operation on a custom key store with a
     *         <code>ConnectionState</code> of <code>DISCONNECTING</code> or <code>FAILED</code>. This operation is
     *         valid for all other <code>ConnectionState</code> values.
     *         </p>
     *         </li>
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.DisconnectCustomKeyStore
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/DisconnectCustomKeyStore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisconnectCustomKeyStoreResponse disconnectCustomKeyStore(
            DisconnectCustomKeyStoreRequest disconnectCustomKeyStoreRequest) throws CustomKeyStoreInvalidStateException,
            CustomKeyStoreNotFoundException, KmsInternalException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectCustomKeyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisconnectCustomKeyStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisconnectCustomKeyStoreRequest, DisconnectCustomKeyStoreResponse>()
                        .withOperationName("DisconnectCustomKeyStore").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disconnectCustomKeyStoreRequest)
                        .withMarshaller(new DisconnectCustomKeyStoreRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the key state of a customer master key (CMK) to enabled. This allows you to use the CMK for cryptographic
     * operations. You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param enableKeyRequest
     * @return Result of the EnableKey operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws LimitExceededException
     *         The request was rejected because a limit was exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html">Limits</a> in the <i>AWS Key
     *         Management Service Developer Guide</i>.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.EnableKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/EnableKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EnableKeyResponse enableKey(EnableKeyRequest enableKeyRequest) throws NotFoundException, InvalidArnException,
            DependencyTimeoutException, KmsInternalException, LimitExceededException, KmsInvalidStateException,
            AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<EnableKeyRequest, EnableKeyResponse>()
                .withOperationName("EnableKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(enableKeyRequest)
                .withMarshaller(new EnableKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables <a href="https://docs.aws.amazon.com/kms/latest/developerguide/rotate-keys.html">automatic rotation of
     * the key material</a> for the specified customer master key (CMK). You cannot perform this operation on a CMK in a
     * different AWS account.
     * </p>
     * <p>
     * You cannot enable automatic rotation of CMKs with imported key material or CMKs in a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param enableKeyRotationRequest
     * @return Result of the EnableKeyRotation operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.EnableKeyRotation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/EnableKeyRotation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EnableKeyRotationResponse enableKeyRotation(EnableKeyRotationRequest enableKeyRotationRequest)
            throws NotFoundException, DisabledException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, UnsupportedOperationException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableKeyRotationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableKeyRotationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<EnableKeyRotationRequest, EnableKeyRotationResponse>()
                .withOperationName("EnableKeyRotation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(enableKeyRotationRequest)
                .withMarshaller(new EnableKeyRotationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Encrypts plaintext into ciphertext by using a customer master key (CMK). The <code>Encrypt</code> operation has
     * two primary use cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You can encrypt up to 4 kilobytes (4096 bytes) of arbitrary data such as an RSA key, a database password, or
     * other sensitive information.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can use the <code>Encrypt</code> operation to move encrypted data from one AWS region to another. In the
     * first region, generate a data key and use the plaintext key to encrypt the data. Then, in the new region, call
     * the <code>Encrypt</code> method on same plaintext data key. Now, you can safely move the encrypted data and
     * encrypted data key to the new region, and decrypt in the new region when necessary.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You don't need use this operation to encrypt a data key within a region. The <a>GenerateDataKey</a> and
     * <a>GenerateDataKeyWithoutPlaintext</a> operations return an encrypted data key.
     * </p>
     * <p>
     * Also, you don't need to use this operation to encrypt data in your application. You can use the plaintext and
     * encrypted data keys that the <code>GenerateDataKey</code> operation returns.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN or alias ARN in the value of
     * the KeyId parameter.
     * </p>
     *
     * @param encryptRequest
     * @return Result of the Encrypt operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws KeyUnavailableException
     *         The request was rejected because the specified CMK was not available. The request can be retried.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidKeyUsageException
     *         The request was rejected because the specified <code>KeySpec</code> value is not valid.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.Encrypt
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/Encrypt" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EncryptResponse encrypt(EncryptRequest encryptRequest) throws NotFoundException, DisabledException,
            KeyUnavailableException, DependencyTimeoutException, InvalidKeyUsageException, InvalidGrantTokenException,
            KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EncryptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EncryptResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<EncryptRequest, EncryptResponse>().withOperationName("Encrypt")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(encryptRequest)
                .withMarshaller(new EncryptRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Generates a unique data key. This operation returns a plaintext copy of the data key and a copy that is encrypted
     * under a customer master key (CMK) that you specify. You can use the plaintext key to encrypt your data outside of
     * KMS and store the encrypted data key with the encrypted data.
     * </p>
     * <p>
     * <code>GenerateDataKey</code> returns a unique data key for each request. The bytes in the key are not related to
     * the caller or CMK that is used to encrypt the data key.
     * </p>
     * <p>
     * To generate a data key, you need to specify the customer master key (CMK) that will be used to encrypt the data
     * key. You must also specify the length of the data key using either the <code>KeySpec</code> or
     * <code>NumberOfBytes</code> field (but not both). For common key lengths (128-bit and 256-bit symmetric keys), we
     * recommend that you use <code>KeySpec</code>. To perform this operation on a CMK in a different AWS account,
     * specify the key ARN or alias ARN in the value of the KeyId parameter.
     * </p>
     * <p>
     * You will find the plaintext copy of the data key in the <code>Plaintext</code> field of the response, and the
     * encrypted copy of the data key in the <code>CiphertextBlob</code> field.
     * </p>
     * <p>
     * We recommend that you use the following pattern to encrypt data locally in your application:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Use the <code>GenerateDataKey</code> operation to get a data encryption key.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the plaintext data key (returned in the <code>Plaintext</code> field of the response) to encrypt data
     * locally, then erase the plaintext data key from memory.
     * </p>
     * </li>
     * <li>
     * <p>
     * Store the encrypted data key (returned in the <code>CiphertextBlob</code> field of the response) alongside the
     * locally encrypted data.
     * </p>
     * </li>
     * </ol>
     * <p>
     * To decrypt data locally:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Use the <a>Decrypt</a> operation to decrypt the encrypted data key. The operation returns a plaintext copy of the
     * data key.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the plaintext data key to decrypt data locally, then erase the plaintext data key from memory.
     * </p>
     * </li>
     * </ol>
     * <p>
     * To get only an encrypted copy of the data key, use <a>GenerateDataKeyWithoutPlaintext</a>. To get a
     * cryptographically secure random byte string, use <a>GenerateRandom</a>.
     * </p>
     * <p>
     * You can use the optional encryption context to add additional security to your encryption operation. When you
     * specify an <code>EncryptionContext</code> in the <code>GenerateDataKey</code> operation, you must specify the
     * same encryption context (a case-sensitive exact match) in your request to <a>Decrypt</a> the data key. Otherwise,
     * the request to decrypt fails with an <code>InvalidCiphertextException</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption Context</a>
     * in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param generateDataKeyRequest
     * @return Result of the GenerateDataKey operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws KeyUnavailableException
     *         The request was rejected because the specified CMK was not available. The request can be retried.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidKeyUsageException
     *         The request was rejected because the specified <code>KeySpec</code> value is not valid.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.GenerateDataKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/GenerateDataKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GenerateDataKeyResponse generateDataKey(GenerateDataKeyRequest generateDataKeyRequest) throws NotFoundException,
            DisabledException, KeyUnavailableException, DependencyTimeoutException, InvalidKeyUsageException,
            InvalidGrantTokenException, KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GenerateDataKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GenerateDataKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GenerateDataKeyRequest, GenerateDataKeyResponse>()
                .withOperationName("GenerateDataKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(generateDataKeyRequest)
                .withMarshaller(new GenerateDataKeyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Generates a unique data key. This operation returns a data key that is encrypted under a customer master key
     * (CMK) that you specify. <code>GenerateDataKeyWithoutPlaintext</code> is identical to <a>GenerateDataKey</a>
     * except that returns only the encrypted copy of the data key.
     * </p>
     * <p>
     * Like <code>GenerateDataKey</code>, <code>GenerateDataKeyWithoutPlaintext</code> returns a unique data key for
     * each request. The bytes in the key are not related to the caller or CMK that is used to encrypt the data key.
     * </p>
     * <p>
     * This operation is useful for systems that need to encrypt data at some point, but not immediately. When you need
     * to encrypt the data, you call the <a>Decrypt</a> operation on the encrypted copy of the key.
     * </p>
     * <p>
     * It's also useful in distributed systems with different levels of trust. For example, you might store encrypted
     * data in containers. One component of your system creates new containers and stores an encrypted data key with
     * each container. Then, a different component puts the data into the containers. That component first decrypts the
     * data key, uses the plaintext data key to encrypt data, puts the encrypted data into the container, and then
     * destroys the plaintext data key. In this system, the component that creates the containers never sees the
     * plaintext data key.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param generateDataKeyWithoutPlaintextRequest
     * @return Result of the GenerateDataKeyWithoutPlaintext operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws KeyUnavailableException
     *         The request was rejected because the specified CMK was not available. The request can be retried.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidKeyUsageException
     *         The request was rejected because the specified <code>KeySpec</code> value is not valid.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.GenerateDataKeyWithoutPlaintext
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/GenerateDataKeyWithoutPlaintext"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GenerateDataKeyWithoutPlaintextResponse generateDataKeyWithoutPlaintext(
            GenerateDataKeyWithoutPlaintextRequest generateDataKeyWithoutPlaintextRequest) throws NotFoundException,
            DisabledException, KeyUnavailableException, DependencyTimeoutException, InvalidKeyUsageException,
            InvalidGrantTokenException, KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GenerateDataKeyWithoutPlaintextResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GenerateDataKeyWithoutPlaintextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GenerateDataKeyWithoutPlaintextRequest, GenerateDataKeyWithoutPlaintextResponse>()
                        .withOperationName("GenerateDataKeyWithoutPlaintext").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(generateDataKeyWithoutPlaintextRequest)
                        .withMarshaller(new GenerateDataKeyWithoutPlaintextRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a random byte string that is cryptographically secure.
     * </p>
     * <p>
     * By default, the random byte string is generated in AWS KMS. To generate the byte string in the AWS CloudHSM
     * cluster that is associated with a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>,
     * specify the custom key store ID.
     * </p>
     * <p>
     * For more information about entropy and random number generation, see the <a
     * href="https://d0.awsstatic.com/whitepapers/KMS-Cryptographic-Details.pdf">AWS Key Management Service
     * Cryptographic Details</a> whitepaper.
     * </p>
     *
     * @param generateRandomRequest
     * @return Result of the GenerateRandom operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws CustomKeyStoreInvalidStateException
     *         The request was rejected because of the <code>ConnectionState</code> of the custom key store. To get the
     *         <code>ConnectionState</code> of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.</p>
     *         <p>
     *         This exception is thrown under the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You requested the <a>CreateKey</a> or <a>GenerateRandom</a> operation in a custom key store that is not
     *         connected. These operations are valid only when the custom key store <code>ConnectionState</code> is
     *         <code>CONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>UpdateCustomKeyStore</a> or <a>DeleteCustomKeyStore</a> operation on a custom key
     *         store that is not disconnected. This operation is valid only when the custom key store
     *         <code>ConnectionState</code> is <code>DISCONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>ConnectCustomKeyStore</a> operation on a custom key store with a
     *         <code>ConnectionState</code> of <code>DISCONNECTING</code> or <code>FAILED</code>. This operation is
     *         valid for all other <code>ConnectionState</code> values.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.GenerateRandom
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/GenerateRandom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GenerateRandomResponse generateRandom(GenerateRandomRequest generateRandomRequest) throws DependencyTimeoutException,
            KmsInternalException, CustomKeyStoreNotFoundException, CustomKeyStoreInvalidStateException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GenerateRandomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GenerateRandomResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GenerateRandomRequest, GenerateRandomResponse>()
                .withOperationName("GenerateRandom").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(generateRandomRequest)
                .withMarshaller(new GenerateRandomRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a key policy attached to the specified customer master key (CMK). You cannot perform this operation on a CMK
     * in a different AWS account.
     * </p>
     *
     * @param getKeyPolicyRequest
     * @return Result of the GetKeyPolicy operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.GetKeyPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/GetKeyPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKeyPolicyResponse getKeyPolicy(GetKeyPolicyRequest getKeyPolicyRequest) throws NotFoundException,
            InvalidArnException, DependencyTimeoutException, KmsInternalException, KmsInvalidStateException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKeyPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKeyPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetKeyPolicyRequest, GetKeyPolicyResponse>()
                .withOperationName("GetKeyPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getKeyPolicyRequest)
                .withMarshaller(new GetKeyPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a Boolean value that indicates whether <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/rotate-keys.html">automatic rotation of the key
     * material</a> is enabled for the specified customer master key (CMK).
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Disabled: The key rotation status does not change when you disable a CMK. However, while the CMK is disabled, AWS
     * KMS does not rotate the backing key.
     * </p>
     * </li>
     * <li>
     * <p>
     * Pending deletion: While a CMK is pending deletion, its key rotation status is <code>false</code> and AWS KMS does
     * not rotate the backing key. If you cancel the deletion, the original key rotation status is restored.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN in the value of the
     * <code>KeyId</code> parameter.
     * </p>
     *
     * @param getKeyRotationStatusRequest
     * @return Result of the GetKeyRotationStatus operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.GetKeyRotationStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/GetKeyRotationStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKeyRotationStatusResponse getKeyRotationStatus(GetKeyRotationStatusRequest getKeyRotationStatusRequest)
            throws NotFoundException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, UnsupportedOperationException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKeyRotationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetKeyRotationStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetKeyRotationStatusRequest, GetKeyRotationStatusResponse>()
                .withOperationName("GetKeyRotationStatus").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getKeyRotationStatusRequest)
                .withMarshaller(new GetKeyRotationStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the items you need in order to import key material into AWS KMS from your existing key management
     * infrastructure. For more information about importing key material into AWS KMS, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * You must specify the key ID of the customer master key (CMK) into which you will import key material. This CMK's
     * <code>Origin</code> must be <code>EXTERNAL</code>. You must also specify the wrapping algorithm and type of
     * wrapping key (public key) that you will use to encrypt the key material. You cannot perform this operation on a
     * CMK in a different AWS account.
     * </p>
     * <p>
     * This operation returns a public key and an import token. Use the public key to encrypt the key material. Store
     * the import token to send with a subsequent <a>ImportKeyMaterial</a> request. The public key and import token from
     * the same response must be used together. These items are valid for 24 hours. When they expire, they cannot be
     * used for a subsequent <a>ImportKeyMaterial</a> request. To get new ones, send another
     * <code>GetParametersForImport</code> request.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param getParametersForImportRequest
     * @return Result of the GetParametersForImport operation returned by the service.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.GetParametersForImport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/GetParametersForImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetParametersForImportResponse getParametersForImport(GetParametersForImportRequest getParametersForImportRequest)
            throws InvalidArnException, UnsupportedOperationException, DependencyTimeoutException, NotFoundException,
            KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetParametersForImportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetParametersForImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetParametersForImportRequest, GetParametersForImportResponse>()
                .withOperationName("GetParametersForImport").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getParametersForImportRequest)
                .withMarshaller(new GetParametersForImportRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Imports key material into an existing AWS KMS customer master key (CMK) that was created without key material.
     * You cannot perform this operation on a CMK in a different AWS account. For more information about creating CMKs
     * with no key material and then importing key material, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Before using this operation, call <a>GetParametersForImport</a>. Its response includes a public key and an import
     * token. Use the public key to encrypt the key material. Then, submit the import token from the same
     * <code>GetParametersForImport</code> response.
     * </p>
     * <p>
     * When calling this operation, you must specify the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The key ID or key ARN of a CMK with no key material. Its <code>Origin</code> must be <code>EXTERNAL</code>.
     * </p>
     * <p>
     * To create a CMK with no key material, call <a>CreateKey</a> and set the value of its <code>Origin</code>
     * parameter to <code>EXTERNAL</code>. To get the <code>Origin</code> of a CMK, call <a>DescribeKey</a>.)
     * </p>
     * </li>
     * <li>
     * <p>
     * The encrypted key material. To get the public key to encrypt the key material, call
     * <a>GetParametersForImport</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The import token that <a>GetParametersForImport</a> returned. This token and the public key used to encrypt the
     * key material must have come from the same response.
     * </p>
     * </li>
     * <li>
     * <p>
     * Whether the key material expires and if so, when. If you set an expiration date, you can change it only by
     * reimporting the same key material and specifying a new expiration date. If the key material expires, AWS KMS
     * deletes the key material and the CMK becomes unusable. To use the CMK again, you must reimport the same key
     * material.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When this operation is successful, the key state of the CMK changes from <code>PendingImport</code> to
     * <code>Enabled</code>, and you can use the CMK. After you successfully import key material into a CMK, you can
     * reimport the same key material into that CMK, but you cannot import different key material.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param importKeyMaterialRequest
     * @return Result of the ImportKeyMaterial operation returned by the service.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws InvalidCiphertextException
     *         The request was rejected because the specified ciphertext, or additional authenticated data incorporated
     *         into the ciphertext, such as the encryption context, is corrupted, missing, or otherwise invalid.
     * @throws IncorrectKeyMaterialException
     *         The request was rejected because the provided key material is invalid or is not the same key material
     *         that was previously imported into this customer master key (CMK).
     * @throws ExpiredImportTokenException
     *         The request was rejected because the provided import token is expired. Use <a>GetParametersForImport</a>
     *         to get a new import token and public key, use the new public key to encrypt the key material, and then
     *         try the request again.
     * @throws InvalidImportTokenException
     *         The request was rejected because the provided import token is invalid or is associated with a different
     *         customer master key (CMK).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ImportKeyMaterial
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ImportKeyMaterial" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ImportKeyMaterialResponse importKeyMaterial(ImportKeyMaterialRequest importKeyMaterialRequest)
            throws InvalidArnException, UnsupportedOperationException, DependencyTimeoutException, NotFoundException,
            KmsInternalException, KmsInvalidStateException, InvalidCiphertextException, IncorrectKeyMaterialException,
            ExpiredImportTokenException, InvalidImportTokenException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportKeyMaterialResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ImportKeyMaterialResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ImportKeyMaterialRequest, ImportKeyMaterialResponse>()
                .withOperationName("ImportKeyMaterial").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(importKeyMaterialRequest)
                .withMarshaller(new ImportKeyMaterialRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of aliases in the caller's AWS account and region. You cannot list aliases in other accounts. For
     * more information about aliases, see <a>CreateAlias</a>.
     * </p>
     * <p>
     * By default, the ListAliases command returns all aliases in the account and region. To get only the aliases that
     * point to a particular customer master key (CMK), use the <code>KeyId</code> parameter.
     * </p>
     * <p>
     * The <code>ListAliases</code> response can include aliases that you created and associated with your customer
     * managed CMKs, and aliases that AWS created and associated with AWS managed CMKs in your account. You can
     * recognize AWS aliases because their names have the format <code>aws/&lt;service-name&gt;</code>, such as
     * <code>aws/dynamodb</code>.
     * </p>
     * <p>
     * The response might also include aliases that have no <code>TargetKeyId</code> field. These are predefined aliases
     * that AWS has created but has not yet associated with a CMK. Aliases that AWS creates in your account, including
     * predefined aliases, do not count against your <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#aliases-limit">AWS KMS aliases limit</a>.
     * </p>
     *
     * @param listAliasesRequest
     * @return Result of the ListAliases operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListAliases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListAliases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAliasesResponse listAliases(ListAliasesRequest listAliasesRequest) throws DependencyTimeoutException,
            InvalidMarkerException, KmsInternalException, InvalidArnException, NotFoundException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListAliasesRequest, ListAliasesResponse>()
                .withOperationName("ListAliases").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listAliasesRequest)
                .withMarshaller(new ListAliasesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of aliases in the caller's AWS account and region. You cannot list aliases in other accounts. For
     * more information about aliases, see <a>CreateAlias</a>.
     * </p>
     * <p>
     * By default, the ListAliases command returns all aliases in the account and region. To get only the aliases that
     * point to a particular customer master key (CMK), use the <code>KeyId</code> parameter.
     * </p>
     * <p>
     * The <code>ListAliases</code> response can include aliases that you created and associated with your customer
     * managed CMKs, and aliases that AWS created and associated with AWS managed CMKs in your account. You can
     * recognize AWS aliases because their names have the format <code>aws/&lt;service-name&gt;</code>, such as
     * <code>aws/dynamodb</code>.
     * </p>
     * <p>
     * The response might also include aliases that have no <code>TargetKeyId</code> field. These are predefined aliases
     * that AWS has created but has not yet associated with a CMK. Aliases that AWS creates in your account, including
     * predefined aliases, do not count against your <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html#aliases-limit">AWS KMS aliases limit</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAliases(software.amazon.awssdk.services.kms.model.ListAliasesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListAliasesIterable responses = client.listAliasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kms.paginators.ListAliasesIterable responses = client.listAliasesPaginator(request);
     *     for (software.amazon.awssdk.services.kms.model.ListAliasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListAliasesIterable responses = client.listAliasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAliases(software.amazon.awssdk.services.kms.model.ListAliasesRequest)} operation.</b>
     * </p>
     *
     * @param listAliasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListAliases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListAliases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAliasesIterable listAliasesPaginator(ListAliasesRequest listAliasesRequest) throws DependencyTimeoutException,
            InvalidMarkerException, KmsInternalException, InvalidArnException, NotFoundException, AwsServiceException,
            SdkClientException, KmsException {
        return new ListAliasesIterable(this, applyPaginatorUserAgent(listAliasesRequest));
    }

    /**
     * <p>
     * Gets a list of all grants for the specified customer master key (CMK).
     * </p>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN in the value of the
     * <code>KeyId</code> parameter.
     * </p>
     *
     * @param listGrantsRequest
     * @return Result of the ListGrants operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListGrants
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListGrants" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGrantsResponse listGrants(ListGrantsRequest listGrantsRequest) throws NotFoundException,
            DependencyTimeoutException, InvalidMarkerException, InvalidArnException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGrantsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGrantsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListGrantsRequest, ListGrantsResponse>()
                .withOperationName("ListGrants").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listGrantsRequest)
                .withMarshaller(new ListGrantsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of all grants for the specified customer master key (CMK).
     * </p>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN in the value of the
     * <code>KeyId</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGrants(software.amazon.awssdk.services.kms.model.ListGrantsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListGrantsIterable responses = client.listGrantsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kms.paginators.ListGrantsIterable responses = client.listGrantsPaginator(request);
     *     for (software.amazon.awssdk.services.kms.model.ListGrantsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListGrantsIterable responses = client.listGrantsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGrants(software.amazon.awssdk.services.kms.model.ListGrantsRequest)} operation.</b>
     * </p>
     *
     * @param listGrantsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListGrants
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListGrants" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGrantsIterable listGrantsPaginator(ListGrantsRequest listGrantsRequest) throws NotFoundException,
            DependencyTimeoutException, InvalidMarkerException, InvalidArnException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        return new ListGrantsIterable(this, applyPaginatorUserAgent(listGrantsRequest));
    }

    /**
     * <p>
     * Gets the names of the key policies that are attached to a customer master key (CMK). This operation is designed
     * to get policy names that you can use in a <a>GetKeyPolicy</a> operation. However, the only valid policy name is
     * <code>default</code>. You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     *
     * @param listKeyPoliciesRequest
     * @return Result of the ListKeyPolicies operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListKeyPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListKeyPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKeyPoliciesResponse listKeyPolicies(ListKeyPoliciesRequest listKeyPoliciesRequest) throws NotFoundException,
            InvalidArnException, DependencyTimeoutException, KmsInternalException, KmsInvalidStateException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKeyPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKeyPoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListKeyPoliciesRequest, ListKeyPoliciesResponse>()
                .withOperationName("ListKeyPolicies").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listKeyPoliciesRequest)
                .withMarshaller(new ListKeyPoliciesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the names of the key policies that are attached to a customer master key (CMK). This operation is designed
     * to get policy names that you can use in a <a>GetKeyPolicy</a> operation. However, the only valid policy name is
     * <code>default</code>. You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listKeyPolicies(software.amazon.awssdk.services.kms.model.ListKeyPoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListKeyPoliciesIterable responses = client.listKeyPoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kms.paginators.ListKeyPoliciesIterable responses = client.listKeyPoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.kms.model.ListKeyPoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListKeyPoliciesIterable responses = client.listKeyPoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listKeyPolicies(software.amazon.awssdk.services.kms.model.ListKeyPoliciesRequest)} operation.</b>
     * </p>
     *
     * @param listKeyPoliciesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListKeyPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListKeyPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKeyPoliciesIterable listKeyPoliciesPaginator(ListKeyPoliciesRequest listKeyPoliciesRequest)
            throws NotFoundException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        return new ListKeyPoliciesIterable(this, applyPaginatorUserAgent(listKeyPoliciesRequest));
    }

    /**
     * <p>
     * Gets a list of all customer master keys (CMKs) in the caller's AWS account and region.
     * </p>
     *
     * @param listKeysRequest
     * @return Result of the ListKeys operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListKeys
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKeysResponse listKeys(ListKeysRequest listKeysRequest) throws DependencyTimeoutException, KmsInternalException,
            InvalidMarkerException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKeysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListKeysRequest, ListKeysResponse>().withOperationName("ListKeys")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listKeysRequest)
                .withMarshaller(new ListKeysRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of all customer master keys (CMKs) in the caller's AWS account and region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listKeys(software.amazon.awssdk.services.kms.model.ListKeysRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListKeysIterable responses = client.listKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kms.paginators.ListKeysIterable responses = client.listKeysPaginator(request);
     *     for (software.amazon.awssdk.services.kms.model.ListKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kms.paginators.ListKeysIterable responses = client.listKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listKeys(software.amazon.awssdk.services.kms.model.ListKeysRequest)} operation.</b>
     * </p>
     *
     * @param listKeysRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListKeys
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKeysIterable listKeysPaginator(ListKeysRequest listKeysRequest) throws DependencyTimeoutException,
            KmsInternalException, InvalidMarkerException, AwsServiceException, SdkClientException, KmsException {
        return new ListKeysIterable(this, applyPaginatorUserAgent(listKeysRequest));
    }

    /**
     * <p>
     * Returns a list of all tags for the specified customer master key (CMK).
     * </p>
     * <p>
     * You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     *
     * @param listResourceTagsRequest
     * @return Result of the ListResourceTags operation returned by the service.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListResourceTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListResourceTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourceTagsResponse listResourceTags(ListResourceTagsRequest listResourceTagsRequest)
            throws KmsInternalException, NotFoundException, InvalidArnException, InvalidMarkerException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourceTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListResourceTagsRequest, ListResourceTagsResponse>()
                .withOperationName("ListResourceTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listResourceTagsRequest)
                .withMarshaller(new ListResourceTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of all grants for which the grant's <code>RetiringPrincipal</code> matches the one specified.
     * </p>
     * <p>
     * A typical use is to list all grants that you are able to retire. To retire a grant, use <a>RetireGrant</a>.
     * </p>
     *
     * @param listRetirableGrantsRequest
     * @return Result of the ListRetirableGrants operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidMarkerException
     *         The request was rejected because the marker that specifies where pagination should next begin is not
     *         valid.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ListRetirableGrants
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ListRetirableGrants" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRetirableGrantsResponse listRetirableGrants(ListRetirableGrantsRequest listRetirableGrantsRequest)
            throws DependencyTimeoutException, InvalidMarkerException, InvalidArnException, NotFoundException,
            KmsInternalException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRetirableGrantsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRetirableGrantsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListRetirableGrantsRequest, ListRetirableGrantsResponse>()
                .withOperationName("ListRetirableGrants").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listRetirableGrantsRequest)
                .withMarshaller(new ListRetirableGrantsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Attaches a key policy to the specified customer master key (CMK). You cannot perform this operation on a CMK in a
     * different AWS account.
     * </p>
     * <p>
     * For more information about key policies, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html">Key Policies</a> in the <i>AWS Key
     * Management Service Developer Guide</i>.
     * </p>
     *
     * @param putKeyPolicyRequest
     * @return Result of the PutKeyPolicy operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws MalformedPolicyDocumentException
     *         The request was rejected because the specified policy is not syntactically or semantically correct.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws UnsupportedOperationException
     *         The request was rejected because a specified parameter is not supported or a specified resource is not
     *         valid for this operation.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws LimitExceededException
     *         The request was rejected because a limit was exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html">Limits</a> in the <i>AWS Key
     *         Management Service Developer Guide</i>.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.PutKeyPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/PutKeyPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutKeyPolicyResponse putKeyPolicy(PutKeyPolicyRequest putKeyPolicyRequest) throws NotFoundException,
            InvalidArnException, MalformedPolicyDocumentException, DependencyTimeoutException, UnsupportedOperationException,
            KmsInternalException, LimitExceededException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutKeyPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutKeyPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutKeyPolicyRequest, PutKeyPolicyResponse>()
                .withOperationName("PutKeyPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putKeyPolicyRequest)
                .withMarshaller(new PutKeyPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Encrypts data on the server side with a new customer master key (CMK) without exposing the plaintext of the data
     * on the client side. The data is first decrypted and then reencrypted. You can also use this operation to change
     * the encryption context of a ciphertext.
     * </p>
     * <p>
     * You can reencrypt data using CMKs in different AWS accounts.
     * </p>
     * <p>
     * Unlike other operations, <code>ReEncrypt</code> is authorized twice, once as <code>ReEncryptFrom</code> on the
     * source CMK and once as <code>ReEncryptTo</code> on the destination CMK. We recommend that you include the
     * <code>"kms:ReEncrypt*"</code> permission in your <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html">key policies</a> to permit
     * reencryption from or to the CMK. This permission is automatically included in the key policy when you create a
     * CMK through the console. But you must include it manually when you create a CMK programmatically or when you set
     * a key policy with the <a>PutKeyPolicy</a> operation.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param reEncryptRequest
     * @return Result of the ReEncrypt operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DisabledException
     *         The request was rejected because the specified CMK is not enabled.
     * @throws InvalidCiphertextException
     *         The request was rejected because the specified ciphertext, or additional authenticated data incorporated
     *         into the ciphertext, such as the encryption context, is corrupted, missing, or otherwise invalid.
     * @throws KeyUnavailableException
     *         The request was rejected because the specified CMK was not available. The request can be retried.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidKeyUsageException
     *         The request was rejected because the specified <code>KeySpec</code> value is not valid.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ReEncrypt
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ReEncrypt" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ReEncryptResponse reEncrypt(ReEncryptRequest reEncryptRequest) throws NotFoundException, DisabledException,
            InvalidCiphertextException, KeyUnavailableException, DependencyTimeoutException, InvalidKeyUsageException,
            InvalidGrantTokenException, KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReEncryptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReEncryptResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ReEncryptRequest, ReEncryptResponse>()
                .withOperationName("ReEncrypt").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(reEncryptRequest)
                .withMarshaller(new ReEncryptRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retires a grant. To clean up, you can retire a grant when you're done using it. You should revoke a grant when
     * you intend to actively deny operations that depend on it. The following are permitted to call this API:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The AWS account (root user) under which the grant was created
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>RetiringPrincipal</code>, if present in the grant
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>GranteePrincipal</code>, if <code>RetireGrant</code> is an operation specified in the grant
     * </p>
     * </li>
     * </ul>
     * <p>
     * You must identify the grant to retire by its grant token or by a combination of the grant ID and the Amazon
     * Resource Name (ARN) of the customer master key (CMK). A grant token is a unique variable-length base64-encoded
     * string. A grant ID is a 64 character unique identifier of a grant. The <a>CreateGrant</a> operation returns both.
     * </p>
     *
     * @param retireGrantRequest
     * @return Result of the RetireGrant operation returned by the service.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws InvalidGrantTokenException
     *         The request was rejected because the specified grant token is not valid.
     * @throws InvalidGrantIdException
     *         The request was rejected because the specified <code>GrantId</code> is not valid.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.RetireGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/RetireGrant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RetireGrantResponse retireGrant(RetireGrantRequest retireGrantRequest) throws InvalidArnException,
            InvalidGrantTokenException, InvalidGrantIdException, NotFoundException, DependencyTimeoutException,
            KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RetireGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RetireGrantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RetireGrantRequest, RetireGrantResponse>()
                .withOperationName("RetireGrant").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(retireGrantRequest)
                .withMarshaller(new RetireGrantRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Revokes the specified grant for the specified customer master key (CMK). You can revoke a grant to actively deny
     * operations that depend on it.
     * </p>
     * <p>
     * To perform this operation on a CMK in a different AWS account, specify the key ARN in the value of the
     * <code>KeyId</code> parameter.
     * </p>
     *
     * @param revokeGrantRequest
     * @return Result of the RevokeGrant operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws InvalidGrantIdException
     *         The request was rejected because the specified <code>GrantId</code> is not valid.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.RevokeGrant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/RevokeGrant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RevokeGrantResponse revokeGrant(RevokeGrantRequest revokeGrantRequest) throws NotFoundException,
            DependencyTimeoutException, InvalidArnException, InvalidGrantIdException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RevokeGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RevokeGrantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RevokeGrantRequest, RevokeGrantResponse>()
                .withOperationName("RevokeGrant").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(revokeGrantRequest)
                .withMarshaller(new RevokeGrantRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Schedules the deletion of a customer master key (CMK). You may provide a waiting period, specified in days,
     * before deletion occurs. If you do not provide a waiting period, the default period of 30 days is used. When this
     * operation is successful, the key state of the CMK changes to <code>PendingDeletion</code>. Before the waiting
     * period ends, you can use <a>CancelKeyDeletion</a> to cancel the deletion of the CMK. After the waiting period
     * ends, AWS KMS deletes the CMK and all AWS KMS data associated with it, including all aliases that refer to it.
     * </p>
     * <important>
     * <p>
     * Deleting a CMK is a destructive and potentially dangerous operation. When a CMK is deleted, all data that was
     * encrypted under the CMK is unrecoverable. To prevent the use of a CMK without deleting it, use <a>DisableKey</a>.
     * </p>
     * </important>
     * <p>
     * If you schedule deletion of a CMK from a <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>,
     * when the waiting period expires, <code>ScheduleKeyDeletion</code> deletes the CMK from AWS KMS. Then AWS KMS
     * makes a best effort to delete the key material from the associated AWS CloudHSM cluster. However, you might need
     * to manually <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-orphaned-key">delete
     * the orphaned key material</a> from the cluster and its backups.
     * </p>
     * <p>
     * You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     * <p>
     * For more information about scheduling a CMK for deletion, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/deleting-keys.html">Deleting Customer Master Keys</a>
     * in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param scheduleKeyDeletionRequest
     * @return Result of the ScheduleKeyDeletion operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.ScheduleKeyDeletion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/ScheduleKeyDeletion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ScheduleKeyDeletionResponse scheduleKeyDeletion(ScheduleKeyDeletionRequest scheduleKeyDeletionRequest)
            throws NotFoundException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ScheduleKeyDeletionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ScheduleKeyDeletionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ScheduleKeyDeletionRequest, ScheduleKeyDeletionResponse>()
                .withOperationName("ScheduleKeyDeletion").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(scheduleKeyDeletionRequest)
                .withMarshaller(new ScheduleKeyDeletionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds or edits tags for a customer master key (CMK). You cannot perform this operation on a CMK in a different AWS
     * account.
     * </p>
     * <p>
     * Each tag consists of a tag key and a tag value. Tag keys and tag values are both required, but tag values can be
     * empty (null) strings.
     * </p>
     * <p>
     * You can only use a tag key once for each CMK. If you use the tag key again, AWS KMS replaces the current tag
     * value with the specified value.
     * </p>
     * <p>
     * For information about the rules that apply to tag keys and tag values, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/allocation-tag-restrictions.html">User-Defined
     * Tag Restrictions</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws LimitExceededException
     *         The request was rejected because a limit was exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/limits.html">Limits</a> in the <i>AWS Key
     *         Management Service Developer Guide</i>.
     * @throws TagException
     *         The request was rejected because one or more tags are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws KmsInternalException, NotFoundException,
            InvalidArnException, KmsInvalidStateException, LimitExceededException, TagException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the specified tags from the specified customer master key (CMK). You cannot perform this operation on a
     * CMK in a different AWS account.
     * </p>
     * <p>
     * To remove a tag, specify the tag key. To change the tag value of an existing tag key, use <a>TagResource</a>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws TagException
     *         The request was rejected because one or more tags are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws KmsInternalException,
            NotFoundException, InvalidArnException, KmsInvalidStateException, TagException, AwsServiceException,
            SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Associates an existing alias with a different customer master key (CMK). Each CMK can have multiple aliases, but
     * the aliases must be unique within the account and region. You cannot perform this operation on an alias in a
     * different AWS account.
     * </p>
     * <p>
     * This operation works only on existing aliases. To change the alias of a CMK to a new value, use
     * <a>CreateAlias</a> to create a new alias and <a>DeleteAlias</a> to delete the old alias.
     * </p>
     * <p>
     * Because an alias is not a property of a CMK, you can create, update, and delete the aliases of a CMK without
     * affecting the CMK. Also, aliases do not appear in the response from the <a>DescribeKey</a> operation. To get the
     * aliases of all CMKs in the account, use the <a>ListAliases</a> operation.
     * </p>
     * <p>
     * The alias name must begin with <code>alias/</code> followed by a name, such as <code>alias/ExampleAlias</code>.
     * It can contain only alphanumeric characters, forward slashes (/), underscores (_), and dashes (-). The alias name
     * cannot begin with <code>alias/aws/</code>. The <code>alias/aws/</code> prefix is reserved for <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk">AWS managed CMKs</a>.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param updateAliasRequest
     * @return Result of the UpdateAlias operation returned by the service.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.UpdateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/UpdateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAliasResponse updateAlias(UpdateAliasRequest updateAliasRequest) throws DependencyTimeoutException,
            NotFoundException, KmsInternalException, KmsInvalidStateException, AwsServiceException, SdkClientException,
            KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateAliasRequest, UpdateAliasResponse>()
                .withOperationName("UpdateAlias").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateAliasRequest)
                .withMarshaller(new UpdateAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Changes the properties of a custom key store. Use the <code>CustomKeyStoreId</code> parameter to identify the
     * custom key store you want to edit. Use the remaining parameters to change the properties of the custom key store.
     * </p>
     * <p>
     * You can only update a custom key store that is disconnected. To disconnect the custom key store, use
     * <a>DisconnectCustomKeyStore</a>. To reconnect the custom key store after the update completes, use
     * <a>ConnectCustomKeyStore</a>. To find the connection state of a custom key store, use the
     * <a>DescribeCustomKeyStores</a> operation.
     * </p>
     * <p>
     * Use the parameters of <code>UpdateCustomKeyStore</code> to edit your keystore settings.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use the <b>NewCustomKeyStoreName</b> parameter to change the friendly name of the custom key store to the value
     * that you specify.
     * </p>
     * <p>
     * </p></li>
     * <li>
     * <p>
     * Use the <b>KeyStorePassword</b> parameter tell AWS KMS the current password of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-store-concepts.html#concept-kmsuser">
     * <code>kmsuser</code> crypto user (CU)</a> in the associated AWS CloudHSM cluster. You can use this parameter to
     * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/fix-keystore.html#fix-keystore-password">fix
     * connection failures</a> that occur when AWS KMS cannot log into the associated cluster because the
     * <code>kmsuser</code> password has changed. This value does not change the password in the AWS CloudHSM cluster.
     * </p>
     * <p>
     * </p></li>
     * <li>
     * <p>
     * Use the <b>CloudHsmClusterId</b> parameter to associate the custom key store with a different, but related, AWS
     * CloudHSM cluster. You can use this parameter to repair a custom key store if its AWS CloudHSM cluster becomes
     * corrupted or is deleted, or when you need to create or restore a cluster from a backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the operation succeeds, it returns a JSON object with no properties.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     *
     * @param updateCustomKeyStoreRequest
     * @return Result of the UpdateCustomKeyStore operation returned by the service.
     * @throws CustomKeyStoreNotFoundException
     *         The request was rejected because AWS KMS cannot find a custom key store with the specified key store name
     *         or ID.
     * @throws CloudHsmClusterNotFoundException
     *         The request was rejected because AWS KMS cannot find the AWS CloudHSM cluster with the specified cluster
     *         ID. Retry the request with a different cluster ID.
     * @throws CloudHsmClusterNotRelatedException
     *         The request was rejected because the specified AWS CloudHSM cluster has a different cluster certificate
     *         than the original cluster. You cannot use the operation to specify an unrelated cluster.</p>
     *         <p>
     *         Specify a cluster that shares a backup history with the original cluster. This includes clusters that
     *         were created from a backup of the current cluster, and clusters that were created from the same backup
     *         that produced the current cluster.
     *         </p>
     *         <p>
     *         Clusters that share a backup history have the same cluster certificate. To view the cluster certificate
     *         of a cluster, use the <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeClusters.html"
     *         >DescribeClusters</a> operation.
     * @throws CustomKeyStoreInvalidStateException
     *         The request was rejected because of the <code>ConnectionState</code> of the custom key store. To get the
     *         <code>ConnectionState</code> of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
     *         </p>
     *         <p>
     *         This exception is thrown under the following conditions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You requested the <a>CreateKey</a> or <a>GenerateRandom</a> operation in a custom key store that is not
     *         connected. These operations are valid only when the custom key store <code>ConnectionState</code> is
     *         <code>CONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>UpdateCustomKeyStore</a> or <a>DeleteCustomKeyStore</a> operation on a custom key
     *         store that is not disconnected. This operation is valid only when the custom key store
     *         <code>ConnectionState</code> is <code>DISCONNECTED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You requested the <a>ConnectCustomKeyStore</a> operation on a custom key store with a
     *         <code>ConnectionState</code> of <code>DISCONNECTING</code> or <code>FAILED</code>. This operation is
     *         valid for all other <code>ConnectionState</code> values.
     *         </p>
     *         </li>
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws CloudHsmClusterNotActiveException
     *         The request was rejected because the AWS CloudHSM cluster that is associated with the custom key store is
     *         not active. Initialize and activate the cluster and try the command again. For detailed instructions, see
     *         <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/getting-started.html">Getting Started</a>
     *         in the <i>AWS CloudHSM User Guide</i>.
     * @throws CloudHsmClusterInvalidConfigurationException
     *         The request was rejected because the associated AWS CloudHSM cluster did not meet the configuration
     *         requirements for a custom key store.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The cluster must be configured with private subnets in at least two different Availability Zones in the
     *         Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">security group for
     *         the cluster</a> (cloudhsm-cluster-<i>&lt;cluster-id&gt;</i>-sg) must include inbound rules and outbound
     *         rules that allow TCP traffic on ports 2223-2225. The <b>Source</b> in the inbound rules and the
     *         <b>Destination</b> in the outbound rules must match the security group ID. These rules are set by default
     *         when you create the cluster. Do not delete or change them. To get information about a particular security
     *         group, use the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html"
     *         >DescribeSecurityGroups</a> operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster must contain at least as many HSMs as the operation requires. To add HSMs, use the AWS
     *         CloudHSM <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html">CreateHsm</a>
     *         operation.
     *         </p>
     *         <p>
     *         For the <a>CreateCustomKeyStore</a>, <a>UpdateCustomKeyStore</a>, and <a>CreateKey</a> operations, the
     *         AWS CloudHSM cluster must have at least two active HSMs, each in a different Availability Zone. For the
     *         <a>ConnectCustomKeyStore</a> operation, the AWS CloudHSM must contain at least one active HSM.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the requirements for an AWS CloudHSM cluster that is associated with a custom key
     *         store, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/create-keystore.html#before-keystore"
     *         >Assemble the Prerequisites</a> in the <i>AWS Key Management Service Developer Guide</i>. For information
     *         about creating a private subnet for an AWS CloudHSM cluster, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/create-subnets.html">Create a Private
     *         Subnet</a> in the <i>AWS CloudHSM User Guide</i>. For information about cluster security groups, see <a
     *         href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/configure-sg.html">Configure a Default
     *         Security Group</a> in the <i> <i>AWS CloudHSM User Guide</i> </i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.UpdateCustomKeyStore
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/UpdateCustomKeyStore" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateCustomKeyStoreResponse updateCustomKeyStore(UpdateCustomKeyStoreRequest updateCustomKeyStoreRequest)
            throws CustomKeyStoreNotFoundException, CloudHsmClusterNotFoundException, CloudHsmClusterNotRelatedException,
            CustomKeyStoreInvalidStateException, KmsInternalException, CloudHsmClusterNotActiveException,
            CloudHsmClusterInvalidConfigurationException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCustomKeyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCustomKeyStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateCustomKeyStoreRequest, UpdateCustomKeyStoreResponse>()
                .withOperationName("UpdateCustomKeyStore").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateCustomKeyStoreRequest)
                .withMarshaller(new UpdateCustomKeyStoreRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the description of a customer master key (CMK). To see the description of a CMK, use <a>DescribeKey</a>.
     * </p>
     * <p>
     * You cannot perform this operation on a CMK in a different AWS account.
     * </p>
     * <p>
     * The result of this operation varies with the key state of the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of a
     * Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     *
     * @param updateKeyDescriptionRequest
     * @return Result of the UpdateKeyDescription operation returned by the service.
     * @throws NotFoundException
     *         The request was rejected because the specified entity or resource could not be found.
     * @throws InvalidArnException
     *         The request was rejected because a specified ARN, or an ARN in a key policy, is not valid.
     * @throws DependencyTimeoutException
     *         The system timed out while trying to fulfill the request. The request can be retried.
     * @throws KmsInternalException
     *         The request was rejected because an internal exception occurred. The request can be retried.
     * @throws KmsInvalidStateException
     *         The request was rejected because the state of the specified resource is not valid for this request.</p>
     *         <p>
     *         For more information about how key state affects the use of a CMK, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">How Key State Affects Use of
     *         a Customer Master Key</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KmsClient.UpdateKeyDescription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kms-2014-11-01/UpdateKeyDescription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateKeyDescriptionResponse updateKeyDescription(UpdateKeyDescriptionRequest updateKeyDescriptionRequest)
            throws NotFoundException, InvalidArnException, DependencyTimeoutException, KmsInternalException,
            KmsInvalidStateException, AwsServiceException, SdkClientException, KmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKeyDescriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKeyDescriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateKeyDescriptionRequest, UpdateKeyDescriptionResponse>()
                .withOperationName("UpdateKeyDescription").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateKeyDescriptionRequest)
                .withMarshaller(new UpdateKeyDescriptionRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KmsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmClusterNotFoundException")
                                .exceptionBuilderSupplier(CloudHsmClusterNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredImportTokenException")
                                .exceptionBuilderSupplier(ExpiredImportTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomKeyStoreNotFoundException")
                                .exceptionBuilderSupplier(CustomKeyStoreNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException")
                                .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectKeyMaterialException")
                                .exceptionBuilderSupplier(IncorrectKeyMaterialException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidImportTokenException")
                                .exceptionBuilderSupplier(InvalidImportTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidStateException")
                                .exceptionBuilderSupplier(KmsInvalidStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArnException")
                                .exceptionBuilderSupplier(InvalidArnException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmClusterNotRelatedException")
                                .exceptionBuilderSupplier(CloudHsmClusterNotRelatedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomKeyStoreInvalidStateException")
                                .exceptionBuilderSupplier(CustomKeyStoreInvalidStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectTrustAnchorException")
                                .exceptionBuilderSupplier(IncorrectTrustAnchorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledException")
                                .exceptionBuilderSupplier(DisabledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KeyUnavailableException")
                                .exceptionBuilderSupplier(KeyUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInternalException")
                                .exceptionBuilderSupplier(KmsInternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmClusterInUseException")
                                .exceptionBuilderSupplier(CloudHsmClusterInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCiphertextException")
                                .exceptionBuilderSupplier(InvalidCiphertextException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidGrantIdException")
                                .exceptionBuilderSupplier(InvalidGrantIdException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomKeyStoreHasCMKsException")
                                .exceptionBuilderSupplier(CustomKeyStoreHasCmKsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidGrantTokenException")
                                .exceptionBuilderSupplier(InvalidGrantTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomKeyStoreNameInUseException")
                                .exceptionBuilderSupplier(CustomKeyStoreNameInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagException").exceptionBuilderSupplier(TagException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKeyUsageException")
                                .exceptionBuilderSupplier(InvalidKeyUsageException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmClusterInvalidConfigurationException")
                                .exceptionBuilderSupplier(CloudHsmClusterInvalidConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMarkerException")
                                .exceptionBuilderSupplier(InvalidMarkerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidAliasNameException")
                                .exceptionBuilderSupplier(InvalidAliasNameException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyTimeoutException")
                                .exceptionBuilderSupplier(DependencyTimeoutException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmClusterNotActiveException")
                                .exceptionBuilderSupplier(CloudHsmClusterNotActiveException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends KmsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
