/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateGrantRequest extends KmsRequest implements
        ToCopyableBuilder<CreateGrantRequest.Builder, CreateGrantRequest> {
    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateGrantRequest::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final SdkField<String> GRANTEE_PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateGrantRequest::granteePrincipal)).setter(setter(Builder::granteePrincipal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GranteePrincipal").build()).build();

    private static final SdkField<String> RETIRING_PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateGrantRequest::retiringPrincipal)).setter(setter(Builder::retiringPrincipal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetiringPrincipal").build()).build();

    private static final SdkField<List<String>> OPERATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateGrantRequest::operationsAsStrings))
            .setter(setter(Builder::operationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<GrantConstraints> CONSTRAINTS_FIELD = SdkField
            .<GrantConstraints> builder(MarshallingType.SDK_POJO).getter(getter(CreateGrantRequest::constraints))
            .setter(setter(Builder::constraints)).constructor(GrantConstraints::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Constraints").build()).build();

    private static final SdkField<List<String>> GRANT_TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateGrantRequest::grantTokens))
            .setter(setter(Builder::grantTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GrantTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateGrantRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_ID_FIELD,
            GRANTEE_PRINCIPAL_FIELD, RETIRING_PRINCIPAL_FIELD, OPERATIONS_FIELD, CONSTRAINTS_FIELD, GRANT_TOKENS_FIELD,
            NAME_FIELD));

    private final String keyId;

    private final String granteePrincipal;

    private final String retiringPrincipal;

    private final List<String> operations;

    private final GrantConstraints constraints;

    private final List<String> grantTokens;

    private final String name;

    private CreateGrantRequest(BuilderImpl builder) {
        super(builder);
        this.keyId = builder.keyId;
        this.granteePrincipal = builder.granteePrincipal;
        this.retiringPrincipal = builder.retiringPrincipal;
        this.operations = builder.operations;
        this.constraints = builder.constraints;
        this.grantTokens = builder.grantTokens;
        this.name = builder.name;
    }

    /**
     * <p>
     * The unique identifier for the customer master key (CMK) that the grant applies to.
     * </p>
     * <p>
     * Specify the key ID or the Amazon Resource Name (ARN) of the CMK. To specify a CMK in a different AWS account, you
     * must use the key ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>.
     * </p>
     * 
     * @return The unique identifier for the customer master key (CMK) that the grant applies to.</p>
     *         <p>
     *         Specify the key ID or the Amazon Resource Name (ARN) of the CMK. To specify a CMK in a different AWS
     *         account, you must use the key ARN.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>.
     */
    public String keyId() {
        return keyId;
    }

    /**
     * <p>
     * The principal that is given permission to perform the operations that the grant permits.
     * </p>
     * <p>
     * To specify the principal, use the <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name (ARN)</a>
     * of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, IAM roles, federated users, and
     * assumed role users. For examples of the ARN syntax to use for specifying a principal, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS Identity and
     * Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
     * </p>
     * 
     * @return The principal that is given permission to perform the operations that the grant permits.</p>
     *         <p>
     *         To specify the principal, use the <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, IAM roles,
     *         federated users, and assumed role users. For examples of the ARN syntax to use for specifying a
     *         principal, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS
     *         Identity and Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
     */
    public String granteePrincipal() {
        return granteePrincipal;
    }

    /**
     * <p>
     * The principal that is given permission to retire the grant by using <a>RetireGrant</a> operation.
     * </p>
     * <p>
     * To specify the principal, use the <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name (ARN)</a>
     * of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, federated users, and assumed
     * role users. For examples of the ARN syntax to use for specifying a principal, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS Identity and
     * Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
     * </p>
     * 
     * @return The principal that is given permission to retire the grant by using <a>RetireGrant</a> operation.</p>
     *         <p>
     *         To specify the principal, use the <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     *         (ARN)</a> of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, federated
     *         users, and assumed role users. For examples of the ARN syntax to use for specifying a principal, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS
     *         Identity and Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
     */
    public String retiringPrincipal() {
        return retiringPrincipal;
    }

    /**
     * <p>
     * A list of operations that the grant permits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of operations that the grant permits.
     */
    public List<GrantOperation> operations() {
        return GrantOperationListCopier.copyStringToEnum(operations);
    }

    /**
     * <p>
     * A list of operations that the grant permits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of operations that the grant permits.
     */
    public List<String> operationsAsStrings() {
        return operations;
    }

    /**
     * <p>
     * Allows a cryptographic operation only when the encryption context matches or includes the encryption context
     * specified in this structure. For more information about encryption context, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption Context</a>
     * in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * 
     * @return Allows a cryptographic operation only when the encryption context matches or includes the encryption
     *         context specified in this structure. For more information about encryption context, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
     *         Context</a> in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     */
    public GrantConstraints constraints() {
        return constraints;
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in the
     * <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of grant tokens.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a>
     *         in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public List<String> grantTokens() {
        return grantTokens;
    }

    /**
     * <p>
     * A friendly name for identifying the grant. Use this value to prevent the unintended creation of duplicate grants
     * when retrying this request.
     * </p>
     * <p>
     * When this value is absent, all <code>CreateGrant</code> requests result in a new grant with a unique
     * <code>GrantId</code> even if all the supplied parameters are identical. This can result in unintended duplicates
     * when you retry the <code>CreateGrant</code> request.
     * </p>
     * <p>
     * When this value is present, you can retry a <code>CreateGrant</code> request with identical parameters; if the
     * grant already exists, the original <code>GrantId</code> is returned without creating a new grant. Note that the
     * returned grant token is unique with every <code>CreateGrant</code> request, even when a duplicate
     * <code>GrantId</code> is returned. All grant tokens obtained in this way can be used interchangeably.
     * </p>
     * 
     * @return A friendly name for identifying the grant. Use this value to prevent the unintended creation of duplicate
     *         grants when retrying this request.</p>
     *         <p>
     *         When this value is absent, all <code>CreateGrant</code> requests result in a new grant with a unique
     *         <code>GrantId</code> even if all the supplied parameters are identical. This can result in unintended
     *         duplicates when you retry the <code>CreateGrant</code> request.
     *         </p>
     *         <p>
     *         When this value is present, you can retry a <code>CreateGrant</code> request with identical parameters;
     *         if the grant already exists, the original <code>GrantId</code> is returned without creating a new grant.
     *         Note that the returned grant token is unique with every <code>CreateGrant</code> request, even when a
     *         duplicate <code>GrantId</code> is returned. All grant tokens obtained in this way can be used
     *         interchangeably.
     */
    public String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        hashCode = 31 * hashCode + Objects.hashCode(granteePrincipal());
        hashCode = 31 * hashCode + Objects.hashCode(retiringPrincipal());
        hashCode = 31 * hashCode + Objects.hashCode(operationsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(constraints());
        hashCode = 31 * hashCode + Objects.hashCode(grantTokens());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateGrantRequest)) {
            return false;
        }
        CreateGrantRequest other = (CreateGrantRequest) obj;
        return Objects.equals(keyId(), other.keyId()) && Objects.equals(granteePrincipal(), other.granteePrincipal())
                && Objects.equals(retiringPrincipal(), other.retiringPrincipal())
                && Objects.equals(operationsAsStrings(), other.operationsAsStrings())
                && Objects.equals(constraints(), other.constraints()) && Objects.equals(grantTokens(), other.grantTokens())
                && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateGrantRequest").add("KeyId", keyId()).add("GranteePrincipal", granteePrincipal())
                .add("RetiringPrincipal", retiringPrincipal()).add("Operations", operationsAsStrings())
                .add("Constraints", constraints()).add("GrantTokens", grantTokens()).add("Name", name()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        case "GranteePrincipal":
            return Optional.ofNullable(clazz.cast(granteePrincipal()));
        case "RetiringPrincipal":
            return Optional.ofNullable(clazz.cast(retiringPrincipal()));
        case "Operations":
            return Optional.ofNullable(clazz.cast(operationsAsStrings()));
        case "Constraints":
            return Optional.ofNullable(clazz.cast(constraints()));
        case "GrantTokens":
            return Optional.ofNullable(clazz.cast(grantTokens()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateGrantRequest, T> g) {
        return obj -> g.apply((CreateGrantRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateGrantRequest> {
        /**
         * <p>
         * The unique identifier for the customer master key (CMK) that the grant applies to.
         * </p>
         * <p>
         * Specify the key ID or the Amazon Resource Name (ARN) of the CMK. To specify a CMK in a different AWS account,
         * you must use the key ARN.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>.
         * </p>
         * 
         * @param keyId
         *        The unique identifier for the customer master key (CMK) that the grant applies to.</p>
         *        <p>
         *        Specify the key ID or the Amazon Resource Name (ARN) of the CMK. To specify a CMK in a different AWS
         *        account, you must use the key ARN.
         *        </p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);

        /**
         * <p>
         * The principal that is given permission to perform the operations that the grant permits.
         * </p>
         * <p>
         * To specify the principal, use the <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, IAM roles,
         * federated users, and assumed role users. For examples of the ARN syntax to use for specifying a principal,
         * see <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS
         * Identity and Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
         * </p>
         * 
         * @param granteePrincipal
         *        The principal that is given permission to perform the operations that the grant permits.</p>
         *        <p>
         *        To specify the principal, use the <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         *        (ARN)</a> of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, IAM roles,
         *        federated users, and assumed role users. For examples of the ARN syntax to use for specifying a
         *        principal, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS
         *        Identity and Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder granteePrincipal(String granteePrincipal);

        /**
         * <p>
         * The principal that is given permission to retire the grant by using <a>RetireGrant</a> operation.
         * </p>
         * <p>
         * To specify the principal, use the <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, federated users,
         * and assumed role users. For examples of the ARN syntax to use for specifying a principal, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS Identity
         * and Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General Reference</i>.
         * </p>
         * 
         * @param retiringPrincipal
         *        The principal that is given permission to retire the grant by using <a>RetireGrant</a> operation.</p>
         *        <p>
         *        To specify the principal, use the <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         *        (ARN)</a> of an AWS principal. Valid AWS principals include AWS accounts (root), IAM users, federated
         *        users, and assumed role users. For examples of the ARN syntax to use for specifying a principal, see
         *        <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-iam">AWS
         *        Identity and Access Management (IAM)</a> in the Example ARNs section of the <i>AWS General
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retiringPrincipal(String retiringPrincipal);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationsWithStrings(Collection<String> operations);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationsWithStrings(String... operations);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operations(Collection<GrantOperation> operations);

        /**
         * <p>
         * A list of operations that the grant permits.
         * </p>
         * 
         * @param operations
         *        A list of operations that the grant permits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operations(GrantOperation... operations);

        /**
         * <p>
         * Allows a cryptographic operation only when the encryption context matches or includes the encryption context
         * specified in this structure. For more information about encryption context, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         * Context</a> in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * </p>
         * 
         * @param constraints
         *        Allows a cryptographic operation only when the encryption context matches or includes the encryption
         *        context specified in this structure. For more information about encryption context, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         *        Context</a> in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constraints(GrantConstraints constraints);

        /**
         * <p>
         * Allows a cryptographic operation only when the encryption context matches or includes the encryption context
         * specified in this structure. For more information about encryption context, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
         * Context</a> in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * </p>
         * This is a convenience that creates an instance of the {@link GrantConstraints.Builder} avoiding the need to
         * create one manually via {@link GrantConstraints#builder()}.
         *
         * When the {@link Consumer} completes, {@link GrantConstraints.Builder#build()} is called immediately and its
         * result is passed to {@link #constraints(GrantConstraints)}.
         * 
         * @param constraints
         *        a consumer that will call methods on {@link GrantConstraints.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #constraints(GrantConstraints)
         */
        default Builder constraints(Consumer<GrantConstraints.Builder> constraints) {
            return constraints(GrantConstraints.builder().applyMutation(constraints).build());
        }

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(Collection<String> grantTokens);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(String... grantTokens);

        /**
         * <p>
         * A friendly name for identifying the grant. Use this value to prevent the unintended creation of duplicate
         * grants when retrying this request.
         * </p>
         * <p>
         * When this value is absent, all <code>CreateGrant</code> requests result in a new grant with a unique
         * <code>GrantId</code> even if all the supplied parameters are identical. This can result in unintended
         * duplicates when you retry the <code>CreateGrant</code> request.
         * </p>
         * <p>
         * When this value is present, you can retry a <code>CreateGrant</code> request with identical parameters; if
         * the grant already exists, the original <code>GrantId</code> is returned without creating a new grant. Note
         * that the returned grant token is unique with every <code>CreateGrant</code> request, even when a duplicate
         * <code>GrantId</code> is returned. All grant tokens obtained in this way can be used interchangeably.
         * </p>
         * 
         * @param name
         *        A friendly name for identifying the grant. Use this value to prevent the unintended creation of
         *        duplicate grants when retrying this request.</p>
         *        <p>
         *        When this value is absent, all <code>CreateGrant</code> requests result in a new grant with a unique
         *        <code>GrantId</code> even if all the supplied parameters are identical. This can result in unintended
         *        duplicates when you retry the <code>CreateGrant</code> request.
         *        </p>
         *        <p>
         *        When this value is present, you can retry a <code>CreateGrant</code> request with identical
         *        parameters; if the grant already exists, the original <code>GrantId</code> is returned without
         *        creating a new grant. Note that the returned grant token is unique with every <code>CreateGrant</code>
         *        request, even when a duplicate <code>GrantId</code> is returned. All grant tokens obtained in this way
         *        can be used interchangeably.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private String keyId;

        private String granteePrincipal;

        private String retiringPrincipal;

        private List<String> operations = DefaultSdkAutoConstructList.getInstance();

        private GrantConstraints constraints;

        private List<String> grantTokens = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateGrantRequest model) {
            super(model);
            keyId(model.keyId);
            granteePrincipal(model.granteePrincipal);
            retiringPrincipal(model.retiringPrincipal);
            operationsWithStrings(model.operations);
            constraints(model.constraints);
            grantTokens(model.grantTokens);
            name(model.name);
        }

        public final String getKeyId() {
            return keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public final String getGranteePrincipal() {
            return granteePrincipal;
        }

        @Override
        public final Builder granteePrincipal(String granteePrincipal) {
            this.granteePrincipal = granteePrincipal;
            return this;
        }

        public final void setGranteePrincipal(String granteePrincipal) {
            this.granteePrincipal = granteePrincipal;
        }

        public final String getRetiringPrincipal() {
            return retiringPrincipal;
        }

        @Override
        public final Builder retiringPrincipal(String retiringPrincipal) {
            this.retiringPrincipal = retiringPrincipal;
            return this;
        }

        public final void setRetiringPrincipal(String retiringPrincipal) {
            this.retiringPrincipal = retiringPrincipal;
        }

        public final Collection<String> getOperationsAsStrings() {
            return operations;
        }

        @Override
        public final Builder operationsWithStrings(Collection<String> operations) {
            this.operations = GrantOperationListCopier.copy(operations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operationsWithStrings(String... operations) {
            operationsWithStrings(Arrays.asList(operations));
            return this;
        }

        @Override
        public final Builder operations(Collection<GrantOperation> operations) {
            this.operations = GrantOperationListCopier.copyEnumToString(operations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operations(GrantOperation... operations) {
            operations(Arrays.asList(operations));
            return this;
        }

        public final void setOperationsWithStrings(Collection<String> operations) {
            this.operations = GrantOperationListCopier.copy(operations);
        }

        public final GrantConstraints.Builder getConstraints() {
            return constraints != null ? constraints.toBuilder() : null;
        }

        @Override
        public final Builder constraints(GrantConstraints constraints) {
            this.constraints = constraints;
            return this;
        }

        public final void setConstraints(GrantConstraints.BuilderImpl constraints) {
            this.constraints = constraints != null ? constraints.build() : null;
        }

        public final Collection<String> getGrantTokens() {
            return grantTokens;
        }

        @Override
        public final Builder grantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder grantTokens(String... grantTokens) {
            grantTokens(Arrays.asList(grantTokens));
            return this;
        }

        public final void setGrantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateGrantRequest build() {
            return new CreateGrantRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
