/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReEncryptRequest extends KmsRequest implements ToCopyableBuilder<ReEncryptRequest.Builder, ReEncryptRequest> {
    private static final SdkField<SdkBytes> CIPHERTEXT_BLOB_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(ReEncryptRequest::ciphertextBlob)).setter(setter(Builder::ciphertextBlob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CiphertextBlob").build()).build();

    private static final SdkField<Map<String, String>> SOURCE_ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(ReEncryptRequest::sourceEncryptionContext))
            .setter(setter(Builder::sourceEncryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceEncryptionContext").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> DESTINATION_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReEncryptRequest::destinationKeyId)).setter(setter(Builder::destinationKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationKeyId").build()).build();

    private static final SdkField<Map<String, String>> DESTINATION_ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(ReEncryptRequest::destinationEncryptionContext))
            .setter(setter(Builder::destinationEncryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationEncryptionContext")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> GRANT_TOKENS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ReEncryptRequest::grantTokens))
            .setter(setter(Builder::grantTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GrantTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CIPHERTEXT_BLOB_FIELD,
            SOURCE_ENCRYPTION_CONTEXT_FIELD, DESTINATION_KEY_ID_FIELD, DESTINATION_ENCRYPTION_CONTEXT_FIELD, GRANT_TOKENS_FIELD));

    private final SdkBytes ciphertextBlob;

    private final Map<String, String> sourceEncryptionContext;

    private final String destinationKeyId;

    private final Map<String, String> destinationEncryptionContext;

    private final List<String> grantTokens;

    private ReEncryptRequest(BuilderImpl builder) {
        super(builder);
        this.ciphertextBlob = builder.ciphertextBlob;
        this.sourceEncryptionContext = builder.sourceEncryptionContext;
        this.destinationKeyId = builder.destinationKeyId;
        this.destinationEncryptionContext = builder.destinationEncryptionContext;
        this.grantTokens = builder.grantTokens;
    }

    /**
     * <p>
     * Ciphertext of the data to reencrypt.
     * </p>
     * 
     * @return Ciphertext of the data to reencrypt.
     */
    public SdkBytes ciphertextBlob() {
        return ciphertextBlob;
    }

    /**
     * <p>
     * Encryption context used to encrypt and decrypt the data specified in the <code>CiphertextBlob</code> parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Encryption context used to encrypt and decrypt the data specified in the <code>CiphertextBlob</code>
     *         parameter.
     */
    public Map<String, String> sourceEncryptionContext() {
        return sourceEncryptionContext;
    }

    /**
     * <p>
     * A unique identifier for the CMK that is used to reencrypt the data.
     * </p>
     * <p>
     * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias name,
     * prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must use the key ARN or
     * alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
     * alias ARN, use <a>ListAliases</a>.
     * </p>
     * 
     * @return A unique identifier for the CMK that is used to reencrypt the data.</p>
     *         <p>
     *         To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
     *         alias name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must
     *         use the key ARN or alias ARN.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name
     *         and alias ARN, use <a>ListAliases</a>.
     */
    public String destinationKeyId() {
        return destinationKeyId;
    }

    /**
     * <p>
     * Encryption context to use when the data is reencrypted.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Encryption context to use when the data is reencrypted.
     */
    public Map<String, String> destinationEncryptionContext() {
        return destinationEncryptionContext;
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in the
     * <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of grant tokens.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a>
     *         in the <i>AWS Key Management Service Developer Guide</i>.
     */
    public List<String> grantTokens() {
        return grantTokens;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ciphertextBlob());
        hashCode = 31 * hashCode + Objects.hashCode(sourceEncryptionContext());
        hashCode = 31 * hashCode + Objects.hashCode(destinationKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(destinationEncryptionContext());
        hashCode = 31 * hashCode + Objects.hashCode(grantTokens());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReEncryptRequest)) {
            return false;
        }
        ReEncryptRequest other = (ReEncryptRequest) obj;
        return Objects.equals(ciphertextBlob(), other.ciphertextBlob())
                && Objects.equals(sourceEncryptionContext(), other.sourceEncryptionContext())
                && Objects.equals(destinationKeyId(), other.destinationKeyId())
                && Objects.equals(destinationEncryptionContext(), other.destinationEncryptionContext())
                && Objects.equals(grantTokens(), other.grantTokens());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ReEncryptRequest").add("CiphertextBlob", ciphertextBlob())
                .add("SourceEncryptionContext", sourceEncryptionContext()).add("DestinationKeyId", destinationKeyId())
                .add("DestinationEncryptionContext", destinationEncryptionContext()).add("GrantTokens", grantTokens()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CiphertextBlob":
            return Optional.ofNullable(clazz.cast(ciphertextBlob()));
        case "SourceEncryptionContext":
            return Optional.ofNullable(clazz.cast(sourceEncryptionContext()));
        case "DestinationKeyId":
            return Optional.ofNullable(clazz.cast(destinationKeyId()));
        case "DestinationEncryptionContext":
            return Optional.ofNullable(clazz.cast(destinationEncryptionContext()));
        case "GrantTokens":
            return Optional.ofNullable(clazz.cast(grantTokens()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReEncryptRequest, T> g) {
        return obj -> g.apply((ReEncryptRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, ReEncryptRequest> {
        /**
         * <p>
         * Ciphertext of the data to reencrypt.
         * </p>
         * 
         * @param ciphertextBlob
         *        Ciphertext of the data to reencrypt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ciphertextBlob(SdkBytes ciphertextBlob);

        /**
         * <p>
         * Encryption context used to encrypt and decrypt the data specified in the <code>CiphertextBlob</code>
         * parameter.
         * </p>
         * 
         * @param sourceEncryptionContext
         *        Encryption context used to encrypt and decrypt the data specified in the <code>CiphertextBlob</code>
         *        parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceEncryptionContext(Map<String, String> sourceEncryptionContext);

        /**
         * <p>
         * A unique identifier for the CMK that is used to reencrypt the data.
         * </p>
         * <p>
         * To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an alias
         * name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you must use the key
         * ARN or alias ARN.
         * </p>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name and
         * alias ARN, use <a>ListAliases</a>.
         * </p>
         * 
         * @param destinationKeyId
         *        A unique identifier for the CMK that is used to reencrypt the data.</p>
         *        <p>
         *        To specify a CMK, use its key ID, Amazon Resource Name (ARN), alias name, or alias ARN. When using an
         *        alias name, prefix it with <code>"alias/"</code>. To specify a CMK in a different AWS account, you
         *        must use the key ARN or alias ARN.
         *        </p>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN: <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get the key ID and key ARN for a CMK, use <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias
         *        name and alias ARN, use <a>ListAliases</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationKeyId(String destinationKeyId);

        /**
         * <p>
         * Encryption context to use when the data is reencrypted.
         * </p>
         * 
         * @param destinationEncryptionContext
         *        Encryption context to use when the data is reencrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationEncryptionContext(Map<String, String> destinationEncryptionContext);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(Collection<String> grantTokens);

        /**
         * <p>
         * A list of grant tokens.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant Tokens</a> in
         * the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param grantTokens
         *        A list of grant tokens.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token">Grant
         *        Tokens</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantTokens(String... grantTokens);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private SdkBytes ciphertextBlob;

        private Map<String, String> sourceEncryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private String destinationKeyId;

        private Map<String, String> destinationEncryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private List<String> grantTokens = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReEncryptRequest model) {
            super(model);
            ciphertextBlob(model.ciphertextBlob);
            sourceEncryptionContext(model.sourceEncryptionContext);
            destinationKeyId(model.destinationKeyId);
            destinationEncryptionContext(model.destinationEncryptionContext);
            grantTokens(model.grantTokens);
        }

        public final ByteBuffer getCiphertextBlob() {
            return ciphertextBlob == null ? null : ciphertextBlob.asByteBuffer();
        }

        @Override
        public final Builder ciphertextBlob(SdkBytes ciphertextBlob) {
            this.ciphertextBlob = StandardMemberCopier.copy(ciphertextBlob);
            return this;
        }

        public final void setCiphertextBlob(ByteBuffer ciphertextBlob) {
            ciphertextBlob(ciphertextBlob == null ? null : SdkBytes.fromByteBuffer(ciphertextBlob));
        }

        public final Map<String, String> getSourceEncryptionContext() {
            return sourceEncryptionContext;
        }

        @Override
        public final Builder sourceEncryptionContext(Map<String, String> sourceEncryptionContext) {
            this.sourceEncryptionContext = EncryptionContextTypeCopier.copy(sourceEncryptionContext);
            return this;
        }

        public final void setSourceEncryptionContext(Map<String, String> sourceEncryptionContext) {
            this.sourceEncryptionContext = EncryptionContextTypeCopier.copy(sourceEncryptionContext);
        }

        public final String getDestinationKeyId() {
            return destinationKeyId;
        }

        @Override
        public final Builder destinationKeyId(String destinationKeyId) {
            this.destinationKeyId = destinationKeyId;
            return this;
        }

        public final void setDestinationKeyId(String destinationKeyId) {
            this.destinationKeyId = destinationKeyId;
        }

        public final Map<String, String> getDestinationEncryptionContext() {
            return destinationEncryptionContext;
        }

        @Override
        public final Builder destinationEncryptionContext(Map<String, String> destinationEncryptionContext) {
            this.destinationEncryptionContext = EncryptionContextTypeCopier.copy(destinationEncryptionContext);
            return this;
        }

        public final void setDestinationEncryptionContext(Map<String, String> destinationEncryptionContext) {
            this.destinationEncryptionContext = EncryptionContextTypeCopier.copy(destinationEncryptionContext);
        }

        public final Collection<String> getGrantTokens() {
            return grantTokens;
        }

        @Override
        public final Builder grantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder grantTokens(String... grantTokens) {
            grantTokens(Arrays.asList(grantTokens));
            return this;
        }

        public final void setGrantTokens(Collection<String> grantTokens) {
            this.grantTokens = GrantTokenListCopier.copy(grantTokens);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ReEncryptRequest build() {
            return new ReEncryptRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
