/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateKeyRequest extends KmsRequest implements ToCopyableBuilder<CreateKeyRequest.Builder, CreateKeyRequest> {
    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateKeyRequest::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateKeyRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> KEY_USAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateKeyRequest::keyUsageAsString)).setter(setter(Builder::keyUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyUsage").build()).build();

    private static final SdkField<String> ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateKeyRequest::originAsString)).setter(setter(Builder::origin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Origin").build()).build();

    private static final SdkField<String> CUSTOM_KEY_STORE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateKeyRequest::customKeyStoreId)).setter(setter(Builder::customKeyStoreId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomKeyStoreId").build()).build();

    private static final SdkField<Boolean> BYPASS_POLICY_LOCKOUT_SAFETY_CHECK_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateKeyRequest::bypassPolicyLockoutSafetyCheck))
            .setter(setter(Builder::bypassPolicyLockoutSafetyCheck))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BypassPolicyLockoutSafetyCheck")
                    .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateKeyRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_FIELD,
            DESCRIPTION_FIELD, KEY_USAGE_FIELD, ORIGIN_FIELD, CUSTOM_KEY_STORE_ID_FIELD,
            BYPASS_POLICY_LOCKOUT_SAFETY_CHECK_FIELD, TAGS_FIELD));

    private final String policy;

    private final String description;

    private final String keyUsage;

    private final String origin;

    private final String customKeyStoreId;

    private final Boolean bypassPolicyLockoutSafetyCheck;

    private final List<Tag> tags;

    private CreateKeyRequest(BuilderImpl builder) {
        super(builder);
        this.policy = builder.policy;
        this.description = builder.description;
        this.keyUsage = builder.keyUsage;
        this.origin = builder.origin;
        this.customKeyStoreId = builder.customKeyStoreId;
        this.bypassPolicyLockoutSafetyCheck = builder.bypassPolicyLockoutSafetyCheck;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The key policy to attach to the CMK.
     * </p>
     * <p>
     * If you provide a key policy, it must meet the following criteria:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the principal
     * that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a> request on the CMK.
     * This reduces the risk that the CMK becomes unmanageable. For more information, refer to the scenario in the <a
     * href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     * >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each statement in the key policy must contain one or more principals. The principals in the key policy must exist
     * and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or role), you might need
     * to enforce a delay before including the new principal in a key policy because the new principal might not be
     * immediately visible to AWS KMS. For more information, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
     * >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access Management User
     * Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more information, see
     * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default">Default Key
     * Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * The key policy size limit is 32 kilobytes (32768 bytes).
     * </p>
     * 
     * @return The key policy to attach to the CMK.</p>
     *         <p>
     *         If you provide a key policy, it must meet the following criteria:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the
     *         principal that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a>
     *         request on the CMK. This reduces the risk that the CMK becomes unmanageable. For more information, refer
     *         to the scenario in the <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     *         >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each statement in the key policy must contain one or more principals. The principals in the key policy
     *         must exist and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or
     *         role), you might need to enforce a delay before including the new principal in a key policy because the
     *         new principal might not be immediately visible to AWS KMS. For more information, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
     *         >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access Management
     *         User Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default">Default
     *         Key Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         The key policy size limit is 32 kilobytes (32768 bytes).
     */
    public String policy() {
        return policy;
    }

    /**
     * <p>
     * A description of the CMK.
     * </p>
     * <p>
     * Use a description that helps you decide whether the CMK is appropriate for a task.
     * </p>
     * 
     * @return A description of the CMK.</p>
     *         <p>
     *         Use a description that helps you decide whether the CMK is appropriate for a task.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The cryptographic operations for which you can use the CMK. The only valid value is <code>ENCRYPT_DECRYPT</code>,
     * which means you can use the CMK to encrypt and decrypt data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyUsage} will
     * return {@link KeyUsageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyUsageAsString}.
     * </p>
     * 
     * @return The cryptographic operations for which you can use the CMK. The only valid value is
     *         <code>ENCRYPT_DECRYPT</code>, which means you can use the CMK to encrypt and decrypt data.
     * @see KeyUsageType
     */
    public KeyUsageType keyUsage() {
        return KeyUsageType.fromValue(keyUsage);
    }

    /**
     * <p>
     * The cryptographic operations for which you can use the CMK. The only valid value is <code>ENCRYPT_DECRYPT</code>,
     * which means you can use the CMK to encrypt and decrypt data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyUsage} will
     * return {@link KeyUsageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyUsageAsString}.
     * </p>
     * 
     * @return The cryptographic operations for which you can use the CMK. The only valid value is
     *         <code>ENCRYPT_DECRYPT</code>, which means you can use the CMK to encrypt and decrypt data.
     * @see KeyUsageType
     */
    public String keyUsageAsString() {
        return keyUsage;
    }

    /**
     * <p>
     * The source of the key material for the CMK. You cannot change the origin after you create the CMK.
     * </p>
     * <p>
     * The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key store.
     * </p>
     * <p>
     * When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that you can
     * import key material from your existing key management infrastructure. For more information about importing key
     * material into AWS KMS, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * and creates its key material in the associated AWS CloudHSM cluster. You must also use the
     * <code>CustomKeyStoreId</code> parameter to identify the custom key store.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The source of the key material for the CMK. You cannot change the origin after you create the CMK.</p>
     *         <p>
     *         The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key store.
     *         </p>
     *         <p>
     *         When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that you
     *         can import key material from your existing key management infrastructure. For more information about
     *         importing key material into AWS KMS, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
     *         Material</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> and creates its key material in the associated AWS CloudHSM cluster. You must also use the
     *         <code>CustomKeyStoreId</code> parameter to identify the custom key store.
     * @see OriginType
     */
    public OriginType origin() {
        return OriginType.fromValue(origin);
    }

    /**
     * <p>
     * The source of the key material for the CMK. You cannot change the origin after you create the CMK.
     * </p>
     * <p>
     * The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key store.
     * </p>
     * <p>
     * When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that you can
     * import key material from your existing key management infrastructure. For more information about importing key
     * material into AWS KMS, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * and creates its key material in the associated AWS CloudHSM cluster. You must also use the
     * <code>CustomKeyStoreId</code> parameter to identify the custom key store.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The source of the key material for the CMK. You cannot change the origin after you create the CMK.</p>
     *         <p>
     *         The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key store.
     *         </p>
     *         <p>
     *         When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that you
     *         can import key material from your existing key management infrastructure. For more information about
     *         importing key material into AWS KMS, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
     *         Material</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> and creates its key material in the associated AWS CloudHSM cluster. You must also use the
     *         <code>CustomKeyStoreId</code> parameter to identify the custom key store.
     * @see OriginType
     */
    public String originAsString() {
        return origin;
    }

    /**
     * <p>
     * Creates the CMK in the specified <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key store, you must also
     * specify the <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>. The AWS CloudHSM cluster
     * that is associated with the custom key store must have at least two active HSMs, each in a different Availability
     * Zone in the Region.
     * </p>
     * <p>
     * To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
     * </p>
     * <p>
     * The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     * 
     * @return Creates the CMK in the specified <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key
     *         store, you must also specify the <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>.
     *         The AWS CloudHSM cluster that is associated with the custom key store must have at least two active HSMs,
     *         each in a different Availability Zone in the Region.</p>
     *         <p>
     *         To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
     *         </p>
     *         <p>
     *         The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
     *         </p>
     *         <p>
     *         This operation is part of the <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key
     *         Store feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS
     *         with the isolation and control of a single-tenant key store.
     */
    public String customKeyStoreId() {
        return customKeyStoreId;
    }

    /**
     * <p>
     * A flag to indicate whether to bypass the key policy lockout safety check.
     * </p>
     * <important>
     * <p>
     * Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value to true
     * indiscriminately.
     * </p>
     * <p>
     * For more information, refer to the scenario in the <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     * >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * </important>
     * <p>
     * Use this parameter only when you include a policy in the request and you intend to prevent the principal that is
     * making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
     * </p>
     * <p>
     * The default value is false.
     * </p>
     * 
     * @return A flag to indicate whether to bypass the key policy lockout safety check.</p> <important>
     *         <p>
     *         Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value to
     *         true indiscriminately.
     *         </p>
     *         <p>
     *         For more information, refer to the scenario in the <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     *         >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     *         </p>
     *         </important>
     *         <p>
     *         Use this parameter only when you include a policy in the request and you intend to prevent the principal
     *         that is making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
     *         </p>
     *         <p>
     *         The default value is false.
     */
    public Boolean bypassPolicyLockoutSafetyCheck() {
        return bypassPolicyLockoutSafetyCheck;
    }

    /**
     * <p>
     * One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both required, but
     * tag values can be empty (null) strings.
     * </p>
     * <p>
     * Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and instead tag
     * the CMK after it is created using <a>TagResource</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both
     *         required, but tag values can be empty (null) strings.</p>
     *         <p>
     *         Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and
     *         instead tag the CMK after it is created using <a>TagResource</a>.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(keyUsageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(originAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customKeyStoreId());
        hashCode = 31 * hashCode + Objects.hashCode(bypassPolicyLockoutSafetyCheck());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateKeyRequest)) {
            return false;
        }
        CreateKeyRequest other = (CreateKeyRequest) obj;
        return Objects.equals(policy(), other.policy()) && Objects.equals(description(), other.description())
                && Objects.equals(keyUsageAsString(), other.keyUsageAsString())
                && Objects.equals(originAsString(), other.originAsString())
                && Objects.equals(customKeyStoreId(), other.customKeyStoreId())
                && Objects.equals(bypassPolicyLockoutSafetyCheck(), other.bypassPolicyLockoutSafetyCheck())
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateKeyRequest").add("Policy", policy()).add("Description", description())
                .add("KeyUsage", keyUsageAsString()).add("Origin", originAsString()).add("CustomKeyStoreId", customKeyStoreId())
                .add("BypassPolicyLockoutSafetyCheck", bypassPolicyLockoutSafetyCheck()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "KeyUsage":
            return Optional.ofNullable(clazz.cast(keyUsageAsString()));
        case "Origin":
            return Optional.ofNullable(clazz.cast(originAsString()));
        case "CustomKeyStoreId":
            return Optional.ofNullable(clazz.cast(customKeyStoreId()));
        case "BypassPolicyLockoutSafetyCheck":
            return Optional.ofNullable(clazz.cast(bypassPolicyLockoutSafetyCheck()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateKeyRequest, T> g) {
        return obj -> g.apply((CreateKeyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateKeyRequest> {
        /**
         * <p>
         * The key policy to attach to the CMK.
         * </p>
         * <p>
         * If you provide a key policy, it must meet the following criteria:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the principal
         * that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a> request on the
         * CMK. This reduces the risk that the CMK becomes unmanageable. For more information, refer to the scenario in
         * the <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         * >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each statement in the key policy must contain one or more principals. The principals in the key policy must
         * exist and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or role), you
         * might need to enforce a delay before including the new principal in a key policy because the new principal
         * might not be immediately visible to AWS KMS. For more information, see <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
         * >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access Management User
         * Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default">Default Key
         * Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * The key policy size limit is 32 kilobytes (32768 bytes).
         * </p>
         * 
         * @param policy
         *        The key policy to attach to the CMK.</p>
         *        <p>
         *        If you provide a key policy, it must meet the following criteria:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the
         *        principal that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a>
         *        request on the CMK. This reduces the risk that the CMK becomes unmanageable. For more information,
         *        refer to the scenario in the <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         *        >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each statement in the key policy must contain one or more principals. The principals in the key policy
         *        must exist and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or
         *        role), you might need to enforce a delay before including the new principal in a key policy because
         *        the new principal might not be immediately visible to AWS KMS. For more information, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
         *        >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access
         *        Management User Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default"
         *        >Default Key Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        The key policy size limit is 32 kilobytes (32768 bytes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);

        /**
         * <p>
         * A description of the CMK.
         * </p>
         * <p>
         * Use a description that helps you decide whether the CMK is appropriate for a task.
         * </p>
         * 
         * @param description
         *        A description of the CMK.</p>
         *        <p>
         *        Use a description that helps you decide whether the CMK is appropriate for a task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The cryptographic operations for which you can use the CMK. The only valid value is
         * <code>ENCRYPT_DECRYPT</code>, which means you can use the CMK to encrypt and decrypt data.
         * </p>
         * 
         * @param keyUsage
         *        The cryptographic operations for which you can use the CMK. The only valid value is
         *        <code>ENCRYPT_DECRYPT</code>, which means you can use the CMK to encrypt and decrypt data.
         * @see KeyUsageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyUsageType
         */
        Builder keyUsage(String keyUsage);

        /**
         * <p>
         * The cryptographic operations for which you can use the CMK. The only valid value is
         * <code>ENCRYPT_DECRYPT</code>, which means you can use the CMK to encrypt and decrypt data.
         * </p>
         * 
         * @param keyUsage
         *        The cryptographic operations for which you can use the CMK. The only valid value is
         *        <code>ENCRYPT_DECRYPT</code>, which means you can use the CMK to encrypt and decrypt data.
         * @see KeyUsageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyUsageType
         */
        Builder keyUsage(KeyUsageType keyUsage);

        /**
         * <p>
         * The source of the key material for the CMK. You cannot change the origin after you create the CMK.
         * </p>
         * <p>
         * The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key store.
         * </p>
         * <p>
         * When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that you can
         * import key material from your existing key management infrastructure. For more information about importing
         * key material into AWS KMS, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a>
         * in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> and creates its key material in the associated AWS CloudHSM cluster. You must also use the
         * <code>CustomKeyStoreId</code> parameter to identify the custom key store.
         * </p>
         * 
         * @param origin
         *        The source of the key material for the CMK. You cannot change the origin after you create the CMK.</p>
         *        <p>
         *        The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key
         *        store.
         *        </p>
         *        <p>
         *        When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that
         *        you can import key material from your existing key management infrastructure. For more information
         *        about importing key material into AWS KMS, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
         *        Material</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> and creates its key material in the associated AWS CloudHSM cluster. You must also use the
         *        <code>CustomKeyStoreId</code> parameter to identify the custom key store.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(String origin);

        /**
         * <p>
         * The source of the key material for the CMK. You cannot change the origin after you create the CMK.
         * </p>
         * <p>
         * The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key store.
         * </p>
         * <p>
         * When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that you can
         * import key material from your existing key management infrastructure. For more information about importing
         * key material into AWS KMS, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a>
         * in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> and creates its key material in the associated AWS CloudHSM cluster. You must also use the
         * <code>CustomKeyStoreId</code> parameter to identify the custom key store.
         * </p>
         * 
         * @param origin
         *        The source of the key material for the CMK. You cannot change the origin after you create the CMK.</p>
         *        <p>
         *        The default is <code>AWS_KMS</code>, which means AWS KMS creates the key material in its own key
         *        store.
         *        </p>
         *        <p>
         *        When the parameter value is <code>EXTERNAL</code>, AWS KMS creates a CMK without key material so that
         *        you can import key material from your existing key management infrastructure. For more information
         *        about importing key material into AWS KMS, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
         *        Material</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        When the parameter value is <code>AWS_CLOUDHSM</code>, AWS KMS creates the CMK in an AWS KMS <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> and creates its key material in the associated AWS CloudHSM cluster. You must also use the
         *        <code>CustomKeyStoreId</code> parameter to identify the custom key store.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(OriginType origin);

        /**
         * <p>
         * Creates the CMK in the specified <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key store,
         * you must also specify the <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>. The AWS
         * CloudHSM cluster that is associated with the custom key store must have at least two active HSMs, each in a
         * different Availability Zone in the Region.
         * </p>
         * <p>
         * To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
         * </p>
         * <p>
         * The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
         * </p>
         * <p>
         * This operation is part of the <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
         * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
         * isolation and control of a single-tenant key store.
         * </p>
         * 
         * @param customKeyStoreId
         *        Creates the CMK in the specified <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key
         *        store, you must also specify the <code>Origin</code> parameter with a value of
         *        <code>AWS_CLOUDHSM</code>. The AWS CloudHSM cluster that is associated with the custom key store must
         *        have at least two active HSMs, each in a different Availability Zone in the Region.</p>
         *        <p>
         *        To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
         *        </p>
         *        <p>
         *        The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
         *        </p>
         *        <p>
         *        This operation is part of the <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key
         *        Store feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS
         *        KMS with the isolation and control of a single-tenant key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeyStoreId(String customKeyStoreId);

        /**
         * <p>
         * A flag to indicate whether to bypass the key policy lockout safety check.
         * </p>
         * <important>
         * <p>
         * Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value to
         * true indiscriminately.
         * </p>
         * <p>
         * For more information, refer to the scenario in the <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         * >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * </p>
         * </important>
         * <p>
         * Use this parameter only when you include a policy in the request and you intend to prevent the principal that
         * is making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
         * </p>
         * <p>
         * The default value is false.
         * </p>
         * 
         * @param bypassPolicyLockoutSafetyCheck
         *        A flag to indicate whether to bypass the key policy lockout safety check.</p> <important>
         *        <p>
         *        Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value
         *        to true indiscriminately.
         *        </p>
         *        <p>
         *        For more information, refer to the scenario in the <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         *        >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         *        </p>
         *        </important>
         *        <p>
         *        Use this parameter only when you include a policy in the request and you intend to prevent the
         *        principal that is making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
         *        </p>
         *        <p>
         *        The default value is false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bypassPolicyLockoutSafetyCheck(Boolean bypassPolicyLockoutSafetyCheck);

        /**
         * <p>
         * One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both required,
         * but tag values can be empty (null) strings.
         * </p>
         * <p>
         * Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and instead
         * tag the CMK after it is created using <a>TagResource</a>.
         * </p>
         * 
         * @param tags
         *        One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both
         *        required, but tag values can be empty (null) strings.</p>
         *        <p>
         *        Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and
         *        instead tag the CMK after it is created using <a>TagResource</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both required,
         * but tag values can be empty (null) strings.
         * </p>
         * <p>
         * Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and instead
         * tag the CMK after it is created using <a>TagResource</a>.
         * </p>
         * 
         * @param tags
         *        One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both
         *        required, but tag values can be empty (null) strings.</p>
         *        <p>
         *        Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and
         *        instead tag the CMK after it is created using <a>TagResource</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * One or more tags. Each tag consists of a tag key and a tag value. Tag keys and tag values are both required,
         * but tag values can be empty (null) strings.
         * </p>
         * <p>
         * Use this parameter to tag the CMK when it is created. Alternately, you can omit this parameter and instead
         * tag the CMK after it is created using <a>TagResource</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private String policy;

        private String description;

        private String keyUsage;

        private String origin;

        private String customKeyStoreId;

        private Boolean bypassPolicyLockoutSafetyCheck;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateKeyRequest model) {
            super(model);
            policy(model.policy);
            description(model.description);
            keyUsage(model.keyUsage);
            origin(model.origin);
            customKeyStoreId(model.customKeyStoreId);
            bypassPolicyLockoutSafetyCheck(model.bypassPolicyLockoutSafetyCheck);
            tags(model.tags);
        }

        public final String getPolicy() {
            return policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKeyUsageAsString() {
            return keyUsage;
        }

        @Override
        public final Builder keyUsage(String keyUsage) {
            this.keyUsage = keyUsage;
            return this;
        }

        @Override
        public final Builder keyUsage(KeyUsageType keyUsage) {
            this.keyUsage(keyUsage == null ? null : keyUsage.toString());
            return this;
        }

        public final void setKeyUsage(String keyUsage) {
            this.keyUsage = keyUsage;
        }

        public final String getOriginAsString() {
            return origin;
        }

        @Override
        public final Builder origin(String origin) {
            this.origin = origin;
            return this;
        }

        @Override
        public final Builder origin(OriginType origin) {
            this.origin(origin == null ? null : origin.toString());
            return this;
        }

        public final void setOrigin(String origin) {
            this.origin = origin;
        }

        public final String getCustomKeyStoreId() {
            return customKeyStoreId;
        }

        @Override
        public final Builder customKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
            return this;
        }

        public final void setCustomKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
        }

        public final Boolean getBypassPolicyLockoutSafetyCheck() {
            return bypassPolicyLockoutSafetyCheck;
        }

        @Override
        public final Builder bypassPolicyLockoutSafetyCheck(Boolean bypassPolicyLockoutSafetyCheck) {
            this.bypassPolicyLockoutSafetyCheck = bypassPolicyLockoutSafetyCheck;
            return this;
        }

        public final void setBypassPolicyLockoutSafetyCheck(Boolean bypassPolicyLockoutSafetyCheck) {
            this.bypassPolicyLockoutSafetyCheck = bypassPolicyLockoutSafetyCheck;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateKeyRequest build() {
            return new CreateKeyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
