/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use this structure to allow cryptographic operations in the grant only when the operation request includes the
 * specified <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">encryption
 * context</a>.
 * </p>
 * <p>
 * AWS KMS applies the grant constraints only when the grant allows a cryptographic operation that accepts an encryption
 * context as input, such as the following.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>Encrypt</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>Decrypt</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GenerateDataKey</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GenerateDataKeyWithoutPlaintext</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ReEncrypt</a>
 * </p>
 * </li>
 * </ul>
 * <p>
 * AWS KMS does not apply the grant constraints to other operations, such as <a>DescribeKey</a> or
 * <a>ScheduleKeyDeletion</a>.
 * </p>
 * <important>
 * <p>
 * In a cryptographic operation, the encryption context in the decryption operation must be an exact, case-sensitive
 * match for the keys and values in the encryption context of the encryption operation. Only the order of the pairs can
 * vary.
 * </p>
 * <p>
 * However, in a grant constraint, the key in each key-value pair is not case sensitive, but the value is case
 * sensitive.
 * </p>
 * <p>
 * To avoid confusion, do not use multiple encryption context pairs that differ only by case. To require a fully
 * case-sensitive encryption context, use the <code>kms:EncryptionContext:</code> and
 * <code>kms:EncryptionContextKeys</code> conditions in an IAM or key policy. For details, see <a href=
 * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-context"
 * >kms:EncryptionContext:</a> in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
 * </p>
 * </important>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GrantConstraints implements SdkPojo, Serializable,
        ToCopyableBuilder<GrantConstraints.Builder, GrantConstraints> {
    private static final SdkField<Map<String, String>> ENCRYPTION_CONTEXT_SUBSET_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(GrantConstraints::encryptionContextSubset))
            .setter(setter(Builder::encryptionContextSubset))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionContextSubset").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ENCRYPTION_CONTEXT_EQUALS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(GrantConstraints::encryptionContextEquals))
            .setter(setter(Builder::encryptionContextEquals))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionContextEquals").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ENCRYPTION_CONTEXT_SUBSET_FIELD, ENCRYPTION_CONTEXT_EQUALS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, String> encryptionContextSubset;

    private final Map<String, String> encryptionContextEquals;

    private GrantConstraints(BuilderImpl builder) {
        this.encryptionContextSubset = builder.encryptionContextSubset;
        this.encryptionContextEquals = builder.encryptionContextEquals;
    }

    /**
     * <p>
     * A list of key-value pairs that must be included in the encryption context of the cryptographic operation request.
     * The grant allows the cryptographic operation only when the encryption context in the request includes the
     * key-value pairs specified in this constraint, although it can include additional key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of key-value pairs that must be included in the encryption context of the cryptographic operation
     *         request. The grant allows the cryptographic operation only when the encryption context in the request
     *         includes the key-value pairs specified in this constraint, although it can include additional key-value
     *         pairs.
     */
    public Map<String, String> encryptionContextSubset() {
        return encryptionContextSubset;
    }

    /**
     * <p>
     * A list of key-value pairs that must match the encryption context in the cryptographic operation request. The
     * grant allows the operation only when the encryption context in the request is the same as the encryption context
     * specified in this constraint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of key-value pairs that must match the encryption context in the cryptographic operation request.
     *         The grant allows the operation only when the encryption context in the request is the same as the
     *         encryption context specified in this constraint.
     */
    public Map<String, String> encryptionContextEquals() {
        return encryptionContextEquals;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encryptionContextSubset());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionContextEquals());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GrantConstraints)) {
            return false;
        }
        GrantConstraints other = (GrantConstraints) obj;
        return Objects.equals(encryptionContextSubset(), other.encryptionContextSubset())
                && Objects.equals(encryptionContextEquals(), other.encryptionContextEquals());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GrantConstraints").add("EncryptionContextSubset", encryptionContextSubset())
                .add("EncryptionContextEquals", encryptionContextEquals()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EncryptionContextSubset":
            return Optional.ofNullable(clazz.cast(encryptionContextSubset()));
        case "EncryptionContextEquals":
            return Optional.ofNullable(clazz.cast(encryptionContextEquals()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GrantConstraints, T> g) {
        return obj -> g.apply((GrantConstraints) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GrantConstraints> {
        /**
         * <p>
         * A list of key-value pairs that must be included in the encryption context of the cryptographic operation
         * request. The grant allows the cryptographic operation only when the encryption context in the request
         * includes the key-value pairs specified in this constraint, although it can include additional key-value
         * pairs.
         * </p>
         * 
         * @param encryptionContextSubset
         *        A list of key-value pairs that must be included in the encryption context of the cryptographic
         *        operation request. The grant allows the cryptographic operation only when the encryption context in
         *        the request includes the key-value pairs specified in this constraint, although it can include
         *        additional key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContextSubset(Map<String, String> encryptionContextSubset);

        /**
         * <p>
         * A list of key-value pairs that must match the encryption context in the cryptographic operation request. The
         * grant allows the operation only when the encryption context in the request is the same as the encryption
         * context specified in this constraint.
         * </p>
         * 
         * @param encryptionContextEquals
         *        A list of key-value pairs that must match the encryption context in the cryptographic operation
         *        request. The grant allows the operation only when the encryption context in the request is the same as
         *        the encryption context specified in this constraint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContextEquals(Map<String, String> encryptionContextEquals);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> encryptionContextSubset = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> encryptionContextEquals = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GrantConstraints model) {
            encryptionContextSubset(model.encryptionContextSubset);
            encryptionContextEquals(model.encryptionContextEquals);
        }

        public final Map<String, String> getEncryptionContextSubset() {
            return encryptionContextSubset;
        }

        @Override
        public final Builder encryptionContextSubset(Map<String, String> encryptionContextSubset) {
            this.encryptionContextSubset = EncryptionContextTypeCopier.copy(encryptionContextSubset);
            return this;
        }

        public final void setEncryptionContextSubset(Map<String, String> encryptionContextSubset) {
            this.encryptionContextSubset = EncryptionContextTypeCopier.copy(encryptionContextSubset);
        }

        public final Map<String, String> getEncryptionContextEquals() {
            return encryptionContextEquals;
        }

        @Override
        public final Builder encryptionContextEquals(Map<String, String> encryptionContextEquals) {
            this.encryptionContextEquals = EncryptionContextTypeCopier.copy(encryptionContextEquals);
            return this;
        }

        public final void setEncryptionContextEquals(Map<String, String> encryptionContextEquals) {
            this.encryptionContextEquals = EncryptionContextTypeCopier.copy(encryptionContextEquals);
        }

        @Override
        public GrantConstraints build() {
            return new GrantConstraints(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
