/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lakeformation.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure containing information about the query plan.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryPlanningContext implements SdkPojo, Serializable,
        ToCopyableBuilder<QueryPlanningContext.Builder, QueryPlanningContext> {
    private static final SdkField<String> CATALOG_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CatalogId").getter(getter(QueryPlanningContext::catalogId)).setter(setter(Builder::catalogId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CatalogId").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(QueryPlanningContext::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<Instant> QUERY_AS_OF_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("QueryAsOfTime").getter(getter(QueryPlanningContext::queryAsOfTime))
            .setter(setter(Builder::queryAsOfTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryAsOfTime").build()).build();

    private static final SdkField<Map<String, String>> QUERY_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("QueryParameters")
            .getter(getter(QueryPlanningContext::queryParameters))
            .setter(setter(Builder::queryParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TRANSACTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TransactionId").getter(getter(QueryPlanningContext::transactionId))
            .setter(setter(Builder::transactionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransactionId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CATALOG_ID_FIELD,
            DATABASE_NAME_FIELD, QUERY_AS_OF_TIME_FIELD, QUERY_PARAMETERS_FIELD, TRANSACTION_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String catalogId;

    private final String databaseName;

    private final Instant queryAsOfTime;

    private final Map<String, String> queryParameters;

    private final String transactionId;

    private QueryPlanningContext(BuilderImpl builder) {
        this.catalogId = builder.catalogId;
        this.databaseName = builder.databaseName;
        this.queryAsOfTime = builder.queryAsOfTime;
        this.queryParameters = builder.queryParameters;
        this.transactionId = builder.transactionId;
    }

    /**
     * <p>
     * The ID of the Data Catalog where the partition in question resides. If none is provided, the Amazon Web Services
     * account ID is used by default.
     * </p>
     * 
     * @return The ID of the Data Catalog where the partition in question resides. If none is provided, the Amazon Web
     *         Services account ID is used by default.
     */
    public final String catalogId() {
        return catalogId;
    }

    /**
     * <p>
     * The database containing the table.
     * </p>
     * 
     * @return The database containing the table.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * The time as of when to read the table contents. If not set, the most recent transaction commit time will be used.
     * Cannot be specified along with <code>TransactionId</code>.
     * </p>
     * 
     * @return The time as of when to read the table contents. If not set, the most recent transaction commit time will
     *         be used. Cannot be specified along with <code>TransactionId</code>.
     */
    public final Instant queryAsOfTime() {
        return queryAsOfTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the QueryParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasQueryParameters() {
        return queryParameters != null && !(queryParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map consisting of key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQueryParameters} method.
     * </p>
     * 
     * @return A map consisting of key-value pairs.
     */
    public final Map<String, String> queryParameters() {
        return queryParameters;
    }

    /**
     * <p>
     * The transaction ID at which to read the table contents. If this transaction is not committed, the read will be
     * treated as part of that transaction and will see its writes. If this transaction has aborted, an error will be
     * returned. If not set, defaults to the most recent committed transaction. Cannot be specified along with
     * <code>QueryAsOfTime</code>.
     * </p>
     * 
     * @return The transaction ID at which to read the table contents. If this transaction is not committed, the read
     *         will be treated as part of that transaction and will see its writes. If this transaction has aborted, an
     *         error will be returned. If not set, defaults to the most recent committed transaction. Cannot be
     *         specified along with <code>QueryAsOfTime</code>.
     */
    public final String transactionId() {
        return transactionId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(catalogId());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(queryAsOfTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasQueryParameters() ? queryParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transactionId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryPlanningContext)) {
            return false;
        }
        QueryPlanningContext other = (QueryPlanningContext) obj;
        return Objects.equals(catalogId(), other.catalogId()) && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(queryAsOfTime(), other.queryAsOfTime()) && hasQueryParameters() == other.hasQueryParameters()
                && Objects.equals(queryParameters(), other.queryParameters())
                && Objects.equals(transactionId(), other.transactionId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QueryPlanningContext").add("CatalogId", catalogId()).add("DatabaseName", databaseName())
                .add("QueryAsOfTime", queryAsOfTime()).add("QueryParameters", hasQueryParameters() ? queryParameters() : null)
                .add("TransactionId", transactionId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CatalogId":
            return Optional.ofNullable(clazz.cast(catalogId()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "QueryAsOfTime":
            return Optional.ofNullable(clazz.cast(queryAsOfTime()));
        case "QueryParameters":
            return Optional.ofNullable(clazz.cast(queryParameters()));
        case "TransactionId":
            return Optional.ofNullable(clazz.cast(transactionId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QueryPlanningContext, T> g) {
        return obj -> g.apply((QueryPlanningContext) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryPlanningContext> {
        /**
         * <p>
         * The ID of the Data Catalog where the partition in question resides. If none is provided, the Amazon Web
         * Services account ID is used by default.
         * </p>
         * 
         * @param catalogId
         *        The ID of the Data Catalog where the partition in question resides. If none is provided, the Amazon
         *        Web Services account ID is used by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalogId(String catalogId);

        /**
         * <p>
         * The database containing the table.
         * </p>
         * 
         * @param databaseName
         *        The database containing the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * The time as of when to read the table contents. If not set, the most recent transaction commit time will be
         * used. Cannot be specified along with <code>TransactionId</code>.
         * </p>
         * 
         * @param queryAsOfTime
         *        The time as of when to read the table contents. If not set, the most recent transaction commit time
         *        will be used. Cannot be specified along with <code>TransactionId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryAsOfTime(Instant queryAsOfTime);

        /**
         * <p>
         * A map consisting of key-value pairs.
         * </p>
         * 
         * @param queryParameters
         *        A map consisting of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryParameters(Map<String, String> queryParameters);

        /**
         * <p>
         * The transaction ID at which to read the table contents. If this transaction is not committed, the read will
         * be treated as part of that transaction and will see its writes. If this transaction has aborted, an error
         * will be returned. If not set, defaults to the most recent committed transaction. Cannot be specified along
         * with <code>QueryAsOfTime</code>.
         * </p>
         * 
         * @param transactionId
         *        The transaction ID at which to read the table contents. If this transaction is not committed, the read
         *        will be treated as part of that transaction and will see its writes. If this transaction has aborted,
         *        an error will be returned. If not set, defaults to the most recent committed transaction. Cannot be
         *        specified along with <code>QueryAsOfTime</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionId(String transactionId);
    }

    static final class BuilderImpl implements Builder {
        private String catalogId;

        private String databaseName;

        private Instant queryAsOfTime;

        private Map<String, String> queryParameters = DefaultSdkAutoConstructMap.getInstance();

        private String transactionId;

        private BuilderImpl() {
        }

        private BuilderImpl(QueryPlanningContext model) {
            catalogId(model.catalogId);
            databaseName(model.databaseName);
            queryAsOfTime(model.queryAsOfTime);
            queryParameters(model.queryParameters);
            transactionId(model.transactionId);
        }

        public final String getCatalogId() {
            return catalogId;
        }

        public final void setCatalogId(String catalogId) {
            this.catalogId = catalogId;
        }

        @Override
        @Transient
        public final Builder catalogId(String catalogId) {
            this.catalogId = catalogId;
            return this;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        @Override
        @Transient
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final Instant getQueryAsOfTime() {
            return queryAsOfTime;
        }

        public final void setQueryAsOfTime(Instant queryAsOfTime) {
            this.queryAsOfTime = queryAsOfTime;
        }

        @Override
        @Transient
        public final Builder queryAsOfTime(Instant queryAsOfTime) {
            this.queryAsOfTime = queryAsOfTime;
            return this;
        }

        public final Map<String, String> getQueryParameters() {
            if (queryParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return queryParameters;
        }

        public final void setQueryParameters(Map<String, String> queryParameters) {
            this.queryParameters = QueryParameterMapCopier.copy(queryParameters);
        }

        @Override
        @Transient
        public final Builder queryParameters(Map<String, String> queryParameters) {
            this.queryParameters = QueryParameterMapCopier.copy(queryParameters);
            return this;
        }

        public final String getTransactionId() {
            return transactionId;
        }

        public final void setTransactionId(String transactionId) {
            this.transactionId = transactionId;
        }

        @Override
        @Transient
        public final Builder transactionId(String transactionId) {
            this.transactionId = transactionId;
            return this;
        }

        @Override
        public QueryPlanningContext build() {
            return new QueryPlanningContext(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
